;===================================================================================================
	TITLE	Wave JT
;	
;	LED chaser/sequencor with voltage booster circuit to oparate on one AA battery.
;
;	Copyright 2012 Akimitsu Sadoi - The LED Artist - http://theLEDart.com
;	The use and distribution of this code without explicit permission is prohibited.
;---------------------------------------------------------------------------------------------------
;	Version history
;	1.0	published
;	1.1 demo mode selection
;===================================================================================================
   PROCESSOR 16F1824						; can be either 1823 or 1824
#INCLUDE <P16F1824.INC>
	RADIX	dec
	LIST	b=4
;--- Device Configuration Bits -------------------------------------------------
	__CONFIG _CONFIG1, _FOSC_INTOSC & _WDTE_OFF & _PWRTE_ON & _BOREN_ON & _CLKOUTEN_OFF & _MCLRE_ON & _CPD_OFF
	__CONFIG _CONFIG2, _WRT_OFF & _PLLEN_ON & _STVREN_ON & _BORV_LO & _LVP_ON

;--- Source Configuration Bits -------------------------------------------------
SYNC_ENABLE		equ		0					; enable sync funtions

;===================================================================================================
;	constants
;
ccpcon_pram		equ		b'00001100'			; single PWM, LSBs of duty cycle(bit 5:4), PWM active-high
ccpcon_pram2	equ		b'00001110'			; single PWM, LSBs of duty cycle(bit 5:4), PWM active-low
prescale		equ		1					; 1: prescale 1, 2: 4, 3: 16
postscale		equ		1					; between 1 to 16
t2com_param		equ		((postscale-1)<<3)|4|(prescale-1)
dead_band		equ		23					; time between PWM interrupt and port set
max_duty		equ		0xFF				; duty cycle value for 100% duty (8 bit)
pwm_period		equ		0xFF				; PWM period value (PR2)


auto_adv_time	equ		1					; auto-advance mode interval in minutes (max: 2290)
; ticks = auto_adv_time*60*10^9/(125*(pwm_period+1)*256)
timer_ticks		equ		(auto_adv_time*7324)
sleep_timer0val	equ		(timer_ticks & 0xFF)+1
sleep_timer1val	equ		(timer_ticks / 0x100 & 0xFF)+1
sleep_timer2val	equ		(timer_ticks / 0x10000)+1

;--- FVR (fixed voltage reference) parameter ---
fvr_param		equ		1<<FVREN | 1<<ADFVR1	; 2.048V

;--- ADC parameters ---
AUX_PORT		equ		PORTA
AUX_PIN			equ		4
adc_param0		equ		b'01111101'				; FVR as input, ADC enable
adc_param0A		equ		b'00001101'				; AN3 (AUX) as input, ADC enable
adc_param1		equ		b'00100000'				; left justified, Fosc/32 clock, Vdd as reference
AUX_SENSE		equ		1						; AUX input sensitivity 0:high, 1:low
.if AUX_SENSE
adc_param1A		equ		b'00100011'				; left justified, Fosc/32 clock, FVR as reference
.else
adc_param1A		equ		b'10100011'				; right justified, Fosc/32 clock, FVR as reference
.endif
SPV_THRES		equ		110						; supply voltage thresthold = 512/target_voltage
												;  or target_voltage = 512/SPV_THRES

;--- Others ---
PON_DELAY		equ		100						; delay time before entering roulette mode (x8.2ms)
RLT_SPEED		equ		27						; power on roulette scan speed (higher -> slower)
SPEED_INT		equ		b'00011111'				; initial master speed

;--- EEPROM addresses ---
EE_LAST_MODE	equ		0x00
EE_LAST_SPEED	equ		0x01

;--- push button service -------------------------
SW_PIN			equ		5				; pin # of button switch
SW_PORT			equ		PORTA			; port of button switch

DEBOUNCE_CNT	equ		5				; debounce time = DEBOUNCE_CNT x 8.2ms
EVT_TIMEOUT		equ		80				; event (double/triple click) timeout x 8.2ms
LONG_PUSH		equ		120				; long push threshold x 8.2ms

;-------------------------------------------------
;--- LED drive parameters ------------------------
PWM_PIN			equ		5
PWM_PORT		equ		PORTC

;--- 8 LED rev.2 PCB ---

NUM_OF_COL		equ		8
NUM_OF_ROW		equ		1
NUM_OF_LED		equ		NUM_OF_COL*NUM_OF_ROW

LED_1_PORT		equ		PORTA
LED_2_PORT		equ		PORTA
LED_3_PORT		equ		PORTA
LED_4_PORT		equ		PORTC
LED_5_PORT		equ		PORTC
LED_6_PORT		equ		PORTC
LED_7_PORT		equ		PORTC
LED_8_PORT		equ		PORTC

LED_1_PIN		equ		0
LED_2_PIN		equ		1
LED_3_PIN		equ		2
LED_4_PIN		equ		0
LED_5_PIN		equ		2
LED_6_PIN		equ		1
LED_7_PIN		equ		3
LED_8_PIN		equ		4


;-------------------------------------------------
;--- port states ---
_LED_ON_A		equ		((LED_1_PORT==PORTA)*1<<LED_1_PIN | (LED_2_PORT==PORTA)*1<<LED_2_PIN | (LED_3_PORT==PORTA)*1<<LED_3_PIN | (LED_4_PORT==PORTA)*1<<LED_4_PIN)
LED_ON_A		equ		_LED_ON_A | ((LED_5_PORT==PORTA)*1<<LED_5_PIN | (LED_6_PORT==PORTA)*1<<LED_6_PIN | (LED_7_PORT==PORTA)*1<<LED_7_PIN | (LED_8_PORT==PORTA)*1<<LED_8_PIN)

_LED_ON_C		equ		((LED_1_PORT==PORTC)*1<<LED_1_PIN | (LED_2_PORT==PORTC)*1<<LED_2_PIN | (LED_3_PORT==PORTC)*1<<LED_3_PIN | (LED_4_PORT==PORTC)*1<<LED_4_PIN)
LED_ON_C		equ		_LED_ON_C | ((LED_5_PORT==PORTC)*1<<LED_5_PIN | (LED_6_PORT==PORTC)*1<<LED_6_PIN | (LED_7_PORT==PORTC)*1<<LED_7_PIN | (LED_8_PORT==PORTC)*1<<LED_8_PIN)

;--- global flags --------------------------------
PWM_REF			equ		0				; PWM service refresh occurred
PWM_REF2		equ		1				; PWM duty phase complete
SP_HIGH			equ		2				; supply voltage too high - booster is off
PWM_BLANK		equ		3				; PwM blacking period for polarity change
SYNC_ON			equ		4				; animation sync-mode on
DEMO_ON			equ		5				; demo mode on

;--- push button service flags -------------------
BTN_DWN			equ		0				; button is down (not debounced)
BTN_HLD			equ		1				; button has been held
BTN_PSD			equ		2				; button has been pushed (short or long)
BTN_LNG			equ		3				; long push
BTN_DBL			equ		4				; double click/tap
BTN_TRP			equ		5				; triple click/tap

;--- duty_dir flags ------------------------------
DUTY_DIR		equ		0				; PWM duty sweep up/down direction (0:up, 1:down)
DUTY_HOLD		equ		1				; PWM duty sweep hold (1:hold)
AF_FWDL			equ		2				; change animation direction to fwd at min duty
AF_FWDH			equ		3				; change animation direction to fwd at max duty
AF_REVL			equ		4				; change animation direction to rev at min duty
AF_REVH			equ		5				; change animation direction to rev at max duty

;--- anim_flags ----------------------------------
AF_DIR			equ		0				; animation direction (0:forward, 1:reverse)
AF_RELHD		equ		1				; release duty_hold for the next (internal use)
AF_MIRROR		equ		2				; mirror mode
AF_SPLIT		equ		3				; split mode
AF_DHLDL		equ		4				; reset duty_hold at zero duty
AF_DHLDH		equ		5				; reset duty_hold at max duty
NO_UPDATE		equ		6				; do not copy the duty buffer
AF_ANALOG		equ		7				; analog input meter mode

;===================================================================================================
;	variables
;
				cblock	0x20			; BANK0
	mode_num							; mode (movment) number
	sp_voltage
	aux_voltage
	aux_voltage2
	sync_timeout_l
	sync_timeout_h
	master_update_rate

	anim_flags							; animation options/flags
	anim_process						; 
	update_rate							; (higher - slower animation)
	update								; update counter
	an_update_rate						; (higher - slower animation)
	an_update							; update counter
	master_update						; master update counter
	duty_diff							; duty difference between LEDs
	step_up								; duty sweep step (higher - faster animation)
	step_down							; duty sweep step (higher - faster animation)
										; --- duty level of LEDs ---
	duty_1
	duty_2
	duty_3
	duty_4
	duty_5
	duty_6
	duty_7
	duty_8
	duty_9
	duty_10
	duty_11
	duty_12
	duty_13
	duty_14
	duty_15
	duty_16
										; --- duty flags (dir, hold) ---
	duty_flag_1
	duty_flag_2
	duty_flag_3
	duty_flag_4
	duty_flag_5
	duty_flag_6
	duty_flag_7
	duty_flag_8
	duty_flag_9
	duty_flag_10
	duty_flag_11
	duty_flag_12
	duty_flag_13
	duty_flag_14
	duty_flag_15
	duty_flag_16
				endc

				cblock	0x70			; COMMOM MEMORY
	gflags								; global flags
	bflags								; button service flags
	adc_state							; A/D converter state
										;  0:read supply voltage & start conversion again
										;  1:start conversion of supply voltage
										;  2:read AUX value & start sampling supply voltage
										;  3:start conversion of AUX in
										;  4:read supply voltage & start sampling AUX in
	debounce							; debounce counter
	click_cnt							; click/tap counter
	event_tmr							; click/tap event timer
	;--- PWM service ---
	duty								; duty level sweep
	port_buff_a							; PORTs buffer
	port_buff_c

	sleep_timer0
	sleep_timer1
	sleep_timer2

	loop_counter
	loop_counter2
	random_val_l
	random_val_h
				endc

				cblock	0xA0			; BANK1
	bit_count
	_duty_								; 7 bit duty level for PWM service (1 - 255, odd numbers only)
	_duty_1								; duty level of LED_1 - 8 (shadow copies)
	_duty_2
	_duty_3
	_duty_4
	_duty_5
	_duty_6
	_duty_7
	_duty_8
	_duty_9
	_duty_10
	_duty_11
	_duty_12
	_duty_13
	_duty_14
	_duty_15
	_duty_16
				endc
;
;===================================================================================================
;
				org		0						; reset vector
				nop
				movlw   HIGH main
				movwf   PCLATH
				goto	main
;
;===================================================================================================
;
				org		4						; INT vector
;				------------------------------------------------------
;				STATUS and other registers automatically saved by hardware
;				------------------------------------------------------

.if (NUM_OF_ROW==2)
				btfss	gflags,PWM_BLANK		; if (PWM_BLANK == 1)
				goto	set_tris				; {

				bcf		gflags,PWM_BLANK		;   clear PWM_BLANK flag

				;--- blanking period - switch PWM polarity -----------
				;--- set all LEDs off/high-impedance ---
				banksel	TRISA
				movlw	LED_ON_A
				iorwf	TRISA,f
				movlw	LED_ON_C
				iorwf	TRISC,f

				;--- change port polarity ---
				btfsc	duty,7					;   if (duty < 0x80)
				goto	drive_led_even			;   {
drive_led_odd
				;--- drive odd LEDs ---
				banksel	LATA
				movlw	LED_ON_A
				iorwf	LATA,f
				movlw	LED_ON_C
				iorwf	LATC,f
				goto	next_duty_level
drive_led_even									;   } else {
				;--- drive even LEDs ---
				banksel	LATA
				movlw	~LED_ON_A
				andwf	LATA,f
				movlw	~LED_ON_C
				andwf	LATC,f
				goto	next_duty_level			;   }
.endif
set_tris										; } else {
				;--- set TRIS ----------------------------------------
				;--- modify only the LED pins ---
				banksel	TRISA
				movfw	port_buff_a				;   load next TRISA data into W
				andlw	LED_ON_A				;   set high bits
				iorwf	TRISA,f
				movfw	port_buff_a				;   load next TRISA data into W again
				iorlw	~LED_ON_A				;   clear low bits
				andwf	TRISA,f

				movfw	port_buff_c				;   load next TRISC data into W
				andlw	LED_ON_C				;   set high bits
				iorwf	TRISC,f
				movfw	port_buff_c				;   load next TRISC data into W again
				iorlw	~LED_ON_C				;   clear low bits
				andwf	TRISC,f

				;--- increment duty level ----------------------------
.if (NUM_OF_ROW==2)
				incfsz	duty,f					;   duty++
				goto	check_80				;   if (duty == 0)
				bsf		gflags,PWM_BLANK		;     set PWM_BLANK flag
				goto	finish_int				;     exit PWM service
check_80
				movlw	0x80
				subwf	duty,w
				btfss	STATUS,Z				;   if (duty == 0x80)
				goto	next_duty_level
				bsf		gflags,PWM_BLANK		;     set PWM_BLANK flag
				goto	finish_int				;     exit PWM service
.else
				incf	duty,f					;   duty++
.endif
next_duty_level									; }
				;--- set the next duty cycle for PWM -----------------
				;--- set upper 8 bits of duty cycle ---
				call	get_ccpr1_value
				banksel	CCPR1L
				movwf	CCPR1L

				;--- set lower 2 bits of duty cycle ---
				call	get_ccp1con_value
				banksel	CCP1CON
				movwf	CCP1CON

				;--- prepare PORT data for the next cycle ------------
				movfw	duty
				btfss	STATUS,Z				; if (duty = 0)
				goto	duty_not_zero			; {
				bsf		gflags,PWM_REF2			;   PWM_REF2 flag set every 256th times

				call	check_button			;   check button SW

				;--- copy LED duty buffers ---
				banksel	anim_flags
				btfsc	anim_flags,NO_UPDATE	;   skip copy if No Update flag set
				goto	no_update
				clrf	FSR0H
				clrf	FSR1H
				movlw	duty_1					;   FSR0 -> source
				movwf	FSR0L
				movlw	_duty_1					;   FSR1 -> distination
				movwf	FSR1L

				movlw	NUM_OF_LED
				btfsc	anim_flags,AF_MIRROR	; if mirror mode on
				movlw	NUM_OF_LED/2			;   loop_counter = NUM_OF_LED/2
				btfsc	anim_flags,AF_SPLIT		; if split mode on
				movlw	NUM_OF_LED/2			;   loop_counter = NUM_OF_LED/2
				movwf	loop_counter			; else loop_counter = NUM_OF_LED
copy_duty_buff
				moviw	FSR0++					;   copy data
				movwi	FSR1++
				decfsz	loop_counter,f
				goto	copy_duty_buff

				btfsc	anim_flags,AF_MIRROR	; if mirror mode on
				goto	copy_duty_mirror
				btfsc	anim_flags,AF_SPLIT		; if split mode on
				goto	copy_duty_split
				goto	no_update

copy_duty_mirror
				movlw	NUM_OF_LED/2
				movwf	loop_counter
copy_duty_buff2
				moviw	--FSR0					;   copy data backwards
				movwi	FSR1++					;   into the rest of buffers
				decfsz	loop_counter,f
				goto	copy_duty_buff2

copy_duty_split
				movlw	NUM_OF_LED/2
				movwf	loop_counter
				movlw	duty_1					;   FSR0 -> source
				movwf	FSR0L
copy_duty_split2
				moviw	FSR0++					;   copy data from the top
				movwi	FSR1++					;   into the rest of buffers
				decfsz	loop_counter,f
				goto	copy_duty_split2

no_update
				;--- prepare port buffers ---
				clrf	port_buff_a
				clrf	port_buff_c

				goto	check_duty

duty_not_zero									; } else
.if (NUM_OF_ROW==2)
				sublw	0x80					; if (duty = 0x80)
				btfss	STATUS,Z
				goto	check_duty				; {

				;--- prepare port buffers ---
				clrf	port_buff_a
				clrf	port_buff_c
.endif
check_duty										; }
				;--- check duty levels and set port data -------------
.if (NUM_OF_ROW==2)
				banksel	_duty_					; internal duty level: 1,3,5...255
				bsf		STATUS,C				; set Carry flag
				rlf		duty,w					; W = (duty << 1 + 1) % 256
				movwf	_duty_					; _duty_ = W
.else
				banksel	_duty_					; internal duty level: 0,1,2...255
				movfw	duty
				movwf	_duty_
.endif
				movlw	HIGH(led_port_lookup)	; setup FSR1 for LED port map
				movwf	FSR1H
				bsf		FSR1H,7					; set the 7th bit for program memory access
				movlw	LOW(led_port_lookup)
				movwf	FSR1L

				movlw	_duty_1					; FSR0 -> duty buffer
				movwf	FSR0L

.if (NUM_OF_ROW==2)
				btfsc	duty,7					; if (duty > 0x80)
				addfsr	FSR0,1					;   do even number LEDs - start at LED 2
.endif
				movlw	NUM_OF_COL				; initialize bit_count = 8
				movwf	bit_count
check_duty_loop									; do {
				movf	_duty_,w
				subwf	INDF0,w
.if (NUM_OF_ROW==1)
				addfsr	FSR0,1					;   point to the next duty data
.else
				addfsr	FSR0,2					;   point to the next duty data - skipping one
.endif
				btfsc	STATUS,C				;   if (duty > _duty_x)
				goto	skip_turnoff			;   {
												;     then turn LED off
				moviw	0[FSR1]					;     load LED port map -> W
				iorwf	port_buff_a,f			;     set the port bit high
				moviw	1[FSR1]					;     load LED port map -> W
				iorwf	port_buff_c,f			;     set the port bit high
skip_turnoff									;   }
				ADDFSR	FSR1,2					;   FSR1 += 2 to point to the next port data
				decfsz	bit_count,f				; } while (--bit_count != 0)
				goto	check_duty_loop
check_duty_end

;				----------------------------
finish_int
				call	ad_convert				; take care of ADC service

				bsf		gflags,PWM_REF			; set PWM refreshed flag
				banksel	PIR1
				bcf		PIR1,TMR2IF				; clear timer2 overflow flag
				retfie
;
;===================================================================================================
;	main
;
main
				;--- initialization --------------------------------------------

				;--- initialize system clock -------------------------
				movlw	b'11110000'				; 32MHz internal oscillator
				banksel	OSCCON
				movwf	OSCCON

				;--- clear RAM ---------------------------------------
				movlw	0x20					; starting RAM addr
				movwf	FSR0L
next_ram		clrf	INDF0					; indirect addressing
				incf	FSR0L,f
				btfss	FSR0L,7					; done if the bit7 on (0x80)
				goto	next_ram

				movlw	0xA0					; clear bank 1
				movwf	FSR0L
next_ram1		clrf	INDF0					; indirect addressing
				incfsz	FSR0L,f					; done if FSR0L == 0
				goto	next_ram1

				;--- initialize variables ----------------------------
				movlw	EE_LAST_SPEED			; read last value from EEPROM
				call	read_eeprom
				btfsc	STATUS,Z				; if (W == 0) {
				movlw	SPEED_INT				;   reset/center master speed
				banksel	master_update_rate		; }
				movwf	master_update_rate

				;--- initialize port (tris) buffers ---
				banksel	port_buff_a
				comf	port_buff_a,f			; = 0xFF
				comf	port_buff_c,f

				;--- initialize PWM ----------------------------------
				banksel	PR2						; set PWM period
				movlw	pwm_period
				movwf	PR2
				banksel	CCP1CON
				movlw	ccpcon_pram				; LSBs of duty cycle(bit 5:4), PWM mode set
				movwf	CCP1CON
				clrf	CCPR1L					; clear duty cycles
				movlw	t2com_param				; set postscale, timer on, prescale
				banksel	T2CON
				movwf	T2CON
				banksel	PIR1
				bcf		PIR1,TMR2IF				; clear timer2 overflow flag

				clrwdt							; clear watchdog timer
				btfss	PIR1,TMR2IF				; wait for timer2 overflow
				goto	$-1
				bcf		PIR1,TMR2IF				; clear timer2 overflow flag

				;--- initialize ports --------------------------------
				banksel	LATA
				movlw	LED_ON_A				; initialize LED drive pins
				iorwf	LATA,f
				movlw	LED_ON_C
				iorwf	LATC,f
												; initialize power control/SW port
				bsf		SW_PORT,SW_PIN			; set PWR/SW port HIGH

				;--- set port input (digital/analog) mode ---
				banksel	ANSELA
				bcf		ANSELA,SW_PIN			; set PWR/SW port digital mode
;				bcf		ANSELA,AUX_PIN			; set AUX_PIN port digital mode

				;--- set port input level (TTL or ST)(1824 only) ---
				banksel	INLVLA
				bsf		INLVLA,SW_PIN			; set PWR/SW port Schmitt Trigger mode
;				bsf		INLVLA,AUX_PIN			; set AUX_PIN port Schmitt Trigger mode

				;--- initialize TRIS ---
				banksel	TRISA
				bcf		PWM_PORT,PWM_PIN		; set PWM port output
				bcf		SW_PORT,SW_PIN			; set PWR/SW port output
												; the rest of ports are inputs by default
.if SYNC_ENABLE
				;--- set up pull-up, etc. for AUX sync ---
				banksel	WPUA
				clrf	WPUA					; disable all pull-ups first
				clrf	WPUC
				bsf		AUX_PORT,AUX_PIN		; then turn on the ones you want
				banksel	OPTION_REG
				bcf		OPTION_REG,NOT_WPUEN	; then enable the master pull-up
				banksel	LATA
				bcf		AUX_PORT,AUX_PIN		; set AUX_PIN latch low
				banksel	IOCAP
				bsf		IOCAP,AUX_PIN			; both positive and negative edges trigger IOC
				banksel	IOCAN
				bsf		IOCAN,AUX_PIN
				banksel	IOCAF
				clrf	IOCAF					; clear the IOC flags
.endif
				;--- set up A/D converter ----------------------------
				banksel	FVRCON
				movlw	fvr_param				; set up voltage ref (FVR) module
				movwf	FVRCON

				banksel	ADCON0					; set up A/D converter
				movlw	adc_param0
				movwf	ADCON0
				movlw	adc_param1
				movwf	ADCON1

				banksel	ADCON0
				bsf		ADCON0,GO				; start AD conversion

				;--- retreave saved parametors -----------------------
				movlw	EE_LAST_MODE			; read last mode from EEPROM
				call	read_eeprom
				banksel	mode_num
				movwf	mode_num

				;--- start PWM LED driver service --------------------
				banksel	PIE1
				bsf		PIE1,TMR2IE				; enable timer2 interrupt
				banksel	INTCON
				bsf		INTCON,PEIE				; enable peripheral interrupt
				bsf		INTCON,GIE				; enable interrupt

				;-----------------------------------------------------
				;--- power-on "roulette" mode selection --------------
				btfss	gflags,PWM_REF2			; wait for the PWM phase start
				goto	$-1
				bcf		gflags,PWM_REF2			; clear the flag
posw_check
				btfss	bflags,BTN_DWN			; if button is down
				goto	posw_released

				btfss	bflags,BTN_LNG			; if button is long held
				goto	posw_check
												;   - SW held long enough -
				banksel	mode_num
				clrf	mode_num				; start scan from 0 - clear mode_num
				call	play_roulette			; display mode_num 0 extra time to start
posw_loop
				call	play_roulette			; play roulette - display mode_num
				btfsc	bflags,BTN_DWN			; check the switch status again
				goto	posw_continue
												; SW up
				call	play_roulette			;   display mode_num
				goto	posw_released			;   end the roulette
posw_continue									; SW still down
				banksel	mode_num
				incf	mode_num,f				;   increment mode_num
				movlw	NUM_OF_LED-1			;   take the bottom 3 or 4 bits only (0-7 or 0-15)
				andwf	mode_num,f
				btfss	STATUS,Z				;   if (mode_num == 0)
				goto	posw_loop				;   {
				movlw	1<<DEMO_ON				;     toggle demo mode flag
				xorwf	gflags,f				;   }
				goto	posw_loop				;   continue the roulette
posw_released
				clrf	click_cnt
				clrf	bflags					; clear button flags

				goto	start_up

next_mode		;--- go to next mode ---------------------------------
				banksel	mode_num
				incf	mode_num				; increase mode number

				;--- startup -----------------------------------------
start_up
				;--- send reset signal ----
				bcf		gflags,PWM_REF2			; clear the flag
				btfss	gflags,PWM_REF2			; wait for the PWM phase start
				goto	$-1
.if	SYNC_ENABLE
				banksel	TRISA					; hold AUX_PIN low for 24ms or so
				bcf		AUX_PORT,AUX_PIN
				bcf		gflags,PWM_REF2			; clear the flag
				btfss	gflags,PWM_REF2			; wait for the PWM phase start
				goto	$-1
				bcf		gflags,PWM_REF2			; clear the flag
				btfss	gflags,PWM_REF2			; wait for the PWM phase start
				goto	$-1
				bcf		gflags,PWM_REF2			; clear the flag
				btfss	gflags,PWM_REF2			; wait for the PWM phase start
				goto	$-1
				bsf		AUX_PORT,AUX_PIN		; release AUX_PIN
.endif
				;--- save mode number --------------------------------
				banksel	mode_num
				movfw	mode_num
				banksel	EEDATL
				movwf	EEDATL					; set data to write to
				movlw	EE_LAST_MODE			; set EEPROM address to write to
				banksel	EEADRL
				movwf	EEADRL
				call	write_eeprom
start_up2
				;--- initialize anim params ---
				movlw	anim_flags				; starting memory address
				movwf	FSR0L					; use indirect addressing
next_ram2		clrf	INDF0					; clear memory
				incf	FSR0L,f
				movlw	duty_flag_1+NUM_OF_LED	; done if (FSR > last memory address)
				subwf	FSR0L,w
				btfss	STATUS,C
				goto	next_ram2

				banksel	0
.if auto_adv_time!=0
				;--- initialize sleep timer counters ---
				movlw	sleep_timer0val
				movwf	sleep_timer0
				movlw	sleep_timer1val
				movwf	sleep_timer1
				movlw	sleep_timer2val
				movwf	sleep_timer2
.endif
				;--- set animation parameters ------------------------
				;--- keep mode_num within range ---
				movlw	(mode_tbl_end - mode_tbl)
				subwf	mode_num,w				; if mode_num > num of modes
				btfsc	STATUS,C
				clrf	mode_num				;   reset the mode number
				movfw	mode_num				; table jump
				brw
mode_tbl
				goto	m_analog
				goto	m_larson1
				goto	m_larson2
				goto	m_larson3
				goto	m_larson4
				goto	mode_01a
				goto	m_split1
				goto	m_mirror1
				goto	m_mirror2
				goto	m_mirror2R
				goto	m_mirror3
				goto	m_mirror3R
				goto	mode_01
				goto	mode_01r
;				goto	mode_02
;				goto	mode_02r
;				goto	mode_02a
				goto	m_random
				goto	m_fade_in_out
				goto	m_flashing
				goto	m_stedy_on
mode_tbl_end
				;---------------------------------
				; Larson Scanner 1
m_larson1
				bsf		duty_flag_1,AF_FWDH		; alternate direction at max duty
				bsf		duty_flag_1+NUM_OF_LED-1,AF_REVH	; alternate direction at max duty
				bsf		anim_flags,AF_DHLDL		; reset duty hold at zero

				movlw	49*NUM_OF_ROW			; set update_rate
				movwf	update_rate

				movlw	1						; set step_up
				movwf	step_up
				movlw	1						; set step_down
				movwf	step_down

				movlw	127
				movwf	duty_diff

				call	set_duty_hold
				bcf		duty_flag_1,DUTY_HOLD

				bsf		anim_flags,AF_DHLDL		; reset duty hold at zero

				goto	start_anim

				;---------------------------------
				; Larson Scanner 2 - wider band
m_larson2
				bsf		duty_flag_1,AF_FWDH		; alternate direction at max duty
				bsf		duty_flag_1+NUM_OF_LED-1,AF_REVH	; alternate direction at max duty
				bsf		anim_flags,AF_DHLDL		; reset duty hold at zero

				movlw	33*NUM_OF_ROW			; set update_rate
				movwf	update_rate

				movlw	1						; set step_up
				movwf	step_up
				movlw	1						; set step_down
				movwf	step_down

				movlw	72
				movwf	duty_diff

				call	set_duty_hold
				bcf		duty_flag_1,DUTY_HOLD

				bsf		anim_flags,AF_DHLDL		; reset duty hold at zero

				goto	start_anim

				;---------------------------------
				; Larson Scanner 3 - dot
m_larson3
				bsf		duty_flag_1,AF_FWDH		; alternate direction at max duty
				bsf		duty_flag_1+NUM_OF_LED-1,AF_REVH	; alternate direction at max duty
				bsf		anim_flags,AF_DHLDL		; reset duty hold at zero

				movlw	127*NUM_OF_ROW+1		; set update_rate
				movwf	update_rate

				movlw	1						; set step_up
				movwf	step_up
				movlw	1						; set step_down
				movwf	step_down

				movlw	245
				movwf	duty_diff

				call	set_duty_hold
				bcf		duty_flag_1,DUTY_HOLD

				bsf		anim_flags,AF_DHLDL		; reset duty hold at zero

				goto	start_anim

				;---------------------------------
				; Larson Scanner 4 - trailing star
m_larson4
				bsf		duty_flag_1,AF_FWDL		; alternate direction at min duty
				bsf		duty_flag_1+NUM_OF_LED-1,AF_REVL	; alternate direction at min duty

				movlw	13*NUM_OF_ROW			; set update_rate
				movwf	update_rate

				movlw	15						; set step_up
				movwf	step_up
				movlw	1						; set step_down
				movwf	step_down

				movlw	255						; 15*n
				movwf	duty_diff

				call	set_duty_hold
				bcf		duty_flag_1,DUTY_HOLD

				bsf		anim_flags,AF_DHLDL		; reset duty hold at zero

				goto	start_anim

				;---------------------------------
				; Split Scanner 1
m_split1
				bsf		duty_flag_1,AF_FWDH		; alternate direction at max duty
				bsf		duty_flag_1+NUM_OF_LED/2-1,AF_REVH		; alternate direction at max duty
				bsf		anim_flags,AF_DHLDL		; reset duty hold at zero
				bsf		anim_flags,AF_SPLIT		; split mode on

				movlw	37*NUM_OF_ROW			; set update_rate
				movwf	update_rate

				movlw	1						; set step_up
				movwf	step_up
				movlw	1						; set step_down
				movwf	step_down

				movlw	191
				movwf	duty_diff

				call	set_duty_hold
				bcf		duty_flag_1,DUTY_HOLD

				bsf		anim_flags,AF_DHLDL		; reset duty hold at zero

				goto	start_anim

				;---------------------------------
				; Mirror Scanner 1
m_mirror1
				bsf		duty_flag_1,AF_FWDH		; alternate direction at max duty
				bsf		duty_flag_1+NUM_OF_LED/2-1,AF_REVH		; alternate direction at max duty
				bsf		anim_flags,AF_DHLDL		; reset duty hold at zero
				bsf		anim_flags,AF_MIRROR	; mirror mode on

				movlw	37*NUM_OF_ROW			; set update_rate
				movwf	update_rate

				movlw	1						; set step_up
				movwf	step_up
				movlw	1						; set step_down
				movwf	step_down

				movlw	191
				movwf	duty_diff

				call	set_duty_hold
				bcf		duty_flag_1+NUM_OF_LED/2-1,DUTY_HOLD

				bsf		anim_flags,AF_DHLDL		; reset duty hold at zero

				goto	start_anim

				;---------------------------------
				; Mirror Scanner 2
m_mirror2
				bsf		anim_flags,AF_DIR		; reverse direction
				call	set_duty_hold
				bcf		duty_flag_1+NUM_OF_LED/2-1,DUTY_HOLD	; start from center
				goto	m_mirror2_
m_mirror2R
				call	set_duty_hold
				bcf		duty_flag_1,DUTY_HOLD	; start from left
m_mirror2_
				bsf		anim_flags,AF_DHLDL		; reset duty hold at zero
				bsf		anim_flags,AF_MIRROR	; mirror mode on

				movlw	49*NUM_OF_ROW			; set update_rate
				movwf	update_rate

				movlw	1						; set step_up
				movwf	step_up
				movlw	1						; set step_down
				movwf	step_down

				movlw	96
				movwf	duty_diff

				bsf		anim_flags,AF_DHLDL		; reset duty hold at zero

				goto	start_anim

				;---------------------------------
				; Mirror Scanner 3
m_mirror3										; center -> out
				bsf		anim_flags,AF_DIR		; reverse direction
				call	set_duty_hold
				bcf		duty_flag_1+NUM_OF_LED/2-1,DUTY_HOLD	; start from center
				goto	m_mirror3_
m_mirror3R										; out -> center
				call	set_duty_hold
				bcf		duty_flag_1,DUTY_HOLD	; start from left
m_mirror3_
				bsf		anim_flags,AF_DHLDL		; reset duty hold at zero
;				bsf		anim_flags,AF_DHLDH		; reset duty hold at max
				bsf		anim_flags,AF_MIRROR	; mirror mode on

				movlw	101*NUM_OF_ROW			; set update_rate
				movwf	update_rate

				movlw	1						; set step_up
				movwf	step_up
				movlw	1						; set step_down
				movwf	step_down

				movlw	245
				movwf	duty_diff

				bsf		anim_flags,AF_DHLDL		; reset duty hold at zero

				goto	start_anim

				;---------------------------------
				; full wave
mode_01a
				bsf		duty_flag_1,AF_FWDL		; alternate direction at min duty
				bsf		duty_flag_1+NUM_OF_LED-1,AF_REVL	; alternate direction at min duty
				bsf		anim_flags,AF_DHLDL		; reset duty hold at zero
				goto	mode_01

mode_01r		bsf		anim_flags,AF_DIR		; reverse direction
				call	set_duty_hold
				bcf		duty_flag_1+NUM_OF_LED-1,DUTY_HOLD
				goto	mode_01_
mode_01
				call	set_duty_hold
				bcf		duty_flag_1,DUTY_HOLD
mode_01_
				movlw	15						; set update_rate
				movwf	update_rate

				movlw	1						; set step_up/down
				movwf	step_up
				movwf	step_down

				movlw	256/NUM_OF_LED
				movwf	duty_diff

				goto	start_anim

				;---------------------------------
				; half wave
mode_02a
				bsf		duty_flag_1,AF_FWDH			; alternate direction at max duty
				bsf		duty_flag_1+NUM_OF_LED-1,AF_REVH	; alternate direction at max duty
				goto	mode_02

mode_02r		bsf		anim_flags,AF_DIR			; reverse direction
				call	set_duty_hold
				bcf		duty_flag_1+NUM_OF_LED-1,DUTY_HOLD
				goto	mode_02_
mode_02
				call	set_duty_hold
				bcf		duty_flag_1,DUTY_HOLD
mode_02_
				movlw	42/NUM_OF_ROW				; set update_rate
				movwf	update_rate

				movlw	1							; set step_up/down
				movwf	step_up
				movwf	step_down

				movlw	256*2/NUM_OF_LED
				movwf	duty_diff

				bsf		anim_flags,AF_DHLDL		; reset duty hold at zero

				goto	start_anim

				;---------------------------------
				; flashing
m_flashing
				movlw	15						; set update_rate
				movwf	update_rate

				movlw	255						; set step_up
				movwf	step_up
				movlw	1						; set step_down
				movwf	step_down

				clrf	duty_diff

				goto	start_anim

				;---------------------------------
				; fade in/out
m_fade_in_out
				movlw	4						; set update_rate
				movwf	update_rate

				movlw	1						; set step_up
				movwf	step_up
				movlw	1						; set step_down
				movwf	step_down

				clrf	duty_diff

				goto	start_anim

				;---------------------------------
				; stedy on
m_stedy_on
				movlw	11						; set update_rate
				movwf	update_rate

				movlw	1						; set step_up
				movwf	step_up
				movlw	0						; set step_down
				movwf	step_down

				clrf	duty_diff

				goto	start_anim

				;---------------------------------
				; Random flashes
m_random
				movlw	1						; use custom process 1
				movwf	anim_process

				movlw	15						; set update_rate
				movwf	update_rate

				movlw	12						; set update_rate on custom process
				movwf	an_update_rate

				movlw	255						; set step_up
				movwf	step_up
				movlw	2						; set step_down
				movwf	step_down

				movlw	254
				movwf	duty_diff

				call	set_duty_hold

				bsf		anim_flags,AF_DHLDL		; reset duty hold at zero

				goto	start_anim

				;---------------------------------
				; Analog mode
m_analog
				banksel	ANSELA
				bsf		ANSELA,AUX_PIN			; set AUX_PIN port analog mode
				banksel	WPUA
				bcf		AUX_PORT,AUX_PIN		; turn off pull-up
				banksel	TRISA
				bsf		AUX_PORT,AUX_PIN		; set AUX_PIN input mode

				banksel	0
				bsf		anim_flags,AF_ANALOG	; analog animation mode on

				movlw	0xFF					; set update_rate
				movwf	an_update_rate

				goto	start_anim

				;---------------------------------
				; power down
power_down
				banksel	INTCON
				bcf		INTCON,GIE				; disable all INT

				banksel	CCP1CON
				clrf	CCP1CON					; reset PWM

				banksel	TRISA
				bsf		SW_PORT,SW_PIN			; turn off boost circuit
				bsf		PWM_PORT,PWM_PIN		; turn off LEDs

				banksel	LATA					; set all LED drive pins low
				movlw	~LED_ON_A
				andwf	LATA,f
				movlw	~LED_ON_C
				andwf	LATC,f

				banksel	SW_PORT
				clrwdt
				btfsc	SW_PORT,SW_PIN			; wait for the button release
				goto	$-2

				;--- wait for the SW chatter to pass ---
				banksel	TMR1
				clrf	TMR1H
				clrf	TMR1L
				banksel	T1CON
				bsf		T1CON,T1CKPS0
				bsf		T1CON,T1CKPS1
				bsf		T1CON,TMR1ON			; start timer1
				banksel	PIR1
				bcf		PIR1,TMR1IF				; clear timer1 overflow flag
				clrwdt
				btfss	PIR1,TMR1IF				; wait for the timer overflow (64 ms)
				goto	$-2
				banksel	T1CON
				bcf		T1CON,TMR1ON			; stop timer1

				;--- set up interrupt ---
				banksel	IOCAP
				bsf		IOCAP,SW_PIN			; enable positive edge trigger IOC
				banksel	IOCAF
				clrf	IOCAF					; clear the IOC flags

				banksel	INTCON
				clrf	INTCON					; disable all INT
				bcf		INTCON,IOCIF			; clear IOCIF flag
				bsf		INTCON,IOCIE			; enable IOC interrupt
				bsf		INTCON,GIE				; enable all INT

				sleep							; SW1 push will wake this up

				reset

				;---------------------------------
start_anim

;---------------------------------------------------------------------------------------------------
;	main loop

main_loop
.if SYNC_ENABLE
				btfss	gflags,SYNC_ON			; if (SYNC_ON) {
				goto	anim_sync_off			;   --- sync with the external clock ---
				banksel	IOCAF
				btfsc	IOCAF,AUX_PIN			;   wait for a sync signal pulse
				goto	sync_pulse_in			;
				btfss	gflags,PWM_REF			;     no pulse - wait for timer2 overflow
				goto	main_loop
				bcf		gflags,PWM_REF			;     clear timer2 overflow flag
				goto	sync_timeout_chk		;     do the timeout check
sync_pulse_in									;   pulse came in
				bcf		IOCAF,AUX_PIN			;     clear IOC flag
				banksel	sync_timeout_l
				clrf	sync_timeout_l			;     clear timeout count
				clrf	sync_timeout_h
				goto	anim_continue2			;     and animate
sync_timeout_chk								;   --- take care of sync timeout ---
				banksel	sync_timeout_l
				decfsz	sync_timeout_l,f		;   
				goto	main_loop2
				decfsz	sync_timeout_h,f		;   count down the timer for timeout
				goto	main_loop2				;   if no pulse for over 8 mS
				bcf		gflags,SYNC_ON			;     sync-mode off
				goto	anim_continue2
.endif
anim_sync_off									; } else {
				btfss	gflags,PWM_REF			;   wait for timer2 overflow
				goto	main_loop
				bcf		gflags,PWM_REF			;   clear timer2 overflow flag
.if SYNC_ENABLE
				banksel	TRISA					;   send out sync signal
				movlw	1<<AUX_PIN				;   toggle AUX_PIN's TRIS
				xorwf	AUX_PORT,f
				btfss	AUX_PORT,AUX_PIN		;   if (AUX_TRIS == high) {
				goto	anim_continue2
				nop								;     give extra time for AUX_PIN to recover
				banksel	0
				btfsc	AUX_PORT,AUX_PIN		;     if (AUX_PORT == low) {
				goto	sync_high
				decfsz	sync_timeout_l,f		;       count down the timer for timeout
				goto	anim_continue2			;       if AUX_PIN held low for over 8 mS
				bsf		gflags,SYNC_ON			;         sync-mode on
				goto	start_up2				;         restart animation
sync_high										;     } else {
				clrf	sync_timeout_l			;       clear timeout count
												;     }
												;   }
.endif
anim_continue2									; }

				;--- animate LEDs ------------------------------------
				;--- update rate adjust --------------------
				banksel	0
				movfw	master_update_rate		; master update rate
				addwf	master_update,f
				btfss	STATUS,C
				goto	main_loop2
				movfw	update_rate
				addwf	update,f				; only update if update overflows
				btfss	STATUS,C
				goto	main_loop2

				;--- animate each LED duty -----------------
				movfw	anim_process
				brw
				goto	default_process
				goto	anim_process1
;				goto	anim_process2
;				goto	anim_process3

anim_process1
				call	random_blinks
				goto	default_process

;anim_process2
;				call	process2
;				goto	main_loop2

default_process
				btfss	anim_flags,AF_DIR		; check animation direction
				call	forward_anim
				btfsc	anim_flags,AF_DIR		; check animation direction
				call	reverse_anim

				;---------------------------------------------------------------
				; take care of other things every 256th times
main_loop2
				btfss	gflags,PWM_REF2			; PWM duty = 0
				goto	main_loop
				bcf		gflags,PWM_REF2			; clear the flag

				banksel	anim_flags
				btfss	anim_flags,AF_ANALOG	; if analog mode
				goto	analog_skip				; {

				movfw	an_update_rate
				addwf	an_update,f				;   only update if update overflows
				btfss	STATUS,C
				goto	analog_skip

				movlw	5						;   set ADC state to sample aux input
				movwf	adc_state
				call	analog_anim				;   do analog thing
analog_skip										; }

.if (auto_adv_time!=0)
				btfss	gflags,DEMO_ON
				bra		end_demo_cnt
				;--- handle auto-changer timer -----------------------
				decfsz	sleep_timer0
				goto	end_demo_cnt
				decfsz	sleep_timer1
				goto	end_demo_cnt
				decfsz	sleep_timer2
				goto	end_demo_cnt
				goto	next_mode
end_demo_cnt
.endif
				;--- check button SW events --------------------------
				btfsc	bflags,BTN_LNG			; check if the button long pushed
				goto	power_down

				btfsc	bflags,BTN_DBL			; double click
				call	speed_up
				bcf		bflags,BTN_DBL

				btfsc	bflags,BTN_TRP			; triple click
				call	speed_down
				bcf		bflags,BTN_TRP

				btfss	bflags,BTN_PSD			; check if the button pushed
				goto	main_loop
				bcf		bflags,BTN_PSD
				goto	next_mode


;-------------------------------------------------------------------------------
;	random blinks

random_blinks
				banksel	0
				movfw	an_update_rate
				addwf	an_update,f				; only update if an_update overflows
				btfss	STATUS,C
				return

				movlw	duty_flag_1				; prepare indirect addressing pointer -> duty_flag_1
				movwf	FSR0L

				call	random_number			; get a random number
				movfw	random_val_l
				andlw	NUM_OF_LED-1			; only use the last n bits
				addwf	FSR0L,f					; index duty buffer -> duty_flag_[w]
				btfsc	INDF0,DUTY_HOLD			; if duty hold is set
				bcf		INDF0,DUTY_HOLD			;   clear the duty hold
				return


;-------------------------------------------------------------------------------
;	analog mode - meter function

analog_anim
				banksel	0
.if	(NUM_OF_ROW == 1)
				lsrf	aux_voltage,w
.else
				movfw	aux_voltage
.endif
				movwf	aux_voltage2

				bsf		anim_flags,NO_UPDATE	; prohibit duty buffer update
.if 0
				;--- bar mode display ---
				movlw	duty_1					; prepare indirect addressing pointer
				movwf	FSR0L
				swapf	aux_voltage2,w			; take the upper 4 bits
				andlw	NUM_OF_LED-1			; into lower 3 or 4 bits
				addwf	FSR0L,f					; index duty buffer - duty_[w]
				movfw	FSR0L					; copy FSR0L -> FSR1L
				movwf	FSR1L

				swapf	aux_voltage2,w			; take lower 4 bits
				andlw	0xF0					; into high nibble
				movwf	INDF0					; use that as the LED brightness
analog_fill
				decf	FSR0L,f					; fill lower number LEDs with max duty
				movlw	duty_1-1
				subwf	FSR0L,w
				btfsc	STATUS,Z
				goto	analog_fill2
				movlw	max_duty
				movwf	INDF0
				goto	analog_fill
analog_fill2
				incf	FSR1L,f					; fill higher number LEDs with min duty
				movlw	duty_1+NUM_OF_LED		; finish if FSR1L == duty_1+NUM_OF_LED
				subwf	FSR1L,w
				btfsc	STATUS,Z
				goto	analog_end
				clrf	INDF1
				goto	analog_fill2
.else
				;--- dot mode display ---
				movlw	duty_1					; prepare indirect addressing pointer
				movwf	FSR0L
				swapf	aux_voltage2,w			; take the upper 4 bits
				andlw	NUM_OF_LED-1			; into lower 3 or 4 bits
				addwf	FSR0L,f					; index duty buffer - duty_[w]
				movfw	FSR0L					; copy FSR0L -> FSR1L
				movwf	FSR1L
				decf	FSR0L,f

				swapf	aux_voltage2,w			; take lower 4 bits
				andlw	0xF0					; into high nibble
				movwf	INDF1					; use that as the LED brightness
				comf	INDF1,w					; complement that value
				movwf	INDF0					; use that as the lower LED brightness

analog_fill										; fill lower number LEDs with min duty
				decf	FSR0L,f
				movlw	duty_1-1
				subwf	FSR0L,w
				btfss	STATUS,C
				goto	analog_fill2
				clrf	INDF0
				goto	analog_fill

analog_fill2
				incf	FSR1L,f					; clear higher number LEDs duty
				movlw	duty_1+NUM_OF_LED		; finish if FSR1L == duty_1+NUM_OF_LED
				subwf	FSR1L,w
				btfsc	STATUS,Z
				goto	analog_end
				clrf	INDF1
				goto	analog_fill2
.endif
analog_end
				bcf		anim_flags,NO_UPDATE	; enable duty buffer update
				return

;-------------------------------------------------------------------------------

forward_anim
				movlw	duty_1					; set up indirect access of duty buffers
				movwf	FSR0L
				movlw	duty_flag_1
				movwf	FSR1L

forward_anim_loop
				;--- LED 1 -----------------------
				btfss	anim_flags,AF_RELHD		; if release duty_hold is set
				goto	LED_1_release			; {
				bcf		INDF1,DUTY_HOLD			;   release duty_hold
				bcf		anim_flags,AF_RELHD		;   clear the flag
LED_1_release									; }
				movfw	duty_diff				; if (duty == duty_diff)
				subwf	INDF0,w
				btfss	STATUS,Z
				goto	LED_1_skip				; 
				btfsc	INDF1,AF_REVL			; and !(duty_flag.AF_REVL)
				goto	LED_1_skip
				btfsc	INDF1,AF_REVH			; and !(duty_flag.AF_REVH)
				goto	LED_1_skip				; {
				bsf		anim_flags,AF_RELHD		;   release duty_hold of next LED
LED_1_skip										; }
				btfsc	INDF1,DUTY_HOLD			; skip if (DUTY_HOLD ==1)
				goto	LED_1_end
				btfsc	INDF1,DUTY_DIR			; if (DUTY_DIR == 0) {
				goto	LED_1_rampDown
LED_1_rampUp	;--- UP ----------------
				bsf		anim_flags,NO_UPDATE	; prohibit duty buffer update
				movfw	step_up
				addwf	INDF0,f					;   increase LED duty
				btfss	STATUS,C				;   if (duty > max)
				goto	LED_1_end				;   {
				movlw	max_duty				;     set duty = max
				movwf	INDF0
				btfss	INDF1,AF_REVH			;     if change animation dir flag is set
				goto	LED_1_hold_h			;     {
				bsf		anim_flags,AF_DIR		;       change direction -> reverse
				call	duty_dir_up				;       duty up for other LEDs
LED_1_hold_h									;     }
				bsf		INDF1,DUTY_DIR			;     duty dir -> down
				btfsc	anim_flags,AF_DHLDH		;     if duty_hold_rs flag is set
				bsf		INDF1,DUTY_HOLD			;       hold the fade
				goto	LED_1_end				;   }
												; } else {
LED_1_rampDown	;--- DOWN --------------
				bsf		anim_flags,NO_UPDATE	; prohibit duty buffer update
				movfw	step_down
				subwf	INDF0,f					;   decrease LED duty
				btfsc	STATUS,C				;   if (duty < 0)
				goto	LED_1_end				;   {
				clrf	INDF0					;     set duty = 0
				bcf		INDF1,DUTY_DIR			;     duty dir -> up
				btfss	INDF1,AF_REVL			;     if change animation dir flag is set
				goto	LED_1_hold_l			;     {
				bsf		anim_flags,AF_DIR		;       change direction -> reverse
				goto	LED_1_end				; 
LED_1_hold_l									;     } else
				btfsc	anim_flags,AF_DHLDL		;     if duty_hold_rs flag is set
				bsf		INDF1,DUTY_HOLD			;       hold the fade
												;   }
LED_1_end										; }
				bcf		anim_flags,NO_UPDATE	; enable duty buffer update

				movlw	duty_1+NUM_OF_LED-1		; return if (FSR0 == address_of_last_duty)
				subwf	FSR0L,w
				btfsc	STATUS,Z
				return

				;--- point to the next duty data ---
				addfsr	FSR0,1
				addfsr	FSR1,1
				goto	forward_anim_loop		; continue loop

;-------------------------------------------------------------------------------

reverse_anim
				movlw	duty_1+NUM_OF_LED-1					; set up indirect access of duty buffers
				movwf	FSR0L
				movlw	duty_flag_1+NUM_OF_LED-1
				movwf	FSR1L
reverse_anim_loop
				;--- LED 16 -----------------------
				btfss	anim_flags,AF_RELHD		; if release duty_hold is set
				goto	LED_16_release			; {
				bcf		INDF1,DUTY_HOLD			;   release duty_hold
				bcf		anim_flags,AF_RELHD		;   clear the flag
LED_16_release									; }
				movfw	duty_diff				; if (duty == duty_diff)
				subwf	INDF0,w
				btfss	STATUS,Z
				goto	LED_16_skip				; 
				btfsc	INDF1,AF_FWDL			; and !(AF_FWDL)
				goto	LED_16_skip
				btfsc	INDF1,AF_FWDH			; and !(AF_FWDH)
				goto	LED_16_skip				; {
				bsf		anim_flags,AF_RELHD		;   release duty_hold of prev LED
LED_16_skip										; }
				btfsc	INDF1,DUTY_HOLD			; skip if (DUTY_HOLD == 1)
				goto	LED_16_end
				btfsc	INDF1,DUTY_DIR			; if (DUTY_DIR == 0) {
				goto	LED_16_rampDown
LED_16_rampUp	;--- UP ----------------
				bsf		anim_flags,NO_UPDATE	; prohibit duty buffer update
				movfw	step_up
				addwf	INDF0,f					;   increase LED duty
				btfss	STATUS,C				;   if (duty > max)
				goto	LED_16_end				;   {
				movlw	max_duty				;     set duty = max
				movwf	INDF0
				bsf		INDF1,DUTY_DIR			;     duty dir -> down

				btfss	INDF1,AF_FWDH			;     if alternate dir flag is set
				goto	LED_16_hold_h			;     {
				bcf		anim_flags,AF_DIR		;       change direction -> forward
				call	duty_dir_up				;       duty up for other LEDs
				bsf		INDF1,DUTY_DIR			;       duty dir -> down
				goto	LED_16_end				;     } else
LED_16_hold_h
				btfsc	anim_flags,AF_DHLDH		;     if duty_hold_rs flag is set
				bsf		INDF1,DUTY_HOLD			;       hold the fade
				goto	LED_16_end				;   }
												; } else {
LED_16_rampDown	;--- DOWN --------------
				bsf		anim_flags,NO_UPDATE	; prohibit duty buffer update
				movfw	step_down
				subwf	INDF0,f					;   decrease LED duty
				btfsc	STATUS,C				;   if (duty < 0)
				goto	LED_16_end				;   {
				clrf	INDF0					;     set duty = 0
				bcf		INDF1,DUTY_DIR			;     duty dir -> up

				btfss	INDF1,AF_FWDL			;     if alternate dir flag is set
				goto	LED_16_hold_l			;     {
				bcf		anim_flags,AF_DIR		;       change direction -> forward
				goto	LED_16_end				;     }
LED_16_hold_l
				btfsc	anim_flags,AF_DHLDL		;     if duty_hold_rs flag is set
				bsf		INDF1,DUTY_HOLD			;       hold the fade
												;   }
LED_16_end										; }
				bcf		anim_flags,NO_UPDATE	; enable duty buffer update

				movlw	duty_1					; return if (FSR0 == duty_1)
				subwf	FSR0L,w
				btfsc	STATUS,Z
				return

				;--- point to the previous duty data ---
				addfsr	FSR0,-1
				addfsr	FSR1,-1

				goto	reverse_anim_loop		; continue loop


				;-----------------------------------------------------

duty_dir_up		;--- change all DUTY_DIR to UP (0)
				bcf		duty_flag_1,DUTY_DIR
				bcf		duty_flag_2,DUTY_DIR
				bcf		duty_flag_3,DUTY_DIR
				bcf		duty_flag_4,DUTY_DIR
				bcf		duty_flag_5,DUTY_DIR
				bcf		duty_flag_6,DUTY_DIR
				bcf		duty_flag_7,DUTY_DIR
				bcf		duty_flag_8,DUTY_DIR
.if (NUM_OF_ROW==2)
				bcf		duty_flag_9,DUTY_DIR
				bcf		duty_flag_10,DUTY_DIR
				bcf		duty_flag_11,DUTY_DIR
				bcf		duty_flag_12,DUTY_DIR
				bcf		duty_flag_13,DUTY_DIR
				bcf		duty_flag_14,DUTY_DIR
				bcf		duty_flag_15,DUTY_DIR
				bcf		duty_flag_16,DUTY_DIR
.endif
				return

				;-----------------------------------------------------
set_duty_hold	; set DUTY_HOLD bit on all LEDs
				bsf		duty_flag_1,DUTY_HOLD
				bsf		duty_flag_2,DUTY_HOLD
				bsf		duty_flag_3,DUTY_HOLD
				bsf		duty_flag_4,DUTY_HOLD
				bsf		duty_flag_5,DUTY_HOLD
				bsf		duty_flag_6,DUTY_HOLD
				bsf		duty_flag_7,DUTY_HOLD
				bsf		duty_flag_8,DUTY_HOLD
.if (NUM_OF_ROW==2)
				bsf		duty_flag_9,DUTY_HOLD
				bsf		duty_flag_10,DUTY_HOLD
				bsf		duty_flag_11,DUTY_HOLD
				bsf		duty_flag_12,DUTY_HOLD
				bsf		duty_flag_13,DUTY_HOLD
				bsf		duty_flag_14,DUTY_HOLD
				bsf		duty_flag_15,DUTY_HOLD
				bsf		duty_flag_16,DUTY_HOLD
.endif
				return

;-------------------------------------------------------------------------------

speed_up
				banksel	master_update_rate
				bsf		STATUS,C				; set carry flag
				RLF		master_update_rate,f	; master update rate << 1 + 1

				goto	save_speed

				;-----------------------------------------------------
speed_down
				banksel	master_update_rate
				LSRF	master_update_rate,f	; master update rate >> 1
				bsf		master_update_rate,0	; keep it above 1

save_speed
				;--- save speed adjust value -------------------------
				banksel	master_update_rate
				movfw	master_update_rate
				banksel	EEDATL
				movwf	EEDATL					; set data to write to
				movlw	EE_LAST_SPEED			; set EEPROM address to write to
				banksel	EEADRL
				movwf	EEADRL
				call	write_eeprom

				return
;
;-------------------------------------------------------------------------------
;	PWM value lookup tables
;
get_ccpr1_value
				;--- translate PWM value to CCPR1 value ---
.if (NUM_OF_ROW==2)
				movfw	duty
				andlw	0x7F					; use lower 7 bits only
				brw								; table jump

				; table to expand 8 bit value to 10 bit (linear)
	variable x = 1
	while	x <= max_duty/2+1
y =	pwm_period*4+3-((pwm_period-dead_band)*4*x/(max_duty/2+1))
				dt		y>>2					; upper 8 bit
x++
	endw

.else			;--- full 8 bit for 8 LED type ---
				movfw	duty
				brw								; table jump

				; table to expand 8 bit value to 10 bit (linear)
	variable x = 0
	while	x <= max_duty
y =	pwm_period*4+3-((pwm_period-dead_band)*4*x/(max_duty))
				dt		y>>2					; upper 8 bit
x++
	endw
.endif
;
;
;
get_ccp1con_value
				;--- translate PWM value to CCP1CON value ---
.if (NUM_OF_ROW==2)
				movfw	duty
				brw								; table jump

				; table to expand 8 bit value to 10 bit (linear)
	variable x = 1
	while	x <= max_duty/2+1
y =	pwm_period*4+3-((pwm_period-dead_band)*4*x/(max_duty/2+1))
				dt		(y&0x03)<<4|ccpcon_pram		; lower 2 bit for CCP1CON - active-high
x++
	endw

	variable x = 1
	while	x <= max_duty/2+1
y =	pwm_period*4-((pwm_period-dead_band)*4*x/(max_duty/2+1))
				dt		(y&0x03)<<4|ccpcon_pram2	; lower 2 bit for CCP1CON - active-low
x++
	endw

.else			;--- full 8 bit for 8 LED type ---
				movfw	duty
				brw								; table jump

				; table to expand 8 bit value to 10 bit (linear)
	variable x = 0
	while	x <= max_duty
y =	pwm_period*4+3-((pwm_period-dead_band)*4*x/(max_duty))
				dt		(y&0x03)<<4|ccpcon_pram		; lower 2 bit for CCP1CON - active-high
x++
	endw
.endif
;
;===================================================================================================
;
;	A/D service
;
ad_convert
				banksel	ADCON0
				btfsc	ADCON0,GO				; is AD conversion done?
				goto	ad_skip
												; conversion done
				movfw	adc_state
				btfss	STATUS,Z				; if (adc_state != 0)
				decf	adc_state,f				;   adc_state--
				brw								; branch based on ADC state
				goto	ad_supply_read
				goto	ad_supply_convert
				goto	ad_aux_read
				goto	ad_aux_convert
;				goto	ad_aux_sample

ad_aux_sample
				banksel	ADCON0					; set up ADC for AUX input
				movlw	adc_param0A
				movwf	ADCON0
				movlw	adc_param1A
				movwf	ADCON1
				goto	ad_supply_read2			; continue on to read supply voltage

ad_aux_read
.if AUX_SENSE	; low sensitivity
				banksel	ADRESH
				movfw	ADRESH					; read the high byte
.else			; high sensitivity - ADC result aligned right
				banksel	ADRESH
				movfw	ADRESH					; read the high byte
				btfss	STATUS,Z
				goto	ad_overflow
				movfw	ADRESL					; read the low byte
				goto	ad_aux_save
ad_overflow		movlw	0xFF					; set the result = 0xFF if overflown
ad_aux_save
.endif
				banksel	0
				subwf	aux_voltage,f			; 
				btfsc	STATUS,C
				goto	ad_aux_high
				movwf	aux_voltage				; set the new value -> aux_voltage
				goto	ad_aux_done
ad_aux_high
				addwf	aux_voltage,f			; add W back
				btfss	STATUS,Z				; if (aux_voltage != 0)
				decf	aux_voltage,f			;   aux_voltage--
				btfss	STATUS,Z				; if (aux_voltage != 0)
				decf	aux_voltage,f			;   aux_voltage--
ad_aux_done
				banksel	ADCON0					; set up ADC for supply voltage
				movlw	adc_param0
				movwf	ADCON0
				movlw	adc_param1
				movwf	ADCON1
				goto	ad_skip

ad_supply_read
				banksel	ADCON0
				bsf		ADCON0,GO				; start the next AD conversion
ad_supply_read2
				banksel	ADRESH
				movfw	ADRESH					; read the high byte
				banksel	0
				movwf	sp_voltage				; save the value
				sublw	SPV_THRES				; if value is higher than thres (thres - sp_voltage) < 0
				banksel	TRISA					; (which means supply voltage is too low)
				btfsc	STATUS,C
				goto	ad_boost_off			; {
				bcf		SW_PORT,SW_PIN			;   turn on boost circuit
				bcf		gflags,SP_HIGH			;   turn off high voltage flag
				goto	ad_skip
ad_boost_off									; } else {
				bsf		SW_PORT,SW_PIN			;   turn off boost circuit
				bsf		gflags,SP_HIGH			;   turn on high voltage flag
												; }
				goto	ad_skip

ad_aux_convert
ad_supply_convert
				banksel	ADCON0
				bsf		ADCON0,GO				; start the next AD conversion
ad_skip
				return

;
;===================================================================================================
;
;	Button switch service
;
check_button
				banksel	TRISA
				bsf		SW_PORT,SW_PIN			; change SW port to input mode
				nop								; give some time for the port to settle
				nop
				banksel	PORTA
				btfss	SW_PORT,SW_PIN			; if (SW port = high)  then SW is on
				goto	sw1_released			; {
				bsf		bflags,BTN_DWN			;   set button down flag
				btfsc	bflags,BTN_HLD			;   if SW had not been held
				goto	sw1_end					;   {
				incf	debounce,f				;     increase sw counter
				movlw	LONG_PUSH				;     if (sw count > LONG_PUSH)
				subwf	debounce,w
				btfss	STATUS,C
				goto	sw1_end					;     {
				bsf		bflags,BTN_LNG			;       set SW long pushed flag
				bsf		bflags,BTN_HLD			;       set SW held flag
				goto	sw1_end					;     }
												;   }
sw1_released									; } else {
				bcf		bflags,BTN_DWN			;   clear button down flag
				btfsc	bflags,BTN_HLD			;   if SW had not been held
				goto	sw1_deffer				;   {
				movlw	DEBOUNCE_CNT			;     if (sw count > debounce count)
				subwf	debounce,w
				btfss	STATUS,C
				goto	sw1_deffer				;     {
				incf	click_cnt,f				;       count the click
				movlw	EVT_TIMEOUT				;       reset event timer
				movwf	event_tmr				;     }
												;   }
sw1_deffer		bcf		bflags,BTN_HLD			;   clear SW held flag
				clrf	debounce				;   clear SW counter
sw1_end											; }
				btfsc	gflags,SP_HIGH			; if high voltage flag not set
				goto	sw1_end2
				banksel	TRISA
				bcf		SW_PORT,SW_PIN			;   change SW port back to output mode
sw1_end2
				decfsz	event_tmr,f				; check event timeout
				return							; if timed out {
				movlw	EVT_TIMEOUT				;   reset event timer
				movwf	event_tmr
				movfw	click_cnt
				btfsc	STATUS,Z				;   click_cnt == 0
				return							;     no event
				decfsz	click_cnt,f
				goto	double_clk
				bsf		bflags,BTN_PSD			;   single click
				return
double_clk
				decfsz	click_cnt,f
				goto	triple_clk
				bsf		bflags,BTN_DBL			;   double click
				return
triple_clk
				clrf	click_cnt
				bsf		bflags,BTN_TRP			;   triple click

				return							; }
;
;===================================================================================================
;	EEPROM read/write
;

read_eeprom
				; read address in W
				banksel	EEADRL
				movwf	EEADRL
				bcf		EECON1,CFGS				; deselect config space
				bcf		EECON1,EEPGD			; select EEPROM
				bsf		EECON1,RD				; initiate read
				movf	EEDATL,W				; put data in W

				banksel	0
				return
;-------------------------------------------------------------------------------

write_eeprom
				; load address->EEADRL and data->EEDATL before call
				banksel	EECON1
				bcf		EECON1,CFGS				; deselect config space
				bcf		EECON1,EEPGD			; select EEPROM
				bsf		EECON1,WREN				; enable writes
				;--- EEPROM write sequence ---
				bcf		INTCON,GIE				; disable interrupts
				movlw	0x55
				movwf	EECON2
				movlw	0xAA
				movwf	EECON2
				bsf		EECON1,WR				; initiate write
				bsf		INTCON,GIE				; enable interrupts
				bcf		EECON1,WREN				; disable write
				clrwdt							; clear watchdog timer
				btfsc	EECON1,WR				; wait till write completes
				goto	$-2

				banksel	0
				return

;===================================================================================================
;	"roulette" quick scanning mode selection
;

play_roulette
				banksel	0
				btfsc	gflags,DEMO_ON			; if demo mode flag on
				goto	roulette_demo

				movfw	mode_num
				andlw	NUM_OF_LED-1			; take the bottom 3 or 4 bits only (only up to the number of LEDs)
				addlw	duty_1					; add to the address of duty_1
				movwf	FSR0L					; INDF0 -> duty_[mode_num]

				comf	INDF0,f					; invert the duty_x - turns on the LED

				movlw	RLT_SPEED				; set loop counter
				movwf	update
				bcf		gflags,PWM_REF2			; clear the flag
roulette_loop
				btfss	gflags,PWM_REF2			; wait for the PWM phase start
				goto	$-1
				bcf		gflags,PWM_REF2			; clear the flag
				decfsz	update,f				; then loop x times
				goto	roulette_loop

				btfss	gflags,DEMO_ON			; if demo mode flag off
				comf	INDF0,f					;   invert the duty_x - turns off the LED

				return

roulette_demo
				movlw	duty_1					; 
				movwf	FSR0L					; INDF0 -> duty_[]

				movlw	NUM_OF_LED				; loop_counter2 = NUM_OF_LED
				movwf	loop_counter2

				movlw	0xFF
roulette_loop2									; do {
				movwi	FSR0++					;   duty_x = w
				decfsz	loop_counter2,f			; } while (--loop_counter2 != 0)
				goto	roulette_loop2

				movlw	RLT_SPEED/2				; set loop counter
				movwf	update
				bcf		gflags,PWM_REF2			; clear the flag
				call	roulette_loop

				movlw	NUM_OF_LED				; loop_counter2 = NUM_OF_LED
				movwf	loop_counter2

				movlw	0x00
roulette_loop3									; do {
				movwi	--FSR0					;   duty_x = w
				decfsz	loop_counter2,f			; } while (--loop_counter2 != 0)
				goto	roulette_loop3

				movlw	RLT_SPEED/2				; set loop counter
				movwf	update
				bcf		gflags,PWM_REF2			; clear the flag
				call	roulette_loop
;				comf	INDF0,f					; invert the duty_x - turns off the LED

				return


;===================================================================================================
;	bitmap -> LED port lookup table
;
led_port_lookup
				dt		(LED_1_PORT==PORTA)*(1<<LED_1_PIN), (LED_1_PORT==PORTC)*(1<<LED_1_PIN)
				dt		(LED_2_PORT==PORTA)*(1<<LED_2_PIN), (LED_2_PORT==PORTC)*(1<<LED_2_PIN)
				dt		(LED_3_PORT==PORTA)*(1<<LED_3_PIN), (LED_3_PORT==PORTC)*(1<<LED_3_PIN)
				dt		(LED_4_PORT==PORTA)*(1<<LED_4_PIN), (LED_4_PORT==PORTC)*(1<<LED_4_PIN)
				dt		(LED_5_PORT==PORTA)*(1<<LED_5_PIN), (LED_5_PORT==PORTC)*(1<<LED_5_PIN)
				dt		(LED_6_PORT==PORTA)*(1<<LED_6_PIN), (LED_6_PORT==PORTC)*(1<<LED_6_PIN)
				dt		(LED_7_PORT==PORTA)*(1<<LED_7_PIN), (LED_7_PORT==PORTC)*(1<<LED_7_PIN)
				dt		(LED_8_PORT==PORTA)*(1<<LED_8_PIN), (LED_8_PORT==PORTC)*(1<<LED_8_PIN)

;===================================================================================================
;	random number generator
;

random_number

RND_SEED		equ		0xB4					; seed can be other values as well

				movfw	random_val_l			; test random_val
				iorwf	random_val_h,w			; 
				btfss	STATUS,Z				; if (random_val == 0)
				goto	random_cont				; {

				movlw	RND_SEED				;   random_val = RND_SEED
				movwf	random_val_l
												; }
random_cont
				bcf		STATUS,C				; clear carry flag
				rrf		random_val_h,f			; random_val = random_val>>1 -> carry
				rrf		random_val_l,f			; 
				btfss	STATUS,C				; if carry is set
				return
				movlw	RND_SEED				;   W = W XOR RND_SEED
				xorwf	random_val_h,f
;				xorwf	random_val_l,f
				return

;===================================================================================================
				end
