"""
    tinyLiDAR functions in micropython for ESP32
    Last Edit: April 12, 2018 by Dinesh Bhatia
    Copyright © 2018 MicroElectronicDesign, Inc. (www.microed.co)
    This work is licensed under Attribution-ShareAlike 3.0 Unported (CC BY-SA 3.0)
    https://creativecommons.org/licenses/by-sa/3.0/legalcode
"""

import machine, utime, gc
from machine import Pin, I2C
from machine import Timer
import tinyLiDAR_var as tlv
import tinyLiDAR_menu as tlm
gc.collect()

class Clock:
	def __init__(self, i2c):		
		self.i2c = i2c
		self.seconds = 0

	def _seconds_handler(self, alarm):		
		alarm.cancel() 
		self.seconds = 0
		self.clearI2Cbegin(self.i2c)

	def start(self):
		self.__alarm = Timer.Alarm(self._seconds_handler, ms=35, periodic=True) #set alarm for 35ms I2C lockup max

	def stop(self, alarm):
		alarm.cancel()
		self.seconds = 0

	def I2C_ClearBus(self, i2c):
		self.i2c = i2c
		_SDApin = Pin('P9', mode=Pin.IN, pull=Pin.PULL_UP)
		_SCLpin = Pin('P10', mode=Pin.IN, pull=Pin.PULL_UP)

		if (not _SCLpin()):
			return 1  #I2C bus error. Could not clear SCL since clock line is held low

		clkCount = 0 
		if (not _SDApin()): # only if SDA is low
			clkCount = 9

		_SCLpin = Pin('P10', mode=Pin.OUT, pull=Pin.PULL_UP) 
		_SCLpin(0) # send low for START or repated START

		while (( not _SDApin() ) and (clkCount > 0) ):			
			clkCount -= 1
			_SCLpin(0) 
			utime.sleep_us(1) # is approx 27us
			_SCLpin(1) 
			utime.sleep_us(1)  # is approx 27us

		if (not _SDApin()):			
			return 2 # I2C bus error. Could not clear. SDA data line held low

		else: 
			_SDApin = Pin('P9', mode=Pin.OUT, pull=Pin.PULL_UP) 
			_SDApin(0) # send low for START or repated START
			utime.sleep_us(1) # is approx 27us
			_SCLpin(1) # need the SCL to be high for STOP to register when pulling the SDA high
			utime.sleep_us(1) # is approx 27us
			_SDApin(1) # back to high for STOP
		return 0

	def clearI2Cbegin(self, i2c):
		self.i2c = i2c
		
		self.i2c.deinit() 
		i = self.I2C_ClearBus(i2c) 

		tlm.home_clr(4)
		if(i != 0):
			if(i == 1):
				print(" SCL stuck low                                                    ")
			elif(i == 2):
				print(" SDA stuck low                                                    ")
		else:
			print("                                                                      ")
		i2c = I2C(0, I2C.MASTER, baudrate=100000)

class tinyLiDAR():
	def __init__(self, i2c, targetAddr = 0x10):
		self.i2c = i2c
		try:
			self.I2Caddr = targetAddr
		except NameError:
			self.I2Caddr = 0x10  # set to the tinyLiDAR default address if none given

	def confirm(self,msg):
		i = ""
		while i not in ["y", "n"]:
			i = input(msg).lower()
		return i == "y"

	def Read_Distance(self, waitTime, i2c, targetAddr = 0x10):		
		self.clk = Clock(self.i2c)		
		self.i2c = i2c
		try:
			i2cAddress = targetAddr
		except NameError:
			i2cAddress = 0x10  # set to the tinyLiDAR default address if none given

		# only use delay before reading result for Real Time mode which is neg waitTime
		if (waitTime <0): 
			self.i2c.writeto(i2cAddress, b'D') 
			try:
				data = self.i2c.readfrom(i2cAddress, 2)       
			except:
				pass
			finally:
				utime.sleep_ms(tlv._bufferTime-1*waitTime ) 
				self.clk.start() # start timeout alarm
				self.i2c.writeto(i2cAddress, b'D') 
				data = self.i2c.readfrom(i2cAddress, 2)

		else: 		# for SS/ULP mode use delay after reading
			try: 
				self.clk.start() # make sure the read doesn't get locked 
				self.i2c.writeto(i2cAddress, b'D') 
				data = self.i2c.readfrom(i2cAddress, 2)       
			except:
				self.clk.clearI2Cbegin(self.i2c) 
				utime.sleep_ms(100)	
				return -1  				# to say invalid data
		self.clk.stop(self.clk.__alarm)  
		return data[0]*256+data[1]

	def Dcommand(self, i2c, numBytes, delayBetween, targetAddr = 0x10):  # num of bytes, up to 255ms delay between and from targetAddr

		for i in range(0, numBytes):
			d = self.Read_Distance( delayBetween, self.i2c, targetAddr)
			tlm.home(3)
			if (d > 9):
				print(u"                                                    \r")            	
				print(u"  Distance = %d mm               \r" % d)            	
			elif (d == 1):
				print(u" VL53L0 Status Code: Sigma Fail           \r")
			elif (d == 2):
				print(" VL53L0 Status Code: Signal Fail           \r")
			elif (d == 3):
				print(" VL53L0 Status Code: Min Range Fail           \r")
			elif (d == 4):
				print(" VL53L0 Status Code: Phase Fail [Out of Range]           \r")
			else: 
				print(" -                                        \r") 			

			if (delayBetween > 0 ):  # use delay only if not in RT mode (ie if delay is pos)
				utime.sleep_ms(delayBetween+tlv._bufferTime) 

	def I2C_Write(self, i2c, address, command, dbyte = None):
		self.i2c = i2c
		try:
			i2cAddress = address
		except NameError:
			i2cAddress = 0x10  # set to the tinyLiDAR default address if none given

		for i in range(3):
			try:
				if dbyte:
					self.i2c.writeto(i2cAddress, bytes([command, dbyte]) )   
				else:
					self.i2c.writeto(i2cAddress, command )   
			except:
				utime.sleep_ms(100) 
			else:
				break

		if (i > 3):
			self.i2c.deinit() 
			utime.sleep_ms(50)	
			i2c = I2C(0, I2C.MASTER, baudrate=100000) 

		return 0	
				
	def ContinuousReadingForTerminal (self,i2c, address, time): 
		self.i2c = i2c
		tlm.home_clr(3)
		try:
			while True:
				self.Dcommand(i2c, 1, 0, address) 
				utime.sleep_ms(time+tlv._bufferTime) 
		except:
			raise KeyboardInterrupt # will reboot ESP32 

	def Qcommand(self, i2c, targetAddr = 0x10):
		self.i2c = i2c
		try:
			i2cAddress = targetAddr
		except NameError:
			i2cAddress = 0x10  # set to the tinyLiDAR default address if none given
		print(": Query tinyLiDAR Configuration Parameters ")
		self.i2c.writeto(i2cAddress, b'Q')         
		data = self.i2c.readfrom(i2cAddress, 23)
		try:
			return (data)
		except IndexError:
			print(" error in returned data! ")
			return 0	
		utime.sleep_ms(tlv.rebootTime)

if __name__ == "__main__":
    main()

    # ============================================================================
    # ============================================================================
    # ============================================================================ 