"""
    tinyLiDAR IoT example code in micropython for ESP32 WiPy3.0 module
    Last Edit: April 12, 2018 by Dinesh Bhatia
    Copyright © 2018 MicroElectronicDesign, Inc. (www.microed.co)
    This work is licensed under Attribution-ShareAlike 3.0 Unported (CC BY-SA 3.0)
    https://creativecommons.org/licenses/by-sa/3.0/legalcode

    Wiring details:

    wipy3.0  to  tinyLiDAR
        P9 = SDA (White)
        P10 = SCL (Yellow)
        P26 = +3.3V (Red)
        P27 = GND (Black)

        P2 (bootloader sw) momentary connection to GND as required for f/w update of wipy3.0 only 

        P28 Vin from a 5v USB Phone charger
            (5v at 100mA nominal drain was measured in lab)

        P27 common ground for system

    Optically isolated SSR for garage door opener:        
        P3 (ES32_GPIO4) to 470ohm resistor and then to G3VM-201AY1 SSR anode pin 1, SSR cathode pin 2 to GND.
               SSR pins 3 and 4 to switch contacts of garage door receiver

    wipy3.0 program port:
        P0 (RX0) From Tx pin of Serial USB cable
        P1 (TX0) To Rx pin of Serial USB cable

"""

import sys, machine, gc, utime, struct 
from machine import Pin, I2C
import tinyLiDAR_var as tlv
from tinyLiDAR_class import tinyLiDAR
from microWebSrv import MicroWebSrv
import pycom
gc.collect()

    # ============================================================================
    # ===( Variables )============================================================
    # ============================================================================

SignalRateLimit = 10
SignalEstimateLimit = 25
TimingBudget = 200 
VCELperiod = 18 
_timeBudget = tlv.TimingBudget_max  
RTmode = 1  # RTmode = -1 for RealTime mode else must be 1
buttonPressed = True
wsflag = False

    # ============================================================================
    # ===( Functions )============================================================
    # ============================================================================
                        
def _acceptWebSocketCallback(webSocket, httpClient) :
  print("WS ACCEPT")
  webSocket.RecvTextCallback   = _recvTextCallback
  webSocket.RecvBinaryCallback = _recvBinaryCallback
  webSocket.ClosedCallback     = _closedCallback
    
	# ----------------------------------------------------------------------------
    
def _recvTextCallback(webSocket, msg) :
  print("WS RECV TEXT : %s" % msg)  
  webSocket.SendText("%s_OK" % msg)
  global wsflag, ws, buttonPressed, _ssr
  ws = webSocket
  wsflag = True

  if(msg == 'press'):
    pycom.rgbled(0x007f00) # Green LED only
    _ssr.value(1) # turn on SSR
    now = utime.ticks_ms()
    buttonPressed = True

  if(msg == 'pressup'):    
    pycom.rgbled(0) # turn off RGB LED
    _ssr.value(0) # turn off SSR

  if(msg == 'pressRD'):
    pycom.rgbled(0x7f0000) # Red LED only
    now = utime.ticks_ms()
    buttonPressed = True
        
  if(msg == 'pressupRD'):    
    pycom.rgbled(0) # turn off RGB LED
    
	# ----------------------------------------------------------------------------
    
def _recvBinaryCallback(webSocket, data) :
  print("WS RECV DATA : %s" % data)

	# ----------------------------------------------------------------------------
    
def _closedCallback(webSocket) :
  buttonPressed = False # i.e. stop sending dist data
  wsflag = False
  print("WS CLOSED")

    # ============================================================================
    # ===( Code )=================================================================
    # ============================================================================
    
try:
    _ssr = Pin('P3', mode=Pin.OUT) # GPIO pin used to control the SSR
    pycom.heartbeat(False) # optional - to turn off slow blinking on wipy3.0 
    
    i2c = I2C(0, I2C.MASTER, baudrate=100000)
    tof = tinyLiDAR(i2c)  # 0x10 is the default I2C address for tinyLiDAR

    try:
        i2c.writeto(0, 6) # RESET command for tinyLiDAR    
        utime.sleep_ms(tlv.rebootTime) 
        RTmode = 1  # RTmode = -1 for RealTime mode else 1 
    except:
        pass

    # then write command "w 10 25 200 18" for longer range and better accuracy 
    param = []
    param.append((int(SignalRateLimit) & 0xff00) >> 8 )  #MSB
    param.append(int(SignalRateLimit) & 0x00ff)  #LSB
    param.append(int(SignalEstimateLimit))
    param.append((int(TimingBudget) & 0xff00) >> 8 )  #MSB
    param.append( int(TimingBudget) & 0x00ff )  #LSB
    param.append( int(VCELperiod) )

    tof.I2C_Write(i2c, tlv.I2C_Address, 'MS')  # set to SS mode for W command parameters
    utime.sleep_ms(tlv.rebootTime)					

    tof.I2C_Write(i2c, tlv.I2C_Address, bytearray([0x57] + param) )  
    utime.sleep_ms(tlv.rebootTime)					

    # finally, set tinyLiDAR to Real Time mode 
    tof.I2C_Write(i2c, tlv.I2C_Address,'MR')
    RTmode = -1  # RTmode = -1 for RealTime mode else 1 
    utime.sleep_ms(tlv.rebootTime) 

    # now start microwebserver
    mws = MicroWebSrv()                                    # TCP port 80 and files in /flash/www
    mws.MaxWebSocketRecvLen     = 256                      # Default is set to 1024
    mws.WebSocketThreaded       = False                    # WebSockets without new threads
    mws.AcceptWebSocketCallback = _acceptWebSocketCallback # Function to receive WebSockets
    mws.Start() 

    while not wsflag : pass  

    while (wsflag == True):
      if (not ws.IsClosed()): 
            if(buttonPressed):
                d = tof.Read_Distance( TimingBudget * RTmode, i2c, tlv.I2C_Address)
                if(d>10):           
                	ws.SendText("dist%d"%int(d/10)) # distance data to send in cm

except: # reboot ESP32 if any exceptions at all 
    RTmode = 1  # reset to normal mode instead of RT mode
    machine.reset()
            
    # ============================================================================
    # ============================================================================
    # ============================================================================ 