
/* ATTiny84 sketch for 78.125 KHz LED driver.
   External oscillator: 20 MHz
   Code contains settings for brightness levels.
   Can set drive current to 3A if circuit is updated.
   Code can be updated for heatsink temperature detection and fan control.

   Clicking the power button changes the brightness of the LEDs. Honding and releasing it turns it on or off.
*/

#include <avr/sleep.h>

// Utility macros
#define adc_disable() (ADCSRA &= ~(1<<ADEN)) // disable ADC
#define adc_enable()  (ADCSRA |=  (1<<ADEN)) // re-enable ADC

int onTime = 0;
const int period = 255; // Determines the frequency and resolution.
const int maxOnTime = period; // Keep it equal to period for buck driver.

const int brightnessTop = 4;
const int buckPwmPin = 5;
const int powerButtonPin = 8;
const int feedbackPin = A4;
const int batteryMonitorPin = A0;
const int batteryStatusLEDPin = 6;

int batteryMonitorReading;
int feedbackReading;
int desiredFeedback;
boolean buttonReading;

int brightness;
const int debounceDelay = 300; // Compensated for prescalar selection and the change of clock speed.
int delayCounts = 0;

int deadBatteryADC = 518; // 10.5V
int chargeReminderADC = 603; // 12.24V
int manualSelectADC = 742; // Manual or 14V

unsigned long elapsedMillis = 0;
unsigned long prevMillis = 0;

void setup() {
  pinMode(buckPwmPin, INPUT);
  pinMode(powerButtonPin, INPUT_PULLUP);
  pinMode(feedbackPin, INPUT);
  pinMode(batteryMonitorPin, INPUT);
  pinMode(batteryStatusLEDPin, OUTPUT);
  TCCR1A = _BV(COM1B1) | _BV(WGM11);
  TCCR1B = _BV(WGM13) | _BV(WGM12) | _BV(CS10);
  ICR1 = period;
  OCR1B = 0; // Start with minimum on time.
  analogReference(INTERNAL);
  sleepNow();
}

void loop() {
  // put your main code here, to run repeatedly:
  buttonReading = digitalRead(powerButtonPin);
  feedbackReading = analogRead(feedbackPin);
  batteryMonitorReading = analogRead(batteryMonitorPin);

  if (buttonReading == LOW) { // Pressing the button cycles through the drive current levels.
    delayCounts = 0;
    OCR1B = 0;
    onTime = 0;
    
    while (buttonReading == LOW) {
      delay(debounceDelay);
      delayCounts++;
      buttonReading = digitalRead(powerButtonPin);
    }
    // Cycles through the brightness levels.
    if (delayCounts <= 1) {
      if (brightness < brightnessTop) {
        brightness++;
      }
      else if (brightness >= brightnessTop) {
        brightness = 0;
      }
    }
    // Go to sleep.
    else if (delayCounts > 1) {
      sleepNow();
    }
  }

  // Determines the selected drive current.
  if (brightness == 0) {
    desiredFeedback = 0;
  }
  else if (brightness == 1) {
    desiredFeedback = 10; // 32 mA
  }
  else if (brightness == 2) {
    desiredFeedback = 100; // 325mA
  }
  else if (brightness == 3) {
    desiredFeedback = 210; // 650mA
  }
  else if (brightness == 4) {
    desiredFeedback = 435; // 1.3A
  }
  
  // Keeps the drive current stable.
  if (feedbackReading < desiredFeedback && onTime < maxOnTime) {
    onTime++;
  }
  else if (feedbackReading > desiredFeedback && onTime > 0) {
    onTime--;
  }
  OCR1B = onTime;

  if (batteryMonitorReading > manualSelectADC) { // Manual power source selected. Solid indicator.
    digitalWrite(batteryStatusLEDPin, HIGH);
  }
  else if (batteryMonitorReading > chargeReminderADC) { // Battery healthy. Indicator off.
    digitalWrite(batteryStatusLEDPin, LOW);
  }
  else if (batteryMonitorReading > deadBatteryADC) { // Battery low. Indicator blinks. Please recharge the battery.
    blinkBatteryIndicator();
  }
  else if (batteryMonitorReading <= deadBatteryADC) { // Battery dead. Go to sleep.
    sleepNow();
  }
}

void blinkBatteryIndicator() {
  elapsedMillis = millis() - prevMillis;
  if (elapsedMillis > 1000) {
    prevMillis = millis();
    digitalWrite(batteryStatusLEDPin, LOW);
  }
  else if (elapsedMillis > 500) {
    digitalWrite(batteryStatusLEDPin, HIGH);
  }
}
