#include <RunningAverage.h>
#include <ESP8266WiFi.h>
#include <WiFiClient.h>
#include <ESP8266WebServer.h>
#include <ESP8266mDNS.h>


const char* ssid = "NAAM_WIFI_NETWERK";
const char* password = "WACHTWOORD";

ESP8266WebServer server(8011);              // start webserver


// thanks Rob Tillaart >> http://playground.arduino.cc/Main/RunningAverage
RunningAverage myRA(10);
int samples = 0;

// DHT-code originally written by ladyada, public domain
#include "DHT.h"
#define DHTPIN 12     // what digital pin we're connected to
#define DHTTYPE DHT22   // DHT 22  (AM2302)

// Initialize DHT sensor
DHT dht(DHTPIN, DHTTYPE);

const int relaisPin =  16;      // the number of the relay pin
float defaultTemperature = 5.00;

float h, t;  // Values read from sensor
String webString=""; // String to display

unsigned long millisecondsSinceSwitch = 0;        // will store last time the heater was turned on or off
unsigned long minimumCycle = 60000;

unsigned long previousMeasurement = 0;        // last time temp and humidity were measured
const long intervalReadings = 6000;           // min time between readings


void readSensor() {
  
  unsigned long currentMillis = millis();
  if (currentMillis - previousMeasurement >= intervalReadings) {
      
      // Reading temperature or humidity takes about 250 milliseconds!
      // Sensor readings may also be up to 2 seconds 'old' (its a very slow sensor)
      h = dht.readHumidity();
      // Read temperature as Celsius
      t = dht.readTemperature();
      Serial.println("Temp: " + String(t));
    
      // Check if any reads failed and exit early (to try again).
      if (isnan(h) || isnan(t) ) {
        Serial.println("Failed to read from DHT sensor!");
        digitalWrite(relaisPin, HIGH);
        return;
      }
      
      myRA.addValue(t); // compute running average
      samples++;
      Serial.print("Running Average: ");
      Serial.println(myRA.getAverage(), 2);
      if (samples > 20) // reset the average at least every three minutes
      {
        samples = 0;
        myRA.clear();
      }
      
      previousMeasurement = currentMillis; // save the last time we read the sensor
  
  }
  
}

void switchHeater() {
  
  unsigned long currentMillis = millis();
  int state = digitalRead(relaisPin);
  
  // if we switched the heater on or off just a few minutes ago, then wait a while before doing it again 
  if (currentMillis - millisecondsSinceSwitch >= minimumCycle) {
    
    // if the temperature is below the minimum and the relay is turned off, turn on the heater
    if (t < defaultTemperature && state == HIGH) {
      
      digitalWrite(relaisPin, LOW);
      Serial.println("Relais: LOW");
      
    } else if ( t > defaultTemperature && state == LOW ) { // in the opposite case, turn the heater off
      
      digitalWrite(relaisPin, HIGH);
      Serial.println("Relais: HIGH");
      
    }
    
    millisecondsSinceSwitch = currentMillis; // remember when the heater was last turned on or off
    
  }
  
}

void handleRoot() {
  server.send(200, "text/plain", "hello from the esp8266 thermostat!");
}

void handleNotFound(){
  String message = "File Not Found\n\n";
  message += "URI: ";
  message += server.uri();
  message += "\nMethod: ";
  message += (server.method() == HTTP_GET)?"GET":"POST";
  message += "\nArguments: ";
  message += server.args();
  message += "\n";
  for (uint8_t i=0; i<server.args(); i++){
    message += " " + server.argName(i) + ": " + server.arg(i) + "\n";
  }
  server.send(404, "text/plain", message);
}

void setup(void){

  Serial.begin(115200);
  WiFi.begin(ssid, password);
  
  IPAddress ip(192,168,1,100);
  IPAddress gateway(192,168,1,1);
  IPAddress subnet(255,255,255,0);
  WiFi.config(ip, gateway, subnet);
  
  pinMode(relaisPin, OUTPUT);
  digitalWrite(relaisPin, HIGH); // initially, keep the relay turned off
  
  dht.begin();
  
  myRA.clear(); // explicitly start running average clean

  // Wait for connection
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }
  Serial.println("");
  Serial.print("Connected to ");
  Serial.println(ssid);
  Serial.print("IP address: ");
  Serial.println(WiFi.localIP());

  if (MDNS.begin("esp8266")) {
    Serial.println("MDNS responder started");
  }

  server.on("/", handleRoot);

  server.on("/dht", [](){ // give out human-readable info about the thermostat status
  webString="Temperature: "+String(t)+" C\nHumidity: "+String((int)h)+"%";
  server.send(200, "text/plain", webString); // send to someones browser when asked
  });
   
  server.on("/status", [](){ // give out info about the thermostat status as a pipe delimited string
  float t_average = myRA.getAverage();
  myRA.clear(); // reset average
  webString=String(t_average)+"|"+String((int)h)+"|"+String(digitalRead(relaisPin))+"|"+String(defaultTemperature);
  server.send(200, "text/plain", webString); // send to someones browser when asked
  });
   
  server.on("/goal", [](){ // allow the goal temperature to be changed
  float prevTemp = defaultTemperature;
  String goal = server.arg("goal");
  defaultTemperature = goal.toFloat(); // set goal temperature
  webString="Previous goal temperature: "+String(prevTemp)+"\nNew goal temperature: "+String(defaultTemperature)+"\nCurrent temperature: "+String(t);
  server.send(200, "text/plain", webString); // send to someones browser when asked
  });

  server.onNotFound(handleNotFound);

  server.begin();
  Serial.println("HTTP server started");
}

void loop(void){
  server.handleClient();
  
  readSensor();
  switchHeater();
  
}
