
#include <ESP8266WiFi.h>
#include <WiFiClient.h>
#include <ESP8266WebServer.h>
#include <ESP8266mDNS.h>

#include <Servo.h>
#include <RunningAverage.h> // by Rob Tillaart, http://playground.arduino.cc/Main/RunningAverage
#include <CapacitiveSensor.h> // CapitiveSense Library by Paul Badger 2008

const char* ssid = "NAAM_WIFI_NETWERK";
const char* password = "WACHTWOORD";

ESP8266WebServer server(8011);              // start webserver

// DHT-code originally written by ladyada, public domain
#include "DHT.h"
#define DHTPIN 12     // what digital pin we're connected to
#define DHTTYPE DHT22   // our sensor type

// Initialize DHT sensor
DHT dht(DHTPIN, DHTTYPE);

Servo servoActueel;  // servo for the actual temperature indicator
Servo servoDoel;  // servo for the goal temperature indicator

bool aanraking = false;
int drempel = 150;
long tijd_sinds_laatste_aanraking = millis();
int ledstatus = 0;
int aantal_tikjes = 0;
int meter_positie = 0;

int maximum_temperatuur = 22; // no need to ever have the thermostat higher than this. put on a sweater!
int minimum_temperatuur = 16; // lower than this isn't necessary, because it rarely gets colder than this in a well-isolated house anyway.
float doeltemperatuur = 16.0; // initial value
float actuele_temperatuur = 16.0; // initial value
float luchtvochtigheid; // air humidity
String webString="";

CapacitiveSensor   cs_4_2 = CapacitiveSensor(15, 14);       // digital pins that the resistor connects to eachother

RunningAverage myRA(10); // start a running average for the temperature readings
int samples = 0;

const int relaisPin =  16;      // digital pin that switches the relay

unsigned long millisecondsSinceSwitch = 0;        // this variabel stores when the heater was last turned on or off
unsigned long minimumCycle = 60000;               // minimal time between switching

unsigned long previousMeasurement = 0;        // time since last measuring the temperature and humidity
const long intervalReadings = 6000;           // minimal time between readings


void geefTemperatuurWeerOpMeter(int meterselectie, float temperatuur) {

  // pins for the servos
  servoDoel.attach(4);
  servoActueel.attach(5);

  // convert temperature to a setting of the servo
  // multiply temperatures by 10, because the map function cannot work with decimals
  meter_positie =  map((temperatuur * 10), (minimum_temperatuur * 10), (maximum_temperatuur * 10), 82, 0);

  // make sure the servo can't turn too far beyond the scale
  meter_positie = constrain(meter_positie, -10, 86);

  if ( meterselectie == 1 ) {

      servoDoel.write(meter_positie); // send a new position to the goal temperature indicating servo
      Serial.print("\ndoeltemperatuur_meter_positie: ");

  } else {

      servoActueel.write(meter_positie); // send a new position to the actual temperature indicating servo
      Serial.print("\nactueletemperatuur_meter_positie: ");
    
  }

  Serial.print(meter_positie);
  Serial.print("\n");

  delay(500); // give the servo a while to reach the right position
  
  // disconnect servos to prevent jitter
  servoActueel.detach();
  servoDoel.detach();

}

void readSensor() {
  
  unsigned long currentMillis = millis();
  if (currentMillis - previousMeasurement >= intervalReadings) {
      
      // Reading temperature or humidity takes about 250 milliseconds!
      // Sensor readings may also be up to 2 seconds 'old' (its a very slow sensor)
      luchtvochtigheid = dht.readHumidity();
      // Read temperature as Celsius
      actuele_temperatuur = dht.readTemperature();
      Serial.println("Temp: " + String(actuele_temperatuur));
    
      // Check if any reads failed and exit early (to try again).
      if (isnan(luchtvochtigheid) || isnan(actuele_temperatuur) ) {
        Serial.println("Failed to read from DHT sensor!");
        digitalWrite(relaisPin, HIGH);
        return;
      }
      
      myRA.addValue(actuele_temperatuur); // compute running average
      samples++;
      Serial.print("Running Average: ");
      Serial.println(myRA.getAverage(), 2);
      if (samples > 20) // reset the average at least every three minutes
      {
        samples = 0;
        myRA.clear();
      }
      
      previousMeasurement = currentMillis; // save the last time we read the sensor
  
  }
  
}

void switchHeater() {
  
  unsigned long currentMillis = millis();
  int state = digitalRead(relaisPin);
  
  // if we switched the heater on or off just a few minutes ago, then wait a while before doing it again 
  if (currentMillis - millisecondsSinceSwitch >= minimumCycle) {
    
    // if the temperature is below the minimum and the relay is turned off, turn on the heater
    if (actuele_temperatuur < doeltemperatuur && state == HIGH) {
      
      digitalWrite(relaisPin, LOW);
      Serial.println("Relais: LOW");
      
    } else if ( actuele_temperatuur > doeltemperatuur && state == LOW ) { // in the opposite case, turn the heater off
      
      digitalWrite(relaisPin, HIGH);
      Serial.println("Relais: HIGH");
      
    }
    
    millisecondsSinceSwitch = currentMillis; // remember when the heater was last turned on or off
    
  }
  
}

void handleRoot() {
  server.send(200, "text/plain", "hello from the esp8266 thermostat!");
}

void handleNotFound(){
  String message = "File Not Found\n\n";
  message += "URI: ";
  message += server.uri();
  message += "\nMethod: ";
  message += (server.method() == HTTP_GET)?"GET":"POST";
  message += "\nArguments: ";
  message += server.args();
  message += "\n";
  for (uint8_t i=0; i<server.args(); i++){
    message += " " + server.argName(i) + ": " + server.arg(i) + "\n";
  }
  server.send(404, "text/plain", message);
}

void setup(void){

  Serial.begin(115200);
  WiFi.begin(ssid, password);
  
  IPAddress ip(192,168,1,100);
  IPAddress gateway(192,168,1,1);
  IPAddress subnet(255,255,255,0);
  WiFi.config(ip, gateway, subnet);
  
  pinMode(relaisPin, OUTPUT);
  digitalWrite(relaisPin, HIGH); // initially, keep the relay turned off

  // led-pinnen aanzetten
  pinMode(13, OUTPUT);
  
  dht.begin();
  
  myRA.clear(); // explicitly start running average clean

  // Wait for connection
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }
  Serial.println("");
  Serial.print("Connected to ");
  Serial.println(ssid);
  Serial.print("IP address: ");
  Serial.println(WiFi.localIP());

  if (MDNS.begin("esp8266")) {
    Serial.println("MDNS responder started");
  }

  server.on("/", handleRoot);

  server.on("/dht", [](){ // give out human-readable info about the thermostat status
  webString="Temperature: "+String(actuele_temperatuur)+" C\nHumidity: "+String((int)luchtvochtigheid)+"%";
  server.send(200, "text/plain", webString); // send to someones browser when asked
  });
   
  server.on("/status", [](){ // give out info about the thermostat status as a pipe delimited string
  float t_average = myRA.getAverage();
  myRA.clear(); // reset average
  webString=String(t_average)+"|"+String((int)luchtvochtigheid)+"|"+String(digitalRead(relaisPin))+"|"+String(doeltemperatuur);
  server.send(200, "text/plain", webString); // send to someones browser when asked
  });
   
  server.on("/goal", [](){ // allow the goal temperature to be changed
  float prevTemp = doeltemperatuur;
  String goal = server.arg("goal");
  doeltemperatuur = goal.toFloat(); // set goal temperature
  webString="Previous goal temperature: "+String(prevTemp)+"\nNew goal temperature: "+String(doeltemperatuur)+"\nCurrent temperature: "+String(actuele_temperatuur);
  server.send(200, "text/plain", webString); // send to someones browser when asked
  });

  server.onNotFound(handleNotFound);

  server.begin();
  Serial.println("HTTP server started");
}

void loop(void){
  server.handleClient();
  
  readSensor();
  switchHeater();

  long start = millis();
  long total =  cs_4_2.capacitiveSensor(30);

  // Serial.print(millis() - start);        // check on performance in milliseconds
  // Serial.print("\t");                    // tab character for debug windown spacing

  // if the thermostat has just been touched (and we account for a little bounce), then turn on the LEDs
  if ( total > drempel && aanraking == false && ((millis() - tijd_sinds_laatste_aanraking) > 200) ) {

    tijd_sinds_laatste_aanraking = millis();             // reset counter

    if ( ledstatus == 0 ) {   // turn on LEDs and update gauges

        digitalWrite(13, HIGH);
        ledstatus = 1;

        geefTemperatuurWeerOpMeter(1, actuele_temperatuur);
        geefTemperatuurWeerOpMeter(2, doeltemperatuur);

    } else {                  // if the LEDs already were on, then count the taps towards increasing or decreasing the goal temperature

        aanraking = true;
        aantal_tikjes++;
      
    }
    
  }


  // when the last tap is more than 0,8 seconds ago, then let's do something
  if ( (millis() - tijd_sinds_laatste_aanraking) > 800 && aantal_tikjes > 0)  { // if the number of knocks > 0 and we have more to do

    Serial.print("\nI felt ");
    Serial.print(aantal_tikjes);
    Serial.print(" knocks.");

    if ( aantal_tikjes == 1 ) { // when tapped once, decrease by half a degree

      if ( doeltemperatuur > minimum_temperatuur ) {  // as lang as we haven't reached minimum temperature...
        doeltemperatuur -= 0.5;       // ... decrease goal temperature.
        Serial.print("\nGoal temperature decreased by half a degree to ");
        Serial.print(doeltemperatuur);
        Serial.print(".\n");
      } else {
        Serial.print("\nIt doesn't get colder then this anyway.\n");
      }

    } else if ( aantal_tikjes == 2 ) { // when tapped twice, increase by half a degree

      if ( doeltemperatuur < maximum_temperatuur ) {  // as lang as we haven't reached maximum temperature...
        doeltemperatuur += 0.5;       // ... increase goal temperature.
        Serial.print("\nGoal temperature went up by half a degree to ");
        Serial.print(doeltemperatuur);
        Serial.print(".\n");
      } else {
        Serial.print("\nYou just put on a sweater, alright?\n");
      }

    } else if ( aantal_tikjes > 2 ) { // when tapped more than twice, effectively turn the heater on or off.

      if ( doeltemperatuur > 19 ) {
        doeltemperatuur = 15.0;
        Serial.print("\nHeater off. (");
        Serial.print(doeltemperatuur);
        Serial.print(" degrees)\n");
      } else {
        doeltemperatuur = 20.5;
        Serial.print("\nHeater on! (");
        Serial.print(doeltemperatuur);
        Serial.print(" degrees)\n");
      }

    }

    geefTemperatuurWeerOpMeter(1, actuele_temperatuur); // hand the actuel temperature to one gauge
    geefTemperatuurWeerOpMeter(2, doeltemperatuur); // hand goal temperature to the other gauge

    aantal_tikjes = 0; // reset number of taps so we can start a new round

  }

  // when the thermostat has not been touched for more than a minute, we turn off the LEDs
  if ( ((millis() - tijd_sinds_laatste_aanraking) > 60000) && (ledstatus == 1) )  {

    Serial.print("Lampjes weer uit.\n\n\n");

    // turn off LEDs
    digitalWrite(13, LOW);

    ledstatus = 0;

  }

  // reset when there is no touching, so we can start registering new taps
  if (total < drempel) {
    aanraking = false;
  }
  
}
