// Kickstarter Tracker
// Garrett Mace October 2014
// All code is in the public domain

// Check Kickstarter stats once per minute (milliseconds)
#define checkInterval 60000

// Leave light on for ten seconds after new backer celebration (milliseconds)
#define lightDuration 10000


// Include necessary files
#include <Process.h> // Allows Arduino to run a program on the Linux side
#include "pitches.h" // Maps musical notes to Tone values

const byte PowerSwitchTail = 12; // The PowerSwitch Tail is located on pin 12
const byte piezoSpeaker = 10;    // The piezo speaker is located on pin 10

// Musical tones for new backer celebration
// Modified from toneMelody example by Tom Igoe
// notes in the melody:
int melody[] = {
  NOTE_G6, NOTE_G6, NOTE_G6, NOTE_G6, NOTE_DS6, NOTE_F6, NOTE_G6, 0, NOTE_F6, NOTE_G6
};

// note durations: 4 = quarter note, 8 = eighth note, etc.:
int noteDurations[] = {
  8, 8, 8, 4, 4, 4, 8, 8, 8, 2
};

// Strings for processing received console data
String pledgedRawStr = "";
String pledgedFormattedStr = "";
String backersStr = "";
String percentFundedFormatted = "";

// Store last known backer count and the current backer count
int backers = 0;
int oldBackers = 0;

// Initialize I/O pins and libraries
void setup() {

  // Configure outputs
  pinMode(PowerSwitchTail, OUTPUT);
  pinMode(piezoSpeaker, OUTPUT);

  // Initialize outputs
  digitalWrite(PowerSwitchTail, LOW);
  digitalWrite(piezoSpeaker, LOW);

  // Start Bridge (links the Arduino chip to the Linux processor)
  Bridge.begin();

}


unsigned long checkTimer = 0;
unsigned long lightTimer = 0;
boolean lightOn = false;
boolean initProgram = false;

void loop() {

  // Wait for the specified interval and then check Kickstarter
  if (((millis() - checkTimer) > checkInterval) || (initProgram == false)) {
    checkTimer = millis();
    initProgram = true;
    
    runKicktracker();
    if (backers > oldBackers) {
      celebration();
      lightOn = true;
      digitalWrite(PowerSwitchTail, lightOn);
      lightTimer = millis();
    }
  }
  
  // Keep the light on for a while after new backer celebration
  if (((millis() - lightTimer) > lightDuration) && (lightOn == true)) {
    lightOn = false;
    digitalWrite(PowerSwitchTail, lightOn);
  }

}

// Request JSON data from Kickstarter and place into variables the Arduino can use
void runKicktracker() {
  Process p; // Create Process object
  p.begin("python");  // Start Python
  p.addParameter("/root/kicktracker.py"); // Run the kicktracker.py script
  p.addParameter("https://www.kickstarter.com/projects/macetech/rgb-led-shades/stats.json"); // Specify the Kickstarter stats URL
  p.addParameter("12000"); // Specify the Kickstarter goal amount
  p.run(); // Run the process

  while (p.running()); // Wait for all console data

  // initialize parser variables
  byte wordcount = 0;
  pledgedRawStr = "";
  pledgedFormattedStr = "";
  backersStr = "";
  percentFundedFormatted = "";

  // Read incoming bytes and split into strings when forward slash is found
  // (Stream).readStringUntil() does not seem to work here
  while (p.available() > 0) {
    char c = p.read();
    if (wordcount < 5) {
      if (c == '/') {
        wordcount++;
      } else {
        switch (wordcount) {
          case 0:
            pledgedRawStr += c;
            break;

          case 1:
            pledgedFormattedStr += c;
            break;

          case 2:
            backersStr += c;
            break;

          case 3:
            percentFundedFormatted += c;
            break;
        }
      }



    }
  }
      p.flush();
      oldBackers = backers;
      backers = backersStr.toInt();
}

// Modified from toneMelody example by Tom Igoe
// Plays fanfare and toggles light at every note change
void celebration() {
   lightOn = true;

  // iterate over the notes of the melody:
  for (int thisNote = 0; thisNote < 10; thisNote++) {
    digitalWrite(PowerSwitchTail, lightOn);
    lightOn = !lightOn;
    // to calculate the note duration, take one second
    // divided by the note type.
    //e.g. quarter note = 1000 / 4, eighth note = 1000/8, etc.
    int noteDuration = 1000 / noteDurations[thisNote];
    tone(piezoSpeaker, melody[thisNote], noteDuration);

    // to distinguish the notes, set a minimum time between them.
    // the note's duration + 30% seems to work well:
    int pauseBetweenNotes = noteDuration * 1.30;
    delay(pauseBetweenNotes);
    // stop the tone playing:
    noTone(piezoSpeaker);
  }
}



