/// Code by Michael J
/// Please do not re-distribute without credit!

using System;
using System.IO.Ports;
using System.Text;
using Microsoft.SPOT;
using SecretLabs.NETMF.Hardware.Netduino;
using System.Threading;

namespace ioRX
{
    /// <summary>
    /// Original from http://forums.netduino.com/index.php?/topic/366-netduino-serial-port-code-review/
    /// </summary>
    public class SerialPortHelper
    {
        SerialPort serialPort;

        const int bufferMax = 1024;
        byte[] buffer;
        int bufferLength = 0;

        public SerialPortHelper(string portName = SerialPorts.COM1, int baudRate = 9600, Parity parity = Parity.None, int dataBits = 8, StopBits stopBits = StopBits.One, bool listen = true)
        {
            this.serialPort = new SerialPort(portName, baudRate, parity, dataBits, stopBits);
            this.serialPort.ReadTimeout = 10000; // Set to 10ms. Default is -1?!

            if (listen)
            {
                this.buffer = new Byte[bufferMax];
                this.serialPort.DataReceived += new SerialDataReceivedEventHandler(serialPort_DataReceived);
            }

            this.serialPort.Open();
        }

        private void serialPort_DataReceived(object sender, SerialDataReceivedEventArgs e)
        {
            lock (buffer)
            {
                int bytesReceived = 0;

                try
                {
                    bytesReceived = serialPort.Read(buffer, bufferLength, bufferMax - bufferLength);
                }
                catch { }

                if (bytesReceived > 0)
                {
                    bufferLength += bytesReceived;
                    if (bufferLength >= bufferMax)
                    {
                        Debug.Print("Buffer Overflow, clearing buffer...");

                        try
                        {
                            buffer = null;
                            buffer = new Byte[bufferMax];
                            bufferLength = 0;
                        }
                        catch (Exception ex) { Debug.Print("Failed to clear buffer: " + ex.Message.ToString()); }
                    }
                }
            }
        }

        public string ReadLine()
        {
            string line = "";

            lock (buffer)
            {
                //-- Look for Return char in buffer --
                for (int i = 0; i < bufferLength; i++)
                {
                    //-- Consider EITHER CR or LF as end of line, so if both were received it would register as an extra blank line. --
                    if (buffer[i] == '\r' || buffer[i] == '\n')
                    {
                        buffer[i] = 0; // Turn NewLine into string terminator
                        line = "" + new string(Encoding.UTF8.GetChars(buffer)); // The "" ensures that if we end up copying zero characters, we'd end up with blank string instead of null string.
                        bufferLength = bufferLength - i - 1;
                        Array.Copy(buffer, i + 1, buffer, 0, bufferLength); // Shift everything past NewLine to beginning of buffer
                        break;
                    }
                }
            }

            return line;
        }

        public void Print(string line)
        {
            System.Text.UTF8Encoding encoder = new System.Text.UTF8Encoding();
            byte[] bytesToSend = encoder.GetBytes(line);
            serialPort.Write(bytesToSend, 0, bytesToSend.Length);
        }

        public void PrintLine(string line)
        {
            Print(line + "\r");
        }

        public int bufferCount()
        {
            return bufferLength;
        }

    }
}
