/// Code by Michael J
/// Please do not re-distribute without credit!
using System;
using Microsoft.SPOT;
using System.Collections;

namespace ioRX
{
    /// <summary>
    /// Sensor Data Class (holds info on individual sensor)
    /// </summary>
    public class SensorData
    {
        public string sensorName { get; set; }

        public string sensorData { get; set; }

        public string sensorSource { get; set; }

        public DateTime sensorTime { get; set; }

        public SensorData(string sName, string sData, string sSource, DateTime sTime)
        {
            this.sensorName = sName;
            this.sensorData = sData;
            this.sensorSource = sSource;
            this.sensorTime = sTime;
        }
    }

    /// <summary>
    /// SensorStore Class, holds data for all sensors
    /// </summary>
    public class SensorStore
    {
        private Hashtable store = new Hashtable();

        /// <summary>
        /// Updates a sensor's data in the store
        /// </summary>
        /// <param name="sensorName">Name of the Sensor</param>
        /// <param name="sensorData">Data to store</param>
        /// <param name="sensorSource">Source of the sensor</param>
        /// <returns></returns>
        public bool updateSensor(string sensorName, string sensorData, string sensorSource)
        {
            if (sensorName.Length < 0 || sensorData.Length <= 0 || sensorSource.Length <= 0)
            {
                return false;
            }

            lock (store)
            {
                try
                {
                    if (store.Contains(sensorSource + "." + sensorName))
                    {
                        store[sensorSource + "." + sensorName] = new SensorData(sensorName, sensorData, sensorSource, DateTime.Now);

                        return true;
                    }
                    else
                    {
                        store.Add(sensorSource + "." + sensorName, new SensorData(sensorName, sensorData, sensorSource, DateTime.Now));

                        return true;
                    }
                }
                catch { return false; }
            }
        }

        /// <summary>
        /// Get data from a single sensor
        /// </summary>
        /// <param name="sensorName">Name of the sensor</param>
        /// <param name="sensorSource">Source of the sensor</param>
        /// <returns>SensorData object</returns>
        public SensorData getSensorData(string sensorName, string sensorSource)
        {
            lock (store)
            {
                try
                {
                    if (store.Contains(sensorSource + "." + sensorName))
                    {
                        return store[sensorSource + "." + sensorName] as SensorData;
                    }
                }
                catch { }
            }

            return null;
        }

        /// <summary>
        /// Returns all sensor data as a hashtable
        /// </summary>
        /// <param name="clearData">false: keep data in store, true: clear store</param>
        /// <returns></returns>
        public Hashtable getAllSensorData(bool clearData)
        {
            lock (store)
            {
                Hashtable tempStore = new Hashtable();

                tempStore = store.Clone() as Hashtable;

                if (clearData)
                {
                    store.Clear();
                }

                return tempStore;
            }
        }

        /// <summary>
        /// Get the current sensor store count
        /// </summary>
        /// <returns>int</returns>
        public int sensorDataCount()
        {
            lock (store)
            {
                return store.Count;
            }
        }
    }
}
