
//** pxnet.c **//

#include "pxcommon.h"
#include "pxmac.h"
#include "pxnet.h"

#if defined(__XC8) || defined(__PICC__)
#define FIX_XC8_BUGS
#endif

#define ETYPE_IPV4	(0x0800)
#define ETYPE_ARP	(0x0806)
#define ETYPE_802_1Q	(0x8100)

#define IPV4_PROTO_ICMP (1)
#define IPV4_PROTO_TCP 	(6)
#define IPV4_PROTO_UDP 	(17)

#define DEFAULT_TTL	(64)

#define ETH_HEADER_LEN	(14)
#define IPV4_HEADER_LEN	(20)

#define ETH_MTU         (1500)

static uint8_t	tmp[20];

#define DEST_OTHER	(0)
#define DEST_UNICAST	(1)
#define DEST_MULTICAST	(2)
#define DEST_BROADCAST	(3)

typedef struct {
    // from ethernet frame
    TMacAddr	en_dstMac;
    TMacAddr	en_srcMac;
    uint16_t	en_etype;
    uint8_t	en_dest;
    // from IPv4 header
    TIpAddr	ip_dstIp;
    TIpAddr	ip_srcIp;
    uint8_t	ip_ttl;
    uint16_t	ip_len;
} TPacket;


static void setu16( uint8_t* dst, uint16_t u16 )
{
    memcpy( (void*) dst, (void*) &u16, 2 );
}


static void setu32( uint8_t *dst, uint32_t u32 )
{
    memcpy( (void*) dst, (void*) &u32, 4 );
}


static uint16_t getu16( uint8_t *src )
{
    uint16_t result;
    memcpy( (void*) &result, (void*) src, 2 );
    return result;
}


static uint32_t getu32( uint8_t *src )
{
    uint32_t result;
    memcpy( (void*) &result, (void*) src, 4 );
    return result;
}


static uint16_t htons( uint16_t n )
{
    uint16_t r = 0;
    uint8_t *src = (uint8_t*) &n;
    uint8_t *dst = (uint8_t*) &r;
    dst[0] = src[1];
    dst[1] = src[0];
    return r;
}

#define ntohs htons

static uint32_t htonl( uint32_t n )
{
    uint32_t r = 0;
    uint8_t *src = (uint8_t*) &n;
    uint8_t *dst = (uint8_t*) &r;
    dst[0] = src[3];
    dst[1] = src[2];
    dst[2] = src[1];
    dst[3] = src[0];
    return r;
}

#define ntohl htonl


static uint32_t gCalcSum	= 0;
static bool	gEvenByte	= false;


static void clrChkSum( void )
{
    gCalcSum = 0;
    gEvenByte = false;
}

static void addChkSum( const void *dataPtr, uint16_t dataLen )
{
    const uint8_t* ptr = (const uint8_t *) dataPtr;
    while ( dataLen > 0 ) {
	if ( gEvenByte )
	    gCalcSum += 256U * *ptr;
	else
	    gCalcSum += *ptr;
	ptr++;
	dataLen--;
	gEvenByte = ! gEvenByte;
    }
}

static uint16_t calcChkSum( void )
{
    while ( gCalcSum >> 16 )
	gCalcSum = ( gCalcSum & 0xFFFF ) + ( gCalcSum >> 16 );
    return ~gCalcSum;
}


uint16_t cksum( const void *src, int len )
{
    clrChkSum();
    addChkSum( src, len );
    return calcChkSum();
}


static void addEthHeader( char* dst, uint16_t etype )
{
    memset( tmp, 0, 14 );

#ifdef FIX_XC8_BUGS
    // NOTE: Workaround XC8 PIC16F pointer bug
    tmp[0] = dst[0];
    tmp[1] = dst[1];
    tmp[2] = dst[2];
    tmp[3] = dst[3];
    tmp[4] = dst[4];
    tmp[5] = dst[5];

    tmp[6]  = gMyMAC[0];
    tmp[7]  = gMyMAC[1];
    tmp[8]  = gMyMAC[2];
    tmp[9]  = gMyMAC[3];
    tmp[10] = gMyMAC[4];
    tmp[11] = gMyMAC[5];
#else
    memcpy( tmp, dst, kMACLEN );
    memcpy( tmp + 6, gMyMAC, kMACLEN );
#endif

    setu16( tmp + 12, htons( etype ) );
    macTxWrite( tmp, 14 );
}


static void handleARP( TPacket* pkt )
{
    // htype=1 ptype=0x0800 hlen=6 plen=4 op=1
    uint8_t arpmsg[] = { 0x00, 0x01, 0x08, 0x00, 0x06, 0x04, 0x00, 0x01 };
    TMacAddr SHA;

    // ARP request header - 28 bytes
    // check first 8 bytes
    macRxRead( tmp, 8 );

    // Ignore non IPv4 ARP requests
    if ( memcmp( tmp, arpmsg, sizeof( arpmsg ) ) ) return;

    // next 20 bytes of header
    // SHA=tmp+0, SPA=tmp+6, THA=tmp+10, TPA=tmp+16
    macRxRead( tmp, 20 );

    memcpy( SHA, tmp + 0, kMACLEN );

    // Ignore requests that aren't for our IP
    if ( memcmp( tmp + 16, gMyIP, sizeof( TIpAddr ) ) ) return;

    macTxBegin();

    // ethernet header
    addEthHeader( SHA, ETYPE_ARP );

    // ARP reply message is 18 bytes
    memset( tmp, 0, 18 );
    memcpy( tmp, arpmsg, 8 );
    tmp[7] = 2; // ARP reply
    memcpy( tmp + 8, gMyMAC, kMACLEN );
    memcpy( tmp + 14, gMyIP, kIPLEN );
    macTxWrite( tmp, 18 );

    macTxSetLen( ETH_HEADER_LEN + 18 );
    macTxSend();
}


static void handleICMP( TPacket* pkt )
{
    uint8_t type, code;
    uint16_t id, seq, left;
    uint16_t icmp_len = pkt->ip_len;

    // read ICMP header - 8 bytes
    memset( tmp, 0, 8 );
    macRxRead( tmp, 8 );

    type = tmp[0];
    code = tmp[1];
    id = ntohs( getu16( tmp + 4 ) );
    seq = ntohs( getu16( tmp + 6 ) );

    if ( type != 8 ) return;

    left = icmp_len - 8;

    macTxBegin();

    // ethernet header
#ifdef FIX_XC8_BUGS
    // NOTE: Workaround XC8 PIC16F pointer bug
    TMacAddr pkt_en_srcMac;
    pkt_en_srcMac[0] = pkt->en_srcMac[0];
    pkt_en_srcMac[1] = pkt->en_srcMac[1];
    pkt_en_srcMac[2] = pkt->en_srcMac[2];
    pkt_en_srcMac[3] = pkt->en_srcMac[3];
    pkt_en_srcMac[4] = pkt->en_srcMac[4];
    pkt_en_srcMac[5] = pkt->en_srcMac[5];
    addEthHeader( pkt_en_srcMac, ETYPE_IPV4 );
#else
    addEthHeader( pkt->en_srcMac, ETYPE_IPV4 );
#endif

    // IPv4 header - 20 bytes
    memset( tmp, 0, 20 );
    tmp[0] = 0x45; // ver=4 ihl=5
    setu16( tmp + 2, htons( 20 + icmp_len ) );
    tmp[6] = 0x40; // DF flag
    tmp[8] = DEFAULT_TTL;
    tmp[9] = IPV4_PROTO_ICMP;
    memcpy( tmp + 12, gMyIP, kIPLEN );

#ifdef FIX_XC8_BUGS
    // NOTE: Workaround XC8 PIC16F pointer bug
    tmp[16] = pkt->ip_srcIp[0];
    tmp[17] = pkt->ip_srcIp[1];
    tmp[18] = pkt->ip_srcIp[2];
    tmp[19] = pkt->ip_srcIp[3];
#else
    memcpy( tmp + 16, pkt->ip_srcIp, kIPLEN );
#endif

    setu16( tmp + 10, cksum( tmp, 20 ) );
    macTxWrite( tmp, 20 );

    // copy remaining ICMP message from Rx to Tx
    macTxSetPtr( ETH_HEADER_LEN + 20 + 8 );
    clrChkSum();
    while ( left > 0 ) {
	uint16_t bytes = left;
	if ( bytes > 20 ) bytes = 20;
	macRxRead( tmp, bytes );
	macTxWrite( tmp, bytes );
	addChkSum( tmp, bytes );
	left -= bytes;
    }

    // write new ICMP header
    memset( tmp, 0, 8 );
    setu16( tmp + 4, htons( id ) );
    setu16( tmp + 6, htons( seq ) );
    addChkSum( tmp, 8 );
    setu16( tmp + 2, calcChkSum() );
    macTxSetPtr( ETH_HEADER_LEN + 20 );
    macTxWrite( tmp, 8 );

    macTxSetLen( ETH_HEADER_LEN + 20 + icmp_len );
    macTxSend();
}


#define TCP_FLAG_FIN	(0x01)
#define TCP_FLAG_SYN	(0x02)
#define TCP_FLAG_RST	(0x04)
#define TCP_FLAG_PSH	(0x08)
#define TCP_FLAG_ACK	(0x10)


typedef enum {
    CLOSED,
    SYN_RECEIVED,
    ESTABLISHED,
    FIN_WAIT_1,
    FIN_WAIT_2,
    CLOSE_WAIT,
    CLOSING,
    LAST_ACK,
    TIME_WAIT
} TcpState;


#define CBF_CONNECT     (0x01)
#define CBF_DISCONNECT  (0x02)

typedef struct {
    TcpState	state;
    uint16_t	locPort;
    uint16_t	remPort;
    TMacAddr	remMac;
    TIpAddr	remIp;
    uint32_t	remSeqNo;
    uint32_t	locSeqNo;
    uint8_t	timer;
    uint8_t     callbacks;
} TSocket;


static TSocket 	gSockets[ kMAXSOCKETS ];
static bool	gInOnData  = false;
static uint8_t	gInSock	   = 0;


static int8_t findSocket( uint16_t locPort, uint16_t remPort, char* remIp )
{
    sock_t sock;

    for ( sock = 0; sock < kMAXSOCKETS; sock++ ) {
	TSocket *srec = gSockets + sock;

#ifdef FIX_XC8_BUGS
        // NOTE: work around XC8/PIC16F pointer bug
        TIpAddr srec_remIp;
        srec_remIp[0] = srec->remIp[0];
        srec_remIp[1] = srec->remIp[1];
        srec_remIp[2] = srec->remIp[2];
        srec_remIp[3] = srec->remIp[3];
	if ( ( srec->locPort == locPort ) &&
	     ( srec->remPort == remPort ) &&
	     ( memcmp( srec_remIp, remIp, kIPLEN ) == 0 ) )
	    return sock;
#else
	if ( ( srec->locPort == locPort ) &&
	     ( srec->remPort == remPort ) &&
	     ( memcmp( srec->remIp, remIp, kIPLEN ) == 0 ) )
	    return sock;
#endif
    }
    return -1;
}


static int8_t newSocket( void )
{
    sock_t sock;

    for ( sock = 0; sock < kMAXSOCKETS; sock++ )
	if ( gSockets[sock].state == CLOSED )
	    return sock;
    return -1;
}


static void callDisconnect( sock_t sock, TSocket *srec )
{
    if ( srec->callbacks & CBF_CONNECT )
        if ( ( srec->callbacks & CBF_DISCONNECT ) == 0 ) {
            onTcpDisconnect( sock );
            srec->callbacks |= CBF_DISCONNECT;
        }
}


static void closeSocket( sock_t sock )
{
    TSocket *srec = gSockets + sock;
    srec->state = CLOSED;
    callDisconnect( sock, srec );
}


static uint16_t gTcpTxBytes   = 0;


static void tcpTxInit( TSocket *srec )
{
    macTxBegin();

    // ethernet header
    addEthHeader( srec->remMac, ETYPE_IPV4 );

    // setup for tcpSendWrite()
    macTxSetPtr( ETH_HEADER_LEN + 20 + 20 );
    clrChkSum();
    gTcpTxBytes = 0;

}


static void tcpTxAppend( TSocket *srec, const void *dataPtr, uint16_t dataLen )
{
    macTxWrite( dataPtr, dataLen );
    addChkSum( dataPtr, dataLen );
    gTcpTxBytes += dataLen;
}


static uint16_t tcpTxFinalize( TSocket *srec, uint8_t flags )
{
    uint16_t wsize;

    // Pseudo-IP-header for TCP checksum - 12 bytes
    memset( tmp, 0, 12 );
    memcpy( tmp, gMyIP, kIPLEN );
    memcpy( tmp + 4, srec->remIp, kIPLEN );
    tmp[9] = IPV4_PROTO_TCP;
    setu16( tmp + 10, htons( 20 + gTcpTxBytes ) );

    gEvenByte = false; // in case dataLen is odd
    addChkSum( tmp, 12 );

    // TCP Header - 20 bytes
    memset( tmp, 0, 20 );
    setu16( tmp, htons( srec->locPort ) );
    setu16( tmp + 2, htons( srec->remPort ) );
    setu32( tmp + 4, htonl( srec->locSeqNo ) );
    setu32( tmp + 8, htonl( srec->remSeqNo ) );
    tmp[ 12 ] = 0x50; // header size = 5 x 32 bits = 20 bytes
    tmp[ 13 ] = flags;

    // Receive window = MAC Free buffer size - TCP header (20)
    //                - IP header (20) - Ethernet header (14)

    // limit one segment at a time from remote host to avoid retries

    wsize = macRxGetBufferFree();
    if ( wsize > ETH_MTU ) wsize = ETH_MTU;
    if ( wsize > 54 ) wsize -= 54;
    else wsize = 0;
    setu16( tmp + 14, htons( wsize ) );

    // Calculate TCP checksum
    addChkSum( tmp, 20 );
    setu16( tmp + 16, calcChkSum() );

    // write TCP header
    macTxSetPtr( ETH_HEADER_LEN + 20 );
    macTxWrite( tmp, 20 );

    // IPv4 header - 20 bytes
    memset( tmp, 0, 20 );
    tmp[0] = 0x45; // ver=4 ihl=5
    setu16( tmp + 2, htons( 20 + 20 + gTcpTxBytes ) );
    tmp[6] = 0x40; // DF flag
    tmp[8] = DEFAULT_TTL;
    tmp[9] = IPV4_PROTO_TCP;
    memcpy( tmp + 12, gMyIP, kIPLEN );
    memcpy( tmp + 16, srec->remIp, kIPLEN );
    setu16( tmp + 10, cksum( tmp, 20 ) );

    // write IP header
    macTxSetPtr( ETH_HEADER_LEN );
    macTxWrite( tmp, 20 );

    macTxSetLen( ETH_HEADER_LEN + 20 + 20 + gTcpTxBytes );
    macTxSend();

    return gTcpTxBytes;
}


static void sendTcpPacket( TSocket *srec, uint8_t flags,
    const void *dataPtr, uint16_t dataLen )
{
    tcpTxInit( srec );
    if ( dataPtr && dataLen )
        tcpTxAppend( srec, dataPtr, dataLen );
    tcpTxFinalize( srec, flags );
}


static void handleTCP( TPacket* pkt )
{
    uint8_t offset, flags, options;
    uint16_t srcPort, dstPort, wsize;
    uint32_t seqNo, ackNo;
    sock_t sock;

    // TCP header without options is 20 bytes
    macRxRead( tmp, 20 );

    srcPort = ntohs( getu16( tmp ) );
    dstPort = ntohs( getu16( tmp + 2 ) );
    seqNo = ntohl(  getu32( tmp + 4 ) );
    ackNo = ntohl(  getu32( tmp + 8 ) );

    offset = tmp[12] >> 4;
    flags = tmp[13];
    wsize = ntohs( getu16( tmp + 14 ) );

    // discard options
    options = offset - 5;
    while ( options > 0 ) {
	macRxRead( tmp, 4 );
	options--;
    }

#ifdef FIX_XC8_BUGS
    // NOTE: work around XC8/PIC16F pointer bug
    TIpAddr pkt_ip_srcIp;
    TMacAddr pkt_en_srcMac;

    pkt_ip_srcIp[0] = pkt->ip_srcIp[0];
    pkt_ip_srcIp[1] = pkt->ip_srcIp[1];
    pkt_ip_srcIp[2] = pkt->ip_srcIp[2];
    pkt_ip_srcIp[3] = pkt->ip_srcIp[3];

    pkt_en_srcMac[0] = pkt->en_srcMac[0];
    pkt_en_srcMac[1] = pkt->en_srcMac[1];
    pkt_en_srcMac[2] = pkt->en_srcMac[2];
    pkt_en_srcMac[3] = pkt->en_srcMac[3];
    pkt_en_srcMac[4] = pkt->en_srcMac[4];
    pkt_en_srcMac[5] = pkt->en_srcMac[5];

    sock = findSocket( dstPort, srcPort, pkt_ip_srcIp );
#else
    sock = findSocket( dstPort, srcPort, pkt->ip_srcIp );
#endif

    if ( ( flags & TCP_FLAG_SYN ) && ( sock < 0 ) ) {
	sock = newSocket();
	if ( sock >= 0 ) {
	    TSocket *srec = gSockets + sock;
	    srec->state = SYN_RECEIVED;
            srec->callbacks = 0;
            srec->timer = 0;
	    srec->locPort = dstPort;
	    srec->remPort = srcPort;
#ifdef FIX_XC8_BUGS
	    memcpy( srec->remIp, pkt_ip_srcIp, kIPLEN );
	    memcpy( srec->remMac, pkt_en_srcMac, kMACLEN );
#else
	    memcpy( srec->remIp, pkt->ip_srcIp, kIPLEN );
	    memcpy( srec->remMac, pkt->en_srcMac, kMACLEN );
#endif
	    srec->remSeqNo = seqNo + 1;
	    // send SYN-ACK
	    sendTcpPacket( srec, TCP_FLAG_SYN | TCP_FLAG_ACK, NULL, 0 );
	    srec->locSeqNo++;
	} else {
            TSocket srec;
            srec.locPort = dstPort;
            srec.remPort = srcPort;
#ifdef FIX_XC8_BUGS
	    memcpy( srec.remIp, pkt_ip_srcIp, kIPLEN );
	    memcpy( srec.remMac, pkt_en_srcMac, kMACLEN );
#else
	    memcpy( srec.remIp, pkt->ip_srcIp, kIPLEN );
	    memcpy( srec.remMac, pkt->en_srcMac, kMACLEN );
#endif
	    srec.remSeqNo = seqNo;
	    sendTcpPacket( &srec, TCP_FLAG_RST, NULL, 0 );
        }
    } else if ( sock >= 0 ) {
	TSocket *srec = gSockets + sock;
        uint16_t dlen;
        uint8_t ackflags, delta;

	// ignore any packets if we're in TIME_WAIT
	if ( srec->state == TIME_WAIT )
	    return;

	srec->timer = 0;

        if ( flags & TCP_FLAG_RST ) {
            closeSocket( sock );
            return;
        }

        if ( flags & TCP_FLAG_ACK ) {
            // TODO: compare ackNo with locSeqNo
            if ( srec->state == SYN_RECEIVED )
		srec->state = ESTABLISHED;
        }

	dlen = pkt->ip_len - ( offset * 4 );
	ackflags = TCP_FLAG_ACK;
	delta = 0;

	if ( ( srec->state == LAST_ACK ) &&
	     ( flags & TCP_FLAG_ACK ) ) {
            closeSocket( sock );
        } else if ( srec->state != CLOSED ) {
	    if ( srec->state == ESTABLISHED )
                if ( ( flags & TCP_FLAG_FIN ) ||
                     ( flags & TCP_FLAG_RST ) ) {
                    ackflags |= TCP_FLAG_FIN;
                    srec->state = LAST_ACK;
                    delta++;
                }
	    if ( ( srec->state == FIN_WAIT_1 ) ||
		 ( srec->state == FIN_WAIT_2 ) ) {
		if ( flags & TCP_FLAG_FIN ) {
		    srec->state = TIME_WAIT;
                    callDisconnect( sock, srec );
		    delta = 1;
		}
		if ( flags & TCP_FLAG_ACK ) {
		    if ( srec->state != TIME_WAIT ) {
			srec->state = FIN_WAIT_2;
			return;
		    }
		}
	    }

	    srec->remSeqNo = seqNo + dlen + delta;
            sendTcpPacket( srec, ackflags, NULL, 0 );

	    if ( dlen > 0 ) {
		gInOnData = true;
		gInSock = sock;
		onTcpData( sock, dlen );
		gInOnData = false;
	    }

	    if ( srec->state == ESTABLISHED )
                if ( ( srec->callbacks & CBF_CONNECT ) == 0 ) {
                    onTcpConnect( sock, srec->locPort );
                    srec->callbacks |= CBF_CONNECT;
                }
	}
    }
}


static void handleIPV4( TPacket* pkt )
{
    uint8_t version, ihl, protocol;
    uint16_t chksum;
    int8_t options;


    // IPv4 header without options is 20 bytes
    macRxRead( tmp, 20 );

    version = tmp[0] >> 4;
    ihl = tmp[0] & 0x0F;
    pkt->ip_len = ntohs( getu16( tmp + 2 ) );

    pkt->ip_ttl = tmp[8];
    protocol = tmp[9];
    chksum = getu16( tmp + 10 );

    pkt->ip_len -= ( ihl * 4 );

    if ( version != 4 )
	return;

    memcpy( pkt->ip_srcIp, tmp + 12, kIPLEN );
    memcpy( pkt->ip_dstIp, tmp + 16, kIPLEN );

    // discard options
    options = ihl - 5;
    while ( options > 0 ) {
	macRxRead( tmp, 4 );
	options--;
    }

    if ( protocol == IPV4_PROTO_ICMP )
	handleICMP( pkt );
    else if ( protocol == IPV4_PROTO_TCP ) {
	if ( pkt->en_dest == DEST_UNICAST )
	    handleTCP( pkt );
    }
}


static void handlePacket( void )
{
    uint8_t bcMAC[kMACLEN] = { 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };
    uint8_t mcMAC[] = { 0x01, 0x00, 0x5E };
    TPacket pkt;

    // Ethernet Frame header
    // Dst MAC, Src MAC, EtherType
    // 6 + 6 + [4] + 2 = 16

    macRxRead( tmp, 14 );

    memcpy( pkt.en_dstMac, tmp, kMACLEN );
    memcpy( pkt.en_srcMac, tmp + 6, kMACLEN );
    pkt.en_etype = ntohs( getu16( tmp + 12 ) );

    // handle 802.1Q frame
    if ( pkt.en_etype == ETYPE_802_1Q ) {
	macRxRead( tmp, 4 );
	pkt.en_etype = ntohs( getu16( tmp + 2 ) );
    }


    pkt.en_dest = DEST_OTHER;
    if ( memcmp( pkt.en_dstMac, gMyMAC, kMACLEN ) == 0 )
	pkt.en_dest = DEST_UNICAST;
    else if ( memcmp( pkt.en_dstMac, bcMAC, kMACLEN ) == 0 )
	pkt.en_dest = DEST_BROADCAST;
    else if ( memcmp( pkt.en_dstMac, mcMAC, 3 ) == 0 )
	pkt.en_dest = DEST_MULTICAST;

    if ( pkt.en_dest ) {
	if ( pkt.en_etype == ETYPE_IPV4 )
	    handleIPV4( &pkt );
	else if ( pkt.en_etype == ETYPE_ARP )
	    handleARP( &pkt );
    }
}


void netInit( void )
{
    memset( &gSockets, 0, sizeof( gSockets ) );
    macInit();
}


void netEnable( void )
{
    macEnable();
}


void netIdle( void )
{
    uint8_t count = macRxPoll();
    if ( count > 0 ) {
	macRxBegin();
	handlePacket();
	macRxEnd();
    }
}


void netTimer( uint32_t seconds )
{
    sock_t sock;
    // expire any inactive sockets and sockets in TIME_WAIT
    for ( sock = 0; sock < kMAXSOCKETS; sock++ ) {
	TSocket *srec = gSockets + sock;
	if ( srec->state != CLOSED ) {
            srec->timer++;
            if ( srec->timer >= kINACTIVE_SECS ) {
                sendTcpPacket( srec, TCP_FLAG_RST, NULL, 0 );
                closeSocket( sock );
            } else if ( srec->state != ESTABLISHED ) {
                if ( srec->timer >= kTIME_WAIT_SECS )
                    closeSocket( sock );
            }
        }
    }
}


void netRead( sock_t sock, uint8_t* dst, uint16_t bytes )
{
    if ( gInOnData )
	if ( gInSock == sock ) {
	    TSocket *srec = gSockets + sock;
	    if ( srec->state == ESTABLISHED )
		macRxRead( dst, bytes );
	}
}


void netClose( sock_t sock )
{
    TSocket *srec = gSockets + sock;
    if ( srec->state == ESTABLISHED ) {
	sendTcpPacket( srec, TCP_FLAG_FIN | TCP_FLAG_ACK, NULL, 0 );
	srec->state = FIN_WAIT_1;
	srec->locSeqNo++;
    }
}


void netGetAddr( int8_t sock, uint16_t* port, uint8_t* ip, uint8_t* mac, uint8_t* state )
{
    TSocket *srec = gSockets + sock;
    *port = srec->remPort;
    memcpy( ip, srec->remIp, kIPLEN );
    memcpy( mac, srec->remMac, kMACLEN );
    *state = srec->state;
}


void netPutBegin( sock_t sock )
{
    TSocket *srec = gSockets + sock;
    tcpTxInit( srec );
}


void netPutAppend( sock_t sock, const uint8_t *src, uint16_t bytes )
{
    TSocket *srec = gSockets + sock;

    const uint8_t *ptr = (const uint8_t *) src;
    int16_t left = bytes;
#ifdef FIX_XC8_BUGS
    char buffer[20];
#endif

    while ( left > 0 ) {
        int16_t free = ETH_MTU - gTcpTxBytes - 54;
        uint16_t count = left;
#ifdef FIX_XC8_BUGS
        if ( count > sizeof( buffer ) )
            count = sizeof( buffer );
#endif
        if ( count > free )
            count = free;

#ifdef FIX_XC8_BUGS
        // NOTE: work around XC8/PIC18F pointer bug
        memcpy( buffer, ptr, count );
        tcpTxAppend( srec, buffer, count );
#else
        tcpTxAppend( srec, ptr, count );
#endif

        left -= count;
        free -= count;
        ptr  += count;

        if ( free <= 0 ) {
            netPutFinalize( sock );
            netPutBegin( sock );
        }
    }
}


void netPutFinalize( sock_t sock )
{
    TSocket *srec = gSockets + sock;
    uint16_t bytes = tcpTxFinalize( srec, TCP_FLAG_PSH | TCP_FLAG_ACK );
    srec->locSeqNo += bytes;
}


void netWrite( sock_t sock, const uint8_t *src, uint16_t bytes )
{
    netPutBegin( sock );
    netPutAppend( sock, src, bytes );
    netPutFinalize( sock );
}
