
//** main.c **//

#include "pxcommon.h"
#include "pxenc.h"
#include "pxnet.h"


////////////////////////////////////////////////////////////////////////////////
//
// Define MAC and static IP address for this node
//
// MAC addresses starting with 80:80:80 are reserved and won't conflict
// with equipment manufacturers
//

TIpAddr  gMyIP  = { 192, 168, 1, 111 };
TMacAddr gMyMAC = { 0x80, 0x80, 0x80, 0x00, 0x00, 0xA1 };


// include Telnet 'netstat' command
#define HAS_NET_COMMAND

// include basic HTTP server
#define HAS_HTTP_SERVER


////////////////////////////////////////////////////////////////////////////////
//
// Telnet & HTTP Handler
//

#define CL_NONE 	(0)
#define CL_TELNET	(1)
#define CL_HTTP         (2)
#define CL_STREAM       (3)

#define CRLF            "\r\n"

// telnet command line buffers
#define kMAXBUFFERS     (4)
#define kBUFFERLEN      (20)


typedef struct {
    bool    used;
    uint8_t bytes;
    char    data[kBUFFERLEN];
} TBuffer;

static TBuffer gBuffers[kMAXBUFFERS];


static void initBuffers( void )
{
    memset( &gBuffers, 0, sizeof( gBuffers ) );
}


static void getBuffer( int8_t* bidx )
{
    int8_t i;

    for ( i = 0; i < kMAXBUFFERS; i++ ) {
        TBuffer *brec = gBuffers + i;
        if ( ! brec->used ) {
            brec->used = true;
            *bidx = i;
            return;
        }
    }
    *bidx = -1;
}


static void freeBuffer( int8_t bidx )
{
    TBuffer *brec = gBuffers + bidx;
    brec->used = false;
    brec->bytes = 0;
}


typedef struct {
    uint8_t	type;
    int8_t      sock;
    int8_t	buffer;
} TClient;


static TClient  gClients[ kMAXSOCKETS ];

static void initClients( void )
{
    int8_t i;

    memset( &gClients, 0, sizeof( gClients ) );
    for ( i = 0; i < kMAXSOCKETS; i++ ) {
        gClients[i].sock = -1;
        gClients[i].buffer = -1;
    }
}


typedef struct {
#ifdef __18CXX
    const far rom char *name;
    volatile near unsigned char *address;
#else
    const char* name;
    volatile unsigned char *address;
#endif
} TRegister;


static TRegister gRegisters[] = {
{ "PORTA" , &PORTA },
{ "LATA"  , &LATA  },
{ "TRISA" , &TRISA },
{ "PORTB" , &PORTB },
{ "LATB"  , &LATB  },
{ "TRISB" , &TRISB },
{ "PORTC" , &PORTC },
{ "LATC"  , &LATC  },
{ "TRISC" , &TRISC },
#if defined(PORTD)
{ "PORTD" , &PORTD },
{ "LATD"  , &LATD  },
{ "TRISD" , &TRISD },
#endif
{ "OSCCON", &OSCCON },
{ "INTCON", &INTCON }
};

const uint8_t gNumRegisters = sizeof( gRegisters ) / sizeof( TRegister );


void dumpAll( sock_t sock )
{
    char s[20] = { 0 };
    int8_t i;

    netPutBegin( sock );

    for ( i = 0; i < gNumRegisters; i++ ) {
        TRegister *rrec = &gRegisters[i];
        uint8_t j, mask = 0x80;

        strcpypgm2ram( s, rrec->name );
        while ( strlen( s ) < 7 ) strcatpgm2ram( s, " " );
        for ( j = 0; j < 8; j++ ) {
            if ( (*rrec->address) & mask )
                strcatpgm2ram( s, "1" );
            else
                strcatpgm2ram( s, "." );
            mask >>= 1;
        }
        strcatpgm2ram( s, CRLF );
        netPutAppend( sock, s, strlen( s ) );
    }
    netPutFinalize( sock );
}


static int8_t strncasecmpram( const char *s1, const char *s2, uint8_t n )
{
    int8_t i;

    for ( i = 0; i < n; i++ ) {
        int8_t c = toupper( *s1++ ) - toupper( *s2++ );
        if ( c ) return c;
    }
    return 0;
}


#ifdef __18CXX
static int8_t strncasecmppgm( const char *s1, const far rom char *s2, uint8_t n )
{
    int8_t i;

    for ( i = 0; i < n; i++ ) {
        int8_t c = toupper( *s1++ ) - toupper( *s2++ );
        if ( c ) return c;
    }
    return 0;
}
#else
#define strncasecmppgm strncasecmpram
#endif


static void soutram( sock_t sock, const char* s )
{
    netPutAppend( sock, s, strlen( s ) );
}


#ifdef __18CXX
static void soutpgm( sock_t sock, const far rom char* s )
{
    uint8_t buffer[8];
    uint8_t count = 0;
    while ( *s ) {
        buffer[count++] = *s;
        if ( count == sizeof( buffer ) ) {
            netPutAppend( sock, buffer, count );
            count = 0;
        }
        s++;
    }
    if ( count )
        netPutAppend( sock, buffer, count );
}
#else
#define soutpgm soutram
#endif


bool checkCommand( sock_t sock, const char* cmd )
{
    int8_t i;

    int cmdlen = strlen( cmd );
    char *space = strchr( cmd, ' ' );
    if ( space == NULL )
        return false;

    for ( i = 0; i < gNumRegisters; i++ ) {
        TRegister *rrec = &gRegisters[i];
        int len = strlenpgm( rrec->name );
        if ( cmdlen > len )
            if ( strncasecmppgm( cmd, rrec->name, len ) == 0 ) {
                if ( isdigit( cmd[len] ) ) {
                    int b = atoi( cmd + len );
                    int v = atoi( space + 1 );
                    uint8_t mask = 0x01 << b;
                    MASKIN( *rrec->address, mask, v ? 0xFF : 0x00 );
                    return true;
                }
            }
    }
    return false;
}


#ifdef HAS_NET_COMMAND

#ifdef __18CXX
const far rom char* TcpStateNames[] = {
#else
const char* TcpStateNames[] = {
#endif
    "CLOSED",
    "SYN_RECEIVED",
    "ESTABLISHED",
    "FIN_WAIT_1",
    "FIN_WAIT_2",
    "CLOSE_WAIT",
    "CLOSING",
    "LAST_ACK",
    "TIME_WAIT"
};


void dumpConnections( sock_t sock )
{
    char s[20] = { 0 };
    sock_t sidx;

    netPutBegin( sock );

    for ( sidx = 0; sidx < kMAXSOCKETS; sidx++ ) {
        uint16_t port = 0;
        TIpAddr ip;
        TMacAddr mac;
        uint8_t state;
        netGetAddr( sidx, &port, ip, mac, &state );
        sprintf( s, "[%2d] %5u", sidx, port );
        soutram( sock, s );
        sprintf( s , "  %3d.%3d.%3d.%3d", ip[0], ip[1], ip[2], ip[3] );
        soutram( sock, s );
        sprintf( s , "  %02X:%02X:%02X:%02X:%02X:%02X",
            mac[0], mac[1], mac[2], mac[3], mac[4], mac[5] );
        soutram( sock, s );
        sprintf( s, " %d-", state );
        soutram( sock, s );
        soutpgm( sock, TcpStateNames[ state ] );
        soutpgm( sock, CRLF );
    }

    netPutFinalize( sock );
}

#endif


static void handleTelnet( TClient *crec, uint8_t sock )
{
    const char prompt[] = "> ";
    const char* cmd = gBuffers[ crec->buffer ].data;

    // quit     - quit session
    // ?        - list SFR values
    // SFRn 0|1 - set bit n of SFR
    // net      - list sockets

    if ( strncasecmppgm( cmd, "quit", 4 ) == 0 ) {
	netClose( sock );
        crec->type = CL_NONE;
	return;
    }
    if ( cmd[0] == '?' ) {
        dumpAll( sock );
    } else if ( checkCommand( sock, cmd ) )
        ;
    else if ( strncasecmppgm( cmd, "net", 3 ) == 0 ) {
#ifdef HAS_NET_COMMAND
        dumpConnections( sock );
#endif
    } else if ( cmd[0] ) {
        const char msg[] = "what?" CRLF;
        netWrite( sock, msg, strlen( msg ) );
    }
    // send prompt
    netWrite( sock, prompt, strlen( prompt ) );
}


#ifdef HAS_HTTP_SERVER

static void handleHTTP( TClient *crec, uint8_t sock )
{
    char s[40] = { 0 };
    int8_t i;
    const char* cmd = gBuffers[ crec->buffer ].data;

    if ( strncasecmppgm( cmd, "GET ", 4 ) ) return;

    netPutBegin( sock );

    soutpgm( sock, "HTTP/1.0 200 OK" CRLF );
    soutpgm( sock, CRLF );
    
    soutpgm( sock, "<html>" CRLF );
    soutpgm( sock, "<head>" CRLF );
    soutpgm( sock, "<title>" CRLF );
    soutpgm( sock, "PIC Status Page" CRLF );
    soutpgm( sock, "</title>" CRLF );
    soutpgm( sock, "</head>" CRLF );
    soutpgm( sock, "<body>" CRLF );
    soutpgm( sock, "<pre>" CRLF );

    for ( i = 0; i < gNumRegisters; i++ ) {
        TRegister *rrec = &gRegisters[i];
        uint8_t j, mask = 0x80;
        strcpypgm2ram( s, rrec->name );
        while ( strlen( s ) < 7 ) strcatpgm2ram( s, " " );
        for ( j = 0; j < 8; j++ ) {
            if ( (*rrec->address) & mask )
                strcatpgm2ram( s, "1" );
            else
                strcatpgm2ram( s, "." );
            mask >>= 1;
        }
        strcatpgm2ram( s, CRLF "<BR>" );
        soutram( sock, s );
    }

    soutpgm( sock, "</pre>" CRLF );

    netPutFinalize( sock );

    netClose( sock );
    crec->type = CL_NONE;
}

#endif


void tcpStream( uint16_t seconds )
{
    char s[10] = { 0 };
    int8_t sock;

    sprintf( s, "%d" CRLF, seconds );
    for ( sock = 0; sock < kMAXSOCKETS; sock++ ) {
        TClient *crec = gClients + sock;
        if ( crec->type == CL_STREAM ) {
            netWrite( sock, s, strlen( s ) );
        }
    }
}


////////////////////////////////////////////////////////////////////////////////
//
// Callbacks from pxnet TCP sockets
//

void     onTcpConnect( sock_t sock, uint16_t port )
{
    TClient *crec = gClients + sock;

    if ( port == 23 ) {
        const char msg[] = "Welcome" CRLF "> ";
        crec->type = CL_TELNET;
        netWrite( sock, msg, strlen( msg ) );
    }

    if ( port == 80 ) {
        crec->type = CL_HTTP;
    }

    if ( port == 8080 ) {
        crec->type = CL_STREAM;
    }

    if ( ! crec->type ) {
        netClose( sock );
    }
}


void     onTcpData( sock_t sock, uint16_t bytes )
{
    TBuffer *brec;
    TClient *crec = gClients + sock;
    uint16_t i;

    if ( crec->buffer < 0 )
        getBuffer( &crec->buffer );

    if ( crec->buffer < 0 ) {
        netClose( sock );
        crec->type = CL_NONE;
        return;
    }

    brec = gBuffers + crec->buffer;

    for ( i = 0; ( i < bytes ) && crec->type; i++ ) {
	uint8_t ch;
	netRead( sock, &ch, 1 );
        // ignore CR
        if ( ch != '\r' ) {
            if ( ch == '\n' ) {
                brec->data[ brec->bytes ] = 0;
		if ( crec->type == CL_TELNET )
                    handleTelnet( crec, sock );
#ifdef HAS_HTTP_SERVER
                else if ( crec->type == CL_HTTP )
                    handleHTTP( crec, sock );
#endif
		brec->bytes = 0;
            } else if ( brec->bytes < ( kBUFFERLEN - 2 ) ) {
                brec->data[ brec->bytes++ ] = ch;
            }
        }
        if ( ( brec->bytes == 3 ) &&
             ( brec->data[0] == 0xFF ) ) {
            brec->bytes = 0;
        }
    }

    if ( brec->bytes == 0 ) {
        freeBuffer( crec->buffer );
        crec->buffer = -1;
    }
}


void     onTcpDisconnect( sock_t sock )
{
    TClient *crec = gClients + sock;
    crec->type = CL_NONE;
    if ( crec->buffer >= 0 ) {
        freeBuffer( crec->buffer );
        crec->buffer = -1;
    }
}


////////////////////////////////////////////////////////////////////////////////
//
// Main
//

#define T1PERSEC    (_XTAL_FREQ/2500000UL)


void initialize( void )
{
    uint8_t i;

#if defined(_16F1938)
    OSCCONbits.IRCF = 0b1110; // 8MHz
    OSCCONbits.SPLLEN = 1;    // 4xPLL 32MHz
    ANSELA = 0x00;
    ANSELB = 0x00;
#endif

#if defined(_18F46K80) || defined(_18F25K80)
    OSCCONbits.IRCF = 0b110; // 8MHz
    OSCTUNEbits.PLLEN = 1;   // 4xPLL 32MHz
    ANCON0 = 0x00;
    ANCON1 = 0x00;
#endif
    
#if defined(_18F2550)
    OSCCONbits.IRCF = 0b111; // 8MHz
    ADCON1bits.PCFG = 0b1111; // Digital I/O
#endif

#if defined(_18F4553)
    OSCCONbits.IRCF = 0b111; // 8MHz
    ADCON1bits.PCFG = 0b1111; // Digital I/O
#endif
    
#if defined(_18F26K22)
    OSCCONbits.IRCF = 0b110; // 8MHz
    OSCTUNEbits.PLLEN = 1;   // 4xPLL 32MHz
    ANSELA = 0x00;
    ANSELB = 0x00;
    ANSELC = 0x00;
#endif

    // wait 1 second for PICkit3 to use MCLR/PGC/PGD
    for ( i = 0; i < 100; i++ ) __delay_ms( 10 );

#ifdef LED_TRIS
    LED_TRIS = 0;
#endif

    T1CONbits.TMR1CS = 0b00;
    T1CONbits.T1CKPS = 0b11;
    T1CONbits.TMR1ON = 1;
}


void main( void )
{
    uint16_t seconds = 0;
    uint8_t i, count = 0;

    initialize();

#ifdef LED_LAT
    // fast flash LED for 2 seconds
    for ( i = 0; i < 50; i++ ) {
        uint8_t j;
        LED_LAT = ! LED_LAT;
        for ( j = 0; j < 5; j++ )
            __delay_ms( 10 );
    }
#endif

    netInit();
    netEnable();

    initBuffers();
    initClients();

    INTCONbits.GIE = 1;
    
    while ( 1 ) {
        if ( PIR1bits.TMR1IF ) {
            PIR1bits.TMR1IF = 0;
            count++;
            if ( count >= T1PERSEC ) {
#ifdef LED_LAT
                LED_LAT ^= 1;
#endif
                count = 0;
                seconds++;
                if ( seconds > 999 )
                    seconds = 0;

                netTimer( seconds );
                tcpStream( seconds );
            }
        }
        netIdle();
    }
}

