/*
 * Show animations on a DIMxDIM led matrix
 *
 * Uses Timer1 library to
 * constantly run an interrupt routine
 * at a specified frequency. This
 * refreshes the display without the
 * main loop having to do anything.
 *
 */
#include <TimerOne.h>

//#define TESTMODE // continuously sequence thru the LED's 

#define DIM 5 // x/y dimension - 5x5 matrix
#define DIM1 (DIM-1)

typedef byte Frame[DIM];

#include "frames.h"

byte row = 0;
byte *curFrame;
int curFrameIdx;

// col[xx] of leds - anodes
// these are the arduino pins assigned to each column
int cols[DIM] = {12,11,10,9,8};

// row[xx] of leds - cathodes
// these are the arduino pins assigned to each row
int rows[DIM] = {7,6,5,4,3};

Frame blankFrame =
{B00000,
 B00000,
 B00000,
 B00000,
 B00000};

// blank the screen
void clearLeds() {
  // blank screen
  curFrame = blankFrame;
}

// select a frame to display
// idx = 0 -> FRAMDECNT-1
void setFrame(int idx) {
  curFrame = Frames[idx];
}

// Interrupt routine
// each time display() is called, it turns off the previous row
// and turns on the next row
byte bitMask = B00000011;
void display() {
  digitalWrite(rows[row], HIGH);  // Turn whole previous row off

  if (bitMask == B00010000) {
    bitMask = B00000011;  // light the right 2 columns (pins 9,8)
    // increment row and wrap if necessary
    if (++row == DIM) {
      row = 0;
    }
  }
  else if (bitMask == B00000011) {
    bitMask = B00001100;  // light the middle 2 columns (pins 11,10)
  }
  else { // bitMaskIdx == B00001100
    bitMask = B00010000;  // light the leftmost column (pin 12)
  }
  
  // direct port manipulation.
  // PORTB is a pseudo variable for digital pins 8 to 13 The two high bits (6 & 7) map to the crystal pins and are not usable
  // the bottom 5 bits are our columns.  We don't want to change the other bits,
  // so we first mask the bits we ignore, and then set the bits we want to light
  PORTB &= B11100000;
  PORTB |= curFrame[row] & bitMask;
  
  digitalWrite(rows[row], LOW); // Turn whole row on at once (for equal lighting times)
}


void setup() {
  int i;
  
  // sets the pins as output
  for (i = 0; i < DIM; i++) {
    pinMode(cols[i], OUTPUT);
    pinMode(rows[i], OUTPUT);
  }

  // set up cols and rows (set display to dark)
  for (i = 0; i < DIM; i++) {
    digitalWrite(cols[i], LOW);
  }

  for (i = 0; i < DIM; i++) {
    digitalWrite(rows[i], HIGH);
  }


  clearLeds();
  
#ifdef TESTMODE  
  while (1) {
  for (i=0;i < DIM;i++) {
    digitalWrite(rows[i],LOW);
    for (int j=0;j < DIM;j++) {
      digitalWrite(cols[j],HIGH);
      delay(250);
      digitalWrite(cols[j],LOW);
    }
    digitalWrite(rows[i],HIGH);
  }
  }
#else // !TESTMODE
  // interrupt interval in uSec
  // this determines how long to keep each row turned on
  // so if we have 5 rows, we redraw the whole screen
  // once every 5 rows * 3 cycles per row * 1000 usec = .015 sec -> 66.67Hz
  // if you perceive flickering you can decrease to 500 for a 133.33Hz rate
  Timer1.initialize(1000);
  
  Timer1.attachInterrupt(display); // ISR

  curFrameIdx = -1;
#endif // TESTMODE
                                                 
}

void loop() {
  // increment the frame index and wrap if necessary
  if (++curFrameIdx == FRAMECNT) {
    curFrameIdx = 0;
  }

  // select frame for display
  setFrame(curFrameIdx);

  delay(400); // wait time between frames in ms
}


