;******************************************************************************************
;  Remote (X-10) AC Power Sensor
;******************************************************************************************
;  Requires: Microchip Assembler/Linker/etc                
;  Target  : Microchip 16F508
;******************************************************************************************
;  Monitor for Power Status Change (2 input channels OR'ed to one X-10 signal channel)
;	- Debounce AC input sensors
;	- Transmit (via X10 contact closure) immediately on state change
;	- Transmit (via X10 contact closure) every hour after initial state change
;	- Delay minimum of 5 seconds between reports
;	- Contact closure = 0.5 seconds
;******************************************************************************************

	list		p=12F508        ; target processor
	#include	<p12F508.inc>   ; processor specific variable definitions

	__CONFIG   _CP_OFF & _WDT_ON & _MCLRE_OFF & _IntRC_OSC 

;**********************************************************************
;     G L O B A L     D E F I N I T I O N S
;**********************************************************************

;-- bit and mask keywords

#DEFINE	B0 0 
#DEFINE	B1 1
#DEFINE	B2 2
#DEFINE	B3 3
#DEFINE	B4 4
#DEFINE	B5 5
#DEFINE	B6 6
#DEFINE	B7 7

#DEFINE MASK(b) (1 << b)

;-- GPIO bits

_AC_SENSE_A	EQU	B0	; GPIO 0 = AC power sensor A
_AC_SENSE_B	EQU	B3	; GPIO 3 = AC power sensor B
_SW_ON		EQU	B1	; GPIO 1 = switch contact ON  \__ NEVER turn both on
_SW_OFF		EQU	B2	; GPIO 2 = switch contact OFF /    at the same time!!!
_DIAG_A		EQU	B5	; GPIO 5 = spare output
_DIAG_B		EQU	B4	; GPIO 4 = spare output

;-- timer rates (based on 3906.25 Hz timebase)

AC_SENSE_TICKS	EQU	65	; number of ticks in 1/60 second
AC_ON_THRESHOLD	EQU	15	; should see AC on for about 1/2 of each 1/60 sec
AC_DEBOUNCE_RAIL EQU	60	; sample threshold (1 second of constant ON/OFF signal)

CONTACT_TICKS	EQU	1953	; X10 contact closure time (1/2 second)

HOLDOFF_TICKS	EQU	19531	; inter-message holdoff delay (5 seconds)

HOUR_TICKS_0	EQU	0xD6	; periodic message rate (14,062,500 ticks = 1 hour)
HOUR_TICKS_1	EQU	0x93
HOUR_TICKS_2	EQU	0xA4

;**********************************************************************
;     R A M
;**********************************************************************
BANK_0	      UDATA	0x07	; 0x07..0x1F (25 bytes max)

pgm_flags     RES	1
_TMR_B7		EQU	B7	; Timer0 B7 change detector
_AC_STATE_A	EQU	B6	; AC power sense channel A
_AC_STATE_B	EQU	B5	; AC power sense channel B
_AC_STATE	EQU	B4	; Overall AC power (A|B) state
_STATE_CHANGED	EQU	B3	; AC state has changed, X-10 signal is pending
_HOLDOFF	EQU	B2	; 5 second delay in progress
_X10_ACTIVE	EQU	B1	; X-10 switch is closed

pwr_counter_a	RES	1	; AC sensor debouncers
pwr_counter_b	RES	1	;   (driven up/down by ac_sense)

ac_sense_timer	RES	1	; 1/60 sec timer
ac_counter_a	RES	1	;   count nbr of AC sense hits per 1/60 sec
ac_counter_b	RES	1	;   count nbr of AC sense hits per 1/60 sec

switch_timer	RES	2	; 1/2 second contact closure timer

holdoff_timer	RES	2	; 5 second inter-message delay

periodic_timer	RES	3	; 1 hour periodic update message timer

;******************************************************************************************
;     M A C R O S
;******************************************************************************************

MOVLF macro	zLITERAL, zREG
	movlw	zLITERAL
	movwf	zREG
    endm

SENSE	macro	zSenseBit, zStateBit, zSenseCounter, zPwrCounter, zDiagBit
  local sense_ac_off, sense_ac_on, sense_exit
	movlw	AC_ON_THRESHOLD		; test for AC ON threshold met
	subwf	zSenseCounter, w
	btfss	STATUS, C
	goto	sense_ac_off
sense_ac_on
	; AC detected on sensor, drive counter upwards
	movlw	AC_DEBOUNCE_RAIL	; limit at high rail
	xorwf	zPwrCounter, w
	btfsc	STATUS, Z
	goto	sense_exit
	; drive counter up towards high rail
	incf	zPwrCounter, f
	movlw	AC_DEBOUNCE_RAIL
	xorwf	zPwrCounter, w
	btfss	STATUS, Z
	goto	sense_exit
	; at high rail
	bsf	pgm_flags, zStateBit
	bsf	GPIO, zDiagBit		; reflect [debounced] sense state on diag output
	goto	sense_exit
sense_ac_off
	; no AC detected on sensor, drive counter down
	movf	zPwrCounter, f		; limit at low rail
	btfsc	STATUS, Z
	goto	sense_exit
	; drive counter towards low rail
	decf	zPwrCounter, f
	btfss	STATUS, Z
	goto	sense_exit
	; at low rail
	bcf	pgm_flags, zStateBit
	bcf	GPIO, zDiagBit		; reflect [debounced] sense state on diag output
sense_exit
	endm


;**********************************************************************
;     P R O G R A M     S T A R T
;**********************************************************************
RESET_VECTOR  CODE	0x1FF	; processor reset vector

;Chip#1	movlw	0x88	; SET BY CHIP MANUFACTURER - MUST READ CHIP TO DETERMINE CORRECT VALUE
	movlw	0x8C	; SET BY CHIP MANUFACTURER - MUST READ CHIP TO DETERMINE CORRECT VALUE

USER_PROGRAM  CODE	0x000	; user code start (after wrapping from reset_vector)

;-- OPTIONAL - use manufacturer's calibration value

	movwf	OSCCAL

;-- set up I/O port pin directions

	clrf	GPIO		; all outputs == 0
	movlw	b'00001001'	; all outputs except GP3, GP0
	tris	GPIO

;-- inertialize TIMER0 as a timebase

	movlw	b'11011111'	; Timer0 clock = fOsc/4 = 1 MHz
	option			; set OPTION register

;-- inertialize RAM variables

	clrf	pgm_flags

	clrf	pwr_counter_a
	clrf	pwr_counter_b

	MOVLF	AC_SENSE_TICKS, ac_sense_timer
	clrf	ac_counter_a
	clrf	ac_counter_b

	; send initial state update after startup stabilization delay
	MOVLF	high(HOLDOFF_TICKS), holdoff_timer+0
	MOVLF	 low(HOLDOFF_TICKS), holdoff_timer+1
	bsf	pgm_flags, _HOLDOFF
	bsf	pgm_flags, _STATE_CHANGED

;**********************************************************************
main_loop	; timebase poll
;**********************************************************************

;-- detect Timer0 B7 transition from 1 => 0

	btfsc	pgm_flags, _TMR_B7
	goto	m10

	; last Timer0 B7 value was 0
	btfss	TMR0, B7		; Timer0 B7 not changed ?
	goto	main_loop
	bsf	pgm_flags, _TMR_B7	; Timer0 B7 changed to 1
	goto	main_loop

m10	; last Timer0 B7 value was 1
	btfsc	TMR0, B7		; Timer0 B7 not changed ?
	goto	main_loop
	bcf	pgm_flags, _TMR_B7	; Timer0 B7 changed to 0

;-- 3906.25 Hz Tick

	clrwdt				; pet the nice watch doggie

;-- AC sensor sampling

	btfss	GPIO, _AC_SENSE_A	; sample AC sensor
	incf	ac_counter_a, f		;   negative logic (0 == AC present)
	btfss	GPIO, _AC_SENSE_B	; sample AC sensor
	incf	ac_counter_b, f		;   negative logic (0 == AC present)

	decfsz	ac_sense_timer, f	; countdown sampling period
	goto	sample_end

	; 1/60 second sampling period complete

	SENSE	_AC_SENSE_A, _AC_STATE_A, ac_counter_a, pwr_counter_a, _DIAG_A
	SENSE	_AC_SENSE_B, _AC_STATE_B, ac_counter_b, pwr_counter_b, _DIAG_B

	MOVLF	AC_SENSE_TICKS, ac_sense_timer
	clrf	ac_counter_a
	clrf	ac_counter_b
sample_end

;-- AC sensor OR'ing and change detection

	btfss	pgm_flags, _AC_STATE_A	; channel A on ?
	btfsc	pgm_flags, _AC_STATE_B	; channel B on ?
	goto	ac_is_on
ac_is_off
	btfss	pgm_flags, _AC_STATE	; already notified via X-10 ?
	goto	ac_done
	bcf	pgm_flags, _AC_STATE	;   remember new state
	goto	ac_state_changed	;   notify via X-10

ac_is_on
	btfsc	pgm_flags, _AC_STATE	; already notified via X-10 ?
	goto	ac_done
	bsf	pgm_flags, _AC_STATE	;   remember new state

ac_state_changed			; notify via X-10 (after possible hold-off delay)
	bsf	pgm_flags, _STATE_CHANGED
ac_done

;-- state reporting & timing

	; X10 contact closure timer
	btfss	pgm_flags, _X10_ACTIVE
	goto	no_x10
	; count down to timeout
	decfsz	switch_timer+1, f
	goto	timer_end
	decfsz	switch_timer+0, f
	goto	timer_end
	; timed out
	bcf	GPIO, _SW_ON
	bcf	GPIO, _SW_OFF
	bcf	pgm_flags, _X10_ACTIVE
	; initiate inter-message delay
	bsf	pgm_flags, _HOLDOFF
	MOVLF	high(HOLDOFF_TICKS), holdoff_timer+0
	MOVLF	 low(HOLDOFF_TICKS), holdoff_timer+1
	goto	timer_end
no_x10

	; consecutive message holdoff timer
	btfss	pgm_flags, _HOLDOFF
	goto	no_holdoff
	; count down holdoff time
	decfsz	holdoff_timer+1, f
	goto	timer_end
	decfsz	holdoff_timer+0, f
	goto	timer_end
	; end holdoff
	bcf	pgm_flags, _HOLDOFF
	; [re]start periodic update timer
	MOVLF	HOUR_TICKS_0, periodic_timer+0
	MOVLF	HOUR_TICKS_1, periodic_timer+1
	MOVLF	HOUR_TICKS_2, periodic_timer+2
no_holdoff

	; state changed detector
	btfss	pgm_flags, _STATE_CHANGED
	goto	no_state_change
send_x10
	bcf	pgm_flags, _STATE_CHANGED
	; send X10 message
	btfsc	pgm_flags, _AC_STATE
	goto	send_x10_on
	bsf	GPIO, _SW_OFF
	goto	send_started
send_x10_on
	bsf	GPIO, _SW_ON
send_started
	; start contact closure timer
	bsf	pgm_flags, _X10_ACTIVE
	MOVLF	high(CONTACT_TICKS), switch_timer+0
	MOVLF	 low(CONTACT_TICKS), switch_timer+1
	goto	timer_end
no_state_change

	; periodic update timer
	decfsz	periodic_timer+2, f
	goto	timer_end
	decfsz	periodic_timer+1, f
	goto	timer_end
	decfsz	periodic_timer+0, f
	goto	timer_end
	; timed out
	goto	send_x10

timer_end

;-- loop forever

	goto	main_loop

;**********************************************************************
	END
;**********************************************************************

