/*
      Sets all the parameters of the MPU6050 accelerometer
      to enable it to wakeup the ESP32 from deep sleep on motion
*/
void SetAcceleroSleep () {
  // Set accelerometer's registers for wake up
  // instructions from : https://lukelectro.wordpress.com/2016/08/11/how-to-enable-motion-detection-interrupt-on-mpu6050/
  accelgyro.resetGyroscopePath();
  accelgyro.resetAccelerometerPath();
  accelgyro.setInterruptDrive(0);
  accelgyro.setInterruptMode(1); // 0=active-high, 1=active-low
  accelgyro.setDHPFMode(MPU6050_DHPF_5);
  accelgyro.setMotionDetectionThreshold(2); // Threshold in 2mg (ajouté by me)
  accelgyro.setMotionDetectionDuration(100); //Duration in ms
  accelgyro.setAccelerometerPowerOnDelay(1); // Set accelerometer power-on delay
  accelgyro.setFreefallDetectionCounterDecrement(1); // Set Free Fall detection counter decrement
  accelgyro.setMotionDetectionCounterDecrement(1); // Set Motion detection counter decrement
  accelgyro.setIntMotionEnabled(true);
}

/*
      Returns the reason for Wake up
      1 : accelero, 3 : timer, 4 : touch (not available)
*/
byte wakeup_reason() {
  esp_sleep_wakeup_cause_t wakeup_reason;
  wakeup_reason = esp_sleep_get_wakeup_cause();
  return (byte)wakeup_reason;
}

/*
   Resets seconds to zero if wake up by timer
*/
void updateTime () {
  getLocalTime(timeinfo);
  if (timeinfo->tm_sec < 41) timeinfo->tm_sec = 0;
  if (timeinfo->tm_sec > 40) timeinfo->tm_sec = 60;
  time_t t = mktime(timeinfo);
  struct timeval now0 = { .tv_sec = t };
  settimeofday(&now0, NULL);
  getLocalTime(timeinfo);
  Serial.printf ("Update time : %s\n", asctime(timeinfo));
}

/*
      Compute and return the time before next wakeup (in microseconds)
*/
uint64_t TimeToSleep_old (unsigned long WakeUpInstant, unsigned long epoch0, byte WakeupReason) {
  WhatTime ();
  Serial.printf ("Date and time : %s\n", asctime(timeinfo));
  unsigned long epoch1 = time(NULL);
  long deltaT = millis() - (epoch1 - epoch0) * 1000;
  if (bootCount == 1) deltaT = 0;
  Serial.printf ("Bootcount = %d DeltaT (ms) = %d\n", bootCount, deltaT);
  byte MinBeforeWakeup = 60 - timeinfo->tm_min;
  uint64_t sleep_time;
  // If we display the time, we need to wakeup next minute
  int SecBeforeWakeup = 60 - timeinfo->tm_sec;
  // otherwise, next plain hour (for wifi updates)
  if (Orientation == Yplus || Orientation == Ymoins) SecBeforeWakeup += MinBeforeWakeup * 60;
  sleep_time = SecBeforeWakeup * 1000000ULL - deltaT * 1000ULL;
  return sleep_time; // microseconds...
}
uint64_t TimeToSleep (unsigned long WakeUpInstant, unsigned long epoch0, byte WakeupReason) {
  uint64_t sleepPeriod1 =  60100000ull;  // 60 seconds sleep
  uint64_t sleepPeriod2 = 120360000ull;  // 2 minutes sleep
  uint64_t sleepPeriod3 = 180767400ull;  // 3 minutes sleep
  uint64_t sleepPeriod5 = 301461000ull;  // 5 minutes sleep
  uint64_t sleep_time;
  int SecBeforeWakeup;
  WhatTime ();
  Serial.printf ("Date and time : %s\n", asctime(timeinfo));
  Serial.printf ("Boot count : %d\n", bootCount);
  if (bootCount == 0 || WakeupReason != 3) { // First run or wake up by motion
    bootCount++;
    SecBeforeWakeup = 60 - timeinfo->tm_sec;
    if (bootCount == 1) SecBeforeWakeup -= 5;
    if (SecBeforeWakeup < 0) SecBeforeWakeup += 60;
    sleep_time = SecBeforeWakeup * 1000000ULL;
  } else {
    // Wake up by timer
    if (Orientation == Xmoins) return sleepPeriod1 - micros();
    if (timeinfo->tm_min % 5 == 0) sleep_time = sleepPeriod5 - micros();
    else if (timeinfo->tm_min % 5 == 4) sleep_time = sleepPeriod1 - micros();
    else if (timeinfo->tm_min % 5 == 3) sleep_time = sleepPeriod2 - micros();
    else if (timeinfo->tm_min % 5 == 2) sleep_time = sleepPeriod3 - micros();
    else if (timeinfo->tm_min % 5 == 1) sleep_time = sleepPeriod1 - micros();
  }
  return sleep_time; // microseconds...
}
/*
    Go to sleep
*/
void GoodNight (unsigned long WakeUpInstant, unsigned long epoch0, byte WakeupReason) {
  SetAcceleroSleep ();
  // Configure RTCIO as wakeup source (GPIO 34 connects to INT of accelero)
  esp_sleep_enable_ext0_wakeup(GPIO_NUM_34, 0); //1 = High, 0 = Low
  // Compute the time to sleep before next wake up
  uint64_t sleep_time = TimeToSleep (WakeUpInstant, epoch0, WakeupReason);
  esp_sleep_enable_timer_wakeup(sleep_time);
  Serial.printf("uSeconds before wakeup: %lu ", uint32_t(sleep_time));
  Serial.printf("= %lu min %lu s\n", uint32_t(sleep_time) / 60000000ul, (uint32_t(sleep_time) / 1000000ul) % 60);
  if (DEBUG) delay(20);
  esp_deep_sleep_start(); // Nothing executed after that (ever)
}
/*
      All the stuff necessary for first execution of the code:
         increase boot count
         search available Wifi network and connect
         get time
         display according to orientation
         set sleep parameters and bye...
*/
void FirstInit(unsigned long WakeUpInstant) {
  Serial.printf("First Init - bootCount = %d\n", bootCount);
  // Init sensors & displays
  bool AcceleroSuccess = InitSensorsDisplays ();
  ProcessAccelero (AcceleroSuccess);
  // Init Wifi
  bool wifiOk = WifiInit ();
  // Init time
  SetTimeByWifi ();
  delay(1000);
  WhatTime ();
  unsigned long epoch0 = time(NULL);
  // Get weather
  GetHourlyForecast ();
  GetDailyForecast ();
  GetCurrentMeteo ();
  // Get citations and ephemeris
  totalCitNb = getCitations ();
  GetEphemeris (timeinfo->tm_mon + 1, timeinfo->tm_mday, Ephemeride);
  // Shut down Wifi
  WifiDisconnect ();
  // Display everything
  if (Orientation == Xmoins) GrandTimeH(true);
  SetDisplaysOrientation (Orientation, true);
  // Prepare for deep sleep
  GoodNight (WakeUpInstant, epoch0, 1);
}

