package org.android.uploadthingspeakapp;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.Set;
import java.util.UUID;

import android.app.Activity;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothSocket;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.res.Configuration;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;
import android.util.Log;
import android.view.View;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.ListView;
import android.widget.TextView;
import android.widget.Toast;
import android.widget.AdapterView.OnItemClickListener;

public class UploadThingspeakAppActivity extends Activity {
	
	static final int REQUEST_ENABLE_BT = 1;
	static final String TAG = "UploadThingspeak";
	private final int MESSAGE_READ = 1;
	private final int DEVICE_CONNECTED = 2;
	
	BluetoothAdapter mBluetoothAdapter; //the Bluetooth "function" of the phone
	ArrayAdapter<CharSequence> devicesArrayAdapter; //the list of paired device names
	BluetoothDevice[] pairedDevices; //the list of paired device
	BluetoothSocket socket; //the connection socket
	InputStream inputStream; //the stream from the serial port
	ListView listDevices;
	TextView logText;
	TextView btText;
	
	
    /** Called when the activity is first created. */
	
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.main);
        
        logText = (TextView) findViewById(R.id.log_text);
        btText = (TextView) findViewById(R.id.bt_dialog_text);
        
        listDevices=(ListView) findViewById(R.id.list_devices);
        
        //this binds the arrayAdapter to the listView, enabling display of paired devices
        devicesArrayAdapter = new ArrayAdapter<CharSequence>(this, R.layout.list_item_layout);
        listDevices.setAdapter(devicesArrayAdapter);
        
        //click listener on the devices list, enabling connection
        OnItemClickListener listener = new OnItemClickListener (){
        	public void onItemClick(AdapterView<?> parent, View child, int position, long id) {
        		connectDevice(position);
        	}
        };
        listDevices.setOnItemClickListener(listener);
        
        //the Bluetooth adapter
        mBluetoothAdapter = BluetoothAdapter.getDefaultAdapter();
    }
    
    //1) -------------------------
    //first, handling the Bluetooth on/off status
    //we check it in onResume, in case it has been disable since last onPause
    @Override
    public void onResume(){
    	super.onResume();
    	
        if (mBluetoothAdapter == null) {
            // Device does not support Bluetooth
        	Toast t = Toast.makeText(this, "Device does not support Bluetooth", Toast.LENGTH_SHORT);
        	t.show();
        	btText.setText("Device does not support Bluetooth");
        }
        else{
        	//Device supports BT
        	if (!mBluetoothAdapter.isEnabled()){
        		//if Bluetooth not activated, then request it 
                Intent enableBtIntent = new Intent(BluetoothAdapter.ACTION_REQUEST_ENABLE);
                startActivityForResult(enableBtIntent, REQUEST_ENABLE_BT);
        	}
        	else{
            	//BT activated, then launch the search for device to pair
            	Toast t = Toast.makeText(this, "BT is already activated", Toast.LENGTH_SHORT);
            	t.show();
            	searchBTDevice();
            }
        }
    }
    
    //called only if the BT is not already activated, in order to activate it
	protected void onActivityResult(int requestCode, int resultCode, Intent moreData){
    	if (requestCode == REQUEST_ENABLE_BT){
    		if (resultCode == Activity.RESULT_OK){
    			//BT activated, then launch the search for device to pair
            	Toast t = Toast.makeText(this, "BT has been activated manually", Toast.LENGTH_SHORT);
            	t.show();
            	btText.setText("BT activated");
            	searchBTDevice();
    		}
    		else if (resultCode == Activity.RESULT_CANCELED)
    			Log.v(TAG, "BT not activated");
    		else
    			Log.v(TAG, "result code not known");
    	}
    	else{
    		Log.v(TAG, "request code not known");    	
    	}
     }
	
	//-----------------------------
	//2) then, find the already-paired devices and display their adress and name in the listView
	
	// Create a BroadcastReceiver for when a BT device is found
	private final BroadcastReceiver mReceiver = new BroadcastReceiver() {
	    public void onReceive(Context context, Intent intent) {
	        String action = intent.getAction();
	        // When discovery finds a device
	        if (BluetoothDevice.ACTION_FOUND.equals(action)) {
	            // Get the BluetoothDevice object from the Intent
	            BluetoothDevice device = intent.getParcelableExtra(BluetoothDevice.EXTRA_DEVICE);
	            // Add the name and address to an array adapter to show in a ListView
	            devicesArrayAdapter.add(device.getName() + " " 
	            					+ device.getAddress() + " is active \n");
	            devicesArrayAdapter.notifyDataSetChanged(); //this "refreshes" the listView display
	            btText.setText("Bluetooth devices found");
	        }
	    }
	};
	
    private void searchBTDevice() {
    	//first, reset the device list
    	devicesArrayAdapter.clear();
    	btText.setText("BT activated, searching devices");
    	
    	//start the discovery
    	mBluetoothAdapter.startDiscovery();
    	// Register the BroadcastReceiver
    	IntentFilter filter = new IntentFilter(BluetoothDevice.ACTION_FOUND);
    	registerReceiver(mReceiver, filter); // Don't forget to unregister during onDestroy
    	
    	/*
    	//get paired devices from the BluetoothAdapter
    	//not recommended because, if the device is not activated, clicking on it will result in a crash (app not responsive) 
    	Set<BluetoothDevice> setPairedDevices = mBluetoothAdapter.getBondedDevices();
    	//convert them into an array
    	int nbDevices = setPairedDevices.size();
    	pairedDevices = (BluetoothDevice[]) setPairedDevices.toArray(new BluetoothDevice[nbDevices]);
    	
    	String btDialog = nbDevices + " paired devices in memory of the phone";
    	btText.setText(btDialog);
    	
    	if (nbDevices > 0) {
    		for (int i = 0; i < nbDevices; i++){
    			CharSequence text = pairedDevices[i].getName() + " " 
    							+ pairedDevices[i].getAddress() + " is in memory \n" ;
    			devicesArrayAdapter.add(text); //name + adress are added to the listView through the ArrayAdapter
    		}
    	}
    	devicesArrayAdapter.notifyDataSetChanged(); //this "refreshes" the listView display
    	*/
    	
	}

    //3) -----------------------
    //this part establishes the connection with a paired device, selected from the onItemClickListener
    
	protected void connectDevice(int position) {
		Toast t = Toast.makeText(this, "trying connection with device " + pairedDevices[position].getName(), Toast.LENGTH_SHORT);
    	t.show();
    	
    	//find the BluetoothDevice in the deviceArrayAdapter identifed by position, and connect to it
    	//the BT chip on the weather station is maintaining an open socked with
    	//the UUID 00001101-0000-1000-8000-00805F9B34FB is standard for serial ports
    	//note that if the device is not turned on, this call will crash the application
    	try{
    		socket = pairedDevices[position].createRfcommSocketToServiceRecord(UUID.fromString("00001101-0000-1000-8000-00805F9B34FB"));
    	}
    	catch(IOException e){
    		e.printStackTrace();
    	}
    	//it is recommended to connect it in a separate thread because it can be slow
		ConnectedThread thread = new ConnectedThread(socket, handler);
		thread.start();
	}
	
	//the Handler has limited use.
	//it will only receive confirmation from the ConnectedThread that Thingspeak has been updated, and update the logView
	final Handler handler = new Handler() {
        public void handleMessage(Message message) {
        	if (message.what == DEVICE_CONNECTED){
        		String btDialog = "Now connected with device " + socket.getRemoteDevice().getName();
            	btText.setText(btDialog);
        	}
        	if (message.what == MESSAGE_READ){
        		String data = (String) message.obj;
                logText.setText("ThingSpeak updated with line \n" + data);
        	}
        }
	};
	
	@Override
	protected void onPause(){
		super.onPause();
	}
	
	//close the socket when the app is done
	@Override
	public void onDestroy(){
		super.onDestroy();
		unregisterReceiver(mReceiver);
		if (socket != null){
			try {
				socket.close();
			}
			catch (IOException e) {
				e.printStackTrace();
			}
	    }
	}
	
	
}