﻿
Imports System.Runtime.InteropServices
Imports System.Text


Module MemoryMappedFiles

    ' ======================================================================================
    '  MemoryMappedFile "Theremino1"
    ' ======================================================================================

    Friend MMF1 As MemoryMappedFile

    Friend Const NAN_Zero As Single = 0 + Single.NaN
    Friend Const NAN_Sleep As Single = 1 + Single.NaN

    Friend Sub MemoryMappedFile_Init()
        MMF1 = New MemoryMappedFile("Theremino1", 4080)
    End Sub

    Friend Function ReadSlot(ByVal Slot As Int32) As Single
        Dim n As Single = MMF1.ReadSingle(Slot * 4)
        Return n
    End Function

    Friend Function ReadSlot_NoNan(ByVal Slot As Int32) As Single
        Dim n As Single = MMF1.ReadSingle(Slot * 4)
        If Single.IsNaN(n) Then n = 0
        Return n
    End Function

    Friend Sub WriteSlot(ByVal Slot As Int32, ByVal Value As Single)
        MMF1.WriteSingle(Slot * 4, Value)
    End Sub

    Friend Sub MemoryMappedFile_FillWithNanSleep()
        For i As Int32 = 0 To 4080 \ 4 - 1
            MMF1.WriteSingle(i * 4, NAN_Sleep)
        Next
    End Sub



    ' ======================================================================================
    '   CLASS MemoryMappedFile
    ' ======================================================================================
    Public Class MemoryMappedFile

        Private Structure SECURITY_ATTRIBUTES
            Const nLength As Int32 = 12
            Public lpSecurityDescriptor As Int32
            Public bInheritHandle As Int32
        End Structure

        ' ---------------------------------------------------------------- private declararations
        <DllImport("Kernel32")> _
        Private Shared Function CloseHandle(ByVal intPtrFileHandle As Int32) As Boolean
        End Function

        <DllImport("Kernel32", EntryPoint:="CreateFileMappingA")> _
        Private Shared Function CreateFileMapping(ByVal hFile As Int32, _
                                                  ByRef lpFileMappigAttributes As SECURITY_ATTRIBUTES, _
                                                  ByVal flProtect As Int32, _
                                                  ByVal dwMaximumSizeHigh As Int32, _
                                                  ByVal dwMaximumSizeLow As Int32, _
                                                  ByVal lpname As String) As Int32
        End Function

        <DllImport("Kernel32")> _
        Private Shared Function MapViewOfFile(ByVal hFileMappingObject As Int32, _
                                              ByVal dwDesiredAccess As Int32, _
                                              ByVal dwFileOffsetHigh As Int32, _
                                              ByVal dwFileOffsetLow As Int32, _
                                              ByVal dwNumberOfBytesToMap As Int32) As IntPtr
        End Function

        <DllImport("Kernel32")> _
        Private Shared Function UnmapViewOfFile(ByVal lpBaseAddress As IntPtr) As Int32
        End Function

        Private Const PAGE_READWRITE As Int32 = 4
        Private Const FILE_MAP_ALL_ACCESS As Int32 = &H1 Or &H2 Or &H4 Or &H8 Or &H10 Or &HF0000

        Private Const INVALID_HANDLE_VALUE As Int32 = -1

        ' ---------------------------------------------------------------- private members

        Private FileHandle As Int32 = 0
        Private MappingAddress As IntPtr = IntPtr.Zero
        Private FileLength As Int32 = 0

        ' ---------------------------------------------------------------- construction / destruction

        Friend Sub New(ByVal Filename As String, Optional ByVal Length As Int32 = 1024)

            FileHandle = CreateFileMapping(INVALID_HANDLE_VALUE, _
                                                 Nothing, _
                                                 PAGE_READWRITE, _
                                                 0, _
                                                 Length, _
                                                 Filename)

            If FileHandle = 0 Then
                MessageBox.Show("Unable to create the MemoryMappedFile: " & Filename, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error)
                Destroy()
                Return
            End If

            MappingAddress = MapViewOfFile(FileHandle, FILE_MAP_ALL_ACCESS, 0, 0, 0)

            If MappingAddress = IntPtr.Zero Then
                MessageBox.Show("Unable to map the MemoryMappedFile: " & Filename, "Error", MessageBoxButtons.OK, MessageBoxIcon.Error)
                Destroy()
                Return
            End If

            FileLength = Length
        End Sub

        Protected Overrides Sub Finalize()
            Destroy()
            MyBase.Finalize()
        End Sub

        Friend Sub Destroy()
            If MappingAddress <> IntPtr.Zero Then
                UnmapViewOfFile(MappingAddress)
            End If
            If FileHandle <> 0 Then
                CloseHandle(FileHandle)
                FileHandle = 0
            End If
        End Sub




        ' ======================================================================================
        '  PUBLIC FUNCTIONS
        ' ======================================================================================

        ' ---------------------------------------------------------------------------- String
        Friend Sub WriteString(ByVal Offset As Int32, ByVal Text As String)
            If MappingAddress = IntPtr.Zero Then Return
            If Offset < 0 OrElse Offset + Text.Length >= FileLength Then Return
            Text &= Chr(0)
            For i As Int32 = 0 To Text.Length - 1
                Marshal.WriteByte(MappingAddress, i + Offset, CByte(Asc(Text(i))))
            Next
        End Sub
        Friend Function ReadString(ByVal Offset As Int32) As String
            If MappingAddress = IntPtr.Zero Then Return ""
            If Offset < 0 OrElse Offset >= FileLength Then Return ""
            ReadString = ""
            Dim b As Byte
            For i As Int32 = Offset To Offset + FileLength - 1
                b = Marshal.ReadByte(MappingAddress, i)
                If b = 0 Then Exit For
                ReadString &= Chr(b)
            Next
        End Function

        ' ---------------------------------------------------------------------------- Int32
        Friend Sub WriteInt32(ByVal Offset As Int32, ByVal Value As Int32)
            If MappingAddress = IntPtr.Zero Then Return
            If Offset < 0 OrElse Offset >= FileLength Then Return
            Marshal.WriteInt32(MappingAddress, Offset, Value)
        End Sub
        Friend Function ReadInt32(ByVal Offset As Int32) As Int32
            If MappingAddress = IntPtr.Zero Then Return 0
            If Offset < 0 OrElse Offset >= FileLength Then Return 0
            ReadInt32 = Marshal.ReadInt32(MappingAddress, Offset)
        End Function

        ' ---------------------------------------------------------------------------- Single
        Friend Sub WriteSingle(ByVal Offset As Int32, ByVal Value As Single)
            If MappingAddress = IntPtr.Zero Then Return
            If Offset < 0 OrElse Offset >= FileLength Then Return
            Dim i As Int32 = BitConverter.ToInt32(BitConverter.GetBytes(Value), 0)
            Marshal.WriteInt32(MappingAddress, Offset, i)
        End Sub
        Friend Function ReadSingle(ByVal Offset As Int32) As Single
            If MappingAddress = IntPtr.Zero Then Return 0
            If Offset < 0 OrElse Offset >= FileLength Then Return 0
            Dim i As Int32 = Marshal.ReadInt32(MappingAddress, Offset)
            ReadSingle = BitConverter.ToSingle(BitConverter.GetBytes(i), 0)
        End Function

        ' ---------------------------------------------------------------------------- Double
        Friend Sub WriteDouble(ByVal Offset As Int32, ByVal Value As Double)
            If MappingAddress = IntPtr.Zero Then Return
            If Offset < 0 OrElse Offset >= FileLength Then Return
            Dim i As Int64 = BitConverter.DoubleToInt64Bits(Value)
            Marshal.WriteInt64(MappingAddress, Offset, i)
        End Sub
        Friend Function ReadDouble(ByVal Offset As Int32) As Double
            If MappingAddress = IntPtr.Zero Then Return 0
            If Offset < 0 OrElse Offset >= FileLength Then Return 0
            Dim i As Int64 = Marshal.ReadInt64(MappingAddress, Offset)
            ReadDouble = BitConverter.Int64BitsToDouble(i)
        End Function


        ' ---------------------------------------------------------------------------- Int32 ARRAY 
        Friend Sub WriteInt32Array(ByVal Offset As Int32, ByRef Array As Int32())
            If MappingAddress = IntPtr.Zero Then Return
            If Offset < 0 OrElse Offset + 4 * Array.Length >= FileLength Then Return
            Marshal.Copy(Array, 0, New IntPtr(MappingAddress.ToInt32 + Offset), Array.Length)
        End Sub
        Friend Sub ReadInt32Array(ByVal Offset As Int32, ByRef Array As Int32())
            If MappingAddress = IntPtr.Zero Then Return
            If Offset < 0 OrElse Offset + 4 * Array.Length >= FileLength Then Return
            Marshal.Copy(New IntPtr(MappingAddress.ToInt32 + Offset), Array, 0, Array.Length)
        End Sub

        ' ---------------------------------------------------------------------------- Single ARRAY
        Friend Sub WriteSingleArray(ByVal Offset As Int32, ByRef Array As Single())
            If MappingAddress = IntPtr.Zero Then Return
            If Offset < 0 OrElse Offset + 4 * Array.Length >= FileLength Then Return
            Marshal.Copy(Array, 0, New IntPtr(MappingAddress.ToInt32 + Offset), Array.Length)
        End Sub
        Friend Sub ReadSingleArray(ByVal Offset As Int32, ByRef Array As Single())
            If MappingAddress = IntPtr.Zero Then Return
            If Offset < 0 OrElse Offset + 4 * Array.Length >= FileLength Then Return
            Marshal.Copy(New IntPtr(MappingAddress.ToInt32 + Offset), Array, 0, Array.Length)
        End Sub

        ' ---------------------------------------------------------------------------- Double ARRAY
        Friend Sub WriteDoubleArray(ByVal Offset As Int32, ByRef Array As Double())
            If MappingAddress = IntPtr.Zero Then Return
            If Offset < 0 OrElse Offset + 8 * Array.Length >= FileLength Then Return
            Marshal.Copy(Array, 0, New IntPtr(MappingAddress.ToInt32 + Offset), Array.Length)
        End Sub
        Friend Sub ReadDoubleArray(ByVal Offset As Int32, ByRef Array As Double())
            If MappingAddress = IntPtr.Zero Then Return
            If Offset < 0 OrElse Offset + 8 * Array.Length >= FileLength Then Return
            Marshal.Copy(New IntPtr(MappingAddress.ToInt32 + Offset), Array, 0, Array.Length)
        End Sub

        ' ---------------------------------------------------------------------------- Memory area ( untested )
        Private Declare Sub MoveMemory Lib "kernel32" _
                                      Alias "RtlMoveMemory" (ByVal pDest As Int32, _
                                                             ByVal pSource As Int32, _
                                                             ByVal iLength As Int32)

        Friend Sub WriteMemoryArea(ByVal Address As Int32, ByVal Nbyte As Int32, ByVal MMF_Offset As Int32)
            If MappingAddress = IntPtr.Zero Then Return
            If MMF_Offset < 0 OrElse Nbyte < 0 OrElse MMF_Offset + Nbyte >= FileLength Then Return
            MoveMemory(MappingAddress.ToInt32 + MMF_Offset, Address, Nbyte)
        End Sub
        Friend Sub ReadMemoryArea(ByVal Address As Int32, ByVal Nbyte As Int32, ByVal MMF_Offset As Int32)
            If MappingAddress = IntPtr.Zero Then Return
            If MMF_Offset < 0 OrElse Nbyte < 0 OrElse MMF_Offset + Nbyte >= FileLength Then Return
            MoveMemory(Address, MappingAddress.ToInt32 + MMF_Offset, Nbyte)
        End Sub


    End Class


End Module
