// Space Rocks Game
// Simple obstacle avoidance game similar to "Flappy Bird" or "Asteroids".
// Fly a space ship through a maze of floating "space rocks".
// The game speeds up as your score increases.
// Uses a 20x4 LCD display and two push button switches for up/down ship movement.
// First game starts automatically.  Press "Reset" to start a new game.
// Hold either "Up" or "Down" button while pressing "Reset" to clear old high score.
// Written by Boomer48: November 2019

#include <EEPROM.h>
#include "LCD_2004.h"

#define line_len 20
#define Space ' '
#define Rock 0xFF
#define Ship '>'
#define Up_Switch 0 // move ship up
#define Dn_Switch 1 // move ship down

char Line_Data[4][line_len] = {
  {Rock, Space, Space, Space, Rock, Space, Rock, Rock, Space, Space, Space, Rock, Space, Rock, Space, Space, Rock, Space, Rock, Rock},
  {Space, Space, Rock, Space, Space, Space, Rock, Space, Space, Rock, Space, Space, Space, Space, Rock, Space, Space, Space, Rock, Rock},
  {Rock, Rock, Rock, Rock, Space, Rock, Rock, Space, Rock, Rock, Rock, Rock, Rock, Space, Space, Space, Rock, Space, Space, Space},
  {Rock, Rock, Rock, Rock, Space, Space, Space, Space, Space, Rock, Rock, Rock, Space, Space, Space, Rock, Rock, Space, Rock, Space}
};

/*char Line_Data[4][line_len] = {
  {Rock, Space, Space, Space, Rock, Rock, Rock, Rock, Rock, Space, Rock, Space, Space, Rock, Space, Space, Space, Space, Space, Rock},
  {Space, Space, Rock, Space, Space, Rock, Space, Rock, Space, Rock, Space, Rock, Space, Space, Space, Rock, Rock, Rock, Space, Space},
  {Rock, Rock, Space, Space, Space, Space, Rock, Rock, Space, Space, Space, Space, Rock, Space, Rock, Rock, Space, Space, Space, Space},
  {Space, Space, Space, Rock, Rock, Space, Space, Space, Space, Rock, Rock, Space, Space, Space, Rock, Space, Rock, Space, Space, Rock}
  }; */

char Line_Addr[4][line_len] = {
  {0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x8D, 0x8E, 0x8F, 0x90, 0x91, 0x92, 0x93},
  {0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, 0xD0, 0xD1, 0xD2, 0xD3},
  {0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9A, 0x9B, 0x9C, 0x9D, 0x9E, 0x9F, 0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 0xA7},
  {0xD4, 0xD5, 0xD6, 0xD7, 0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF, 0xE0, 0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7}
};

byte Ship_Row = 1;
byte Ship_Col = 0;
byte Prev_Row = 1;
byte Prev_Col = 0;
byte Step_Count;
byte Max_Steps = 20; // used to speed up the game as the score gets higher
float Time_Count = 0;
boolean Swapped = false; // true = rows are in 1, 2, 3, 4 order; false = rows are in 3, 4, 1, 2 order
byte ROM_Byte;
const byte Valid_Score = 0xA5; // EEPROM value indicating valid score has been stored
float High_Score;

void setup() {
  byte i;
  String junk;

  LCD_setup();

  pinMode(Up_Switch, INPUT_PULLUP);
  pinMode(Dn_Switch, INPUT_PULLUP);

  High_Score = 1.0;

  if ((digitalRead(Up_Switch) == LOW) || (digitalRead(Dn_Switch) == LOW)) {
    // clear old high score
    EEPROM.write(0, 0);
    EEPROM.write(1, 0);
  }
  else {
    ROM_Byte = EEPROM.read(0);

    if (ROM_Byte == Valid_Score) {
      ROM_Byte = EEPROM.read(1);
      if (ROM_Byte == Valid_Score)
        EEPROM.get(2, High_Score);
    }
  }

  LCD_Line1();
  Send_Byte('H');
  Send_Byte('I');
  Send_Byte('G');
  Send_Byte('H');
  Send_Byte(' ');
  Send_Byte('S');
  Send_Byte('C');
  Send_Byte('O');
  Send_Byte('R');
  Send_Byte('E');
  Send_Byte(' ');
  Send_Byte('=');
  Send_Byte(' ');

  junk = String(High_Score);
  i = 0;

  while (junk[i] != '.') {
    Send_Byte(junk[i]);
    i++;
  }

  delay(1000);
  delay(1000);
  delay(1000);

  LCD_Line1();
  for (i = 0; i < line_len; i++)
    Send_Byte(Line_Data[0][i]);
  delay(1000);
  LCD_Line2();
  for (i = 0; i < line_len; i++)
    Send_Byte(Line_Data[1][i]);
  delay(1000);
  LCD_Line3();
  for (i = 0; i < line_len; i++)
    Send_Byte(Line_Data[2][i]);
  delay(1000);
  LCD_Line4();
  for (i = 0; i < line_len; i++)
    Send_Byte(Line_Data[3][i]);

  Move_Ship();

  delay(1000);
  delay(1000);
  delay(1000);
}

void loop() {

  Step_Count = 0;

  while (Step_Count < Max_Steps) {
    Check_Switch();
  }

  Ship_Col++;

  if (Ship_Col == line_len) {
    // lines swapped so adjust ship position
    Ship_Col = 0;
    Swapped = !Swapped;
    if (Ship_Row == 0)
      Ship_Row = 2;
    else if (Ship_Row == 1)
      Ship_Row = 3;
    else if (Ship_Row == 2)
      Ship_Row = 0;
    else if (Ship_Row == 3)
      Ship_Row = 1;
  }

  Check_Ship();
  Move_Ship();
  LCD_Left();
  Time_Count++;

  //Speed up game every 20 counts but limit to 500ms
  if ((Max_Steps > 10) && ((long(Time_Count) % 20) == 0))
    Max_Steps--;
}

void Check_Switch() {

  if (digitalRead(Up_Switch) == LOW) {
    if (Swapped == false) {
      if (Ship_Row != 0)  //can't move up from row 0
        Ship_Row--;
    }
    else {
      if (Ship_Row == 0)
        Ship_Row = 3;
      else if (Ship_Row == 1)
        Ship_Row = 0;
      //can't move up from row 2
      else if (Ship_Row == 3)
        Ship_Row = 2;
    }

    Check_Ship();
    Move_Ship();
    while (digitalRead(Up_Switch) == LOW) {
      delay(50); // wait for switch to be released
      Step_Count++;
      if (Step_Count >= Max_Steps)
        return;
    }
  }

  else if (digitalRead(Dn_Switch) == LOW) {
    if (Swapped == false) {
      if (Ship_Row != 3)  //can't move down from row 3
        Ship_Row++;
    }
    else {
      if (Ship_Row == 0)
        Ship_Row = 1;
      //can't move down from row 1
      else if (Ship_Row == 2)
        Ship_Row = 3;
      else if (Ship_Row == 3)
        Ship_Row = 0;
    }

    Check_Ship();
    Move_Ship();
    while (digitalRead(Dn_Switch) == LOW) {
      delay(50); // wait for switch to be released
      Step_Count++;
      if (Step_Count >= Max_Steps)
        return;
    }
  }
  else {
    delay(50);
    Step_Count++;
  }
}

void Move_Ship() {
  byte temp;

  temp = Line_Addr[Prev_Row][Prev_Col];

  LCD_Char_Loc(temp, Space);
  temp = Line_Addr[Ship_Row][Ship_Col];
  LCD_Char_Loc(temp, Ship);
  Prev_Row = Ship_Row;
  Prev_Col = Ship_Col;
}

void Check_Ship() {
  byte temp;
  String junk;

  temp = Line_Data[Ship_Row][Ship_Col];

  if (temp == Rock) {
    temp = 0;
    while (temp < 5) {
      Clear_LCD();
      delay(500);
      Send_Byte(' ');
      Send_Byte(' ');
      Send_Byte(' ');
      Send_Byte('C');
      Send_Byte('R');
      Send_Byte('A');
      Send_Byte('S');
      Send_Byte('H');
      Send_Byte(' ');
      Send_Byte('A');
      Send_Byte('N');
      Send_Byte('D');
      Send_Byte(' ');
      Send_Byte('B');
      Send_Byte('U');
      Send_Byte('R');
      Send_Byte('N');
      delay(500);
      temp++;
    }

    LCD_Line3();
    Send_Byte('S');
    Send_Byte('C');
    Send_Byte('O');
    Send_Byte('R');
    Send_Byte('E');
    Send_Byte(' ');
    Send_Byte('=');
    Send_Byte(' ');
    junk = String(Time_Count);
    temp = 0;

    while (junk[temp] != '.') {
      Send_Byte(junk[temp]);
      temp++;
    }

    if (Time_Count > High_Score) {
      LCD_Line4();
      Send_Byte('N');
      Send_Byte('E');
      Send_Byte('W');
      Send_Byte(' ');
      Send_Byte('H');
      Send_Byte('I');
      Send_Byte('G');
      Send_Byte('H');
      Send_Byte(' ');
      Send_Byte('S');
      Send_Byte('C');
      Send_Byte('O');
      Send_Byte('R');
      Send_Byte('E');
      Send_Byte('!');
      Send_Byte('!');
      Send_Byte('!');

      // Save high score to EEPROM
      EEPROM.write(0, Valid_Score);
      EEPROM.write(1, Valid_Score);
      EEPROM.put(2, Time_Count);
    }

    while (true); // requires reset to start new game
  }
}
