/*
 NTP Client with Time
 Four/Six digit 7 segment display
 with DST

 MTS

 Get the time from a Network Time Protocol (NTP) time server
 Demonstrates use of UDP sendPacket and ReceivePacket
 For more on NTP time servers and the messages needed to communicate with them,
 see http://en.wikipedia.org/wiki/Network_Time_Protocol

 updated for the ESP8266 12 Apr 2015 
 by Ivan Grokhotkov
 * Example showing time sync to NTP time source
 *
 */
 
#include <Time.h> 
#include "LedControl.h"		// For MAX7219 7 segment driver
#include <ESP8266WiFi.h>
#include <WiFiUdp.h>

// Network/ NTP stuff

char ssid[] = "yourSSID";  			//   network SSID (name)
char pass[] = "yourPassword";  //  network password

int size; // NTP Packet variables

// NTP Servers:
IPAddress timeServer(198, 60, 73, 8); // Ntp-nist.ldsbc.net
//IPAddress timeServer(132, 163, 4, 101); // time-a.timefreq.bldrdoc.gov
// IPAddress timeServer(132, 163, 4, 102); // time-b.timefreq.bldrdoc.gov
// IPAddress timeServer(132, 163, 4, 103); // time-c.timefreq.bldrdoc.gov

/*-------- NTP code ----------*/
const int NTP_PACKET_SIZE = 48; // NTP time is in the first 48 bytes of message
byte packetBuffer[NTP_PACKET_SIZE]; //buffer to hold incoming & outgoing packets

unsigned int localPort = 8888;  // local port to listen for UDP packets

bool DST;				// true if Daylight Savings Time
time_t SpringChange;	// Date Spring forward
time_t FallChange;		// Date Fall back
int currentYear;
const int timeZone = -7; 	// Mountain Standard Time (USA)
//const int timeZone = -7;  // Mountain Standard Time (USA)
//const int timeZone = -6;  // Mountain Daylight Time (USA)

// A UDP instance to let us send and receive packets over UDP
WiFiUDP udp;
int AMLEDpin = 0;

/*
 Now we need a LedControl to work with.
 ***** These pin numbers are for my ESP8266 connected to MAX7219
 pin 13 is connected to the DataIn 
 pin 14 is connected to the CLK 
 pin 12 is connected to LOAD 
 We have only a single MAX7219.
 */
LedControl lc=LedControl(13,14,12,1);

time_t prevDisplay = 0; // when the digital clock was displayed

// Prototypes
void findChangeDates();		// Find Spring/Fall change dates
bool IsDST();				// Check if it is DST
void digitalClockDisplay();	//display clock data
void printDigits(int);
time_t getNtpTime();		// get NTP time, converts to local returns as time_t
void sendNTPpacket(IPAddress &address); // send an NTP request to the time server at the given address

void setup(){
	Serial.begin(115200);
	Serial.println("TimeNTP Example");
	pinMode(AMLEDpin, OUTPUT); //set AMLEDpin as output
	
	// The MAX72XX is in power-saving mode we have to do a wakeup call
	lc.shutdown(0,false);
	lc.setIntensity(0,1); // Set the brightness to a low value
	lc.clearDisplay(0);	// and clear the display
 	// We start by connecting to a WiFi network
	WiFi.begin(ssid, pass);
  
	while (WiFi.status() != WL_CONNECTED) {
		delay(500);
		Serial.print(".");
	}
	udp.begin(localPort);	// Get first NTP time 

	setSyncInterval(60);	// time in seconds to update NTP time
	Serial.println("waiting for sync");
	setSyncProvider(getNtpTime);	// Puts local time in now()
}

void loop(){  
	if (timeStatus() != timeNotSet){
		// if SpringChange not set or not current year, findChangeDates
		if (!SpringChange ||(currentYear != year())){
			currentYear = year();
			findChangeDates();
			DST = IsDST();
			if (DST) adjustTime(SECS_PER_HOUR); //This is needed only once
		}
		else{  
			if (now() != prevDisplay){ //update the display only if time has changed
				prevDisplay = now();
				DST = IsDST();	// Check for a change
				digitalClockDisplay();
			}				
		}
	}
}

void digitalClockDisplay(){
	// digital clock display of the time
	Serial.print(hourFormat12());
	printDigits(minute());
	printDigits(second());
	if(isAM()) Serial.print(" AM ");
	else Serial.print(" PM ");
	Serial.print(day());
	Serial.print(" ");
	Serial.print(month());
	Serial.print(" ");
	Serial.print(year()); 
	Serial.println(); 
	
	//  setDigit(int addr, int digit, byte value, boolean dp);
	//  digit left to right 0 1 2 3	
	lc.clearDisplay(0);
  
	// Start with left digit
	lc.setDigit(0,0,int(hourFormat12()/10),false);
	lc.setDigit(0,1,(hourFormat12() % 10),true);
	lc.setDigit(0,2,(minute()/10),false);
	lc.setDigit(0,3,(minute() % 10),true);
	lc.setDigit(0,4,int(second()/10),false);
	lc.setDigit(0,5,(second() % 10),false);
	if (isAM()) digitalWrite(AMLEDpin, LOW);
	else digitalWrite(AMLEDpin, HIGH);
}

void printDigits(int digits){
	// utility for digital clock display: prints preceding colon and leading 0
	Serial.print(":");
	if(digits < 10)
    Serial.print('0');
	Serial.print(digits);
}

// get NTP time, converts to local returns as time_t
time_t getNtpTime(){
	time_t testTime;
	while (udp.parsePacket() > 0) ; // discard any previously received packets
	Serial.println("Transmit NTP Request");
	sendNTPpacket(timeServer);
	delay(1000);
	size = udp.parsePacket();
	if (size >= NTP_PACKET_SIZE) {
		Serial.println("Receive NTP Response");
		udp.read(packetBuffer, NTP_PACKET_SIZE);  // read packet into the buffer
		unsigned long secsSince1900;
		// convert four bytes starting at location 40 to a long integer
		secsSince1900 =  (unsigned long)packetBuffer[40] << 24;
		secsSince1900 |= (unsigned long)packetBuffer[41] << 16;
		secsSince1900 |= (unsigned long)packetBuffer[42] << 8;
		secsSince1900 |= (unsigned long)packetBuffer[43];
		// This is local Daylight time
		testTime = secsSince1900 - 2208988800UL + timeZone * SECS_PER_HOUR;
		if (DST){
			return testTime + SECS_PER_HOUR;	// add an hour if DST
		}
		else return testTime;
  }
  Serial.println("No NTP Response :-(");
  return 0; // return 0 if unable to get the time
}

// send an NTP request to the time server at the given address
void sendNTPpacket(IPAddress &address){
	// set all bytes in the buffer to 0
	memset(packetBuffer, 0, NTP_PACKET_SIZE);
	// Initialize values needed to form NTP request
	// (see URL above for details on the packets)
	packetBuffer[0] = 0b11100011;   // LI, Version, Mode
	packetBuffer[1] = 0;     // Stratum, or type of clock
	packetBuffer[2] = 6;     // Polling Interval
	packetBuffer[3] = 0xEC;  // Peer Clock Precision
	// 8 bytes of zero for Root Delay & Root Dispersion
	packetBuffer[12]  = 49;
	packetBuffer[13]  = 0x4E;
	packetBuffer[14]  = 49;
	packetBuffer[15]  = 52;
	// all NTP fields have been given values, now
	// you can send a packet requesting a timestamp:                 
	udp.beginPacket(address, 123); //NTP requests are to port 123
	udp.write(packetBuffer, NTP_PACKET_SIZE);
	udp.endPacket();
}
// Find Spring/Fall change dates
void findChangeDates(){	// Find Spring/Fall change dates
	/*Currently, daylight saving time starts on the second Sunday 
	in March and ends on the first Sunday in November, with the 
	time changes taking place at 2:00 a.m. local time. With a 
	mnemonic word play referring to seasons, clocks "spring forward 
	and fall back"—that is, in spring (technically late winter) 
	the clocks are moved forward from 2:00 a.m. to 3:00 a.m., 
	and in fall they are moved back from 2:00 am to 1:00 am.
	*/
	time_t TT;	//Temporary time, setTime() changes now()
	TT = now();
	// Set time to March 8 2am. (first possible second Sunday)
	//setTime(int hr,int min,int sec,int day, int month, int yr);
	setTime(2,0,0,8,3,currentYear);
	SpringChange = now();
	// if not Sunday, then make Sunday
	if (weekday(SpringChange) != 1) 
		SpringChange = SpringChange + (8-weekday(SpringChange))*SECS_PER_DAY;
	//SpringChange now has time_t for March change date/time
Serial.println(SpringChange);
	// Set time to Nov 1 (first possible first Sunday)
	//setTime(int hr,int min,int sec,int day, int month, int yr);
	setTime(2,0,0,1,11,currentYear);
	FallChange = now();
	// if not Sunday, then make Sunday
	if (weekday(FallChange) != 1) 
		FallChange = FallChange + (8-weekday(FallChange))*SECS_PER_DAY;
	// FallChange now has time_t for November change date/time
Serial.println(FallChange);
	// now() was changed, put it back sort of
	setTime(hour(TT),minute(TT),second(TT),day(TT),month(TT),year(TT));
}

// Check if it is DST
bool IsDST(){
	// if current date/time is between SpringChange and FallChange, then it's DST
	if (now() >= SpringChange && now() < FallChange) return true;
	else return false;
}
