#include <Wire.h>
#include <Adafruit_Sensor.h>
#include <Adafruit_TSL2561_U.h>

#include <WindowStepper.h>
#include <LightBarrier.h>
#include <Time.h>

//Defines
//-----------------------------------------------------------------------------------
//Debug Mode (when set to 1)
#define DEBUG 1

//Lumacity level values
#define LUMACITY_LEVEL_VERY_BRIGHT   2000
#define LUMACITY_LEVEL_BRIGHT        1000
#define LUMACITY_LEVEL_NORMAL         500
#define LUMACITY_LEVEL_DARK           100

//The minimum amount of time (in seconds) the ring should stay in a specific state
//It's because the ring should not move around between states again and again...
#define MIN_STATE_TIME 20

//The intervall (in ms) we measure the lumacity 
#define LUMACITY_MEASURE_INTERVALL 1000

//The max number of lumacity measures we make to
//get the average light value
//(be careful: >=8912 will result in an overflow!)
#define MAX_NUM_LUMACITY_VAL_MEASURED 10

//Directions
#define RIGHT 0
#define LEFT  1
//-----------------------------------------------------------------------------------

//Enumerations
//-----------------------------------------------------------------------------------
//Light levels
enum LightLevel {DARK = 1,NORMAL = 2, BRIGHT = 3, VERY_BRIGHT = 4};

//Window states
enum WindowState {OPENED, HALFCLOSED, CLOSED};
//-----------------------------------------------------------------------------------

//variables for Debug-Mode
float fDebugTimer = 0.0f;
boolean  bShowDebugInfo = true;

//How long is the ring in its current state?
float stateTime = MIN_STATE_TIME;

WindowState winState = OPENED;

//Time
Time time = Time();

//LightBarriers
//-----------------------------------------------------------------------------------
//Use analog pins!
LightBarrier lightBarrierStart= LightBarrier(2,185);
LightBarrier lightBarrierStop = LightBarrier(3,225);
//-----------------------------------------------------------------------------------

//Stepper
//-----------------------------------------------------------------------------------
//Use digital pins!
WindowStepper stepper = WindowStepper(4,5,6,7);

//Has the ring to be moved?
boolean movingStepper = false;
//-----------------------------------------------------------------------------------

//lumacity sensor
//-----------------------------------------------------------------------------------
Adafruit_TSL2561_Unified tsl = Adafruit_TSL2561_Unified(TSL2561_ADDR_FLOAT, 12345);

//Number of values counted
int lumacityValCounter=0;

//The sum of measured values
int lumacityValueSum = 0;

//An array that holds the measured values
int lumacityValues[MAX_NUM_LUMACITY_VAL_MEASURED];

LightLevel avgLumacityLevel = NORMAL;

//The Timer for measuring differnt lightValues,
//we want to measure every 200ms!
float fLumacityMeasureTimer = 0.0f;
//-----------------------------------------------------------------------------------


/*
  Configures the light sensor
*/
void configureLumacitySensor()
{
  tsl.enableAutoRange(true); 
  tsl.setIntegrationTime(TSL2561_INTEGRATIONTIME_13MS);    
}


/*
  returns the value measured by the light sensor hardware
*/
int getLumacitySensorValue()
{
  sensors_event_t event;
  tsl.getEvent(&event);
  int x = event.light;
  return x;
}


/*
  Initializes the serial monitor, the light sensor, 
  the timer and resets the ring position.
*/
void setup()
{
  //Alow serial monitor
  Serial.begin(9600);
  
  setupValues();
  
  //Configures the light sensor
  configureLumacitySensor();
  
  //stepper zurÃ¼cksetzen
  resetRingPosition();
  
  //Setups the timer to get the elapsed time 
  //between each loop cycle
  time.setup();
}


void setupValues()
{
  //TODO
  stepper.setDelay(2);
 
  for(int i=0; i<MAX_NUM_LUMACITY_VAL_MEASURED; i++)
  {
    lumacityValues[i]=0;
  }
  
  lumacityValCounter=0;
  lumacityValueSum = 0;
}


//Setzt den Ring wieder auf die Startposition(geÃ¶ffnet) zurÃ¼ck
void resetRingPosition()
{
  debug_print("Resetting - Start\n");
  
  stepper.setDirection(0);
  
  while(lightBarrierStart.hasHitAmplitude()==0)
  {
     stepper.step(10);
  }
  
  //new windows state: opened
  winState = OPENED;
  
  //After resetting, don't wait te full minimum
  //state time before moving the ring...
  stateTime = MIN_STATE_TIME-3;
  
  stepper.setDirection(1);
  
  debug_print("Resetting - Completed\n");
}

/*
  Main loop
*/
void loop()
{ 
  updateTime();
  
  updateLumacitySensor();
  
  //moves the ring (if it should)
  updateStepper();
  
  //Check if the ring has to stop and stops it if necessary
  checkRingPosition();
}


/*
  Updates the global time
*/
void updateTime()
{
  time.update();
  
  updateDebugTimer();
  
  //update state time
  if(movingStepper == false)
  {
    stateTime+=time.getElapsed();
    
    debug_print_intervall("stateTime: ", stateTime);
  }
  else
  {
    debug_print_intervall("moving ring...\n");
  }
}


/*
  Updates the lightsensor and orders the ring to move
*/
void updateLumacitySensor()
{
  //Only do something when the ring is not moving anymore!
  if(movingStepper == true)
    return;
  
  //Only do something when the ring stayed in a specific
  //state (by example OPENED, HALFCLOSED,...) for some time...
  if((int)stateTime < MIN_STATE_TIME-(LUMACITY_MEASURE_INTERVALL*MAX_NUM_LUMACITY_VAL_MEASURED)/1000)
  {
    return;
  }
  
  fLumacityMeasureTimer += time.getElapsed();
  
  //not time to measure a new value? return...
  if(fLumacityMeasureTimer*1000<LUMACITY_MEASURE_INTERVALL)
  {
     return;
  }
  
  //reset timer
  fLumacityMeasureTimer=0.0f;
  
  int val = getLumacitySensorValue();
  
  //Print debug infos
  debug_print("Lumacity-Value: ",val);
  
  //Convert real value in lumacity level
  val = (val>=LUMACITY_LEVEL_VERY_BRIGHT) ? (int)VERY_BRIGHT : 
        (val>=LUMACITY_LEVEL_BRIGHT)      ? (int)BRIGHT :
        (val>=LUMACITY_LEVEL_NORMAL)      ? (int)NORMAL : (int)DARK; 
  
  //Substract the old value (which will be overwritten by the new value)
  //from the sum
  if(lumacityValues[lumacityValCounter]!=0)
  {
    lumacityValueSum-=lumacityValues[lumacityValCounter];
  }
  
  //Write the new value into the table and add it to the sum
  //(so the sum has not to be recalculated every time...)
  lumacityValues[lumacityValCounter] = val;
  lumacityValueSum += val;
  
  lumacityValCounter++;
  
  //Print debug infos
  debug_print("Lumacity-Level: ",val);
  
  //Measured enough values? Calculate average lumacity level
  //-----------------------------------------------------------------
  if(lumacityValCounter>=MAX_NUM_LUMACITY_VAL_MEASURED)
  {
    debug_print("Lumacity-Measured values: ");
    
    for(int i=0; i<MAX_NUM_LUMACITY_VAL_MEASURED;i++)
    {
      debug_print(lumacityValues[i]);
      
      if(i<MAX_NUM_LUMACITY_VAL_MEASURED-1)
      {  
         debug_print(", ");
      }
    }
    
    float avg = ((float)lumacityValueSum)/MAX_NUM_LUMACITY_VAL_MEASURED;
    
    avgLumacityLevel =(LightLevel) round(avg);
     
    //Print debug infos
    debug_print("Lumacity-Average level (rounded) ", (int)avgLumacityLevel);
    
    lumacityValCounter = 0;    
  }
  else
  {
    return;
  }
  //-----------------------------------------------------------------
  
  //Process the average value: 
  //DARK or NORMAL -> open window
  //BRIGHT or VERY_BRIGHT -> close window
  //-----------------------------------------------------------------
  switch(avgLumacityLevel)
  {
    case DARK:
    case NORMAL:    
      if (winState!=OPENED)
      {   
        stepper.setDirection(LEFT);
        
        //reset stateTime
        stateTime = 0.0f;
        
        //move the ring
        movingStepper=true;
        
        debug_print("LightSensor: Light - dimmed - open window...\n");
      }  
    break;
    
    case BRIGHT:
    case VERY_BRIGHT:   
      if (winState!=CLOSED)
      { 
        stepper.setDirection(RIGHT);
        
        //reset stateTime
        stateTime = 0.0f;
        
        //move the ring
        movingStepper=true;
        
        debug_print("LightSensor: Light - very bright - close window...\n");
      }   
    break;
    
    default:
    break;
  }
  //-----------------------------------------------------------------
}



/*
  If movingStepper is 'true', the stepper 
  makes a 10ms long step and waits for 2ms
  (delay to slow it down)
*/
void updateStepper()
{
  //If the ring has to move:
  //Let the stepper make a 10ms long step
  if (movingStepper)
  {
    stepper.step(10);
  }
}


/*
  Checks the Lightbarriers and stops the ring if necessary
*/
void checkRingPosition()
{
  //Only do something when the ring IS moving!
  if(movingStepper==false)
    return;
    
   //Window opened? Stop ring!
  if(stepper.getDirection() == LEFT && lightBarrierStart.hasHitAmplitude()==1)
  {   
    winState = OPENED;
    movingStepper=false;
    
    debug_print("Barrier - STOP - Closed\n");
  }
 
  //Window closed? Stop ring!
  if(stepper.getDirection() == RIGHT && lightBarrierStop.hasHitAmplitude()==1)
  {   
    winState = CLOSED;
    movingStepper=false;
    
    debug_print("Barrier - START - Opened\n");
  }
}



//Debug-Helpers
//-----------------------------------------------------------------------------------

//Update debug time intervall (in an intervall of 1 second,
//debug_print_intervall functions are allowed to print)
void updateDebugTimer()
{
  //Only do something in DEBUG-Mode!
  if(DEBUG == false)
    return;
   
  bShowDebugInfo = false;
  
  fDebugTimer += time.getElapsed();
 
  if(fDebugTimer>1.0f)
  {
    fDebugTimer = 0.0f;
    
    bShowDebugInfo = true;
  }
}

inline void debug_print_intervall(char text[], int val)
{
  if(bShowDebugInfo)
  {
    debug_print(text);
    debug_print(val);
    debug_print("\n");
  }
}

inline void debug_print_intervall(char text[], float val)
{
  if(bShowDebugInfo)
  {
    debug_print(text);
    debug_print(val);
    debug_print("\n");
  }
}

inline void debug_print_intervall(char text[])
{
  if(bShowDebugInfo)
    debug_print(text);
}

inline void debug_print_intervall(int val)
{
  if(bShowDebugInfo)
    debug_print(val);
}

inline void debug_print_intervall(float val)
{
  if(bShowDebugInfo)
      debug_print(val);
}

inline void debug_print(char text[], int val)
{
  debug_print(text);
  debug_print(val);
  debug_print("\n");
}

inline void debug_print(char text[], float val)
{
  debug_print(text);
  debug_print(val);
  debug_print("\n");
}

inline void debug_print(char text[])
{
  if(DEBUG)
  {
    Serial.print(text);
  }
}

inline void debug_print(int value)
{
  if(DEBUG)
  {
    Serial.print(value);
  }
}

inline void debug_print(float value)
{
  if(DEBUG)
  {
    Serial.print(value);
  }
}
//-----------------------------------------------------------------------------------

