// 1602 LCD 4-bit interface
// Written by Boomer48: May 2019


#include "LCD_4Bit.h"

// Input switches
#define Init_Sw 11 // manual re-init of tag
#define Inc_Sw 12 // Increment tag credit amount

#define Trail_Blk 7 // ID of Trailer Block (holds keys)
#define Data_Blk 6 // ID of Data Block (holds credits)
#define Bkup_Blk 5 // ID of Backup Data Block (holds credits)
#define Resp_TFI 0xD5 // TFI for response messages from reader

#define Def_Key1 0xFF // first byte of default key
#define Def_Key2 0xFF
#define Def_Key3 0xFF
#define Def_Key4 0xFF
#define Def_Key5 0xFF
#define Def_Key6 0xFF // last byte of default key

#define My_Key1 0x01 // first byte of new key
#define My_Key2 0x02
#define My_Key3 0x00
#define My_Key4 0x01
#define My_Key5 0x04
#define My_Key6 0x08 // last byte of new key

// Set for increment amount (packed BCD)
#define Inc1        0x00 // 10/1
#define Inc100      0x00 // 1000/100
#define Inc10k      0x00 // 100/10 thousand
#define Inc1M       0x02 // 10/1 million

// Set for decrement amount (packed BCD)
#define Dec1        0x00 // 10/1
#define Dec100      0x00 // 1000/100
#define Dec10k      0x50 // 100/10 thousand
#define Dec1M       0x00 // 10/1 million

byte temp;
byte TFI_Found;
byte Tag_Blk_Num; // Tag Block Number
byte UID1;
byte UID2;
byte UID3;
byte UID4;
byte Checksum;
byte Cmd_Echo; // byte after TFI in response
byte Cmd_Stat; // command status in response
byte Error_Flg;
byte Junk_Num = 0; // junk number for write to tag
byte Millions; // Packed BCD (millions and 10 millions)
byte Ten_Thou; // Packed BCD (10 thousands and 100 thousands)
byte Hundreds; // Packed BCD (hundreds and thousands)
byte Ones; // Packed BCD (ones and tens)
byte Carry; // carry flag from DAA routine
byte Borrow; // borrow flag from DAS routine
byte Comp_1; // 1's complement credit amount LS byte
byte Comp_2;
byte Comp_3;
byte Comp_4; // 1's complement credit amount MS byte
byte Try_Def; // if scan fails then try default key
byte Reinit_Tag; // tag data corrupt so try reinit
byte Bad_Credit; // flag set if credit data mismatch
char ASCII[] = "0123456789ABCDEF";

void setup() {
  pinMode(Init_Sw, INPUT_PULLUP);
  pinMode(Inc_Sw, INPUT_PULLUP);
  LCD_setup();
  Serial.begin(115200);
  Flush_RX();

  while (Serial.available() == 0)
    Send_Init();
}

void loop() {
Scan_Loop:
  Tag_Blk_Num = Data_Blk;
  Scan_Tag();
  Disp_UID(); // comment out if not needed
  Auth_Tag();

  if (Error_Flg == true)
    goto Scan_Loop;

  Auth_Msg(); // comment out if not needed

  // Reinit tag based on manual command, failed key access, or corrupt data
  if ((digitalRead(Init_Sw) == LOW) || (Try_Def == true) || (Reinit_Tag == true)) {
    Init_Tag();
    goto Scan_Loop;
  }

  Read_Tag();

  if (Error_Flg == true)
    goto Scan_Loop;

  Get_Credit(); // get stored credit amount

  if (Bad_Credit == true) {
    Match_Fail();
    goto Scan_Loop;
  }

  Disp_Credit(); // display credit amount

  if (digitalRead(Inc_Sw) == LOW)
    Inc_Credit(); // increment credits
  else
    Dec_Credit(); // decrement credits

  Done_Msg();
}

void Flush_RX() {
  while (Serial.available() > 0)
    Serial.read(); // clear receive buffer
}

void Make_ASCII(byte BCD_Byte) {
  Send_Byte(ASCII[BCD_Byte >> 4]); // display upper nibble first
  Send_Byte(ASCII[BCD_Byte & 0x0F]);
}

void Disp_Data() {
  byte i;

  LCD_Line1();

  for (i = 0; i < 8; i++)
    Make_ASCII(Serial.read());

  LCD_Line2();

  for (i = 0; i < 8; i++)
    Make_ASCII(Serial.read());
}
// Supress leading zeros in credits display
void Find_First() {
  if ((Millions & 0xF0) != 0)
    goto Disp_10M;
  else if ((Millions & 0x0F) != 0)
    goto Disp_1M;
  else if ((Ten_Thou & 0xF0) != 0)
    goto Disp_100k;
  else if ((Ten_Thou & 0x0F) != 0)
    goto Disp_10k;
  else if ((Hundreds & 0xF0) != 0)
    goto Disp_1k;
  else if ((Hundreds & 0x0F) != 0)
    goto Disp_100;
  else if ((Ones & 0xF0) != 0)
    goto Disp_10;
  else
    goto Disp_1;

Disp_10M:
  Send_Byte(ASCII[Millions >> 4]);
Disp_1M:
  Send_Byte(ASCII[Millions & 0x0F]);
  Send_Byte(',');
Disp_100k:
  Send_Byte(ASCII[Ten_Thou >> 4]);
Disp_10k:
  Send_Byte(ASCII[Ten_Thou & 0x0F]);
Disp_1k:
  Send_Byte(ASCII[Hundreds >> 4]);
  Send_Byte(',');
Disp_100:
  Send_Byte(ASCII[Hundreds & 0x0F]);
Disp_10:
  Send_Byte(ASCII[Ones >> 4]);
Disp_1:
  Send_Byte(ASCII[Ones & 0x0F]);
}

// ---------------------------------------------------
// RFID Messages
// ---------------------------------------------------

void Preamble() {
  Flush_RX();
  Serial.write(0x00);
  Serial.write(0x00);
  Serial.write(0xFF);
}

void Postamble() {
  Serial.write(0x00);
  delay(100);
}

void Send_ACK() {
  Serial.write(0x00);
  Serial.write(0x00);
  Serial.write(0xFF);
  Serial.write(0x00);
  Serial.write(0xFF);
  Serial.write(0x00);
}

void Do_Cksum() {
  Checksum = (~Checksum) + 1;
  Serial.write(Checksum);
  Postamble();
}
void Send_Init() {
  Serial.write(0x55);
  Serial.write(0x55);
  Serial.write(0x00);
  Serial.write(0x00);
  Serial.write(0x00);
  Serial.write(0x00); // add more zeros if needed to wake up

  // Need to send SAM Config to get out of Low Vbat state
  Preamble();
  Serial.write(0x03);
  Serial.write(0xFD);
  Serial.write(0xD4);
  Serial.write(0x14); // SAMConfiguration
  Serial.write(0x01);
  Serial.write(0x17);
  Postamble();
}

void Scan_Tag() {
  TFI_Found = false;

  while (TFI_Found == false) {
    Preamble();
    Serial.write(0x04);
    Serial.write(0xFC);
    Serial.write(0xD4);
    Serial.write(0x4A); // InListPassiveTarget
    Serial.write(0x01); // read one card (max=2)
    Serial.write(0x00); // 106-kbps type A
    Serial.write(0xE1);
    Postamble();
    Scan_Msg();
    Find_Resp();
    Get_UID();
  }
}

void Find_Resp() {
  TFI_Found = false;

  while ((Serial.available() > 0) && (TFI_Found == false)) {
    temp = Serial.read();
    if (temp == Resp_TFI) {
      TFI_Found = true;
      Cmd_Echo = Serial.read();
      Cmd_Stat = Serial.read();
    }
  }
}

void Get_UID() {
  Serial.read();
  Serial.read();
  Serial.read();
  Serial.read();
  Serial.read();
  UID1 = Serial.read();
  UID2 = Serial.read();
  UID3 = Serial.read();
  UID4 = Serial.read();
}

// Authorize tag data sector using the defined Trailer block ID
void Auth_Tag() {
  Checksum = 0;
  Preamble();
  Serial.write(0x0F);
  Serial.write(0xF1);
  temp = 0xD4;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = 0x40;
  Serial.write(temp); // InDataExchange
  Checksum = Checksum + temp;
  temp = 0x01;
  Serial.write(temp); // number of targets
  Checksum = Checksum + temp;
  temp = 0x60;
  Serial.write(temp); // authenticate with Key A
  Checksum = Checksum + temp;
  temp = Trail_Blk;
  Serial.write(temp);
  Checksum = Checksum + temp;

  if (Try_Def == true) {
    temp = Def_Key1;
    Serial.write(temp);
    Checksum = Checksum + temp;
    temp = Def_Key2;
    Serial.write(temp);
    Checksum = Checksum + temp;
    temp = Def_Key3;
    Serial.write(temp);
    Checksum = Checksum + temp;
    temp = Def_Key4;
    Serial.write(temp);
    Checksum = Checksum + temp;
    temp = Def_Key5;
    Serial.write(temp);
    Checksum = Checksum + temp;
    temp = Def_Key6;
    Serial.write(temp);
    Checksum = Checksum + temp;
  }
  else {
    temp = My_Key1;
    Serial.write(temp);
    Checksum = Checksum + temp;
    temp = My_Key2;
    Serial.write(temp);
    Checksum = Checksum + temp;
    temp = My_Key3;
    Serial.write(temp);
    Checksum = Checksum + temp;
    temp = My_Key4;
    Serial.write(temp);
    Checksum = Checksum + temp;
    temp = My_Key5;
    Serial.write(temp);
    Checksum = Checksum + temp;
    temp = My_Key6;
    Serial.write(temp);
    Checksum = Checksum + temp;
  }

  temp = UID1;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = UID2;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = UID3;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = UID4;
  Serial.write(temp);
  Checksum = Checksum + temp;

  Do_Cksum();
  Find_Resp();

  if ((TFI_Found == false) || (Cmd_Stat != 0))
    Bad_Scan();
  else
    Error_Flg = false;
}

// Initiates a write to the tag
void Write_Tag() {
  Checksum = 0;
  Preamble();
  Serial.write(0x15);
  Serial.write(0xEB);
  temp = 0xD4;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = 0x40;
  Serial.write(temp); // InDataExchange
  Checksum = Checksum + temp;
  temp = 0x01;
  Serial.write(temp); // number of targets
  Checksum = Checksum + temp;
  temp = 0xA0;
  Serial.write(temp); // write block (16 bytes)
  Checksum = Checksum + temp;
  temp = Tag_Blk_Num; // block number (0-63)
  Serial.write(temp);
  Checksum = Checksum + temp;
}

void Send_Data() {
  temp = Millions;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = Ten_Thou;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = Hundreds;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = Ones;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = (~Millions);
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = (~Ten_Thou);
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = (~Hundreds);
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = (~Ones);
  Serial.write(temp);
  Checksum = Checksum + temp;
  Serial.write(0x00);
  Serial.write(0x00);
  Serial.write(0x00);
  Serial.write(0x00);
  Serial.write(0x00);
  Serial.write(0x00);
  Serial.write(0x00);
  Serial.write(0x00);

  Do_Cksum();
  Find_Resp();

  if ((TFI_Found == false) || (Cmd_Stat != 0))
    Bad_Write();
  else
    Error_Flg = false;
}

// Sends key data as part of the Write_Tag command.
// Formated as 6 bytes Key A, 3 bytes write access flags,
// 1 byte Don't Care, 6 bytes Key B.
void Send_Keys() {
  temp = My_Key1; // write Key A
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = My_Key2;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = My_Key3;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = My_Key4;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = My_Key5;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = My_Key6;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = 0xFF; // write access flags
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = 0x07;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = 0x80;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = 0x69; // don't care byte
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = My_Key1; // write Key B
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = My_Key2;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = My_Key3;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = My_Key4;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = My_Key5;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = My_Key6;
  Serial.write(temp);
  Checksum = Checksum + temp;

  Do_Cksum();
  Find_Resp();

  if ((TFI_Found == false) || (Cmd_Stat != 0))
    Bad_Write();
}

// Read tag data from the defined data block of the authorized sector
void Read_Tag() {
  Checksum = 0;
  Preamble();
  Serial.write(0x05);
  Serial.write(0xFB);
  temp = 0xD4;
  Serial.write(temp);
  Checksum = Checksum + temp;
  temp = 0x40;
  Serial.write(temp); // InDataExchange
  Checksum = Checksum + temp;
  temp = 0x01;
  Serial.write(temp); // number of targets
  Checksum = Checksum + temp;
  temp = 0x30;
  Serial.write(temp); // read block (16 bytes)
  Checksum = Checksum + temp;
  temp = Tag_Blk_Num; // block number (0-63)
  Serial.write(temp);
  Checksum = Checksum + temp;

  Do_Cksum();
  Find_Resp();

  if ((TFI_Found == false) || (Cmd_Stat != 0))
    Bad_Read();
  else
    Error_Flg = false;
}

void Init_Tag() {
  Auth_Tag();
  Tag_Blk_Num = Trail_Blk;
  Write_Tag();
  Send_Keys();
  Tag_Blk_Num = Data_Blk;
  Millions = 0;
  Ten_Thou = 0;
  Hundreds = 0;
  Ones = 0;
  Write_Tag();
  Send_Data();
  Tag_Blk_Num = Bkup_Blk;
  Write_Tag();
  Send_Data();
  Try_Def = false;
  Reinit_Tag = false;
  Tag_Done();
}

// Gets the credit value from the tag and verifies it against the
// complemented copy.
void Get_Credit() {
  Millions = Serial.read();
  Ten_Thou = Serial.read();
  Hundreds = Serial.read();
  Ones = Serial.read();

  // read 1's complement version
  Comp_4 = ~Serial.read();
  Comp_3 = ~Serial.read();
  Comp_2 = ~Serial.read();
  Comp_1 = ~Serial.read();

  // verify values
  temp = Millions - Comp_4 + Ten_Thou - Comp_3 + Hundreds - Comp_2 + Ones - Comp_1;
  if (temp != 0)
    Bad_Credit = true;
  else
    Bad_Credit = false;
}

void Match_Fail() {
  Bad_Data();
  Get_Backup();
  Tag_Blk_Num = Data_Blk;
  Write_Tag();
  Send_Data();
  Read_Tag();
  Get_Credit();

  if (Bad_Credit == true)
    Bad_Card();

  Data_Fixed();
  Disp_Credit();
}

void Get_Backup() {
  Tag_Blk_Num = Bkup_Blk;
  Read_Tag();
  Get_Credit();
  if (Bad_Credit == true)
    Bad_Card();
}

void Inc_Credit() {
  Ones = Ones + Inc1;
  Hundreds = Hundreds + Inc100;
  Ten_Thou = Ten_Thou + Inc10k;
  Millions = Millions + Inc1M;

  if (((Ones & 0x0F) > 9) || ((Ones & 0x0F) < (Inc1 & 0x0F)))
    Ones = Ones + 6;
  if (((Ones & 0xF0) > 0x90) || ((Ones & 0xF0) < (Inc1 & 0xF0))) {
    Ones = Ones + 0x60;
    Hundreds++;
  }

  if (((Hundreds & 0x0F) > 9) || ((Hundreds & 0x0F) < (Inc100 & 0x0F)))
    Hundreds = Hundreds + 6;
  if (((Hundreds & 0xF0) > 0x90) || ((Hundreds & 0xF0) < (Inc100 & 0xF0))) {
    Hundreds = Hundreds + 0x60;
    Ten_Thou++;
  }

  if (((Ten_Thou & 0x0F) > 9) || ((Ten_Thou & 0x0F) < (Inc10k & 0x0F)))
    Ten_Thou = Ten_Thou + 6;
  if (((Ten_Thou & 0xF0) > 0x90) || ((Ten_Thou & 0xF0) < (Inc10k & 0xF0))) {
    Ten_Thou = Ten_Thou + 0x60;
    Millions++;
  }

  if (((Millions & 0x0F) > 9) || ((Millions & 0x0F) < (Inc1M & 0x0F)))
    Millions = Millions + 6;
  if (((Millions & 0xF0) > 0x90) || ((Millions & 0xF0) < (Inc1M & 0xF0)))
    Full_Credit();
  else {
    Tag_Blk_Num = Bkup_Blk;
    Write_Tag();
    Send_Data();
    Tag_Blk_Num = Data_Blk;
    Write_Tag();
    Send_Data();
    Read_Tag();
    Get_Credit();

    if (Bad_Credit == true)
      Match_Fail();
    else
      Disp_Credit();
  }
}

void Dec_Credit() {
  if ((Ones & 0x0F) < (Dec1 & 0x0F))
    Ones = Ones - 6;
  if ((Ones & 0xF0) < (Dec1 & 0xF0)) {
    Ones = Ones - 0x60;
    Hundreds--;
  }

  if ((Hundreds & 0x0F) < (Dec100 & 0x0F))
    Hundreds = Hundreds - 6;
  if ((Hundreds & 0xF0) < (Dec100 & 0xF0)) {
    Hundreds = Hundreds - 0x60;
    Ten_Thou--;
  }

  if ((Ten_Thou & 0x0F) < (Dec10k & 0x0F))
    Ten_Thou = Ten_Thou - 6;
  if ((Ten_Thou & 0xF0) < (Dec10k & 0xF0)) {
    Ten_Thou = Ten_Thou - 0x60;
    Millions--;
  }

  if ((Millions & 0x0F) < (Dec1M & 0x0F))
    Millions = Millions - 6;
  if ((Millions & 0xF0) < (Dec1M & 0xF0))
    Low_Credit();
  else {
    Ones = Ones - Dec1;
    Hundreds = Hundreds - Dec100;
    Ten_Thou = Ten_Thou - Dec10k;
    Millions = Millions - Dec1M;
    if ((Millions & 0xF0) > 0x90)
      Low_Credit();
    else {
      Tag_Blk_Num = Bkup_Blk;
      Write_Tag();
      Send_Data();
      Tag_Blk_Num = Data_Blk;
      Write_Tag();
      Send_Data();
      Read_Tag();
      Get_Credit();

      if (Bad_Credit == true)
        Match_Fail();
      else
        Disp_Credit();
    }
  }
}

// ---------------------------------------------------
// Display Messages
// ---------------------------------------------------

void Scan_Msg() {
  Clear_LCD();
  Send_Byte('R');
  Send_Byte('E');
  Send_Byte('A');
  Send_Byte('D');
  Send_Byte('Y');
  Send_Byte(' ');
  Send_Byte('T');
  Send_Byte('O');
  Send_Byte(' ');
  Send_Byte('S');
  Send_Byte('C');
  Send_Byte('A');
  Send_Byte('N');
  delay(1000);
}

void Bad_Scan() {
  Clear_LCD();
  Send_Byte('B');
  Send_Byte('A');
  Send_Byte('D');
  Send_Byte(' ');
  Send_Byte('S');
  Send_Byte('C');
  Send_Byte('A');
  Send_Byte('N');
  Error_Flg = true;
  delay(2000);
}

void Bad_Read() {
  Clear_LCD();
  Send_Byte('B');
  Send_Byte('A');
  Send_Byte('D');
  Send_Byte(' ');
  Send_Byte('R');
  Send_Byte('E');
  Send_Byte('A');
  Send_Byte('D');
  Error_Flg = true;
  delay(2000);
}

void Bad_Card() {
  Clear_LCD();
  Send_Byte('B');
  Send_Byte('A');
  Send_Byte('D');
  Send_Byte(' ');
  Send_Byte('C');
  Send_Byte('A');
  Send_Byte('R');
  Send_Byte('D');
  Error_Flg = true;
  Reinit_Tag = true;
  delay(2000);
}

void Bad_Data() {
  Clear_LCD();
  Send_Byte('D');
  Send_Byte('A');
  Send_Byte('T');
  Send_Byte('A');
  Send_Byte(' ');
  Send_Byte('C');
  Send_Byte('O');
  Send_Byte('R');
  Send_Byte('R');
  Send_Byte('U');
  Send_Byte('P');
  Send_Byte('T');
  Send_Byte(':');
  LCD_Line2();
  Send_Byte('C');
  Send_Byte('H');
  Send_Byte('E');
  Send_Byte('C');
  Send_Byte('K');
  Send_Byte('I');
  Send_Byte('N');
  Send_Byte('G');
  Send_Byte(' ');
  Send_Byte('B');
  Send_Byte('A');
  Send_Byte('C');
  Send_Byte('K');
  Send_Byte('U');
  Send_Byte('P');
  delay(3000);
}

void Low_Credit() {
  Clear_LCD();
  Send_Byte('C');
  Send_Byte('R');
  Send_Byte('E');
  Send_Byte('D');
  Send_Byte('I');
  Send_Byte('T');
  Send_Byte('S');
  Send_Byte(' ');
  Send_Byte('T');
  Send_Byte('O');
  Send_Byte('O');
  Send_Byte(' ');
  Send_Byte('L');
  Send_Byte('O');
  Send_Byte('W');
  Error_Flg = true;
  delay(2000);
}

void Full_Credit() {
  Clear_LCD();
  Send_Byte('W');
  Send_Byte('A');
  Send_Byte('L');
  Send_Byte('L');
  Send_Byte('E');
  Send_Byte('T');
  Send_Byte(' ');
  Send_Byte('F');
  Send_Byte('U');
  Send_Byte('L');
  Send_Byte('L');
  Error_Flg = true;
  delay(2000);
}

void Bad_Write() {
  Clear_LCD();
  Send_Byte('B');
  Send_Byte('A');
  Send_Byte('D');
  Send_Byte(' ');
  Send_Byte('W');
  Send_Byte('R');
  Send_Byte('I');
  Send_Byte('T');
  Send_Byte('E');
  Error_Flg = true;
  delay(2000);
}

void Disp_UID() {
  Clear_LCD();
  Send_Byte('U');
  Send_Byte('I');
  Send_Byte('D');
  Send_Byte(':');
  Send_Byte(' ');
  Make_ASCII(UID1);
  Make_ASCII(UID2);
  Make_ASCII(UID3);
  Make_ASCII(UID4);
  delay(2000);
}

void Disp_Credit() {
  Clear_LCD();
  Send_Byte('C');
  Send_Byte('R');
  Send_Byte('E');
  Send_Byte('D');
  Send_Byte('I');
  Send_Byte('T');
  Send_Byte('S');
  Send_Byte(':');
  LCD_Line2();
  Find_First();
  delay(2000);
}

void Auth_Msg() {
  Clear_LCD();
  Send_Byte('A');
  Send_Byte('U');
  Send_Byte('T');
  Send_Byte('H');
  Send_Byte('E');
  Send_Byte('N');
  Send_Byte('T');
  Send_Byte('I');
  Send_Byte('C');
  Send_Byte('A');
  Send_Byte('T');
  Send_Byte('E');
  Send_Byte('D');
  delay(2000);
}

void Done_Msg() {
  Clear_LCD();
  Send_Byte('R');
  Send_Byte('E');
  Send_Byte('M');
  Send_Byte('O');
  Send_Byte('V');
  Send_Byte('E');
  Send_Byte(' ');
  Send_Byte('T');
  Send_Byte('A');
  Send_Byte('G');
  delay(2000);
}

void Tag_Done() {
  Clear_LCD();
  Send_Byte('T');
  Send_Byte('A');
  Send_Byte('G');
  Send_Byte(' ');
  Send_Byte('I');
  Send_Byte('N');
  Send_Byte('I');
  Send_Byte('T');
  Send_Byte('I');
  Send_Byte('A');
  Send_Byte('L');
  Send_Byte('I');
  Send_Byte('Z');
  Send_Byte('E');
  Send_Byte('D');
  delay(2000);
}

void Data_Fixed() {
  Clear_LCD();
  Send_Byte('D');
  Send_Byte('A');
  Send_Byte('T');
  Send_Byte('A');
  Send_Byte(' ');
  Send_Byte('R');
  Send_Byte('E');
  Send_Byte('P');
  Send_Byte('A');
  Send_Byte('I');
  Send_Byte('R');
  Send_Byte('E');
  Send_Byte('D');
  delay(2000);
}
