/*
  Ohbot_Alexa.ino
  Written by: Martyn Wright 2017

  This project is designed to simulate talking on the Ohbot by 
  sensing the sounds on a speaker cable.
  Primarily, it was designed to listen to a Raspberry Pi running 
  the AleaxPi application and sync to Alexa's voice.
  However, it can sync to any noise over the loudspeaker cable it 
  is plugged into.
  
  Currently the Arduino being used is an Uno clone, with the following 
  pins connected:
  
  Arduino pin 0  RX - Connected to pin 1 TX on the Pro Mini on the Ohbot
    "     Pin 1  TX        "           0 RX             "
    "     Pin    GND       "             GND            "
    "     Pin    GND       "             GND           on the vero board
    "     Pin    +5v       "             +5v                "    
    "     Pin A0           "             0.1 mF capacitor   "
    
 The circuit diagram for the vero board is as follows:

                                 +----------- +5v
                                 |
                      0.1 mF     } 100k Ohm 
                                 |
 Input from speaker -----||------+----------- Pin A0
 (left or right channel)         |
                                 } 100k Ohm  
                                 |
 Input from speaker -------------+----------- GND                                  
    
*/

#include "U8glib.h"
U8GLIB_SSD1306_128X64 u8g(10, 11, 8, 9);
char chUnderline[17]   = "________________";

String inputString = "";         // a string to hold incoming data
boolean stringComplete = false;  // whether the string is complete
boolean bSendCommands = false;
boolean bUseVoice = false;
boolean bEyesLit = false;
boolean bEyesTurned = false;
int lastEyePos = 104;
long nTimeout = 9000;

//int dly = 100;
int dly = 50;

int count = 0;
int tot = 0;
int maxp = 0;
int lastLip = -1;
int loops = 0;
long lastMillis = 0;
long currMillis = 0;
long diffMillis = 0;
long callBack = 0;

void setup() {

  randomSeed(analogRead(A1));
  
  // initialize serial:
  Serial.begin(19200);
    
  // reserve 200 bytes for the inputString:
  inputString.reserve(200);
  Serial.println(F("Ohbot_Alexa.ino"));

  // OLED display setup - for debug only - not required
  setupDisplay(); 
  Serial.println(F("Ready"));
}

void loop() {
    
  // Wait for Ohbot's initial output
  if (stringComplete) {    
    if (inputString.indexOf("Servo: 11 on pin") >= 0) {
      bSendCommands = true;
    }
    if (inputString.indexOf("reset ohbot") >= 0) {
      bSendCommands = true;
    }
    inputString = "";
    stringComplete = false;
  }
  
  if (bSendCommands) {
    delay(1000);
    Serial.println(F("Resetting Ohbot"));
    resetOhbot();
    bSendCommands = false;
    delay(1000);
    bUseVoice = true;
  }
    
  if (bUseVoice) {
    count++;
    delay(10);
    int val = analogRead(A0);          // Read loudspeaker signal
    int peaks = val-509;
    if (peaks < 0) peaks = peaks * -1; // Ensure always positive

    if (peaks < 7) {
      peaks = 0;          // Ignore Noise
    } else {
      if (bEyesLit == false) wakeupOhbot();
      lastMillis = currMillis;
    }
        
    if (count < 8) {
      tot = tot + peaks;
    } else {
      if (tot > maxp) maxp = tot;       // Aggregate last ten samples
            
      int lip = map(tot, 0, 300, 0, 5); // Map data to lip movements
      if (lip > 0) {
        lip = 9 + (lip * 10);
        if (lip > 49) lip = 49;
      }
      if (lip != lastLip) {
        moveLip(lip);
       
        lastLip = lip;
      }
      tot = 0;
      count = 0;
      loops++;
    }

    if (bEyesTurned) {
      if (millis() > callBack) {
        Serial.print(F("a02\nm02,104,00\n"));    // reset EyeTurn servo
        bEyesTurned = false;
      }
    }
      
    // Randomly blink and turn Ohbot's eyes
    currMillis = millis();
    diffMillis = currMillis - lastMillis;
    if (diffMillis > 1000) {
      long randNumber = random(10000);
      if (bEyesLit) {
        if (randNumber > 9995) blinkOhbot();
        if (randNumber < 5) eyeturnOhbot();
      }      
    }
    
    // Shut down servos after 9 seconds of inactivity 
    if ((diffMillis) > nTimeout) {
      if (bEyesLit) sleepOhbot();
    }
  }
}

void moveLip(int lip) {
  int lip2 = lip;
  
  // reverse the values
  switch(lip) {
    case 0:
      lip2 = 49;
      break;
    case 19:
      lip2 = 39;
      break;
    case 39:
      lip2 = 19;
      break;
    case 49:
      lip2 = 0;
      break;
  }

  // Make lips move according to strength of sounds  
  String strLip = String(lip2);
  sendCommand("m04," + strLip + ",00\n");
  sendCommand("m05," + strLip + ",00\n");
}

void sendCommand(String strCmd) {
  Serial.print(strCmd);
  delay(dly);
}

void resetOhbot() {
  Serial.print(F("l00,00,00,00\nl01,00,00,00\n")); // reset Left and Right EyeColour
  Serial.print(F("a01\nm01,90,40\na00\nm00,76,00\na02\nm02,104,00\na06\nm06,129,00\na04\nm04,49,00\na05\nm05,49,00\na03\nm03,61,00\n"));
  sleepOhbot();
}

void blinkOhbot() {
  Serial.print(F("a03\nm03,00,00\n"));    // set LidBlink servo
  delay(150);
  Serial.print(F("m03,61,00\n"));    // reset LidBlink servo
  lastMillis = currMillis;
}

void eyeturnOhbot() {
  long randNumber = random(2);
  if (randNumber < 1) {
    Serial.print(F("a02\nm02,162,00\n"));   // set EyeTurn servo
  } else {
    Serial.print(F("a02\nm02,46,00\n"));    // set EyeTurn servo
  }
  callBack = millis() + 1000;
  bEyesTurned = true;
  lastMillis = currMillis;
}

void wakeupOhbot() {
  Serial.print(F("l00,99,99,99\nl01,99,99,99\n")); // light Left and Right Eyes
  Serial.print(F("a00\na01\na02\na03\na04\na05\na06\n")); // Attach servos
  Serial.print(F("m03,61,00\n"));    // reset LidBlink servo
  bEyesLit = true;
}

void sleepOhbot() {
  Serial.print(F("a02\nm02,104,00\n"));   // reset EyeTurn
  delay(100);
  Serial.print(F("a03\nm03,00,00\n"));    // set LidBlink servo
  delay(100);
  Serial.print(F("l00,00,00,00\nl01,00,00,00\n")); // reset Left and Right EyeColour
  Serial.print(F("d00\nd01\nd02\nd03\nd04\nd05\nd06\n")); // Detach servos
  bEyesLit = false;
}

void serialEvent() {

  while (Serial.available()) {
    // get the new byte:
    char inChar = (char)Serial.read();
    inputString += inChar;
    if (inChar == '\n') {
      Serial.print(inputString);
      stringComplete = true;
    }
  }
}

void setupDisplay() {
  // assign default color value
  if ( u8g.getMode() == U8G_MODE_R3G3B2 ) {
    u8g.setColorIndex(255);     // white
  }
  else if ( u8g.getMode() == U8G_MODE_GRAY2BIT ) {
    u8g.setColorIndex(3);         // max intensity
  }
  else if ( u8g.getMode() == U8G_MODE_BW ) {
    u8g.setColorIndex(1);         // pixel on
  }
  else if ( u8g.getMode() == U8G_MODE_HICOLOR ) {
    u8g.setHiColorByRGB(255,255,255);
  }  

  // graphic commands to redraw the complete screen should be placed here  
  u8g.setFont(u8g_font_unifont);
  
  // picture loop
  u8g.firstPage();  
  do {
    draw();
  } while( u8g.nextPage() );
}

void draw(void) {
  int col = 0;
  int row = 10;

  u8g.drawStr(col, row,    "Ohbot_Alexa     ");            
  u8g.drawStr(col, row+2,  chUnderline); 
  row = row+4;
  for (int i=12; i<49; i=i+12) {
    u8g.drawStr(col, row+i, "                ");
  }
}
