/* Wireless mesh for Arduino and APC220 modules
Each node samples two analog voltages. Each node can output two analog voltages.
Nodes all work on the same RF frequency. They are synchronised so each transmits in a particular time slot.
Time signals are generated by node zero, and if node zero does not exist, by the lowest numbered node.
There are 16 nodes in the mesh. Each timing cycle is 4096ms, and so it takes around 4 to 80 seconds to send data through the mesh.
Each node also transmits its local time (sync within a few ms) so nodes that cannot directly talk to node 0 can still synchronise to the mesh.
Each node samples its analog inputs, and puts a time stamp on these values.
Each node stores all the values of all other nodes, so 16 nodes, 2 values per node is 32 values.
When a node is ready to transmit, it sends all 32 values, with associated time stamps.
When receiving, if a value has a newer time stamp, then replace it.
This means that newer values propogate through the mesh. 
Nodes can be paired together, so adjusting a potentiometer on one node will change the voltage on another node (with 4 to 80s delay)
Each node can either be a repeater, or input values, or output values, or all of these at once. 
The mesh can tolerate repeater nodes being removed. Ideally there are several paths data can take.
APC220 modules have open air ranges of up to 1000m. Trees and metal (eg a shed) can greatly decrease this though, to as low as 50m.
The mesh allows signals to get around buildings and barriers.
The 4096ms time slot was calculated as the time needed to send all node data at 9600 baud, using human readable hex (ascii 32 to 126)
and allowing for delays of about 150ms between each packet. This means there are two bytes per value (eg F1) and theoretically
it would be possible to send this as a single byte. However, then ascii 13 can't be used as an end of line marker, and then packets
have to be constructed with start and finish bytes and they get more complex to code. 
Current consumption of a node is about 25mA for the Arduino, 30mA for the radio module and 40mA for the display.
Arduino nodes powered using solar power are run with a swithching regulator stepping 13V down to 7V which means current draw from
a 12V battery is about half the current going into the arduino. Nodes draw roughly 1 watt.
Solar powered nodes are run from a 20W to 80W solar panel, which is oversized, but allows for energy collection even on cloudy days.
The solar panel goes into a $10 charge regulator which has over and undervoltage protection and temperature adjusts the float voltage.
Hopefully a 12V 7Ah SLA alarm battery will have a reasonable life as it is only being discharged a small percentage.
*/ 


/*
Demonstration sketch for PCF8574T I2C LCD Backpack
http://tronixlabs.com/display/lcd/serial-i2c-backpack-for-hd44780-compatible-lcd-modules/
Uses library from https://bitbucket.org/fmalpartida/new-liquidcrystal/downloads
GNU General Public License, version 3 (GPL-3.0)
*/

#include <Wire.h>
#include <LCD.h> 
#include <LiquidCrystal_I2C.h> // I2C driver, only uses two pins
#include <AltSoftSerial.h>

// AltSoftSerial always uses these pins:
//
// Board          Transmit  Receive   PWM Unusable
// -----          --------  -------   ------------
// Teensy 3.0 & 3.1  21        20         22
// Teensy 2.0         9        10       (none)
// Teensy++ 2.0      25         4       26, 27
// Arduino Uno        9         8         10
// Arduino Leonardo   5        13       (none)
// Arduino Mega      46        48       44, 45
// Wiring-S           5         6          4
// Sanguino          13        14         12

LiquidCrystal_I2C	lcd(0x27,2,1,0,4,5,6,7); // 0x27 is the I2C bus address for an unmodified backpack


AltSoftSerial altSerial;

byte myNode = 1; // my node number
byte partnerNode = 1; // output the value of this node (can output more than one if needed)


// global variables, keeps the stack small and more predictable if all variables are global
char screen[80]; // could fill with ={32,32,32} etc but takes memory and gets overwritten quickly as characters come in
int col = 0; // 0 to 19 column number
unsigned long timeCentral = 0xFFFFF800; // synchronise with node 0, as can't reset the millis counter. start high so can spot rollover errors (otherwise every 49 days)
unsigned long timeOffset  = 0xFFFFF800; // offset from my local millis counter - numbers set so if program node 0, it does a refresh immediately

String radioString = ""; 
String txString = "";
int nodeValuesA0[16]; // 16 nodes in a mesh
int nodeValuesA1[16]; // A1 values
unsigned long nodeTimestamps[16]; // timestamps for each value
int lowestNode = 255; // 0 to 15, 255 is false, 0-15 is node last got a time sync signal from

int inPacketLen = 0;

void setup()
{
  Serial.begin(9600); // debug port and serial data out
  // activate LCD module
  lcd.begin (20,4); // for 16 x 2 LCD module
  lcd.setBacklightPin(3,POSITIVE);
  lcd.setBacklight(HIGH);
  lcd.setCursor(0,3);
  printlcdstringln(""); // clear the screen
  printlcdstringln("");
  printlcdstringln("APC220 Mesh");
  printlcdstring("Node number ");
  printlcdstring((String) myNode);
  pinMode(13, OUTPUT);  // led
  pinMode(3, OUTPUT); // analog output 0
  pinMode(5, OUTPUT); // analog output 1
  altSerial.begin(9600);
  
  Serial.print("Mesh node "); // debug output
  Serial.println(myNode);
  outputRS232();
  //createDataMessage();
  //buildOutpacket();
}

void loop()
{
  rxRadio();
  NodeTimeSlot();
}

void printlcdchar(char c)
{
  if ((c>31) && (c<127))
  {
    screen[col+60] = c;
    lcd.print(c);
    col = col+1;
  }  
  if (col > 19) // end of the line
  {
    linefeed(); // scroll up a line if got to the end of the line
    refreshscreen();
    col = 0;
  }
  if (c == 10)
  {
    linefeed(); // scroll up a line if got to the end of the line
    refreshscreen(); // but col stays the same
  }
  if (c==13)
  {
    col=0; // carriage return, but don't scroll
    lcd.setCursor(0,3); // back to beginning of 4th line  
  }  
}  

void printlcdstring(String s)
{
  int i;
  int j;
  j = s.length();
  for (i=0;i<j;i++)
  {
    printlcdchar(s.charAt(i));
  }  
}  

void printlcdstringln(String s)
{
  printlcdstring(s);
  printlcdchar(13);
  printlcdchar(10); 
}  

void lnprintlcd(String s) // reverse of above, print crlf, then print the line, means has 4 lines visible on a 20x4 display at all times
{
  printlcdchar(13);
  printlcdchar(10); 
  printlcdstring(s);
}  

void linefeed() // scroll up a line
{
  int i;
  for(i=0;i<20;i++) // scroll up a line
    {
      screen[i] = screen[i+20]; // move up a line
      screen[i+20] = screen[i+40]; // move up a line
      screen[i+40] = screen[i+60]; // move up a line
      screen[i+60] = 32; // make bottom line all blank
    }  
    col = 0; // reset column to zero
}    

void refreshscreen()
{
  int i;
  lcd.setCursor (0,0);        // go to start of first line
  for(i=0;i<20;i++)
  {
    lcd.print(screen[i]);
  }
  lcd.setCursor (0,1);        // go to start of second line
  for(i=20;i<40;i++)
  {
    lcd.print(screen[i]);
  }   
  lcd.setCursor (0,2);        // go to start of third line
  for(i=40;i<60;i++)
  {
    lcd.print(screen[i]);
  } 
  lcd.setCursor(0,3); // go to start of 4th line
  for(i=60;i<80;i++)
    {
    lcd.print(screen[i]);
  }    
  lcd.setCursor(0,3); // back to beginning of 4th line  
}  
 // ****************** end LCD routines *******************
 
// start radio routines

void txRadio(char c)
{
  // send a byte via radio, will display on the remote screen
  altSerial.write(c);
}  

void rxRadio()
{
  byte c;
  while (altSerial.available())
  {
    c = altSerial.read(); // fetch the byte
    processRadioString(c); // is it a command?
  }  
  
}  

void processRadioString(char c) // converts a byte to a char, so can work with the number
{
  if (radioString.length() > 30) // string too long
  {
    radioString = ""; 
    lnprintlcd("String too long");    
  }  
  
  if ((c>31) && (c<127)) // only send printable characters and 13,, takes 2 bytes to send a value, but can use 13 as an end of line marker
  {
    radioString += c; // add to the string
    //printlcdchar(c); // print the character so can see data coming in
  }
  if (c == 13) // process the command ir there is a carriage return
  {
   digitalWrite(13,HIGH);
  //if (radioString.startsWith("SSID=")) { processSSID();} // parse commands
   if (radioString.startsWith("Time=")) { processTimeSignal();} // parse command
   if (radioString.startsWith("Data=")) { processDataMessage();} // node,sample,timestamp
   //if (radioString.startsWith("Pack=")) { processPacket();} // all data in one large packet - too large somewhere around 100 bytes starts to error
    // and now delete the radiostring
   radioString = ""; 
   digitalWrite(13,LOW);
  }  
}  

void processTimeSignal() // pass Time=03 00000000 where 03 is the node this came from
{
 String s;
 unsigned long centraltime;
 unsigned long mytime;
 int messageFrom;
 s = stringMid(radioString,6,2);
 s = "000000" + s;
 messageFrom = (int) hexToUlong(s);
 s = radioString;
 s = stringMid(s,9,8);
 lnprintlcd("Time msg from ");
 printlcdstring((String) messageFrom);
 if ((messageFrom <= lowestNode) && (myNode != 0)) // current time slot is less or equal than the last node update number, so refresh time, so minimum number of hops
 {
   mytime = (unsigned long) millis();
   centraltime = hexToUlong(s); // convert back to a number
   centraltime = centraltime + 250; // offset to allow for delay with transmission, determined by trial and error, if slow, then add more
   timeOffset = centraltime - mytime;
   lnprintlcd("New ");
   displayTime();
   lowestNode = messageFrom; // update with the new current lowest node number
 }  
}

 

//void millisRollover() // with rollover protection as rollover happens every 49 days. For reference baldengineer.com
//{
//  unsigned long currentMillis = millis()
//  if (((unsigned long ) (currentMillis - previousMillis) >= interval) {} // handles trigger and rollover
//}  

void refreshTime()
{
  timeCentral = ((unsigned long) millis()) + timeOffset;
}

void displayTime()
{
  refreshTime();
  //printlcdstringln((String) timeCentral); // print in decimal
  printlcdstring("Time ");
  printlcdstring(ulongToHex(timeCentral)); // print in hex
}  

void sendTime() // send unsigned long as 4 hex values
{
  txString="";
  refreshTime(); // get the current time
  //txString = String(timeCentral, HEX); // convert to hex
  //printlcdstringln(txString);
  //ulongToHex(timeCentral);
  lnprintlcd(ulongToHex(timeCentral));
  printlcdstring((String) hexToUlong("000000FF"));
}  

String ulongToHex(unsigned long n) // convert an unsigned long number to a hex string 8 characters long
{
   int i;
   byte x;
   String s = "";
   for (i=0;i<8;i++) {
     x = n & 0x0000000F; // mask
     if (x<10) {
       s = (char)(x+48) + s; // 0 to 9
     }else{
       s = (char)(x+55) + s; // A to F  
     } 
     n = n >> 4; // bit shift right
   } 
  return s;
}  

unsigned long hexToUlong(String s)
{
  unsigned long n = 0; // return value
  int i; // general counter
  byte a; // byte at the position in the string
  unsigned long x = 1; // multiply by this number
  for (i=7;i>=0;i--) { // end to the start of the string
    a = (char) s.charAt(i); // get the character and convert to ascii value
    if (a<65) { // if less than 'A'
      n = n + ((a-48) * x); // convert the number
    }else{
      n = n + ((a-55) * x); // convert the letter
    }
    x = x << 4; // multipy by 16  
  }
  return n;
}  

void transmitTime() // send my local time, will converge on the lowest node number's time
{
  String nodeNumber;
  refreshTime();
  //displayTime();
  nodeNumber = stringMid(ulongToHex(myNode),7,2);
  String t="Time=";
  t = t + nodeNumber; // hex value
  t = t + " ";
  t = t + ulongToHex(timeCentral);
  altSerial.println(t); // transmit via radio
}  

String stringLeft(String s, int i) // stringLeft("mystring",2) returns "my"
{
  String t;
  t = s.substring(0,i);
  return t;
}  

String stringMid(String s, int i, int j) // stringmid("mystring",4,3) returns "tri" (index is 1, not zero)
{
  String t;
  t = s.substring(i-1,i+j-1);
  return t;
}  
  
void processDataMessage() // Data=0312AAAABBBBBBBBcrlf where 03 is from, 12 is node (hex), AAAA is integer data, BBBBBBBB is the time stamp
{
  String s;
  unsigned long node;
  unsigned long valueA0;
  unsigned long valueA1;
  unsigned long timestamp;
  unsigned long from;
  unsigned long age;
  unsigned long previousage;
  //printlcdstring("."); // print a dot as data comes in
  s = "000000" + stringMid(radioString,6,2); // node is 2 bytes
  from = hexToUlong(s); // get where this data came from
  s = "000000" + stringMid(radioString,8,2); // node number
  node = hexToUlong(s);
  s = "0000" + stringMid(radioString,10,4); // get the 2 bytes A0 value
  valueA0 = hexToUlong(s);
  s = "0000" + stringMid(radioString,14,4); // get the 2 bytes A1 value
  valueA1 = hexToUlong(s);
  s = stringMid(radioString,18,8);
  timestamp = hexToUlong(s);
  age = (unsigned long) (timeCentral - timestamp);
  previousage = (unsigned long) (timeCentral - nodeTimestamps[node]);
  if (age < previousage) // more recent data so update
  {
    nodeTimestamps[node] = timestamp; // update the time stamp
    nodeValuesA0[node] = (int) valueA0; // update the values
    nodeValuesA1[node] = (int) valueA1; // A1 as well
    //printlcdstring("Update node");
    //printlcdstringln((String) node);
    analogOutput(); // update the analog outputs
  }  
}  

void createDataMessage() // read A0 and A1 create data string
{
  String s;
  byte i;
  String buildMessage;
  unsigned long u;
  String myNodeString;
  updateMyValue(); // update my analog input
  lnprintlcd("My values=");
  printlcdstring((String) nodeValuesA0[myNode]);
  printlcdstring(",");
  printlcdstring((String) nodeValuesA1[myNode]);
  u = (unsigned long) myNode;
  s = ulongToHex(u);
  myNodeString = "Data=" + stringMid(s,7,2); // from me
  delay(150); // for any previous message to go through
  for (i=0;i<16;i++) {
    buildMessage = myNodeString; // start building a string
    u = (unsigned long) i; // 0 to 15 - 2 bytes for the actual node number
    s = ulongToHex(u);
    buildMessage += stringMid(s,7,2);
    u = (unsigned long) nodeValuesA0[i];
    s = ulongToHex(u);
    buildMessage += stringMid(s,5,4); // data value in hex for A0
    u = (unsigned long) nodeValuesA1[i];
    s = ulongToHex(u);
    buildMessage += stringMid(s,5,4); // data value in hex for A1
    s = ulongToHex(nodeTimestamps[i]); // timestamp value for this node
    buildMessage += s;// add this
    altSerial.println(buildMessage); // transmit via radio
    delay(150); // errors on 75, ok on 100
  }
}  

void outputRS232() // output all mesh values in one long hex string to RS232 port aaaabbbbccccdddd etc where a is node 0 value 0, b is node 0 value 1, c is node 1 value 0
{
  String buildMessage;
  byte i;
  String s;
  unsigned long u;
  buildMessage = "All values=";
  for (i=0;i<16;i++) {
    u = (unsigned long) nodeValuesA0[i];
    s = ulongToHex(u);
    buildMessage += stringMid(s,5,4); // data value in hex for A0
    u = (unsigned long) nodeValuesA1[i];
    s = ulongToHex(u);
    buildMessage += stringMid(s,5,4); // data value in hex for A1
   }   
 Serial.println(buildMessage);  
}  
  
void updateMyValue()
{
  refreshTime(); // timecentral
  int sensorValue = analogRead(A0);
  nodeValuesA0[myNode] = sensorValue;
  sensorValue = analogRead(A1);
  nodeValuesA1[myNode] = sensorValue;
  nodeTimestamps[myNode] = timeCentral; // this was updated now
}  

void NodeTimeSlot() // takes 100ms at beginning of slot so don't tx during this time.
// time slots are 4096 milliseconds - easier to work with binary maths
{
  unsigned long t;
  unsigned long remainder;
  int timeSlot = 0;
  refreshTime(); // update timeCentral
  t = timeCentral >> 12; // 4096ms counter
  t = t << 12; // round
  remainder = timeCentral - t;
  if (( remainder >=0) && (remainder < 100)) // beginning of a 4096ms time slot
  {
     digitalWrite(13,HIGH);
     //printlcdstring("Slot ");
     timeSlot = getTimeSlot();
     lnprintlcd((String) timeSlot);
     printlcdstring(" ");
     //printPartnerNode(); // print my partner's value
     printNode(timeSlot); // print the value of this node
     //printHeapStack(); // heap should always be less than stack
     if ( timeSlot == 0) { lowestNode = 255; } // reset if nod zero get time from the lowest node number, 0 to 15 and if 255 then has been reset
     delay(110);
     digitalWrite(13,LOW);
     if (timeSlot == myNode) // transmit if in my time slot
     {
       lnprintlcd("Sending my data"); // all nodes transmit central time, when listening, take the lowest number and ignore others
       transmitTime();
       digitalWrite(13,HIGH); // led on while tx long data message
       createDataMessage(); // send out all my local data via radio
       outputRS232(); // send out all the node values via RS232 port (eg to a PC)
       digitalWrite(13,LOW); // led off
       //buildPacket(); - too large, deleted this
       //printPartnerNode(); // print my partner's value
      }
  }  
}  

int getTimeSlot() // find out which time slot we are in
{
  int n;
  unsigned long x;
  refreshTime();
  x = timeCentral >> 12; // divide by 4096
  x = x & 0x0000000F; // mask so one of 16
  n = (int) x;
  return n;
}  

//void printPartnerNode() // can output partner node's values or do other things
//{
//  printlcdstring("Mate ");
//  printlcdstring((String) partnerNode);
//  printlcdstring("=");
//  printlcdstring((String) nodeValuesA0[partnerNode]);
//  printlcdstring(",");
//  printlcdstring((String) nodeValuesA1[partnerNode]);
//}  

void printNode(int x) // print current node
{
  printlcdstring("= ");
  printlcdstring((String) nodeValuesA0[x]);
  printlcdstring(",");
  printlcdstring((String) nodeValuesA1[x]);
  
}  
  
/* This function places the current value of the heap and stack pointers in the
 * variables. You can call it from any place in your code and save the data for
 * outputting or displaying later. This allows you to check at different parts of
 * your program flow.
 * The stack pointer starts at the top of RAM and grows downwards. The heap pointer
 * starts just above the static variables etc. and grows upwards. SP should always
 * be larger than HP or you'll be in big trouble! The smaller the gap, the more
 * careful you need to be. Julian Gall 6-Feb-2009.
 */
uint8_t * heapptr, * stackptr;
void check_mem() {
  stackptr = (uint8_t *)malloc(4);          // use stackptr temporarily
  heapptr = stackptr;                     // save value of heap pointer
  free(stackptr);      // free up the memory again (sets stackptr to 0)
  stackptr =  (uint8_t *)(SP);           // save value of stack pointer
}

void printHeapStack()
{
  int n;
  check_mem();
  printlcdstring("Heap/Stack=");
  n = (int) heapptr;
  printlcdstring((String) n);
  printlcdstring(",");
  n = (int) stackptr;
  printlcdstring((String) n);
}    

void analogOutput() // output on pins 3 and 5 my partner's voltages
{
  int x;
  x = nodeValuesA0[partnerNode];
  analogWrite(3, x/4); // analog inputs are 0-1023, outputs are 0-255
  x = nodeValuesA1[partnerNode];
  analogWrite(5, x/4);
}
