% Matlab script to generate a lizard shape that interlocks and tiles the
% plane
%
% Please note that this script is intended to show techniques... It is not
% intended to be a general purpose tool

function Build_Lizard

    close all; clear all; clc;

    %% read the verticies of the traced lizard
    fid                = fopen('Traced_Lizard.txt', 'rt');
    traced_lizard      = fscanf(fid, '%f,%f\n', [2,inf])';
    num_verticies      = size(traced_lizard, 1);
    fclose(fid);

    %% plot the traced lizard
    figure
    fill  (traced_lizard(:,1), traced_lizard(:,2), [0.7, 0.7, 1.0], 'LineStyle', 'none');
    title ('Traced Lizard')
    axis square
    
    % show that verticies 8, 26 and 42 almost form an equilateral triangle
    line([traced_lizard( 8, 1), traced_lizard(26, 1)], [traced_lizard( 8, 2), traced_lizard(26, 2)]) 
    line([traced_lizard(26, 1), traced_lizard(42, 1)], [traced_lizard(42, 2), traced_lizard(26, 2)]) 
    line([traced_lizard(42, 1), traced_lizard( 8, 1)], [traced_lizard(42, 2), traced_lizard( 8, 2)]) 
    
    % label the verticies
    for v = 1:size(traced_lizard, 1)
        text(traced_lizard(v,1), traced_lizard(v,2), num2str(v))
    end
    
    %% scale the lizard to approx 100 px (can always be rescaled later)
    A_000          = 100 * traced_lizard / max(max(traced_lizard));
    % note: lizards in the first part of the code have names like A_000 -
    % the A identifies the lizard and the 000 identifies how far it has
    % been rotated about z relative to the traced lizard
    
    %% adjust vertex 8 such that vertices 8, 26 & 42 form an equlateral triangle
    new_y_26_42        = (A_000(26, 2) + A_000(42, 2)) / 2;
    A_000(26, 2)       = new_y_26_42;
    A_000(42, 2)       = new_y_26_42;
    distance_26_42     = A_000(42, 1) - A_000(26, 1);
    A_000( 8,:)        = A_000(26,:) + distance_26_42 * [ 0.5, -sqrt(3)/2 ];
    
    %% build three rotated lizards
    B_120              = Rotate_About_Z(A_000, 120);
    C_240              = Rotate_About_Z(A_000, 240);
    D_240              = Rotate_About_Z(A_000, 240);

    % move lizard B such that it touches lizard A at vertex 8
    offset_B           = B_120( 8, :) - A_000( 8, :);
    B_120( :, 1)       = B_120( :, 1) - offset_B(1);
    B_120( :, 2)       = B_120( :, 2) - offset_B(2);

    % move lizard C such that it touches lizard A at vertex 26
    offset_C           = C_240(26, :) - A_000(26, :);
    C_240( :, 1)       = C_240( :, 1) - offset_C(1);
    C_240( :, 2)       = C_240( :, 2) - offset_C(2);
    
    % move lizard D such that it touches lizard A at vertex 42
    offset_D           = D_240(42, :) - A_000(42, :);
    D_240( :, 1)       = D_240( :, 1) - offset_D(1);
    D_240( :, 2)       = D_240( :, 2) - offset_D(2);
    
    %% plot the four lizards
    figure
    plot  (A_000(:,1), A_000(:,2), 'k', ...
           B_120(:,1), B_120(:,2), 'r', ...
           C_240(:,1), C_240(:,2), 'g', ...
           D_240(:,1), D_240(:,2), 'b')
    title ('Traced Lizard Resized and Rotated')
    axis square
    
    %% trim lizard A vertices near vertex 8
    A_000( 9: 1:15, :) = B_120( 7:-1: 1, :);
    A_000(16: 1:17, :) = B_120(50:-1:49, :);
    
    %% trim lizard A vertices near vertex 26
    A_000(18: 1:25, :) = C_240(34:-1:27, :);
    
    %% trim lizard A vertices near vertex 42
    A_000(35: 1:41, :) = D_240(49:-1:43, :);

    %% trim around the edge by 0.6 to create a small gap between copies
    % this algorithm selects three consecutive points a, b, and c
    % 1) points a and c are shifted inward creating points a2 and c2
    % 2) m1 & n1 are calculated such that the line y = m1 * x + n1 is
    %    parallel to ab and passes through a2 
    % 3) m2 & n2 are calculated such that the line y = m2 * x + n2 is
    %    parallel to bc and passes through c2 
    % 4) equations 2 & 3 are solved simultaneously to find point b2; this
    %    new point will be 0.6 from both lines ab and bc!
    A_000_trim         = zeros(num_verticies, 2);
    for n = 0:num_verticies - 1
        a                 = cat(2, A_000(mod(n - 1, num_verticies) + 1, :), 0);
        b                 = cat(2, A_000(mod(n + 0, num_verticies) + 1, :), 0);
        c                 = cat(2, A_000(mod(n + 1, num_verticies) + 1, :), 0);
        ab                = b - a;
        bc                = c - b;
        a2                = a + 0.6 * cross([ 0; 0;-1], ab / norm(ab));
        c2                = c + 0.6 * cross([ 0; 0;-1], bc / norm(bc));
        m1                = ab(2) / ab(1); % divide by 0 error if ab is vertical
        n1                = a2(2) - a2(1) * m1;
        m2                = bc(2) / bc(1); % divide by 0 error if bc is vertical
        n2                = c2(2) - c2(1) * m2;
        b2                = [-m1, 1;-m2, 1] \ [n1; n2];
        A_000_trim(n+1,:) = b2(:)';
    end

    %% last plot
    figure
    plot  (A_000(:,1)     , A_000(:,2)     , 'k', ...
           A_000_trim(:,1), A_000_trim(:,2), 'k--')
    title ('Lizard Fixed and Trimmed')
    axis square

    %% build new lizards from the fixed and trimmed copy
    A_120              = Rotate_About_Z(A_000     , 120);
    A_120_trim         = Rotate_About_Z(A_000_trim, 120);
    A_240              = Rotate_About_Z(A_000     , 240);
    A_240_trim         = Rotate_About_Z(A_000_trim, 240);

    %% build an SVG file with 9 trimmed lizards
    fid                = fopen('Nine_Lizards.svg', 'wt');
    Add_SVG_Header(fid);

    % the lizards are placed in three groups of three; the offsets between
    % groups can be calculated from the lizard verticies
    dx                 = A_120( 8, 1) - A_120(27, 1) + A_240(25, 1) - A_240( 8, 1);
    dy                 = A_120( 8, 2) - A_120(27, 2) + A_240(25, 2) - A_240( 8, 2);
    
    xPos               =  30;
    yPos               = 120;
    Add_Lizard(fid, A_000_trim, xPos     , yPos)
    Add_Lizard(fid, A_000_trim, xPos + dx, yPos + dy)
    Add_Lizard(fid, A_000_trim, xPos + dx, yPos - dy)
    
    xPos               = xPos + A_000(15, 1) - A_120( 1, 1);
    yPos               = yPos + A_000(15, 2) - A_120( 1, 2);
    Add_Lizard(fid, A_120_trim, xPos     , yPos)
    Add_Lizard(fid, A_120_trim, xPos + dx, yPos + dy)
    Add_Lizard(fid, A_120_trim, xPos + dx, yPos - dy)
    
    xPos               = xPos + A_120(15, 1) - A_240( 1, 1);
    yPos               = yPos + A_120(15, 2) - A_240( 1, 2);
    Add_Lizard(fid, A_240_trim, xPos     , yPos)
    Add_Lizard(fid, A_240_trim, xPos + dx, yPos + dy)
    Add_Lizard(fid, A_240_trim, xPos + dx, yPos - dy)
    
    Add_SVG_Footer(fid)
    fclose(fid);
    
end

% function that rotates xy pairs about the z axis
function [new_xy] = Rotate_About_Z(xy, theta_deg)
    theta_rad = theta_deg * pi / 180.0;
    new_xy    = cat(2, xy(:,1) * cos(theta_rad) - xy(:,2) * sin(theta_rad), ...
                       xy(:,1) * sin(theta_rad) + xy(:,2) * cos(theta_rad));
end

function Add_SVG_Header(fid)
    fprintf(fid, '<svg\n');
    fprintf(fid, 'xmlns:dc  = "http://purl.org/dc/elements/1.1/"\n');
    fprintf(fid, 'xmlns:cc  = "http://creativecommons.org/ns#"\n');
    fprintf(fid, 'xmlns:rdf = "http://www.w3.org/1999/02/22-rdf-syntax-ns#"\n');
    fprintf(fid, 'xmlns:svg = "http://www.w3.org/2000/svg"\n');
    fprintf(fid, 'xmlns     = "http://www.w3.org/2000/svg"\n');
    fprintf(fid, 'version   = "1.1"\n');
    fprintf(fid, 'width     = "2160"\n');
    fprintf(fid, 'height    = "1620"\n');
    fprintf(fid, 'id        = "svg2">\n');
end

function Add_Lizard(fid, lizard, xPos, yPos)
    fprintf(fid, '<g transform="translate(%6.4f,%6.4f)">\n', xPos, yPos);
    fprintf(fid, '<path d="\n');
    for v = 1:size(lizard, 1)
        if v == 1
            fprintf(fid, 'M %10.5f, %10.5f\n', lizard(v,:));
        else
            fprintf(fid, 'L %10.5f, %10.5f\n', lizard(v,:));
        end
    end
    fprintf(fid, 'z"\n');       
    fprintf(fid, 'id="path%03d%03d"\n', xPos, yPos);
    fprintf(fid, 'style="opacity:1.0;fill:#8080FF;stroke:#000000;stroke-width:0.005" />\n');
    fprintf(fid, '</g>\n');
end

function Add_SVG_Footer(fid)
    fprintf(fid, '</svg>\n');
end
