////////////////////////////////////////////////////////////////////////////
// Name: mastermind.c
// Author: B. Gian James <gian@BattleDroids.net>
// Description: The code behind the famous mastermind logic game.
//
// $Id: mastermind.c,v 1.5 2009/09/02 12:53:22 gian Exp $
////////////////////////////////////////////////////////////////////////////
#include <stdlib.h>		// rand,srand,itoa
#include <string.h>		// strtok
#include "common.h"
#include "hardware.h"
#include "usart.h"
#include "config.h"
#include "progmem.h"
#include "mastermind.h"

ColorStruct ColorInfo[NUM_COLORS] =
{
	{ BLUE, 'B', "blue" },
	{ RED, 'R', "red" },
	{ YELLOW, 'Y', "yellow" },
	{ GREEN, 'G', "green" },
	{ WHITE, 'W', "white" },
	{ BLACK, 'K', "black" },
	{ ORANGE, 'O', "orange" }
};

TurnStruct 		TurnInfo[NUM_TURNS];
Color			ColorCompare[NUM_SLOTS];

static uint8	currTurn;
static void		PrintDisplay(void);

void DisplayTurnInfo()
{
	PrintDisplay();
}

// Calculate the number of correct colors
TurnStruct * CalcCorrect()
{
	uint8 cols[NUM_SLOTS] = { 0,0,0,0 };
	//uint8 choicecols[NUM_SLOTS] = { 0,0,0,0 };
	
	for(uint8 x = 0; x < NUM_SLOTS; x++)	// for each color chosen
	{
		for (uint8 z = 0; z < NUM_SLOTS; z++)	// for each color guessed
			if (ColorCompare[x] == TurnInfo[currTurn].colorChoices[z] && cols[z] == 0)
			{
				++(TurnInfo[currTurn].numColorCorrect);
				cols[z] = 1;	// we've counted that color
				break;
			}
	}
	return &TurnInfo[currTurn];
}

#define WHITE_SPACE " \r\t\n"
// Tokenize the buffer holding the player's color choices
uint8 ProcessColors(char * buff)
{
	// Remove the trailing "\n" from input
	//buff[strlen(buff)-1] = '\0';
	
	char * ptr, * colors[NUM_SLOTS];
	uint8	col = 0, ok = 0;
	
	if (strtok(buff,WHITE_SPACE) == NULL) {
		WriteString(PSTR("You didn't choose *any* colors?\r"));
		return 0;
	}
	colors[col] = buff;
	
	while( (ptr = strtok(NULL,WHITE_SPACE)) != NULL ) {
		if (++col > NUM_SLOTS) {
			WriteString(PSTR("Too many colors. Please only choose four.\r"));
			return 0;
		}
			
		colors[col] = ptr;
	}
	if (col != NUM_SLOTS -1) {
		WriteString(PSTR("You must pick four colors. Not three, not two. Four. Five is right out.\r"));
#ifdef DEBUG
		WriteString(PSTR("You chose: "));
		WriteRAMString(itoa((col+1),buff,10));
#endif
		WriteString(PSTR("\r"));
		return 0;
	}
	
	// Verify the colors
	for( col = 0; col < NUM_SLOTS; col++)	// for each color in the a slot
	{
		for (uint8 x = 0; x < NUM_COLORS; x++)	// for all colors
		{
			if (!strcmp(ColorInfo[x].name, colors[col]))
			{
#ifdef DEBUG
				WriteRAMString("Found: ");
				WriteRAMString(colors[col]);
				WriteRAMString("\r");
#endif
				TurnInfo[currTurn].colorChoices[col] = ColorInfo[x].color;
				if (TurnInfo[currTurn].colorChoices[col] == ColorCompare[col])
					++(TurnInfo[currTurn].numColorPlaceCorrect);
				++ok;
				break;
			}
		}
		if (ok != (col+1)) {
		WriteString(PSTR("Couldn't find a color match for: "));
		WriteRAMString(colors[col]);
		WriteRAMString("\r");
		return 0;
	}
		
		
		
	}
	return 1;
}

void StartRound(uint8 round)
{
	currTurn = round;
	TurnInfo[currTurn].turn = (currTurn + 1);
	TurnInfo[currTurn].numColorCorrect = 0;
	TurnInfo[currTurn].numColorPlaceCorrect = 0;
}



void InitGame(uint8 gameMaster)
{
	if (gameMaster == MCU)
	{
#ifdef DEBUG
		WriteString(PSTR("Random value: "));
		char buff[5];
		uint8 i = GetRandom();
		WriteRAMString(itoa(i,buff,10));
		WriteRAMString("\r");
		srand(i);
#else
		srand(GetRandom());
#endif		
		for (uint8 x = 0; x < NUM_SLOTS; x++)
		{

			uint8 r = (rand() % NUM_COLORS);
			ColorCompare[x] = (Color)r;
#ifdef DEBUG
			WriteRAMString(ColorInfo[r].name);
			WriteRAMString(" ");
#endif
		}
	} else	// MCU is the player
	{
	
	}
}

void PrintDisplay()
{
	char buff[85];
	
	for(uint8 x = 0; x < (TERM_COLS - 2); x++)
		WriteChar('*');
	WriteChar('\r');
	sprintf(buff,"** %72s **\r", ".oO[ Mastermind v1.0 AVR ]Oo.");
	WriteRAMString(buff);	
	sprintf(buff,"** %72s **\r","AVR Mastermind. An 8-bit master of logic.");
	WriteRAMString(buff);
	
	uint8 highScore = ReadEEHighScore();
	if (highScore != 0xFF)
		sprintf(buff,"** Current Round: %d    High Score: %d %38s **\r",TurnInfo[currTurn].turn, highScore, " ");
	else
		sprintf(buff,"** Current Round: %d %55s **\r",TurnInfo[currTurn].turn," ");
	WriteRAMString(buff);
	sprintf(buff,"** %9s  %10s %10s %10s %10s    %6s  %6s **\r","         ", " "," "," "," ","Color","Place");
	WriteRAMString(buff);
	sprintf(buff,"** %8s  %10s %10s %10s %10s    %6s  %6s **\r","         ","Color1","Color2","Color3","Color4","Right","Right");
	WriteRAMString(buff);
	for (uint8 x = 0; x < TurnInfo[currTurn].turn; x++)
	{
		sprintf(buff,"** Round %d: %11s %11s %10s %10s  %6d  %6d   **\r",TurnInfo[x].turn,ColorInfo[TurnInfo[x].colorChoices[0]].name,
			ColorInfo[TurnInfo[x].colorChoices[1]].name,ColorInfo[TurnInfo[x].colorChoices[2]].name,ColorInfo[TurnInfo[x].colorChoices[3]].name,
			TurnInfo[x].numColorCorrect, TurnInfo[x].numColorPlaceCorrect);
		WriteRAMString(buff);
	}
	if (TurnInfo[currTurn].turn == NUM_TURNS)
	{
		sprintf(buff,"** MCU picked: %8s %11s %10s %10s  %6s  %6s   **\r", ColorInfo[ColorCompare[0]].name ,ColorInfo[ColorCompare[1]].name,
			ColorInfo[ColorCompare[2]].name, ColorInfo[ColorCompare[3]].name," ", " ");
		WriteRAMString(buff);
	}
	
	for(uint8 x = 0; x < (TERM_COLS - 2); x++)
		WriteChar('*');
	WriteChar('\r');	
	
}