/************************************************************************/
/*																		*/
/*	LS1.cpp		--		Definition for LS1 library 	    				*/
/*																		*/
/************************************************************************/
/*	Author:		Eric Marsh												*/
/*	Copyright 2016, Digilent Inc.										*/
/************************************************************************/
/*  File Description:													*/
/*		This file defines functions for the PmodLS1						*/
/*																		*/
/************************************************************************/
/*  Revision History:													*/
/*																		*/
/*	06/22/2016(EricM): created											*/
/*																		*/
/************************************************************************/


/* ------------------------------------------------------------ */
/*				Include File Definitions						*/
/* ------------------------------------------------------------ */
#include "LS1.h"
#include <DSPI.h>

/* ------------------------------------------------------------ */
/*					Definitions									*/
/* ------------------------------------------------------------ */


/* ------------------------------------------------------------ */
/*				Procedure Definitions							*/
/* ------------------------------------------------------------ */


/* ------------------------------------------------------------ */
/*        LS1::LS1
**
**        Synopsis:
**				
**        Parameters:
**
**
**
**        Return Values:
**                void 
**
**        Errors:
**
**
**        Description:
**			Class constructor. Performs variables initialization tasks
**
**
*/
LS1::LS1()
{
	pdspi = NULL;
	
}


/* ------------------------------------------------------------ */
/*        LS1::begin
**
**        Synopsis:
**				myLS1.begin(PAR_ACCESS_SPI0);
**        Parameters:
**				uint8_t bAccessType	- the SPI interface where the Pmod is connected. It can be one of:
**					0	PAR_ACCESS_SPI0	- indicates SPI port 2, connetor JB
**					1	PAR_ACCESS_SPI1	- indicates SPI port 1, connector J1

**				uint8_t spd - The frequency at which the SPI clock should be set to. 1 Mhz by default
**
**        Return Values:
**                void 
**
**        Errors:
**
**
**        Description:
**				This function initializes the specific SPI interface used, setting the SPI frequency to a default value of 1 MHz.
**
**
*/
void LS1::begin(uint8_t bAccessType, uint32_t spd)
{
	if(bAccessType == PAR_ACCESS_SPI0)
	{
		pdspi = new DSPI0();
		m_SSPin = PIN_DSPI0_SS;	// default SS pin for SPI0
	}
	if(bAccessType == PAR_ACCESS_SPI1)
	{
		pdspi = new DSPI1();
		m_SSPin = PIN_DSPI1_SS;	// default SS pin for SPI1
	}
	if(pdspi != NULL)
	{
		pdspi->begin(m_SSPin);
		pdspi->setMode(DSPI_MODE3);
		pdspi->setSpeed(spd);
		
		// set up input pins
		S1 = m_SSPin;
		S2 = m_SSPin + 1;
		S3 = m_SSPin + 2;
		S4 = m_SSPin + 3;
	}
}

/* ------------------------------------------------------------ */
/*        LS1::AllSensorsEnabled
**
**        Synopsis:
**				myLS1.AllSensorsEnabled();
**        Parameters:
**				None
**
**        Return Values:
**                bool - True if every sensor has reached their threshold, and outputs a 1
**						 False otherwise 
**
**        Errors:
**
**
**        Description:
**				If every sensor is past its indicator threshold and outputs a 1, then this function returns true
**
**
*/
bool LS1::AllSensorsEnabled()
{
	// updates sensors and assign their outputs to sensor_status
	UpdateSensorsStatus();
	
	// if sensor status is 15, then each sensor is activated (0x1111 is 15 in binary)
	return sensor_status == 15;
}

/* ------------------------------------------------------------ */
/*        LS1::AllSensorsDisabled
**
**        Synopsis:
**				myLS1.AllSensorsDisabled();
**        Parameters:
**				None
**
**        Return Values:
**                bool - True if every sensor has not reached their threshold, and outputs a 0
**						 False otherwise 
**
**        Errors:
**
**
**        Description:
**				If every sensor has not passed their threshold, then this function returns True
**
**
*/
bool LS1::AllSensorsDisabled()
{
	// updates sensors and assign their outputs to sensor_status
	UpdateSensorsStatus();
	
	// if sensor_status is 0, then that means none of the sensors are activated
	return sensor_status == 0;
}

/* ------------------------------------------------------------ */
/*        LS1::SensorIsEnabled
**
**        Synopsis:
**				myLS1.SensorIsEnabled(0);
**        Parameters:
**				int - the number of the indicator pin to check (0 - 3). 0 is S1, 1 is S2 and so forth
**
**        Return Values:
**                bool - True if the sensor with the peramator value is active
**						 False otherwise 
**
**        Errors:
**
**
**        Description:
**				Checks the sensor specified in the parameter. If the sensor has reached its threshold and is therefore activated, return true. Otherwise return false
**
**
*/
bool LS1::SensorIsEnabled(int sensor_num)
{
	return digitalRead(m_SSPin + sensor_num) == 1;
}

/* ------------------------------------------------------------ */
/*        LS1::UpdateSensorsStatus
**
**        Synopsis:
**				myLS1.UpdateSensorsStatus();
**        Parameters:
**				None
**
**        Return Values:
**                None
**
**        Errors:
**
**
**        Description:
**				Updates sensor_status byte with the status of the sensors. 1 if sensor is activated, 0 if not. Sensor 1 is the first bit, sensor 2 is the second, and so on.
**
**
*/
void LS1::UpdateSensorsStatus()
{
	byte bS1 = digitalRead(S1)*(0x1);
    byte bS2 = digitalRead(S2)*(0x2);
    byte bS3 = digitalRead(S3)*(0x4);
    byte bS4 = digitalRead(S4)*(0x8);
	sensor_status = (bS1 | bS2 | bS3 | bS4);
}

/* ------------------------------------------------------------ */
/*        LS1::GetSensorsStatus
**
**        Synopsis:
**				myLS1.GetSensorsStatus();
**        Parameters:
**				None
**
**        Return Values:
**                byte sensor_status - a 4 bit value with the status of the sensors
**
**        Errors:
**
**
**        Description:
**				Updates sensor_status and returns it. If it returns 15, that means all sensors are activated. If 0, then none of them are. 
**
**
*/
byte LS1::GetSensorsStatus()
{
	// updates sensors and assign their outputs to sensor_status
	UpdateSensorsStatus();
	
	return sensor_status;
}

