;-----------------------------------------------------------------------------------------------;
;		LANC Controller																			;
;		---------------																			;
;																								;
;	Author: Craig Smith																			;
;	Date:	August 2008																			;
;																								;
;	4MHz internal resonator used 																;
;																								;
;																								;
;																								;
;	The program reads and writes the LANC line of a Sony camcorder. It has a record and a stop  ;
;	button and will turn on an LED when the camera is actually recording. This will be used		;
;	to control the camera when it is in a bag with a helmet camera attacthed.					;
;																								;
;	RB4	-	Record button																		;
;	RB5	-	Stop Button																			;
;	RB7	-	Record LED																			;
;	RA0	-	LANC																				;
;	RA5	-	Status LED (on - all OK, flashing - battery low)									;
;																								;
;	When reading and writing the LANC codes, the codes are inverted. So if we read a 1, we save	;
;	a 0 etc.																					;
;-----------------------------------------------------------------------------------------------;


#include <p16F690.inc>
    __config (_INTRC_OSC_NOCLKOUT & _WDT_OFF & _PWRTE_OFF & _MCLRE_OFF & _CP_OFF & _BOR_OFF & _IESO_OFF & _FCMEN_OFF)


cblock 0x20
	Byte					; holds the Byte we are reading on the LANC line

	ReadError				; bit 0 is used to hold a 1 if there is a read error

	Byte0					; holds Byte0
	Byte1					; holds Byte1
	Byte2					; holds Byte2
	Byte3					; holds Byte3
	Byte4					; holds Byte4
	Byte5					; holds Byte5
	Byte6					; holds Byte6
	Byte7					; holds Byte7

	SendByte0				; used to hold the value that will be sent in Byte0
	SendByte1				; used to hold the value that will be sent in Byte1


	Buttons					; bit 0 holds a 1 when a button has been pressed
	Flash					; Flash incrtements every program loop, and used to flash status LED
	Sender					; used when sending a command. A command need to be sent 3 - 5 times
							; to the camera, so this is used as a counter

	BitCount				; counter used in bit counting
	Count					; counter
	Temp					; tempory register



endc


	org 0
	goto	Start			; jump to Start


;-----------------------------------------------------------------------------------------------;
; Subroutines are here at the top																;
;-----------------------------------------------------------------------------------------------;

;-----------------------------------------------------------------------------------------------;
; Read and Write LANC subs setup the TRISA register to change them for reading and writing		;
;-----------------------------------------------------------------------------------------------;
ReadLANC:
    bsf		STATUS,RP0			; register page 1
	movlw	1
	movwf	TRISA				; set RA0 as an input
	bcf		STATUS,RP0			; back to Register Page 0
	return

WriteLANC:
    bsf		STATUS,RP0			; register page 1
	movlw	0
	movwf	TRISA				; set RA0 as an output
	bcf		STATUS,RP0			; back to Register Page 0
	return


;-----------------------------------------------------------------------------------------------;
; BatteryFlash flashes the status LED if the battery is low										;
;-----------------------------------------------------------------------------------------------;
BatteryFlash:
	btfsc	Flash,4			; is bit 2 of the Flasher register a 1
	bsf		PORTA,5
	btfss	Flash,4			; is bit 2 of the Flasher register a 0
	bcf		PORTA,5
	return


;-----------------------------------------------------------------------------------------------;
; TapeEnd flashes the recording LED if the camera is at the end of the tape						;
;-----------------------------------------------------------------------------------------------;
TapeEnd:
	btfsc	Flash,4			; is bit 2 of the Flasher register a 1
	bsf		PORTB,7
	btfss	Flash,4			; is bit 2 of the Flasher register a 0
	bcf		PORTB,7
	return

;-----------------------------------------------------------------------------------------------;
; RecordButton is a debouce debouce sub for the for the Record button							;
;-----------------------------------------------------------------------------------------------;
RecordButton:
	movlw	0xFF			; put FFh into W
	movwf	Temp			; put W into Temp

RecordButton2:
	btfsc	PORTB,4			; see if RB4 is low
	return					; if not return
	decfsz	Temp			; if so, decrement Temp
	goto	RecordButton2	; jump back up for another decrement
	bsf		Buttons,0		; if we counted all the way from 256 down to 0, set bit 0 of Buttons reg


RecordButton3:				; make sure the button has been released for a 256 count
	movlw	0xFF			; put FFh into W
	movwf	Temp

RecordButton4:
	btfss	PORTB,4			; make sure the button has been released
	goto	RecordButton3
	decfsz	Temp
	goto	RecordButton4

	movlw	0x18
	movwf	SendByte0
	movlw	0x3A			; record
	movwf	SendByte1		; the 2 bytes to command the camera to record

	movlw	5
	movwf	Sender			; put 5 into the Sender reg
	return



;-----------------------------------------------------------------------------------------------;
; StopButton is a debouce debouce sub for the for the Stop button								;
;-----------------------------------------------------------------------------------------------;
StopButton:
	movlw	0xFF			; put FFh into W
	movwf	Temp			; put W into Temp

StopButton2:
	btfsc	PORTB,5			; see if RB5 is low
	return					; if not return
	decfsz	Temp			; if so, decrement Temp
	goto	StopButton2		; jump back up for another decrement
	bsf		Buttons,0		; if we counted all the way from 256 down to 0, set bit 0 of Buttons reg


StopButton3:				; make sure the button has been released for a 256 count
	movlw	0xFF			; put FFh into W
	movwf	Temp

StopButton4:
	btfss	PORTB,5			; make sure the button has been released
	goto	StopButton3
	decfsz	Temp
	goto	StopButton4

	movlw	0x18
	movwf	SendByte0
	movlw	0x30			; stop
	movwf	SendByte1		; the 2 bytes to command the camera to stop

	movlw	5
	movwf	Sender			; put 5 into the Sender reg
	return



;-----------------------------------------------------------------------------------------------;
; The waitHalf sub delays the program by 48uS. This is used when the start bit of each byte is	;
; found so we are reading the bits in the middle, not on the transistion. Half a bit length is	;
; actually 52uS, but most lines of code take 1uS, these lines of code in the program have been	;
; worked to make sure the delay is correct.														;
;-----------------------------------------------------------------------------------------------;
WaitHalf:
	movlw	.14
	movwf	Count
	nop

WaitHalfLoop:
	decfsz	Count
	goto	WaitHalfLoop
	return


;-----------------------------------------------------------------------------------------------;
; ReadByte is called to read a byte on the LANC line, the byte is stored in the Byte register	;
; The bits we read are inverted before they are stored.											;
;-----------------------------------------------------------------------------------------------;
ReadByte:
	clrf	Byte				; clear the Byte register
	btfsc	PORTA,0				; skip when the LANC line drops to 0
	goto	$-1

	call	WaitHalf			; wait for 50uS

	btfsc	PORTA,0				; make sure the LANC line is still 0
	bsf		ReadError,0			; set the read error flag

								; now we are in the middle of the start bit, so we wait for 104uS
								; so we are in the middle of Bit0

	call	WaitHalf			; wait for 48uS (about half of a bit length)
	call	WaitHalf			; wait for 48uS (about half of a bit length)
	nop
	nop
	nop
	nop
	nop
	nop							; wait another 5uS


								; CheckBit is looped 8 times in order to read the 8 bits of the byte

	movlw	.8					; put 8 into W
	movwf	BitCount

CheckBit:
	rrf		Byte,1				; rotate the Byte register right
	bcf		Byte,7				; clear bit 7 of the Byte register incase the Carry flag has been rotated into it

	btfss	PORTA,0				; is the LANC line 0
	bsf		Byte,7				; if the LANC line was 'high' clear bit 7 of the Byte register

	call	WaitHalf			; wait for 52uS (half of a bit length)
	call	WaitHalf			; wait for 52uS (half of a bit length)

	decfsz	BitCount			; have we read all 8 bits
	goto	CheckBit
	return



;-----------------------------------------------------------------------------------------------;
; WriteByte is called to write a byte on the LANC line, the Byte register is written to LANC	;
;-----------------------------------------------------------------------------------------------;
WriteByte:
	btfsc	PORTA,0				; skip when the LANC line drops to 0
	goto	$-1

	call	WaitHalf			; wait for 50uS

	btfsc	PORTA,0				; make sure the LANC line is still 0
	bsf		ReadError,0			; set the read error flag

								; now we are in the middle of the start bit, so we wait for 104uS
								; so we are in the middle of Bit0

	call	WaitHalf			; wait for 48uS (about half of a bit length)


	call	WriteLANC			; set up RA0 for writing to LANC

								; WriteBit is looped 8 times in order to write the 8 bits of the byte

	movlw	.8					; put 8 into W
	movwf	BitCount

WriteBit:
	btfss	Byte,0				; is bit 0 of the Byte register a 1
	bsf		PORTA,0				; output a 0
	btfsc	Byte,0				; is bit 0 of the Byte register a 0
	bcf		PORTA,0				; output a 1

	rrf		Byte				; rotate the Byte register to the right

	call	WaitHalf			; wait for 52uS (half of a bit length)
	call	WaitHalf			; wait for 52uS (half of a bit length)

	decfsz	BitCount			; have we read all 8 bits
	goto	WriteBit

	call	ReadLANC
	return






;-----------------------------------------------------------------------------------------------;
; Main program starts here																		;
;-----------------------------------------------------------------------------------------------;

Start:
    bsf		STATUS,RP0			; register page 1
	clrf	TRISC				; Set portC, all output
	clrf	TRISA				; Set portA, all output

	movlw	b'01110000'
	movwf	TRISB				; PORT B output bit 7, input bits 6,5 and 4


	bsf		STATUS,RP1			; select Page 2,
	bcf		STATUS,RP0			; by setting RP1 in Status register and clearing RP0
	clrf	ANSEL				; select Digital I/O on all ports
	clrf	ANSELH				; "		"
    bcf		STATUS,RP1			; back to Register Page 0

	clrf	Sender				; clear the Sender register
	clrf	Buttons				; clear the Buttons register

	clrf	PORTB				; clear PORTB
	clrf	PORTC				; clear PORTC

	movlw	b'00100000'
	movwf	PORTA				; turn on Status LED



Start2:
	btfss	PORTB,4				; is the Record button being pressed
	call	RecordButton
	btfss	PORTB,5				; is the Stop button being pressed
	call	StopButton
	incf	Flash				; increment the Flash register

Sync:
	call	ReadLANC			; set for reading

Sync2:
	clrf	Count				; clear the Count register

SyncLoop:
	btfss	PORTA,0				; see if the LANC line is high
	goto	Sync2

								; If the Count register gets to zero, we know we are in the interframe
								; gap. 256 x 5uS (the time for the loop) = 1280uS. There is only about
								; 200uS between bytes on the LANC line!

	decfsz	Count				; decrement the Count register
	goto	SyncLoop



	movfw	Sender				; if there is a value in the Sender register
	iorlw	0					; that means that we need to write the first
	btfsc	STATUS,Z			; 2 bytes, not read them
	goto	ReadFrom0




; Write byte 0
	clrf	ReadError
	movfw	SendByte0
	movwf	Byte				; put SendByte 0 into Byte
	call	WriteByte
	btfsc	ReadError,0			; did the sub return a read error?
	goto	Start2				; if so, return to Start2

; Write byte 1
	clrf	ReadError
	movfw	SendByte1
	movwf	Byte				; put SendByte 1 into Byte
	call	WriteByte
	btfsc	ReadError,0			; did the sub return a read error?
	goto	Start2				; if so, return to Start2

	decf	Sender				; decrement the Sender register

	goto	ReadFrom2




ReadFrom0:
; Read byte 0
	clrf	ReadError
	call	ReadByte			; read the first byte on the LANC line
	btfsc	ReadError,0			; did the sub return a read error?
	goto	Start2				; if so, return to Start2
	movfw	Byte
	movwf	Byte0

; Read byte 1
	clrf	ReadError
	call	ReadByte			; read the first byte on the LANC line
	btfsc	ReadError,0			; did the sub return a read error?
	goto	Start2				; if so, return to Start2
	movfw	Byte
	movwf	Byte1


ReadFrom2:
; Read byte 2
	clrf	ReadError
	call	ReadByte			; read the first byte on the LANC line
	btfsc	ReadError,0			; did the sub return a read error?
	goto	Start2				; if so, return to Start2
	movfw	Byte
	movwf	Byte2

; Read byte 3
	clrf	ReadError
	call	ReadByte			; read the first byte on the LANC line
	btfsc	ReadError,0			; did the sub return a read error?
	goto	Start2				; if so, return to Start2
	movfw	Byte
	movwf	Byte3

; Read byte 4
	clrf	ReadError
	call	ReadByte			; read the first byte on the LANC line
	btfsc	ReadError,0			; did the sub return a read error?
	goto	Start2				; if so, return to Start2
	movfw	Byte
	movwf	Byte4

; Read byte 5
	clrf	ReadError
	call	ReadByte			; read the first byte on the LANC line
	btfsc	ReadError,0			; did the sub return a read error?
	goto	Start2				; if so, return to Start2
	movfw	Byte
	movwf	Byte5

; Read byte 6
	clrf	ReadError
	call	ReadByte			; read the first byte on the LANC line
	btfsc	ReadError,0			; did the sub return a read error?
	goto	Start2				; if so, return to Start2
	movfw	Byte
	movwf	Byte6

; Read byte 7
	clrf	ReadError
	call	ReadByte			; read the first byte on the LANC line
	btfsc	ReadError,0			; did the sub return a read error?
	goto	Start2				; if so, return to Start2
	movfw	Byte
	movwf	Byte7


; check to see if camera is recording
	movfw	Byte4				; put Byte4 into W
	xorlw	0x04				; xor W ith 04 (recording)
	btfsc	STATUS,Z			; if it's the same, put the recording light on
	bsf		PORTB,7
	btfss	STATUS,Z			; if it's not the same, turn off the recording light
	bcf		PORTB,7

	movfw	Byte4				; portC used to monitor Byte4
	movwf	PORTC

; check to see if the camera has a low battery
	btfsc	Byte5,2				; if the low battery flag set in Byte5
	call	BatteryFlash
	btfss	Byte5,2				; if the low battery flag clear in Byte5
	bsf		PORTA,5				; turn on the status LED

; check to see if the camera is at the end of the tape
	movfw	Byte4				; put Byte4 into W
	xorlw	0x62				; xor W ith 62 (low battery)
	btfsc	STATUS,Z			; if it's the same, call tape end sub
	call	TapeEnd
	goto	Start2


end