/*
 JeonLab Car Digital Compass & Thermometer
 
 Functions:
 LCD display
 Digital compass (MAG3110)
 Interior temperature
 LCD backlight brightness automatic adjust
 
 MAG3110 read: Original code from Sparkfun example code
 (https://www.sparkfun.com/products/10619)
 */

#include <Wire.h>
#include <LiquidCrystal.h>
#include <avr/eeprom.h>

#define MAG_ADDR  0x0E 
int avgX, avgY;

struct settings_t
{
  long maxX, minX, maxY, minY;
} 
settings;

LiquidCrystal lcd(2,3,6,7,8,9);

void setup()
{
  Wire.begin();    
  config(); //configuration of the magnetometer, MAG3110       
  pinMode(10,INPUT); //temperature sensor
  pinMode(5,OUTPUT); //LCD backlight LED
  analogWrite(5, 100);

  //read previously stored calibration data from EEPROM
  eeprom_read_block((void*)&settings, (void*)0, sizeof(settings));
  avgX=(settings.maxX+settings.minX)/2;
  avgY=(settings.maxY+settings.minY)/2;

  lcd.begin(16, 2);
}

void loop()
{
  //read the magnetometer and calculate heading
  float heading = (atan2(readx()-avgX, ready()-avgY))*180/PI;

  //read and calculate the interior temperature in celcius
  float temp = (analogRead(0)/1024.0*5000.0-500)/10;

  //read ambient brightness from the photo-transistor
  int pTRread=analogRead(1);
  int brightness = map(pTRread, 930, 1023, 50, 255); 
  analogWrite(5,brightness); //LCD backlight LED voltage control by PWM 

  //diaplay compass bearing
  lcd.setCursor(0,0);
  lcd.print("                ");
  lcd.setCursor(0,0);
  if (abs(heading) <= 22.5) lcd.print("N ");
  if (abs(heading) >= 157.5) lcd.print("S ");
  if (heading >= 67.5 && heading <= 112.5) lcd.print("E ");
  if (heading <= -67.5 && heading >= -112.5) lcd.print("W ");
  if (heading > 22.5 && heading < 67.5) lcd.print("NE");
  if (heading < -22.5 && heading > -67.5) lcd.print("NW");
  if (heading > 112.5 && heading < 157.5) lcd.print("SE");
  if (heading < -112.5 && heading > -157.5) lcd.print("SW");

  if (heading < 0) heading += 360.0;

  //display heading
  lcd.setCursor(3,0);
  lcd.print(int(heading));

  lcd.setCursor(7,0);
  lcd.print(int(temp));
  lcd.print("C");
  
  lcd.setCursor(13,0);
  lcd.print(int(temp*9.0/5.0+32.0));
  lcd.print("F");

  lcd.setCursor(0,1);
  lcd.print("                ");
  lcd.setCursor(0,1);
  lcd.print("pTR "); //show PWM value for a reference
  lcd.print(pTRread);
  lcd.setCursor(9,1);
  lcd.print("PWM ");
  lcd.print(brightness);

  if (digitalRead(10) == HIGH) //monitor calibration button status
  {
    analogWrite(5,200);
    lcd.setCursor(0,1);
    lcd.print("calibrating.....");  
    delay(1000);
    calXY();
  }
  delay(1000);
}

void calXY() //magnetometer calibration: finding max and min of X, Y axis
{
  int tempXmax, tempXmin, tempYmax, tempYmin, newX, newY;

  lcd.setCursor(0,1);
  lcd.print("Rotate the car  ");
  delay(1000);
  lcd.setCursor(0,1);
  lcd.print("and press button");
  delay(1000);
  lcd.setCursor(0,1);
  lcd.print("Now, begin!     ");
  delay(500);

  tempXmax = tempXmin = readx();
  tempYmax = tempYmin = ready();
  while(digitalRead(10) == LOW) 
  {
    newX = readx();
    newY = ready();
    if (newX > tempXmax) tempXmax = newX;
    if (newX < tempXmin) tempXmin = newX;
    if (newY > tempYmax) tempYmax = newY;
    if (newY < tempYmin) tempYmin = newY;
  }
  settings.maxX = tempXmax;
  settings.minX = tempXmin;
  settings.maxY = tempYmax;
  settings.minY = tempYmin;

  //store new X, Y values in the EEPROM
  eeprom_write_block((const void*)&settings, (void*)0, sizeof(settings));

  avgX=(settings.maxX+settings.minX)/2;
  avgY=(settings.maxY+settings.minY)/2;

  lcd.setCursor(0,1);
  lcd.print("Calibration done");
  delay(3000);
  lcd.setCursor(0,1);
  lcd.print("                ");

}

void config(void) //MAG3110 config taken from Sparkfun example
{
  Wire.beginTransmission(MAG_ADDR); // transmit to device 0x0E
  Wire.write(0x10);              // cntrl register1
  Wire.write(0);                 // send 0x00, standby mode
  Wire.endTransmission();       // stop transmitting

  delay(15);

  Wire.beginTransmission(MAG_ADDR); // transmit to device 0x0E
  Wire.write(0x11);              // cntrl register2
  Wire.write(0x80);              // send 0x80, enable auto resets
  Wire.endTransmission();       // stop transmitting

  delay(15);

  Wire.beginTransmission(MAG_ADDR); // transmit to device 0x0E
  Wire.write(0x10);              // cntrl register1
  Wire.write(0x19);                 // send 0x01, active mode
  Wire.endTransmission();       // stop transmitting

  delay(15);
}

void print_values(void)
{
}

int readx(void) //MAG3110 read X value taken from Sparkfun example
{
  int xl, xh;  //define the MSB and LSB

  Wire.beginTransmission(MAG_ADDR); // transmit to device 0x0E
  Wire.write(0x01);              // x MSB reg
  Wire.endTransmission();       // stop transmitting

  delayMicroseconds(2); //needs at least 1.3us free time between start and stop

  Wire.requestFrom(MAG_ADDR, 1); // request 1 byte
  while(Wire.available())    // slave may send less than requested
  { 
    xh = Wire.read(); // receive the byte
  }

  delayMicroseconds(2); //needs at least 1.3us free time between start and stop

  Wire.beginTransmission(MAG_ADDR); // transmit to device 0x0E
  Wire.write(0x02);              // x LSB reg
  Wire.endTransmission();       // stop transmitting

  delayMicroseconds(2); //needs at least 1.3us free time between start and stop

  Wire.requestFrom(MAG_ADDR, 1); // request 1 byte
  while(Wire.available())    // slave may send less than requested
  { 
    xl = Wire.read(); // receive the byte
  }

  int xout = (xl|(xh << 8)); //concatenate the MSB and LSB
  return abs(xout);
}

int ready(void) //MAG3110 read Y value taken from Sparkfun example
{
  int yl, yh;  //define the MSB and LSB

  Wire.beginTransmission(MAG_ADDR); // transmit to device 0x0E
  Wire.write(0x03);              // y MSB reg
  Wire.endTransmission();       // stop transmitting

  delayMicroseconds(2); //needs at least 1.3us free time between start and stop

  Wire.requestFrom(MAG_ADDR, 1); // request 1 byte
  while(Wire.available())    // slave may send less than requested
  { 
    yh = Wire.read(); // receive the byte
  }

  delayMicroseconds(2); //needs at least 1.3us free time between start and stop

  Wire.beginTransmission(MAG_ADDR); // transmit to device 0x0E
  Wire.write(0x04);              // y LSB reg
  Wire.endTransmission();       // stop transmitting

  delayMicroseconds(2); //needs at least 1.3us free time between start and stop

  Wire.requestFrom(MAG_ADDR, 1); // request 1 byte
  while(Wire.available())    // slave may send less than requested
  { 
    yl = Wire.read(); // receive the byte
  }

  int yout = (yl|(yh << 8)); //concatenate the MSB and LSB
  return abs(yout);
}
