#include <ESP8266WiFi.h>
#include <WiFiClient.h>
#include <EEPROM.h>
#include <ESP8266WebServer.h>
#include <ESP8266HTTPClient.h>
#include <WiFiUdp.h>
#include <ESP8266mDNS.h>
#include <functional>
#include "Declarations.h"
#include <TimeLib.h>
#include "SSD1306Ascii.h"
#include "SSD1306AsciiWire.h"
#include <OneWire.h>
#include <DallasTemperature.h>
#include <Wire.h>
#include <Adafruit_Sensor.h>

#define OLED_I2C_ADDRESS 0x3C
const byte DS3231_I2C_ADDRESS = 0x68; // I2C address for the DS3231 RTC
SSD1306AsciiWire oled;
// ---------------------------------------------------------------
OneWire oneWire(ONE_WIRE_BUS);
DallasTemperature sensors(&oneWire);

ESP8266WebServer serverAP(80);
ESP8266WebServer GeyserPages(80);
// ---------------------------------------------------------------
uint8_t buf[10];
uint8_t buflen = sizeof(buf);
 // --------------------------------------------------------------
IPAddress timeServer(129, 6, 15, 29); // time.nist.gov NTP server
// Alternative NTP servers addresses
// IPAddress timeServer(129, 6, 15, 27);
// IPAddress timeServer(129, 6, 15, 28);
// IPAddress timeServer(129, 6, 15, 30);

// IPAddress timeServerIP; // time.nist.gov NTP server address
const char* ntpServerName = "time.nist.gov";
unsigned int localPort_NTP = 2390;      // local port to listen for UDP packets
const int NTP_PACKET_SIZE = 48; // NTP time stamp is in the first 48 bytes of the message
const int TimeZone = 2;        // Time zone adjustment for NTP time server
byte packetBuffer2[ NTP_PACKET_SIZE]; //buffer to hold incoming and outgoing packets 
const byte NTP_WaitTime = 100; // time in multiples of 10 mS that device will wait 
WiFiUDP udp;          // A UDP instance for NTP clock
boolean wifiConnected = false;
  // ------------------------------------------------------------------
boolean NTP_Timeout;   // if true, NTP attempt timed out
char KeyboardBuffer[MaxStringLength];      // Keyboard buffer
 // -------------------- LED timer and flash rates ----------------------------------
unsigned long LoopTimer;     // used as timer for timed operations. Eg LED flashing
unsigned long NTP_Timer;      // used as timer for the NTP time interval
const unsigned long One_Minute_timer_Tick_Threshold = 60000; // used to determine one minute timer activity (mS)
byte Command = 255;       // input buffer for main screen commands
const boolean AP_SwitchOn = LOW;   // IO status for AP switch ON
const boolean AP_SwitchOff = !AP_SwitchOn;
// --------------------------------------------------------
void setup()
{
  Serial.begin(SERIAL_BAUDRATE);
  EEPROM.begin(512);            // Enable 512 bytes of EEPROM
  Serial.println("");
   // ------------------------------------------------
  pinMode(ONE_WIRE_BUS,INPUT_PULLUP);  
  Wire.begin(Board_SDA,Board_SCL);      
   // ------------ Check if OLED screen fitted ------
      Wire.setClock(400000L);
      Wire.beginTransmission(OLED_I2C_ADDRESS);
      byte error = Wire.endTransmission();
      if (error > 0)
      {
        Serial.println(F("No OLED display fitted"));
        OLED = false;
      }
      else 
      {
        OLED = true;
        Serial.println(F("OLED display detected"));   
      }
   // ------------ Initialise OLED screen ----------
  if(OLED)   // only if OLED display is enabled
  {  
      oled.begin(&Adafruit128x64, OLED_I2C_ADDRESS);  
      oled.setFont(Adafruit5x7);
      // ----------------- OLED boot screen ---------------------
      oled.clear();
      oled.println(Firmware_Version.substring(0,20));
      oled.println("");
      oled.println("Device boot");
      oled.println("");     
  }  // end of   if(OLED)   // only if OLED display is enabled
   // ----------------------------------------------------------- 
  Auto_EEPROM_Check();   // check to see if EEPROM is initialised, if not set to defaults
  pinMode(AP_Switch, INPUT_PULLUP);   
  pinMode(GeyserControlRelay, OUTPUT);           // Initialise Relay output pins
  digitalWrite(GeyserControlRelay, GeyserOff);   // set relay OFF
  pinMode(Heartbeat_LED,OUTPUT);  // Inititialise Heart beat LED   
  Serial.println("");
  Serial.println(Firmware_Version);   // display firmware revision number
  Serial.println(F("Developer: Chris Gimson     chris@gimson.co.uk"));
  EnableMessages = EEPROM.read(EEPROM_EnableMessages);  // set the display messages flag 
  FirstPass = false;    // used in OLED update function
  if (EnableMessages)       // If display messages is enabled
  {
     Serial.println("");
     Serial.println(F("Display messaging enabled"));
  }
  SelfCheck = Enabled;
  //DownloadWatchDogType(false);   // Download type of watchdog in use & which output is Alexi enabled  
  Download_WiFi_Login_Details_From_EEPROM(false);   // Download EEPROM values to working memory
  DownloadFrom_EEPROM_GeyserControlParameters(false); 
  byte i = 0;
  while (i < 5)   // download Temp sensor addresses
  {          
      TempSensorIndex[i] = EEPROM.read(EEPROM_TempSensorIndex + i);  // get stored index number for each sensor
      if(TempSensorIndex[i] < 0 || TempSensorIndex[i] > 4) TempSensorIndex[i] = 0;
      i++;
  } 
  DisplayOffTime = EEPROM.read(EEPROM_DisplayOffTime);
  WiFiManualEnable = EEPROM.read(EEPROM_WiFiManualEnable);
  Soft_AP_Switch = false;    // Access point is disabled 
  if (DisplayOffTime > 24) DisplayOffTime = 23;
  DisplayOnTime = EEPROM.read(EEPROM_DisplayOnTime); 
  if (DisplayOnTime > 24) DisplayOnTime = 8;  
  AutoResetTimer = AutoResetTimeout;  // Timer for auto reset if connection fails
  AutoResetTimerEnable = false;       // disable auto reset timer
  if(WiFiManualEnable == true)
  {
    if(OLED)
    {   
      oled.println(HostName);  // print to OLED the device name - Line 1  
      oled.println("Logging on to network");  
    }     
    WifiConnect ();                 
    udp.begin(localPort_NTP);   // for the NTP server
    GeyserPages.on("/",MainGeyserPage);
    GeyserPages.on("/Timer",Timer);
    GeyserPages.on("/TimerSettingUpload",TimerSettingUpload);     
    GeyserPages.on("/Heater",Heater);
    GeyserPages.on("/HeaterSettingUpload",HeaterSettingUpload);
    GeyserPages.on("/DisplaySystemSettings",DisplaySystemSettings);   
    GeyserPages.on("/ManualSwitchGeyser", ManualSwitchGeyser); 
    GeyserPages.on("/DataLog", DataLog);
    GeyserPages.on("/ResetLog", ResetLog);
    GeyserPages.on("/StartAP", StartAP);   
    GeyserPages.on("/Diagnostics", Diagnostics); 
    GeyserPages.on("/SaveTemperatureSensorAddresses", SaveTemperatureSensorAddresses); 
    GeyserPages.on("/Quit",Quit); 
    GeyserPages.on("/ManualUpdateLog",ManualUpdateLog);   
    GeyserPages.on("/SaveDisplayTimerOnOffSettings",SaveDisplayTimerOnOffSettings); 
    GeyserPages.begin();    
  }   // end of if(WiFiManualEnable == true)  
  else 
  {
    WiFi.disconnect(true);      
    Serial.println(F("WiFi not enabled"));      // connect to wifi network  if enabled          
  }
  serverAP.on("/",MainGeyserPage);
  serverAP.on("/Timer",Timer);
  serverAP.on("/TimerSettingUpload",TimerSettingUpload);  
  serverAP.on("/Heater",Heater);
  serverAP.on("/HeaterSettingUpload",HeaterSettingUpload);  
  serverAP.on("/WiFiSetup",WiFiSetup);    
  serverAP.on("/DisplaySystemSettings", DisplaySystemSettings); 
  serverAP.on("/ManualSwitchGeyser", ManualSwitchGeyser); 
  serverAP.on("/WiFiSettings",WiFiSettings);
  serverAP.on("/Quit",Quit);
  serverAP.on("/SaveTemperatureSensorAddresses", SaveTemperatureSensorAddresses);   
  serverAP.on("/ManualUpdateLog",ManualUpdateLog); 
  serverAP.on("/SaveDisplayTimerOnOffSettings",SaveDisplayTimerOnOffSettings);  
             
   byte PageCounter = 0;   // Initialise (clear) OLED display buffer
   ScreenNo = 2;            // initialise first OLED screen number;
   i=0;
   while(i<10) // Get 10 successive temperature readings (to ramp up average value
   {
      GetGeyserTemp(true);  // include reseting Min & Max Temperatures
      i++; 
   }
   TempFailCounter = 0;           // clear failure counter
   TempFailFlag = false;          // reset temp sensor failure flag
   GeyserManualControlTimer = 0;  // set up timer used for any manual control of geyser
   if (CheckFor_RTC_Fitted() == true) // is RTC fitted?
   {                          // Yes
      RTC_BoardPresent = true; 
      ClockMinute = RTC_Minute;   // load RTC clock to soft clock
      ClockHour = RTC_Hour;
      PreviousHour = ClockHour;
      ClockWeekDay = RTC_Day;
      LongClockMinute = ClockMinute + (ClockHour * 60);   // calc & save updated time value in minutes 
      if (EnableMessages) Serial.println(F("RTC detected"));
   }
   else 
   {
      ClockMinute = 0;   // reset soft clock for next NTP cycle
      ClockHour = 0;
      ClockWeekDay = 0;      
      RTC_BoardPresent = false;
   }
  //  ----- Check for connected DS18B20 sensors
  sensors.begin();  
  deviceCount = sensors.getDeviceCount();  
  if(deviceCount == 0) TempFailFlag = true;
  if(OLED)
  { 
      oled.print(F("DS18B20 Devices=  "));
      oled.println(deviceCount); 
      if(TempFailFlag) 
      {
        oled.println("ERROR-No Temp sensor");
        delay(3000);
      }    
  }
  if (EnableMessages)
  {
      Serial.print(F("DS18B20 Devices=  "));
      Serial.println(deviceCount);
      if(TempFailFlag) 
      {
        Serial.println("ERROR-No Temp sensor");
        delay(5000);
      }         
  }
//     ------------------ Initialise remaining flags and variables ----------------- 
  NTP_Resolved = false;   // true = soft clock successfully by NTP
  NTP_Time_Interval = 3000;   // set up NTP interval timer (mS)
  i = 0;
  while (i < 24)  // reset  all members of Data log array to zero
  {
     GeyserPowerLog[i] = 0;  
     TempLog[i] = 0;   
     TimeLogHr[i] = 0;
     TimeLogMin[i] = 0;  
     i++;    
  }
  //UpdateMinuteLog(true);  // initialise the Hourly average parameters
  LogPointer = 0;      
  NTP_Timer = millis();   // for the NTP time interval timer
  LoopTimer = millis();  // for the LED flash rates
  Timer_Tick = millis();  // for the output and PIR timers 
  AccessPointRunning = false;  
  DisplayToggleRunningTimer = millis();
  GeyserOnTimer = 0; 
  GeyserOnRunningTimer = millis();   // relay is being turned on - reset running timers to current time 
}    // end of Setup
// -------------------------------------------------------- 
void loop()
{
   if(digitalRead(GeyserControlRelay) == GeyserOff) GeyserOnRunningTimer = millis(); // if Geyser is off reset timer every loop
   byte i = 0;
   //------------- Check for Button presses ------------------------------------
   i = Check_AP_Switch(); 
   if (i == 2) Soft_AP_Switch = true;  // set flag if AP switch pressed for > 5 seconds 
   if (i == 1) 
   {  // manual increment of the screen display toggle counter
      ScreenNo ++;
      if (ScreenNo >= NoOf_OLED_Pages) ScreenNo = 0; // page counter overflow - reset to start
      DisplayToggleRunningTimer = millis(); 
      DisplayToggleInterval = 20000; // temporarily set duration interval to 20 seconds
      Update_OLED();  // manual update of OLED screen
      TemporaryDisplayCounter = TemporaryDisplayCounterInterval; // enable temporary display count for approx 15 seconds      
   }                
   // ----------------------- Access Point (AP) ------------------------------------
   if(Soft_AP_Switch == true)    // 
   {   // AP is enabled
      if (AccessPointRunning == false)  // is AP already running?
      {                                   // not running so set up AP
         Set_Up_Access_Point();
         AccessPointRunning = Enabled;    
      }   // end of  if (AccessPointRunning == false) 
      else 
      {
        serverAP.handleClient();  // AP is running so maintain AP web page 
      }
   }  // end of    if(Soft_AP_Switch == true)    // Access Point (AP) 
   else
   {                //  AP is disabled
      WiFi.softAPdisconnect(true);   // disconnect the AP
      serverAP.stop();               // Stop  the AP webserver 
      WiFi.mode(WIFI_STA);           // Switch to station mode only
      AccessPointRunning = false;   
      if(WiFiManualEnable == true) GeyserPages.handleClient();   
   
   }  // end of  else  
   Check_For_Keyboard_Command();       
// ---------------------------- Station mode ------------------------------------------------                  
   if (LoginSuccessful == false && AccessPointRunning == false && WiFiManualEnable == true) 
                                  ResetDevice("Lost Wifi Connection");  // Lost connection (only if AP is not running)  
   // -------------------- Timed Operations -----------------------------
   if(GeyserTempControlEnabled == Disabled && GeyserManualControlTimer == 0)   // Self Check for uncontrolled operation of geyser
   {   // the setpoints are not enabled AND the manual timer is not running
      if(digitalRead(GeyserControlRelay) == GeyserOn) // if Geyser is ON = illegal condition
      {
        ActivateGeyser(GeyserOff);  // try to Switch Geyser off
        Serial.println(F("Error detected - Temp Control enabled without safety timer running"));
        delay(25);
        if(digitalRead(GeyserControlRelay != GeyserOff))  // Reset device if Geyser not switched off within 25 mS
        {
          ResetDevice("Self Check Error");
        }
      }  // end of if(digitalRead(GeyserControlRelay) == GeyserOn) // if Geyser is ON = illegal condition    
   }  // end of if(GeyserTempControlEnabled == Disabled && GeyserManualControlTimer == 0) 
   // -------------------- OLED multi screen toggle feature ---  
    unsigned long DisplayToggleTimer = millis() - DisplayToggleRunningTimer; 
    if (DisplayToggleTimer >= DisplayToggleInterval)
    {     // Display toggle timeout      
      //DisplayToggleInterval = 4000;    //Reload interval time just in case been changed manually
      ScreenNo ++;
      if (ScreenNo >= 2) ScreenNo = 0; // page counter overflow - reset to start      
      if(ScreenNo == 0) DisplayToggleInterval = 6000;
      if(ScreenNo == 1) DisplayToggleInterval = 3000;

      DisplayToggleRunningTimer = millis();     
    }
   // --------------------------------------------------------
   unsigned long LED_Timer = millis() - LoopTimer;   // Grab time value since last scan
   if (digitalRead(Heartbeat_LED) == LED_On)   // If the Heartbeat LED is ON
   {
     if (LED_Timer >= LED_OnFlash) 
     {
       digitalWrite(Heartbeat_LED,LED_Off); // On timer has timed out so flip LED output
       GetGeyserTemp(false);
       CheckMeasurementAgainstSetpoints();  // if enabled check measurement against setpoints      
       Check_Time_Control_Geyser();   // Check clock against Geyser Time clock for Timer Control of Geyser Time switches
       ProcessMinuteTimer();  // check and update minute timer (re output timers and timer control
       if(WiFiManualEnable) Interval_Update_Clock();  // Check soft clock against NTP server    
// ---------------- Update OLED display -----------------------------
       if(!ShutdownCheck())  // check if Geyser has over heated
       {   // not overheated so normal display processing
          if(OLED == Fitted)  // only if display is fitted
          {
              if(TemporaryDisplayCounter > 0)  // if temporary timed display (via manual button)
              {
                Update_OLED();
                TemporaryDisplayCounter = TemporaryDisplayCounter - 1;           
              }
              else
              if(DisplayOnTime!=DisplayOffTime)    
              {
                  if((ClockHour>=DisplayOnTime && ClockHour<DisplayOffTime) || WiFiManualEnable == false) Update_OLED();
                   else oled.clear();
              }  // end of if(OLED == Fitted && DisplayOnTime!=DisplayOffTime) 
          }  // end of if(OLED)          
       }    // end of if(!ShutdownCheck())  // check if Geyser has over heated      
// ------------------------------------------------------------------             
       LoopTimer = millis();              // Set LED flash timer for OFF cycle                
     }  // end of      if (LoopTimer >= LED_OnFlash)      
   } // end of if (digitalRead(Heartbeat_LED))
   else              // If the Heartbeat LED is OFF
   {
    if (LED_Timer >= LED_OffFlash) 
    {
      digitalWrite(Heartbeat_LED,LED_On);  // Heartbeat Off timer has timed out so flip LED output
      LoopTimer = millis();              // Set LED flash timer for ON cycle
    } // end of  if (LoopTimer >= LED_OffTimer)
   }  // end of else  // If the Heartbeat LED is OFF
} // end of void loop()
// ---------------------------------------------------------
void GetGeyserTemp(boolean ResetTempLimits)   // get geyser temperature, calculate rolling average temperature and update Min & Max temperatures
{     
  byte NoOfSensors = sensors.getDeviceCount();  
  static float PreviousAverageGeyserTemperature;
  static byte TempSensorPointer;  // pointer to select the skin sensor
  if(TempSensorPointer == 0) TempSensorPointer = 1;  
  if(TempSensorPointer > NoOfSensors) TempSensorPointer = 1;  
  sensors.requestTemperatures(); // Send the command to get temperatures    
  float TemperatureReading =  sensors.getTempCByIndex(TempSensorIndex[0]);  // get Geyser Temperature
  if (TemperatureReading <= 0.0) // check reading is in valid range
  {
    TempFailCounter ++;  // increment measure fail counter
    TempFailFlag = true;  // set flag for display
    return;   // abandon reading if out of limits
  }
  else  // reading is in valid range
  {
    AverageGeyserTemperature = (0.4 * TemperatureReading) + (0.6 * PreviousAverageGeyserTemperature);  // calculate running average
    PreviousAverageGeyserTemperature = AverageGeyserTemperature;   // save current calculation as new previous average for next calculation loop    
    TempFailFlag = false;  // reset flag for display   
  }  // end of else
  //--------------   Get Skin Temps - 1 per call of this function ---------------------- 
  if (NoOfSensors > 1) // only if more than 1 sensor connected (1st = geyser water temp)
  {
    SkinTemp[TempSensorPointer-1] = sensors.getTempCByIndex(TempSensorIndex[TempSensorPointer]); 
    TempSensorPointer ++;  // increment pointer for next call of this function 
  }
  // -----------------------------------------------------------------------------------  
  if(ResetTempLimits)  // if Reset Limits = true
  {
      TempMin = AverageGeyserTemperature;
      TempMax = AverageGeyserTemperature;
  }  
  else  // Reset not active - just update limts when necessary
  {
      if (AverageGeyserTemperature > TempMax) TempMax = AverageGeyserTemperature;
      else if (AverageGeyserTemperature < TempMin) TempMin = AverageGeyserTemperature;
  }  
} // end of void GetGeyserTemp()
// --------------------------------------------------------------
void CheckMeasurementAgainstSetpoints()
{
  byte SetpointOn;
  byte SetpointOff;  
  if(GeyserTempControlEnabled == Enabled)
  {
    if(GeyserTempMode() == SetpointGroup1) //  settings 1 needed
    {
      SetpointOn = Setting1_ON_Setpoint;
      SetpointOff = Setting1_OFF_Setpoint; 
    }
    else    // Timer 2 settings needed
    {
      SetpointOn = Setting2_ON_Setpoint;
      SetpointOff = Setting2_OFF_Setpoint; 
    }  
    if (byte(AverageGeyserTemperature) >= SetpointOff) 
    {   // Geyser is hot >>> switch off
      if(digitalRead(GeyserControlRelay) == GeyserOn)
      {
        if (EnableMessages) 
        {
          Serial.print(F("Switched OFF by Thermostat @ "));
          Serial.print(AverageGeyserTemperature,1);
          Serial.print("'C @ ");
          Print_Clock();
        }
        ActivateGeyser(GeyserOff);
      }  // end of if(digitalRead(GeyserControlRelay) == GeyserOn);
    }  // end of if (AverageGeyserTemperature >= SetpointOff) 
    else if (byte(AverageGeyserTemperature) <= SetpointOn) 
    {    // Geyser is cool >>>> switch geyser on
      if(digitalRead(GeyserControlRelay) == GeyserOff)
      {
        if (EnableMessages) 
        {
          Serial.print(F("Switched ON by Thermostat @ "));
          Serial.print(AverageGeyserTemperature,1);
          Serial.print("'C @ ");
          Print_Clock();      
        }
        ActivateGeyser(GeyserOn); 
      }  // end of  if(digitalRead(GeyserControlRelay) == GeyserOff);   
    }  // end of else if (AverageGeyserTemperature <= SetpointOn)
  }    // end of if(GeyserTempControlEnabled == Enabled)
}  // end of void CheckMeasurementAgainstSetpoints()
// -------------------------------------------------------- 
byte Check_AP_Switch() // returns value depending on how long the AP button is pressed  
{
  unsigned long Switch_Pressed_Time; 
  if(digitalRead(AP_Switch) == AP_SwitchOff )   // read & test the button status
  {       // if not pressed
      TimeStart = millis();   // get time marker in readiness for switch being pressed
      return 0;
  }
  else // switch is ON
  {
    while (digitalRead(AP_Switch) == AP_SwitchOn)  // wait until button is released
    {
      HeartbeatTick();   // keep HW watchdog alive
      Switch_Pressed_Time = millis() - TimeStart;  // calculate the time since button was pressed(mS)
    }
          // button released
    if(Switch_Pressed_Time >= 10000) ResetDevice("by AP button");   // if AP_Switch is pressed for > 10 seconds = reset 
    else if(Switch_Pressed_Time >= 4000)  return 2;   // Wake up Access point
    else if(Switch_Pressed_Time >= 80 && OLED == Enabled ) return 1;   // Scroll through OLED screens
    return 0;   // button not pressed long enough

  }  // end of else // switch is ON
}     // end of byte Check_AP_Switch()
//  ---------------------------------------------------------------------
//void UpdateMinuteLog(boolean Reset)  // updated with temperature value every minute
//{                                    // and used for calculating the hourly average temperature
  //static byte MinutePointer;
  //if(Reset) // reset all variables
  //{
    //MinutePointer = 0;
    //while (MinutePointer < 60)
    //{
      //MinuteTempLog[MinutePointer] = 0;
     // MinutePointer ++;
    //}
    //MinutePointer = 0;
  //}  // end of if(Reset)
  //else
  //{
   // MinuteTempLog[MinutePointer] = AverageGeyserTemperature;  // store Temperature in averaging array
    //MinutePointer ++;
    //if (MinutePointer > 59) MinutePointer = 0;  // check & correct for overflow    
  //}
//}   // end of void UpdateMinuteLog()
// ----------------------------------------------------------------------
void ProcessMinuteTimer()   // check the various function minute timers inc soft clock
{
  unsigned long Minute_Timer = millis() - Timer_Tick;
  if (Minute_Timer >= One_Minute_timer_Tick_Threshold)   // Check for 1 minute timer tick
  {     // every minute
    Timer_Tick = millis();  
    if(RTC_BoardPresent)  GetTimeDateFromRTC(false);   // if RTC fitted get RTC time parameters
    if(AutoResetTimerEnable)   // if WiFi connect failed at log in
    {
      AutoResetTimer = AutoResetTimer - 1;  // decrement AP timeout counter
      if(AutoResetTimer == 0) ResetDevice("Login Failed, Auto Reset");       
    }      
    CheckGeyserForTimeout();  // check timer every minute, if time out switch output off
    //UpdateMinuteLog(false);   // Update Geyser Temperature Minute Log
    if(RTC_BoardPresent) // RTC is fitted so directly update clock parameters
    {
      ClockMinute = RTC_Minute;
      ClockHour = RTC_Hour; 
      ClockWeekDay = RTC_Day;
      if(ClockHour != PreviousHour)  // When hour changes
      {
        UpdateLogs();  // update Temp & Geyser Power logs   
        PreviousHour = ClockHour;    
      }
    }   // end of if(RTC_BoardPresentClock) // RTC is fitted so directly update clock parameters
    else // RTC not fitted so manage software clock
    {
      ClockMinute ++;    // Service the soft clock
      if (ClockMinute >= 60)  // if on the hour
      {
        ClockMinute = 0;
        ClockHour ++;  // increment the hour value
        if (ClockHour >= 24) // if midnight
        {
          ClockHour = 0; 
          ClockWeekDay ++; //increment week day
          if (ClockWeekDay >= 8 || ClockWeekDay == 0) ClockWeekDay = 1;  // overflow from Saturday to Monday
        }
        UpdateLogs();  // update Temp & Geyser Power logs
      }   // end of  if (ClockMinute >= 60)  // if on the hour
    }  // end of else // RTC not fitted so manage software clock
    // -------------------------------------------------------------------------
    if(ClockHour == 0 && ClockHour == 22 && ClockMinute == 0) GetGeyserTemp(true);   // reset Temp max and mins at start of Night mode
    // -------------------------------------------------------------------------
    LongClockMinute = (ClockHour * 60) + ClockMinute;   // calculate time in minutes   
  }  // end of if (Minute_Timer >= One_Minute_timer_Tick_Threshold)
}  // end of void ProcessMinuteTimer()   // check the various function minute timers inc soft clock
// -------------------------------------------------------- 
boolean GeyserTempMode()
{
    //if(ClockHour >= 22 || ClockHour < 6) return Night; else return Day;  // Set Day/Night mode for set points 
    if(TimerActive[Timer2] == Running) return SetpointGroup2; else return SetpointGroup1;  // Set correct mode for which set points to use 
}
// -------------------------------------------------------
void Check_For_Keyboard_Command()  // Used in loop() - checks keyboard for a single character command
{
  if (Serial.available() > 0) Command = Serial.read();
  {
    if (Command == 'd' || Command == 'D')  // Download and display all EEPROM based values
    {
      //DownloadWatchDogType(true);       
      EnableMessages = EEPROM.read(EEPROM_EnableMessages);
      Serial.print(F("Diagnostic messages flag = ")); Serial.println(EnableMessages);     
      Download_WiFi_Login_Details_From_EEPROM(true);
      DownloadFrom_EEPROM_GeyserControlParameters(true);
      // Serial.print(F("Geyser log Timer = ")); Serial.print(TempLog[0]);Serial.print(":"); Serial.println(TempLog[1]);
      Serial.print(F("Log pointer = ")); Serial.println(LogPointer);      
      byte i = 0;
      if(NTP_Resolved == true || RTC_BoardPresent == true)
      {
        Serial.println(F("24 hour logging array"));
        Serial.println(F("Array No - Time Logged - Temp - Geyser On since last log (mm:ss)"));
        while (i < 24)
        {
          if(TimeLogHr[i] != 0)  // only print valid data
          {
            if(i < 10) Serial.print("0"); Serial.print(i);
            Serial.print("            ");
            if(TimeLogHr[i] < 10) Serial.print("0"); Serial.print(TimeLogHr[i]);
            Serial.print(":");   
            if(TimeLogMin[i] < 10) Serial.print("0"); Serial.print(TimeLogMin[i]); 
            Serial.print("        ");  
            Serial.print(TempLog[i]);
            Serial.print("       ");   
            if(GeyserPowerLog[i] < 10) Serial.print("0"); Serial.println(GeyserPowerLog[i]);
          }       
          i ++;       
        }
        Serial.println("");
      }  // end of if(NTP_Resolved == true || RTC_BoardPresent == true)
    }   // end of if (Command == 'd')  // Download and display all EEPROM based values

    if (Command == 'h' || Command == 'H')  Help();
    //if (Command == 'r')  RemoteReadOfGeyserStatus();   // Send output status values to serial port
    //if (Command == 'w')  RemoteControlOfGeyser(); // Set/reset outputs from serial port
    if (Command == 'W' || Command == 'w')  Display_WiFi_Connection_Status();
    if (Command == 'R' || Command == 'r')  ResetDevice("by keyboard");  // Reset device
    if (Command == 'E' || Command == 'e')  Reset_All_EEPROM();
    if (Command == 'T' || Command == 't')  Print_Clock();  
    if (Command == 'l' || Command == 'L')  KeyboardSetupWiFiLoginParameters();
    if (Command == 'f' || Command == 'F')  Diagnostic_Flag_Dump();
    if (Command == 'm' || Command == 'M')  Toggle_Diagnostic_Messages(); 
    if (Command == 'A' || Command == 'A')  Soft_AP_Switch = true; 
    if (Command == 'b' || Command == 'B')  KeyboardSetup_AP_Password();
    if (Command == 'u')  UpdateLogs();
    if(RTC_BoardPresent)  // only if RTC fitted
    {
      if (Command == 'g')  GetTimeDateFromRTC(true);
      if (Command == 'U')   // Update RTC & soft clock from NTP
      {
        setSyncProvider(Get_NTP_Time);     // Set up NTP time provider for timelib library function   
        UpdateRTC(true);            
      }
    } // end of if(RTC_BoardPresent)  // only if RTC fitted

    
    Command = 255;
  }   // end of if (Serial.available() > 0) Command = Serial.read();
}     // end of void Check_For_Keyboard_Command() 
// -----------------------------------------------------------------------
void Help()  // displays all of the keyboard commands
{
    Serial.println(""); Serial.println(Firmware_Version);Serial.println("");
    Serial.println(F("AVAILABLE KEYBOARD COMMANDS")); Serial.println("");
    Serial.println(F("A = Toggle the soft Access Point Enable Flag")); 
    Serial.println(F("b = Set up Configuration Access Point password"));        
    Serial.println(F("d = Download and display all of the EEPROM stored configuration parameters"));
    Serial.println(F("E = Clear all EEPROM to default values (Must reset afterward)"));
    Serial.println(F("f = Diagnostic Demon - Active Flags & Variables")); 
    Serial.println(F("h = Show this Help menu screen again"));                  
    Serial.println(F("l = Keyboard entry of WiFi login parameters"));
    Serial.println(F("m = Toggle Enable Diagnostic Messsages"));
    Serial.println(F("R = Reset device"));
    Serial.println(F("T = Display System Clock"));           
    //Serial.println(F("w = Direct write to output E.g w10, w11, w20, w21, w30, w31")); 
    Serial.println(F("W = Display WiFi connection status and details"));      
    if(RTC_BoardPresent)  // only if RTC fitted
    {
      Serial.println(F("G = Get Time directly from RTC")); 
      Serial.println(F("U = Update RTC from NTP"));
    }       
    Serial.println("");
}  // end of void Help()
//  ---------------------------------------------------------------------
//void DownloadWatchDogType(boolean Display)
//{
//  WatchDogType = EEPROM.read(EEPROM_WatchDogType);   // set working memory from EEPROM stored parameters

//  if(Display) 
//  {
//    Serial.println("");
//    Serial.print(F("Watchdog type = ")); 
//    if (WatchDogType == Hardware) Serial.println(F("Hardware")); else Serial.println(F("Software")); 
//  }  // end of  if(Display)
//}  // end of void DownloadWatchDogType(boolean Display)
  // --------------------------------------------------------------------
byte Validate_Numerical_Entry_From_Keyboard_Buffer(byte No_Of_Keystrokes, byte Value_Limit) // validate characters in the keyboard buffer and return a value
{
  byte Value;   // numerical value to be returned from the keyboard buffer.  255 = invalid entry
  boolean Invalid_Entry = false;
  // test depending on number of keystrokes
  if (No_Of_Keystrokes == 1) if(isDigit (KeyboardBuffer[0])) Value = KeyboardBuffer[0] - '0'; else Invalid_Entry = true;      
  if (No_Of_Keystrokes == 2)
  {
    if (isDigit(KeyboardBuffer[1]) && isDigit(KeyboardBuffer[0])) 
      Value = (KeyboardBuffer[1] - '0') + ((KeyboardBuffer[0] - '0') * 10); else Invalid_Entry = true;
  }
  if (No_Of_Keystrokes == 3) 
  {
    if (isDigit(KeyboardBuffer[2]) && isDigit(KeyboardBuffer[1]) && isDigit(KeyboardBuffer[0])) 
      Value = (KeyboardBuffer[2] - '0') + ((KeyboardBuffer[1] - '0') * 10) + ((KeyboardBuffer[0] - '0') * 100) ;
    else Invalid_Entry = true;      // if not numeric  
  }             
       // test for validity - range, null entry, too many characters
  if (No_Of_Keystrokes > 3 || No_Of_Keystrokes == 0 || Value > Value_Limit) Invalid_Entry = true;
  if (Invalid_Entry)
  {
    Serial.println(""); Serial.println(F("Invalid Entry")); Serial.println("");
    Value = 255;
    return Value;   // illegal data entry      
  }
  else return Value;   // value = numerical keyboard entry or = 255 if not valid  
}
    // ------------------------------------------------------------------  
void Toggle_Diagnostic_Messages()  // enable/disable display printout of messages
{
  EnableMessages = !EEPROM.read(EEPROM_EnableMessages);  // get and invert the current flag status
  EEPROM.write(EEPROM_EnableMessages, EnableMessages);         // re-write the inverted flag
  Serial.print(F("Diagnostic messages are now "));
  if (EnableMessages) Serial.print(F("enabled")); else Serial.print(F("disabled"));
  EEPROM.commit();
}  // end of void Toggle_Diagnostic_Messages()
//  ---------------------------------------------------------------------
//void Toggle_WatchDogType()  // toggle between Hardware/Software watchdog type
//{
//  WatchDogType = !EEPROM.read(EEPROM_WatchDogType);  // get and invert the current flag status
//  EEPROM.write(EEPROM_WatchDogType, WatchDogType);         // re-write the inverted flag
//  Serial.print(F("Watchdog is now "));
//  if (WatchDogType) Serial.print(F("hardware based")); else Serial.print(F("software based"));
//  EEPROM.commit(); 
//}  // end of void Toggle_WatchDogType()
//  ---------------------------------------------------------------------
void Diagnostic_Flag_Dump()
{
  Serial.println("");
  Serial.println(F("DEMON STATUS STATUS DUMP"));
  //Serial.print(F("Watchdog is "));
  //if (WatchDogType == Hardware) Serial.println(F("Hardware based")); else Serial.println(F("Software based"));
  Serial.println("");
  Serial.print(F("The Setup Access Point is "));
  if(AccessPointRunning) Serial.println(F("running")); else  Serial.println(F("not running")); 
  Serial.print(F("The Soft Access Point Switch is ")); 
  if(Soft_AP_Switch) Serial.println(F("ON")); else  Serial.println(F("OFF"));  
  Serial.println("");  
  Serial.print(F("Initial login was ")); 
  if(LoginSuccessful) Serial.println(F("successful")); else Serial.println(F("not successful"));
  if (WiFi.status() == WL_CONNECTED) Serial.println(F("Current connection is good"));      
  else Serial.println(F("Current connection is broken"));
  Serial.println("");  
  Serial.print(F("The NTP_Time_Interval (mS) is ")); Serial.println(NTP_Time_Interval);
  Serial.print(F("The last successful NTP update was at "));Serial.print(Last_NTP_Hour); Serial.print(":");
  if (Last_NTP_Minute > 9) Serial.print(Last_NTP_Minute); 
  else 
  {
      Serial.print('0'); 
      Serial.print(Last_NTP_Minute);    
  }
  Serial.print(F("  Day:")); Serial.println(Last_NTP_Day); 
  Serial.println("");
  Serial.print(F("Temp Measurement Fail Counter = "));Serial.println(TempFailCounter);
  Serial.println("");  
  Serial.print(F("Geyser Timer 1 is ")); 
  if(TimerActive[0]) Serial.println(F("Running")); else  Serial.println(F("Off"));
  Serial.print(F("Geyser Timer 2 is ")); 
  if(TimerActive[1]) Serial.println(F("Running")); else  Serial.println(F("Off"));
  
  Serial.print(F("Current Geyser ON Timer(S) = ")); Serial.println((millis() - GeyserOnRunningTimer)/1000);    
  Serial.print(F("Serial port diagnostic messaging is ")); 
  if(EnableMessages) Serial.println(F("Enabled")); else  Serial.println(F("Disabled"));  
  Serial.println(""); 
  PrintTimerParameters();  
}   // end of  void Diagnostic_Flag_Dump()
//  ---------------------------------------------------------------------
void CheckGeyserForTimeout()    // every minute check Geyser Output for possible time out (output timers)
{
  if((digitalRead(GeyserControlRelay) == GeyserOff) || GeyserManualControlTimer  == 0) return;  // nothing to do if relay is OFF or timer disabled
  GeyserManualControlTimer = GeyserManualControlTimer - 1;   // decrement timer
  if (GeyserManualControlTimer > 0) return;  // nothing to do if timer has not timed out
  Serial.println(F("Manual Timeout - Geyser OFF"));
  SelfCheck = Enabled;   // re-enable self check
  ActivateGeyser(GeyserOff);  // Switch relay and remote control flag OFF
}  // end of void CheckGeyserForTimeout()
// ----------------------------------------------------------------------
void Reset_All_EEPROM()   // Clear out all of the previous stored configuration entries & reset the device
{                         // to default values
  unsigned int i = EEPROM_Start_Address;   // initial clear out of complete EEPROM block
  while(i < 200)  
  {
     EEPROM.write(EEPROM_Start_Address + i , 255);
     i++;
  }
  i = EEPROM_WiFi_SSID;
  while(i < EEPROM_WiFi_SSID + MaxStringLength)  // stuff default name into SSID
  {
    EEPROM.write(i, 'S');
    i++;
  }
  i = EEPROM_WiFi_Password;
  while(i < (EEPROM_WiFi_Password + MaxStringLength))  // stuff default name into Password
  {
    EEPROM.write(i, 'P'); 
    i++;
  }
  i = EEPROM_HostName;
  while(i < (EEPROM_HostName + MaxStringLength))  // stuff default name into HostName
  {
    EEPROM.write(i, 'N'); 
    i++;
  }

  EnableMessages = Enabled;
  EEPROM.write (EEPROM_EnableMessages,EnableMessages);  // enable disgnostic messages
  EEPROM.write (EEPROM_WiFiManualEnable,false);   // WiFi Not enabled

  EEPROM.write (EEPROM_AP_Password,'p');      // AP default password = password
  EEPROM.write (EEPROM_AP_Password+1,'a'); 
  EEPROM.write (EEPROM_AP_Password+2,'s'); 
  EEPROM.write (EEPROM_AP_Password+3,'s'); 
  EEPROM.write (EEPROM_AP_Password+4,'w'); 
  EEPROM.write (EEPROM_AP_Password+5,'o'); 
  EEPROM.write (EEPROM_AP_Password+6,'r'); 
  EEPROM.write (EEPROM_AP_Password+7,'d'); 
  
  //DisplayToggleInterval = 3;
  //EEPROM.write (EEPROM_DisplayToggleInterval,DisplayToggleInterval); 
  
  Setting1_ON_Setpoint = 20;
  EEPROM.write (EEPROM_Setting1_ON_Setpoint,Setting1_ON_Setpoint); 
  Setting2_ON_Setpoint = 20;
  EEPROM.write (EEPROM_Setting2_ON_Setpoint,Setting2_ON_Setpoint); 
  Setting1_OFF_Setpoint = 30;
  EEPROM.write (EEPROM_Setting1_OFF_Setpoint,Setting1_OFF_Setpoint); 
  Setting2_OFF_Setpoint = 30;
  EEPROM.write (EEPROM_Setting2_OFF_Setpoint,Setting2_OFF_Setpoint);  
  GeyserTempControlEnabled = Disabled;
  EEPROM.write (EEPROM_GeyserTempControlEnabled,GeyserTempControlEnabled);
  TimerEnable[Timer1]  = Disabled;
  EEPROM.write (EEPROM_TimerEnableGeyserTimer1,TimerEnable[Timer1]);
  TimerEnable[Timer2] = Disabled;
  EEPROM.write (EEPROM_TimerEnableGeyserTimer2,TimerEnable[Timer2]);
  EEPROM.write (EEPROM_TempSensorIndex,0);
  EEPROM.write (EEPROM_TempSensorIndex,1);
  EEPROM.write (EEPROM_TempSensorIndex,2);
  EEPROM.write (EEPROM_TempSensorIndex,3);
  EEPROM.write (EEPROM_TempSensorIndex,4);
  EEPROM.write (EEPROM_DisplayOffTime,23); 
  EEPROM.write (EEPROM_DisplayOnTime,8);
      
  i=8;
  while (i < MaxStringLength)
  {
    EEPROM.write (EEPROM_AP_Password+i,255); 
    i++;
  }
  
  i = 0;
  while (i < 14)    // reset boolean flags for each of 6 x timer week day flags and timer enable flags
  {
    EEPROM.write(EEPROM_Geyser_Timer1_Sun + i, false);
    i++;
  }
    GeyserTimeControlParameters[0][0][0] = 0;  // Timer 1 ON Minutes
    EEPROM.write(EEPROM_Geyser_Timer1_TimOn_Min,GeyserTimeControlParameters[0][0][0]);
    GeyserTimeControlParameters[0][0][1] = 12; // Timer 1 ON Hours
    EEPROM.write(EEPROM_Geyser_Timer1_TimOn_Hrs,GeyserTimeControlParameters[0][0][1]);
       
    GeyserTimeControlParameters[1][0][0] = 0;  // Timer 2 ON minutes
    EEPROM.write(EEPROM_Geyser_Timer2_TimOn_Min,GeyserTimeControlParameters[1][0][0]);
    GeyserTimeControlParameters[1][0][1] = 15; // Timer 2 ON Hours
    EEPROM.write(EEPROM_Geyser_Timer2_TimOn_Hrs,GeyserTimeControlParameters[1][0][1]); 
    
    GeyserTimeControlParameters[0][1][0] = 0;  // Timer 1 OFF Minutes
    EEPROM.write(EEPROM_Geyser_Timer1_TimOff_Min,GeyserTimeControlParameters[0][1][0]);
    GeyserTimeControlParameters[0][1][1] = 12; // Timer 1 OFF Hours
    EEPROM.write(EEPROM_Geyser_Timer1_TimOff_Hrs,GeyserTimeControlParameters[0][1][1]);
       
    GeyserTimeControlParameters[1][1][0] = 0;  // Timer 2 OFF Hours
    EEPROM.write(EEPROM_Geyser_Timer2_TimOff_Min,GeyserTimeControlParameters[1][1][0]);
    GeyserTimeControlParameters[1][1][1] = 15; // Timer 2 OFF Hours
    EEPROM.write(EEPROM_Geyser_Timer2_TimOff_Hrs,GeyserTimeControlParameters[1][1][1]); 

  byte Value;         // set up EEPROM header block - indicates that the EEPROM
  i = 0;              // has been initialised
  while (i < 5)
  {
    EEPROM.write(EEPROM_Header_Block + i, i);  // write header byte
    i ++;  
  }  // end of while (i < 5) 
     
  EEPROM.commit();
  Serial.println(F("EEPROM reset to default values"));
   
}   // end of void Reset_All_EEPROM()
// ------------------------------------------------------------
void Download_WiFi_Login_Details_From_EEPROM(boolean Display)   // Download to memory, the EEPROM stored values of SSID and password
{
    unsigned int i = EEPROM_WiFi_SSID;
    esid = "";  //start with empty SSID string
    while (i < EEPROM_WiFi_SSID + MaxStringLength)  // build up SSID string from stored characters that are not FF
    {
      if (EEPROM.read(i) != 255) 
      {
      esid += char (EEPROM.read(i)); 
      } else break;
      i++; 
    }
    i = EEPROM_WiFi_Password;
    epass = "";   //start with empty password string
    while (i < EEPROM_WiFi_Password + MaxStringLength) // build up password string from stored characters that are not FF
    {
      if (EEPROM.read(i) != 255) 
      {
      epass += char(EEPROM.read(i)); 
      } else break; 
      i++;
    }
    i = EEPROM_HostName;
    HostName = "";   //start with empty password string
    while (i < EEPROM_HostName + MaxStringLength) // build up password string from stored characters that are not FF
    {
      if (EEPROM.read(i) != 255) 
      {
      HostName += char(EEPROM.read(i)); 
      } else break; 
      i++;
    }

    i = EEPROM_AP_Password;
    passphrase = "";   //start with empty password string
    while (i < EEPROM_AP_Password + MaxStringLength) // build up password string from stored characters that are not FF
    {
      if (EEPROM.read(i) != 255) 
      {
      passphrase += char(EEPROM.read(i)); 
      } else break; 
      i++;
    } 
    if (Display)
    {
      Serial.println(""); 
      Serial.print(F("EEPROM SSID: ")); 
      Serial.println(esid); // print final SSID string - diagnostic
      Serial.print(F("EEPROM Password: ")); 
      Serial.println(epass);  // print final Password string - diagnostic
      Serial.print(F("EEPROM HostName: ")); 
      Serial.println(HostName);  // print final HostName string - diagnostic
      Serial.print(F("EEPROM Access Point Password: ")); 
      Serial.println(passphrase);     
      Serial.println("");
    }        
}   // end of void Download_WiFi_Login_Details_From_EEPROM()
// ------------------------------------------------------------
void Display_WiFi_Connection_Status()  // display the status of the current WiFi connection
{
    Serial.println(""); 
               // print out to serial monitor, the connected network details 
    Serial.print(F("Device name: "));
    Serial.println(HostName);

    Serial.print(F("SSID: "));      // SSID
    Serial.println(WiFi.SSID());                       
  
    long rssi = WiFi.RSSI();        // print the received signal strength:
    Serial.print(F("signal strength (RSSI):"));
    Serial.println(rssi);            
  
    Serial.print(F("IP Address: ")); // IP address given to the connection
    Serial.println(GetIPaddress());
  
    byte mac[6];                      // MAC address 
    WiFi.macAddress(mac);
    Serial.print(F("MAC address: "));
    Serial.print(mac[0],HEX);
    Serial.print(F(":"));
    Serial.print(mac[1],HEX);
    Serial.print(F(":"));
    Serial.print(mac[2],HEX);
    Serial.print(F(":"));
    Serial.print(mac[3],HEX);
    Serial.print(F(":"));
    Serial.print(mac[4],HEX);
    Serial.print(F(":"));
    Serial.println(mac[5],HEX);

    IPAddress subnet = WiFi.subnetMask();   // Subnet mask
    Serial.print(F("NetMask: "));
    Serial.println(subnet);
  
    IPAddress gateway = WiFi.gatewayIP();   // Gateway address
    Serial.print(F("Gateway: "));
    Serial.println(gateway);
    Serial.println("");
  }  // end of void Display_WiFi_Connection_status

// ---------------------- WIFI --------------------------------------
void WifiConnect()     // Connect to the wifi network using the stored credentials
{        
   WiFi.mode(WIFI_STA);             // Set WIFI module to STA mode
   byte Login_Wait_Timer = 0;    // no of loops waiting for log in success
   LoginSuccessful = true;
   if (EnableMessages) 
   {  
      Serial.println(); 
      Serial.println(F("Connecting to WiFi"));
   }  
   WiFi.begin(esid.c_str(), epass.c_str());   
   delay(2);
   WiFi.hostname(HostName.c_str());  
   while (WiFi.status() != WL_CONNECTED) 
   {                 // wait for connection whilst flashing the heartbeat LED
     Check_For_Keyboard_Command();  // check for any keyboard command     
     digitalWrite(Heartbeat_LED,LED_On);    // flash the Heartbeat LED
     delay(LED_FlashStartup);
     if (EnableMessages) Serial.print(".");
     digitalWrite(Heartbeat_LED,LED_Off);
     delay(LED_FlashStartup );   
     Login_Wait_Timer ++;
     if ( Login_Wait_Timer >= 100)  // login failed after 100 loops
     {
        LoginSuccessful = false;
        Soft_AP_Switch = true;      // enable Access point to allow set up check
        AutoResetTimer = AutoResetTimeout;
        AutoResetTimerEnable = Enabled;  // Connection failed, start timer, timeout = auto reset
        break;
     }         
   }  // end of while (WiFi.status() != WL_CONNECTED) 
   if(LoginSuccessful) 
   {
      if (EnableMessages) 
      {  
          Serial.println("");
          Serial.println(F(" Connected to Network"));
          Display_WiFi_Connection_Status();        
      } 
    }  // end of if(LoginSuccessful)
   else
   {   
      if (EnableMessages)
      {
          Serial.println(); 
          Serial.println(F("Login failed"));
      }
   } // end of else      
 } // end of void WifiConnect()
// ----------------------- NTP functions ---------------------------------------
// send an NTP request to the time server at the given address
unsigned long sendNTPpacket(IPAddress& address)
{       // set all bytes in the buffer to 0
  memset(packetBuffer2, 0, NTP_PACKET_SIZE);
            // Initialize values needed to form NTP request
  packetBuffer2[0] = 0b11100011;   // LI, Version, Mode
  packetBuffer2[1] = 0;     // Stratum, or type of clock
  packetBuffer2[2] = 6;     // Polling Interval
  packetBuffer2[3] = 0xEC;  // Peer Clock Precision
                            // 8 bytes of zero for Root Delay & Root Dispersion
  packetBuffer2[12]  = 49;
  packetBuffer2[13]  = 0x4E;
  packetBuffer2[14]  = 49;
  packetBuffer2[15]  = 52;
                            // you can send a packet requesting a timestamp:
  udp.beginPacket(address, 123); //NTP requests are to port 123
  udp.write(packetBuffer2, NTP_PACKET_SIZE);
  udp.endPacket();
} // end of unsigned long sendNTPpacket(IPAddress& address)
    // ---------------------------------------------------
void Interval_Update_Clock() // update the soft clock with the NTP clock data
{
   unsigned long NTPTimer = millis() - NTP_Timer;
   if (NTPTimer >= NTP_Time_Interval)   // only do this at defined intervals
   {
     NTP_Timer = millis();
     Get_Clock_Data();   // get latest time via NTP and if valid, update soft clock
   }
}   //end of  Interval_Update_Clock()
    // ---------------------------------------------------
void Get_Clock_Data() // update the soft clock with the NTP clock data
{                        // Returns false if NTP fails
          setSyncProvider(Get_NTP_Time);     // Set up NTP time provider for timelib library function  
          if(timeStatus() == 2)  //timeSet)  // only if sync is successful
          {
            if(RTC_BoardPresent)UpdateRTC(false);  // update RTC if fitted
            else 
            {
              ClockMinute = minute();  // Update Soft Clock only
              ClockHour = hour();
              ClockWeekDay = weekday();            
            }
            LongClockMinute = ClockMinute + (ClockHour * 60);   // calc & save updated time value in minutes 
            Last_NTP_Hour = ClockHour;  // Store for diagnostic page
            Last_NTP_Minute = ClockMinute;  // Store for diagnostic page
            Last_NTP_Day = ClockWeekDay;  // Store for diagnostic page 
            NTP_Time_Interval = 14400000; // set next timed update for 4 hour
            NTP_Resolved = true; 
            if (EnableMessages) 
            {
                Serial.println(""); Serial.print(F("NTP successful @ ")); 
                Print_Clock();   // print the current NTP time         
            }        
          }  // end of if(timeStatus() == 2)  //timeSet)  // only if sync is successful
          else  // sync failed
          {
              NTP_Time_Interval = 3000;  
              NTP_Resolved = false;
              if (EnableMessages) 
              {
              Serial.println(""); Serial.print(F("NTP update failed @ ")); 
              Print_Clock();   // print the current NTP time          
              }  
          }  // end of else  // sync failed       
}   // end of Get_Clock_data
    // ---------------------------------------------------
void Print_Clock()
{
      Serial.print(ClockHour); // print the hour
      Serial.print(':');
      if (ClockMinute < 10 ) Serial.print('0');
      Serial.print(ClockMinute); // print the minute 
      if(NTP_Resolved || RTC_BoardPresent) 
      {
        Serial.print(F(" ")); 
        Serial.print(dayStr(ClockWeekDay)); 
      }
      if(RTC_BoardPresent) 
      {
        Serial.print("  ");
        Serial.print(RTC_MonthDay);
        Serial.print("/");
        Serial.print(monthStr(RTC_Month));
        Serial.print("/");
        Serial.print(RTC_Year);          
      }
      Serial.println("");
} // end of void Print_Clock()
    // --------------------------------------------------------------------     
time_t Get_NTP_Time()   // request timestamp from NTP server. returns 0 if fail
{
    while (udp.parsePacket() > 0) ; // discard any previously received packets
    
    udp.begin(localPort_NTP);
    
    // WiFi.hostByName(ntpServerName, timeServerIP); 
    // sendNTPpacket(timeServerIP); // send an NTP packet to a time server
    
    sendNTPpacket(timeServer); // send an NTP packet to a time server
    
    // wait to see if a reply is available
    byte NTP_WaitTimer = 0;
    boolean cb = false;
    unsigned long epoch = 0;
    while (NTP_WaitTimer < NTP_WaitTime && !cb)  // wait upto 1 second for NTP server to respond
    {
      digitalWrite(Heartbeat_LED,LED_On);  // keep HW watchdog alive
      delay(5);
      cb = udp.parsePacket();
      NTP_WaitTimer ++;
      digitalWrite(Heartbeat_LED,LED_Off);
      NTP_Timeout = false;       
      delay(5);
    }
    if (NTP_WaitTimer >= NTP_WaitTime) // check to see if wait timer timed out
    {         // time out - attempt failed so set timer update interval to 15 seconds for next attempt
      NTP_Timeout = true; 
      return epoch; // nothing received back
    }
    else // data was received so read data
    {
      udp.read(packetBuffer2, NTP_PACKET_SIZE); // read the packet into the buffer
      unsigned long highWord = word(packetBuffer2[40], packetBuffer2[41]);
      unsigned long lowWord = word(packetBuffer2[42], packetBuffer2[43]);
      unsigned long secsSince1900 = highWord << 16 | lowWord;
      const unsigned long seventyYears = 2208988800UL;
      epoch = secsSince1900 - seventyYears;    // UTC time seconds
      epoch = epoch + (TimeZone*3600);  // adjust epoch for timezone difference
      return epoch;
    }  // end of else
   
   udp.stop();
   
}   // end of void Get_NTP_Time()
// ------------------------------------------------------------
byte GetKeyboardInput()    // get data string from keyboard - max 20 characters, terminator = carriage return
{                                           
   byte pointer = 0;      // flush keyboard buffer
   while(pointer < 20)
   {
    KeyboardBuffer[pointer] = 255;
    pointer ++;
   }
   while (Serial.available() > 0) {byte Trash = Serial.read();}   // flush out any trash data
   while (Serial.available() == false) HeartbeatTick();  // pause for keyboard input & flash 
                                                          // the Heartbeat LED to keep the watchdog alive   
   char ch;
   pointer = 0;
   while (Serial.available()>0)   //For each character entered
   {
     ch = Serial.read();
     if (pointer < MaxStringLength && byte(ch) != 13)    
     {
       KeyboardBuffer[pointer] = ch;  // save to keyboard buffer
       pointer ++; 
     }
     HeartbeatTick();   // flash the Heartbeat LED to keep the watchdog alive  
   }  // end of    while (Serial.available()>0)   //For each character entered
   return pointer;
}  // end of void GetKeyboardInput
// ------------------------------------------------------------
void KeyboardSetupWiFiLoginParameters()  // Set up new WiFi login - SSID, Password & Host Name
{
  Serial.println(""); Serial.println(F("Enter new SSID - 20 chars max"));
   
  byte Length = GetKeyboardInput() ;   // get input from keyboard, length = number of characters used 
  unsigned int i=EEPROM_WiFi_SSID;
  if (Length >= 8 && Length < 21)  // if entry is > minimum & less than the max no of characters 
  {
    while(i < EEPROM_WiFi_SSID + MaxStringLength)  // Clear out all of the previous stored entriesfor SSID
    {
      EEPROM.write(i, 255);
      i++;
    }
    Serial.print(F("New WiFi SSID = "));
    i=0;
    while(i < Length)   // store new SSID to EEPROM
    {
      EEPROM.write(EEPROM_WiFi_SSID + i, KeyboardBuffer[i]);
      Serial.print(char(EEPROM.read(EEPROM_WiFi_SSID + i))); 
      i++;
    }
  } // end of if (Length >= 8 && Length < 21)  // if entry is > minimum & less than the max no of characters
  else
  Serial.println(F("Invalid - WiFi SSID not changed"));   
  Serial.println("");
    
  Serial.println(F("Enter WiFi Password - 8 chars min, 20 chars max"));
  Length = GetKeyboardInput() ;   // get input from keyboard & how many characters entered   
  if(Length >= 8 && Length < 21)  // if entry is > minimum & less than the max no of characters 
  {   
    i = EEPROM_WiFi_Password;
    while(i < (EEPROM_WiFi_Password + MaxStringLength))
    {
      EEPROM.write(i, 255); // Clear out all of the previous stored entries for password
      i++;
    }
    Serial.print(F("WiFi Password = "));     
    i = 0; 
    while(i < Length)  // store new password to EEPROM
    {
      EEPROM.write(EEPROM_WiFi_Password + i, KeyboardBuffer[i]);
      Serial.print(char(EEPROM.read(EEPROM_WiFi_Password + i))); 
      i++;
    }
  } // end of if(Length >= 8 && Length < 21)  // if entry is > minimum & less than the max no of characters
  else
  Serial.println(F("Invalid - WiFi password not changed"));
  Serial.println("");
  
  Serial.println(F("Enter Device Name - 4 chars min, 20 chars max"));
  Length = GetKeyboardInput() ;   // get input from keyboard  & how many characters entered      
  i = EEPROM_HostName;
  if (Length >= 4 && Length  < 21)  // if entry is > minimum & less than the max no of characters
  {
    while(i < (EEPROM_HostName + MaxStringLength))
    {
      EEPROM.write(i, 255); // Clear out all of the previous stored entries
      i++;
    }
    Serial.print(F("Device Name = "));     
    i = 0; 
    while(i < Length)  // store new password to EEPROM
    {
      EEPROM.write(EEPROM_HostName + i, KeyboardBuffer[i]);
      Serial.print(char(EEPROM.read(EEPROM_HostName + i))); // ------------------
      i++;
    }
    Serial.println("");
  }   // end of  if (Length >= 4)  // if entry is > minimum no of characters
  else
  Serial.println(F("Invalid - Device Name not changed"));
  EEPROM.commit();   // update EEPROM   
  Download_WiFi_Login_Details_From_EEPROM(true); 
}  // end of  void KeyboardSetupWiFiLoginParameters()  
 // ---------------------------------------------------------------------------
void KeyboardSetup_AP_Password()
{  
  Serial.println(F("Enter Configuration Access Point Password - 8 chars min, 20 chars max"));
  byte Length = GetKeyboardInput() ;   // get input from keyboard & how many characters entered   
  byte i = 0;
  if (Length >= 8 && Length < 21)  // if entry is > minimum & less than the max no of characters
  {
    while(i < MaxStringLength)
    {
      EEPROM.write(EEPROM_AP_Password + i, 255); // Clear out all of the previous stored entries
      i++;
    }
    Serial.print(F("New Access Point Password = "));     
    i = 0; 
    while(i < Length)  // store new password to EEPROM
    {
      EEPROM.write(EEPROM_AP_Password + i, KeyboardBuffer[i]);
      Serial.print(char(EEPROM.read(EEPROM_AP_Password + i))); 
      i++;
    }
    Serial.println("");
    EEPROM.commit();   // update EEPROM   
    Download_WiFi_Login_Details_From_EEPROM(true);  
  }  // end of   if (Length >= 8)  // if entry is > minimum no of characters
  else
  Serial.println(F("Invalid number of characters (min 8, max 20 required"));
}
 // ---------------------------------------------------------------------------
void Check_Time_Control_Geyser()  
{
   const byte Timer1 = 0;
   const byte Timer2 = 1;
   const byte Geyser = 0;
  // ---------------------------------- Timer 1 --------------------------------------------
          // only if relay timer is enabled AND clock is valid (weekday not = 0) AND timer 1 not previously active  
   if(TimerEnable[Timer1] && ClockWeekDay != 0 && TimerActive[Timer1] == false)  
   {
     if (GeyserTimeControlParameters_Day_Flags[Timer1][ClockWeekDay-1]) 
     {    // The clock day is enabled for timer control so check the clock against the timer settings
          if (LongClockMinute >= LongTimerON[Timer1] && LongClockMinute < LongTimerOFF[Timer1])
          {  // if clock is inbetween the ON & OFF setpoints then the timer ON should be initiated
             
             GeyserTempControlEnabled = Enabled;
             if (EnableMessages)
             {
               Serial.println(F("Geyser Timer 1 Started"));
               Print_Clock();
             }
             TimerActive[Timer1] = true;    // set relay 1 timer 1 active flag                                
          }   // end of if (LongClockMinute >= LongTimerON[Timer1] && LongClockMinute < LongTimerOFF[Timer1])     
     }  // end of if (GeyserTimeControlParameters_Day_Flags[Timer1][ClockWeekDay-1])
   } // end of if(TimerEnable && ClockWeekDay != 0 && TimerActive[Timer1] == false)  
  // ------------------------- Timer 2 -------------------------------------------- 
  // only if relay timer is enabled AND clock is valid (weekday not = 0) AND timer 2 not previously active  
  if(TimerEnable[Timer2] && ClockWeekDay != 0 && TimerActive[Timer2] == false)  
  {    
    if (GeyserTimeControlParameters_Day_Flags[Timer2][ClockWeekDay-1]) 
    {    // The clock day is enabled for timer control so check the clock against the timer settings
        if (LongClockMinute >= LongTimerON[Timer2] && LongClockMinute < LongTimerOFF[Timer2])
        {  // if clock is inbetween the ON & OFF setpoints then the timer ON should be initiated
          GeyserTempControlEnabled = Enabled;
          if (EnableMessages)
          {
             Serial.println(F("Geyser Timer 2 Started"));
             Print_Clock();
          }
          TimerActive[Timer2] = true;    // set relay 1 timer 2 active flag          
        } // end of if (LongClockMinute >= LongTimerON[Timer2] && LongClockMinute < LongTimerOFF[Timer1])  
   } // end of if (GeyserTimeControlParameters_Day_Flags[Timer2][ClockWeekDay-1]) 
 }    // if(TimerEnable[Timer2] && ClockWeekDay != 0 && TimerActive[Timer2] == false) 
 // *************************** At the end of timer 1 period ***********************
 if (GeyserTimeControlParameters_Day_Flags[Timer1][ClockWeekDay-1] && TimerActive[Timer1] == true) 
 {    // The timer has been running & the clock day is enabled so reset Timer Active flag if the clock is > setpoint OFF and before midnight 
    if (LongClockMinute >= LongTimerOFF[Timer1] && LongClockMinute <= 1439)
    {
      TimerActive[Timer1] = false;
      GeyserTempControlEnabled = Disabled;
      ActivateGeyser(GeyserOff);  // turn off Geyser relay as a precaution (setpoints are now disabled)
      
      if (EnableMessages)
      {
        Serial.println(F("Geyser Timer 1 Stopped"));
        Print_Clock();
      }           
    }  // end of if (LongClockMinute >= LongTimerOFF[Timer1] && LongClockMinute <= 1439)
 }    // end of if (GeyserTimeControlParameters_Day_Flags[Timer1][ClockWeekDay-1]) 
 // *************************** At the end of timer 2 period *********************** 
 if (GeyserTimeControlParameters_Day_Flags[Timer2][ClockWeekDay-1] && TimerActive[Timer2] == true)
 {  // The clock day is enabled so reset Timer Active flag if the clock is > setpoint OFF and before midnight 
    if (LongClockMinute >= LongTimerOFF[Timer2] && LongClockMinute <= 1439) 
    {
      TimerActive[Timer2] = false; 
      GeyserTempControlEnabled = Disabled;
      ActivateGeyser(GeyserOff);  // turn off Geyser relay as a precaution (setpoints are now disabled)
            
      if (EnableMessages)
      {
        Serial.println(F("Geyser Timer 2 Stopped"));
        Print_Clock();
      }          
    }  // end of if (LongClockMinute >= LongTimerOFF[Timer2] && LongClockMinute <= 1439)
 }  // end of if (GeyserTimeControlParameters_Day_Flags[Timer2][ClockWeekDay-1])
}   // end of void Check_Time_Control_Geyser() 
 // ---------------------------------------------------------------------------
void Auto_EEPROM_Check()   // test EEPROM Block for correct ID, set EEPROM to default if 1st run
{
  byte Value;
  byte i = 0;
  while (i < 5)
  {
    Value = EEPROM.read(EEPROM_Header_Block + i);
    if (Value != i) 
    {    // if any one of header block not correct, reset all EEPROM & exit
      Serial.println(""); Serial.println(F("EEPROM not initialised"));
      Reset_All_EEPROM();
      break;
    }  // end of if (Value != i) 
    i ++;  
  }  // end of while (i < 5) 
}   // end of void Auto_EEPROM_Check() 
// --------------------------------------------------------------
void Display_OLED_String(String Text, byte LineNo, byte CharNo, boolean ClearField)
{        
        oled.setCursor(CharNo*oled.fontWidth(),LineNo);       
        if(ClearField) oled.clearToEOL();
        oled.print (Text);               
}
void Display_OLED_StringNL(String Text, byte LineNo, byte CharNo, boolean ClearField)
{        
        oled.setCursor(CharNo*oled.fontWidth(),LineNo);       
        if(ClearField) oled.clearToEOL();
        oled.print (Text);  
        oled.println("");             
}
// ------------------------------------------------------------
String ConvertClockTimeToString()  // Returns OLED version of Time and date
{
    String TimeDate = String(ClockHour) + ":";      
    if (ClockMinute < 10)  TimeDate += "0" ;
    TimeDate += String(ClockMinute) + " ";
    if(RTC_BoardPresent) 
    {
      TimeDate += dayShortStr (ClockWeekDay);
      TimeDate += " " + String(RTC_MonthDay) + "/" +  String(RTC_Month) + "/" + String(RTC_Year) + "   ";
    }
    else if(NTP_Resolved) TimeDate += dayStr (ClockWeekDay);  // Day - Monday, Tuesday, Wednesday etc
    return TimeDate;
}
// ------------------------------------------------------------------------
void Update_OLED()
{       
 byte NoOfSensors;
 String Temp; 
 if(AccessPointRunning) // if Access point is running - display message screen
 {
    if(FirstPass == false)
    {   // once only
      oled.clear();
      oled.println("Web Configurator ON");   // line 1
      oled.println("Connect Phone to ");     // line 2
      oled.println(HostName);                // line 3  
      oled.println(" - Setup wifi & then");   // line 4   
      oled.println("connect a browser to");  // line 5
      oled.println("http://192.168.1.200");  // line 6
      oled.println("Controller will reset"); // line 7
      oled.print  ("on closing connection"); // line 8    
      FirstPass = true;     
    }
 }
 else
 {      // AP is not running = normal display modes
   byte i;
   signed int temp;
              // ------------------------------- Top 2 Lines ------------------
   switch (ScreenNo) 
   {         
       case 0:    
          //oled.clear();
          oled.setCursor(0,0);
          Display_OLED_StringNL(HostName,0,0,true);  // If Home screen, print to OLED the device name - Line 1
// ----------------------------------- Line 2 ----------------------      
          
          Display_OLED_StringNL(ConvertClockTimeToString(),1,0,true);  // Display current time
          
          oled.clearToEOL();                   
          break;          
   }  // end of switch (ScreenNo)    
// ------------------------------------ Data Area-------------------

   switch (ScreenNo) 
   {   
       case 0: 
          // ------------- line 3 of OLED ----------------------         
          oled.setCursor(0,2);
          oled.clearToEOL();
          oled.setCursor(0,3);      // Line 4
          oled.println("Water  ");
          oled.print  ("Temp   ");  // Line 5
          oled.set2X();
          // ------------- line 5   ----------------------             
          oled.setCursor(4 * oled.fontWidth(),3);  
          if(TempFailFlag)  // if temp reading bad
          {
            oled.println("Terror ");
          }
          else  // Temp reading good
          {
            oled.print(AverageGeyserTemperature,1); 
            oled.println("'C    ");
          }
          oled.clearToEOL(); oled.println("");  // line 6
          oled.clearToEOL(); oled.println("");  // line 7                       
          // -------------- line 3 -----------------------
          oled.set1X(); 
          break;
       // -------------------------------------------------------------------------       
       case 1:
          oled.setCursor(0,0);
          oled.print("Temp="); oled.print( AverageGeyserTemperature,1); oled.print(" SETon=  "); 
          if(GeyserTempMode() == SetpointGroup1) oled.println(Setting1_ON_Setpoint); else oled.println(Setting2_ON_Setpoint);
          oled.print("Tmax=");  oled.print( TempMax,1); oled.print(" SEToff= ");
          if(GeyserTempMode()== SetpointGroup1) oled.println(Setting1_OFF_Setpoint); else oled.println(Setting2_OFF_Setpoint);
          oled.print("Tmin=");  oled.print( TempMin,1); oled.print(" Mode= ");
          if(GeyserTempMode() == SetpointGroup1) oled.println("Set 1  "); else oled.println("Set 2 ");
          NoOfSensors = sensors.getDeviceCount();
          if(NoOfSensors > 1)    // Line 4
          {
              oled.println("Skin Temp's 1 .. 4  ");  
              byte i = 1;    
              while(i < NoOfSensors)
              {
                  Temp = String (SkinTemp[i-1]);
                  oled.print( Temp.substring (0,4));
                  if(i < (NoOfSensors-1)) oled.print(":");
                  i++;
              }
              oled.print("'C");             
          }  // end of if(NoOfSensors > 1 
          else   // no skin sensors connected
          {
            oled.clearToEOL(); oled.println("");  
            oled.clearToEOL(); oled.println("");  
          }
          oled.clearToEOL();
          oled.println("");   // Line 5
          //oled.clearToEOL();
          //oled.println("");   // Line 6                                    
        if(WiFi.status() == WL_CONNECTED) // if home screen, print local IP address if connected
        {   
          IPAddress IP = WiFi.localIP();  // Print to OLED the local IP address
          String IPStr = String(IP[0]) + '.' + String(IP[1]) + '.' + String(IP[2]) + '.' + String(IP[3]);   
          Display_OLED_String("IP:" + IPStr,6,0,true); 
        }
        else  Display_OLED_String("WiFI Not Connected  ",7,0,true);                     
          break; 
       
       case 2:
          oled.setCursor(0,0);
          oled.println("24hr Temp&Geyser log ");          
          oled.println("------------------- 1");
          i = 0;
          if (LogPointer == 0) temp = 23; else temp = LogPointer-1;  // set to value of pointer of last stored value
          while (i < 6)
          {
            oled.setCursor(0,i+2);
            if(TimeLogHr[temp] != 0)
            {
              oled.print("T-"); 
              if(TimeLogHr[temp] < 10) oled.print("0");
              oled.print(TimeLogHr[temp]);           
              oled.print(":");
              if(TimeLogMin[temp] < 10) oled.print("0");
              oled.print(TimeLogMin[temp]); 
              oled.print(" "); 
              oled.print(TempLog[temp]); oled.print("'C ");           
              if(GeyserPowerLog[temp] < 10) oled.print("0"); 
              oled.print(GeyserPowerLog[temp]);
              oled.clearToEOL(); oled.println("");
            }
            else oled.print("     --------        ");            
            i ++;  
            temp --;  
            if(temp < 0) temp = 23;                             
          }
          break;  
       case 3:
          oled.setCursor(0,0);
          oled.println("24hr Temp&Geyser log ");          
          oled.println("------------------- 2");
          i = 0;
          temp = LogPointer - 7;
          if (temp < 0) temp = temp + 24;
          while (i < 6)
          {
            oled.setCursor(0,i+2);
            if(TimeLogHr[temp] != 0)
            {
              oled.print("T-"); 
              if(TimeLogHr[temp] < 10) oled.print("0");
              oled.print(TimeLogHr[temp]);           
              oled.print(":");
              if(TimeLogMin[temp] < 10) oled.print("0");
              oled.print(TimeLogMin[temp]); 
              oled.print(" "); 
              oled.print(TempLog[temp]); oled.print("'C ");           
              if(GeyserPowerLog[temp] < 10) oled.print("0"); 
              oled.print(GeyserPowerLog[temp]);
              oled.clearToEOL(); oled.println("");
            }
            else oled.print("     --------        ");            
            i ++;  
            temp --;  
            if(temp < 0) temp = 23;                             
          }
          break;  
       case 4:
          oled.setCursor(0,0);
          oled.println("24hr Temp&Geyser log ");          
          oled.println("------------------- 3");
          i = 0;
          temp = LogPointer - 13;
          if (temp < 0) temp = temp + 24;
          while (i < 6)
          {
            oled.setCursor(0,i+2);
            if(TimeLogHr[temp] != 0)
            {
              oled.print("T-"); 
              if(TimeLogHr[temp] < 10) oled.print("0");
              oled.print(TimeLogHr[temp]);           
              oled.print(":");
              if(TimeLogMin[temp] < 10) oled.print("0");
              oled.print(TimeLogMin[temp]); 
              oled.print(" "); 
              oled.print(TempLog[temp]); oled.print("'C ");           
              if(GeyserPowerLog[temp] < 10) oled.print("0"); 
              oled.print(GeyserPowerLog[temp]);
              oled.clearToEOL(); oled.println("");           
            }
            else oled.print("     --------        ");            
            i ++;  
            temp --;  
            if(temp < 0) temp = 23;                             
          }
          break;  
       case 5:
          oled.setCursor(0,0);
          oled.println("24hr Temp&Geyser log ");          
          oled.println("------------------- 4");
          i = 0;
          temp = LogPointer - 19;
          if (temp < 0) temp = temp + 24;          
          while (i <6)
          {
            oled.setCursor(0,i+2);
            if(TimeLogHr[temp] != 0)
            {
              oled.print("T-"); 
              if(TimeLogHr[temp] < 10) oled.print("0");
              oled.print(TimeLogHr[temp]);           
              oled.print(":");
              if(TimeLogMin[temp] < 10) oled.print("0");
              oled.print(TimeLogMin[temp]); 
              oled.print(" "); 
              oled.print(TempLog[temp]); oled.print("'C ");           
              if(GeyserPowerLog[temp] < 10) oled.print("0"); 
              oled.print(GeyserPowerLog[temp]);
              oled.clearToEOL(); oled.println("");          
            }
            else oled.print("     --------        ");            
            i ++;  
            temp --;  
            if(temp < 0) temp = 23;                            
          }      
          break;             
   }  // end of  switch(ScreenNo)
   
// --------------------  Bottom OLED line (line 7) ------------------------------------
 switch(ScreenNo)
 { 
    case 0:
      if(digitalRead(GeyserControlRelay) == GeyserOff)  // If Geyser is not on
      {
        if(WiFi.status() == WL_CONNECTED) // if home screen, print local IP address if connected
        {   
          IPAddress IP = WiFi.localIP();  // Print to OLED the local IP address
          String IPStr = String(IP[0]) + '.' + String(IP[1]) + '.' + String(IP[2]) + '.' + String(IP[3]);   
          Display_OLED_String("IP:" + IPStr,7,0,true); 
        }
        else  
        {
          Display_OLED_String("WiFI Not Connected  ",7,0,true); 
        }
      }    // end of if(digitalRead(GeyserControlRelay == GeyserOff)  // If Geyser is not on
      else Display_OLED_String("Water Heater is ON",7,0,true); 
      break;
      //---------------------------------------------------------
    case 1:
      //if(digitalRead(GeyserControlRelay) == GeyserOff)  // If Geyser is not on
      //{
        if(WiFi.status() == WL_CONNECTED) // if home screen, print SSID of connected network
        {   
          Display_OLED_String(esid,7,0,true); 
        }
        else  Display_OLED_String("WiFI Not Connected  ",7,0,true); 
      //}    // end of if(digitalRead(GeyserControlRelay == GeyserOff)  // If Geyser is not on
      //else Display_OLED_String("Geyser Heater is ON",7,0,true); 
      break;              
 }  // end of   switch(ScreenNo) 
 }  // end of else    // AP is not running = normal display modes
}  // end of void Update_OLED()
// -----------------------------------------------------------------------------
void HeaterSettingUpload()  
{
   boolean Flash = false;
   String qHeaterONDay;
   String qHeaterOFFDay;
   String qHeaterONNight;
   String qHeaterOFFNight;    
   String qGeyserTempControlEnable;
   
   if(!AccessPointRunning)
   {   
      qHeaterONDay = GeyserPages.arg("HeaterONDay");
      qHeaterOFFDay = GeyserPages.arg("HeaterOFFDay");
      qHeaterONNight = GeyserPages.arg("HeaterONNight");
      qHeaterOFFNight = GeyserPages.arg("HeaterOFFNight");     
      qGeyserTempControlEnable = GeyserPages.arg("GeyserTempControlEnable");
   }
   else
   {
      qHeaterONDay = serverAP.arg("HeaterONDay");
      qHeaterOFFDay = serverAP.arg("HeaterOFFDay");
      qHeaterONNight = serverAP.arg("HeaterONNight");
      qHeaterOFFNight = serverAP.arg("HeaterOFFNight");     
      qGeyserTempControlEnable = serverAP.arg("GeyserTempControlEnable"); 
   }  
      

      if(qGeyserTempControlEnable.length() >0)
      {
        if (qGeyserTempControlEnable == "Disable") 
        {
          GeyserTempControlEnabled = Disabled;
          EEPROM.write(EEPROM_GeyserTempControlEnabled,GeyserTempControlEnabled);
          ActivateGeyser(GeyserOff);  // turn off Geyser as a precaution (Setpoints are now disabled)
          Flash = Enabled;
        }
        if (qGeyserTempControlEnable == "Enable") 
        {
          GeyserTempControlEnabled = Enabled;
          EEPROM.write(EEPROM_GeyserTempControlEnabled,GeyserTempControlEnabled);
          Flash = Enabled;
        }        
      }   // end of if(qGeyserTempControlEnable.length() >0)
    
      if (qHeaterONDay.length() > 0)  // if data has been inputted to field
      {
       Setting1_ON_Setpoint = qHeaterONDay.toInt();
       EEPROM.write(EEPROM_Setting1_ON_Setpoint,Setting1_ON_Setpoint);
       Flash = Enabled;
      }
      
      if (qHeaterOFFDay.length() > 0)  // if data has been inputted to field
      {
       Setting1_OFF_Setpoint = qHeaterOFFDay.toInt();
       EEPROM.write(EEPROM_Setting1_OFF_Setpoint,Setting1_OFF_Setpoint);
       Flash = Enabled;        
      }

      if (qHeaterONNight.length() > 0)  // if data has been inputted to field
      {
       Setting2_ON_Setpoint = qHeaterONNight.toInt();
       EEPROM.write(EEPROM_Setting2_ON_Setpoint,Setting2_ON_Setpoint);
       Flash = Enabled;        
      }
      
      if (qHeaterOFFNight.length() > 0)  // if data has been inputted to field
      {
       Setting2_OFF_Setpoint = qHeaterOFFNight.toInt();
       EEPROM.write(EEPROM_Setting2_OFF_Setpoint,Setting2_OFF_Setpoint);
       Flash = Enabled;          
      }

      if(Flash) EEPROM.commit();  // lock in EEPROM changes if any made
      content = "<!DOCTYPE HTML>\r\n<html><form>";
      content += "<br>Success: Data has been uploaded to the device"; 
      if(!AccessPointRunning)   // if WiFi Connection
      {
        content += "<br><br><a href='http://" + GetIPaddress() + "/Heater'> Go to Heating Settings Screen  </a>";  
        content += "<br><br><a href='http://" + GetIPaddress() + "/Timer'>Go to Timer Settings Screen </a>";
        content += "<br><br><a href='http://" + GetIPaddress() + "/DisplaySystemSettings'>Go to System Settings Screen </a>";
        content += "<br><br><a href='http://" + GetIPaddress() + "/DataLog'>Go to Data Logging Screen </a>";   
        content += "<br><br><a href='http://" + GetIPaddress() + "'>Go to Main Screen </a>";       
        content += "</form></html>";
        GeyserPages.send(200, "text/html", content);
      }
      else    // if AP connection
      {
        content += "<br><br><a href='http://192.168.8.200/Heater'> Go to Heating Settings Screen  </a>";  
        content += "<br><br><a href='http://192.168.8.200/Timer'>Go to Timer Settings Screen </a>";
        content += "<br><br><a href='http://192.168.8.200/DisplaySystemSettings'>Go to System Settings Screen </a>";
        content += "<br><br><a href='http://192.168.8.200/DataLog'>Go to Data Logging Screen </a>";   
        content += "<br><br><a href='http://192.168.8.200'>Go to Main Screen </a>"; 
        content += "</form></html>";
        serverAP.send(200, "text/html", content);
      }      
}  // end of void HeaterSettingUpload()
// ------------------------------------------------------------------------------
void TimerSettingUpload()    // Geyser Timer control configuration page 
{
   String qGeyser_Timer1_Sun;
   String qGeyser_Timer1_Mon;  
   String qGeyser_Timer1_Tue;
   String qGeyser_Timer1_Wed;  
   String qGeyser_Timer1_Thu;
   String qGeyser_Timer1_Fri;
   String qGeyser_Timer1_Sat; 

   String qGeyser_Timer2_Sun;
   String qGeyser_Timer2_Mon;
   String qGeyser_Timer2_Tue;
   String qGeyser_Timer2_Wed;  
   String qGeyser_Timer2_Thu;
   String qGeyser_Timer2_Fri;  
   String qGeyser_Timer2_Sat;
    
   String qGeyser_TIM_1_ON; 
   String qGeyser_TIM_1_OFF;
   String qGeyser_TIM_2_ON;
   String qGeyser_TIM_2_OFF;  
   
   String qGeyser_Timer1_Mode;
   String qGeyser_Timer2_Mode;
     
   if(!AccessPointRunning)    // if WiFi connection
   {     
        qGeyser_Timer1_Sun = GeyserPages.arg("Geyser_Timer1_Sunday");
        qGeyser_Timer1_Mon = GeyserPages.arg("Geyser_Timer1_Monday");   
        qGeyser_Timer1_Tue = GeyserPages.arg("Geyser_Timer1_Tuesday");
        qGeyser_Timer1_Wed = GeyserPages.arg("Geyser_Timer1_Wednesday");   
        qGeyser_Timer1_Thu = GeyserPages.arg("Geyser_Timer1_Thursday");
        qGeyser_Timer1_Fri = GeyserPages.arg("Geyser_Timer1_Friday");   
        qGeyser_Timer1_Sat = GeyserPages.arg("Geyser_Timer1_Saturday");  

        qGeyser_Timer2_Sun = GeyserPages.arg("Geyser_Timer2_Sunday");
        qGeyser_Timer2_Mon = GeyserPages.arg("Geyser_Timer2_Monday");   
        qGeyser_Timer2_Tue = GeyserPages.arg("Geyser_Timer2_Tuesday");
        qGeyser_Timer2_Wed = GeyserPages.arg("Geyser_Timer2_Wednesday");   
        qGeyser_Timer2_Thu = GeyserPages.arg("Geyser_Timer2_Thursday");
        qGeyser_Timer2_Fri = GeyserPages.arg("Geyser_Timer2_Friday");   
        qGeyser_Timer2_Sat = GeyserPages.arg("Geyser_Timer2_Saturday");
      
        qGeyser_TIM_1_ON = GeyserPages.arg("Geyser_Timer_1_ON"); 
        qGeyser_TIM_1_OFF = GeyserPages.arg("Geyser_Timer_1_OFF");
        qGeyser_TIM_2_ON = GeyserPages.arg("Geyser_Timer_2_ON");
        qGeyser_TIM_2_OFF = GeyserPages.arg("Geyser_Timer_2_OFF");   
    
        qGeyser_Timer1_Mode = GeyserPages.arg("Geyser_Timer1_Mode");
        qGeyser_Timer2_Mode = GeyserPages.arg("Geyser_Timer2_Mode"); 
   }
   else  // if AP connection
   {     
        qGeyser_Timer1_Sun = serverAP.arg("Geyser_Timer1_Sunday");
        qGeyser_Timer1_Mon = serverAP.arg("Geyser_Timer1_Monday");   
        qGeyser_Timer1_Tue = serverAP.arg("Geyser_Timer1_Tuesday");
        qGeyser_Timer1_Wed = serverAP.arg("Geyser_Timer1_Wednesday");   
        qGeyser_Timer1_Thu = serverAP.arg("Geyser_Timer1_Thursday");
        qGeyser_Timer1_Fri = serverAP.arg("Geyser_Timer1_Friday");   
        qGeyser_Timer1_Sat = serverAP.arg("Geyser_Timer1_Saturday");  

        qGeyser_Timer2_Sun = serverAP.arg("Geyser_Timer2_Sunday");
        qGeyser_Timer2_Mon = serverAP.arg("Geyser_Timer2_Monday");   
        qGeyser_Timer2_Tue = serverAP.arg("Geyser_Timer2_Tuesday");
        qGeyser_Timer2_Wed = serverAP.arg("Geyser_Timer2_Wednesday");   
        qGeyser_Timer2_Thu = serverAP.arg("Geyser_Timer2_Thursday");
        qGeyser_Timer2_Fri = serverAP.arg("Geyser_Timer2_Friday");   
        qGeyser_Timer2_Sat = serverAP.arg("Geyser_Timer2_Saturday");
      
        qGeyser_TIM_1_ON = serverAP.arg("Geyser_Timer_1_ON"); 
        qGeyser_TIM_1_OFF = serverAP.arg("Geyser_Timer_1_OFF");
        qGeyser_TIM_2_ON = serverAP.arg("Geyser_Timer_2_ON");
        qGeyser_TIM_2_OFF = serverAP.arg("Geyser_Timer_2_OFF");   
    
        qGeyser_Timer1_Mode = serverAP.arg("Geyser_Timer1_Mode");
        qGeyser_Timer2_Mode = serverAP.arg("Geyser_Timer2_Mode"); 
    } 
        
        EEPROM.write(EEPROM_Geyser_Timer1_TimOn_Min, Get_Decimal_Mins_From_Web_Time_Value(qGeyser_TIM_1_ON) ); 
        EEPROM.write(EEPROM_Geyser_Timer1_TimOn_Hrs, Get_Decimal_Hours_From_Web_Time_Value(qGeyser_TIM_1_ON));          
        EEPROM.write(EEPROM_Geyser_Timer1_TimOff_Min,Get_Decimal_Mins_From_Web_Time_Value(qGeyser_TIM_1_OFF) ); 
        EEPROM.write(EEPROM_Geyser_Timer1_TimOff_Hrs, Get_Decimal_Hours_From_Web_Time_Value(qGeyser_TIM_1_OFF));

        // Update Long timer parameters with new values
        GeyserTimeControlParameters[Timer1][TimerOn][Mins] = EEPROM.read(EEPROM_Geyser_Timer1_TimOn_Min);
        GeyserTimeControlParameters[Timer1][TimerOn][Hrs] = EEPROM.read(EEPROM_Geyser_Timer1_TimOn_Hrs); 
        GeyserTimeControlParameters[Timer1][TimerOff][Mins] = EEPROM.read(EEPROM_Geyser_Timer1_TimOff_Min); 
        GeyserTimeControlParameters[Timer1][TimerOff][Hrs] = EEPROM.read(EEPROM_Geyser_Timer1_TimOff_Hrs); 
        LongTimerON[Timer1] = 
          GeyserTimeControlParameters[Timer1][TimerOn][Mins] + (GeyserTimeControlParameters[Timer1][TimerOn][Hrs] * 60);   // = Timer ON in minutes [Geyser][Timer1]  
        LongTimerOFF[Timer1] = 
          GeyserTimeControlParameters[Timer1][TimerOff][Mins] + (GeyserTimeControlParameters[Timer1][TimerOff][Hrs] * 60);

                                   // Timer settings are being changed therefore reset all Timer flags and turn Geyser off
        TimerActive[Timer1] = Stopped;
        TimerActive[Timer2] = Stopped;  
        //GeyserTempControlEnabled = Disabled;
        //ActivateGeyser(GeyserOff);                  
    
        content = "<!DOCTYPE HTML> \r\n<html>";
        content += "<br><br>Geyser Timer 1 days";           
        if (qGeyser_Timer1_Sun == "1") EEPROM.write(EEPROM_Geyser_Timer1_Sun,true);
        else EEPROM.write(EEPROM_Geyser_Timer1_Sun,false);
        if (qGeyser_Timer1_Mon == "2") EEPROM.write(EEPROM_Geyser_Timer1_Mon,true); 
        else EEPROM.write(EEPROM_Geyser_Timer1_Mon,false);
        if (qGeyser_Timer1_Tue == "3") EEPROM.write(EEPROM_Geyser_Timer1_Tue,true);
        else EEPROM.write(EEPROM_Geyser_Timer1_Tue,false);
        if (qGeyser_Timer1_Wed == "4") EEPROM.write(EEPROM_Geyser_Timer1_Wed,true);      
        else EEPROM.write(EEPROM_Geyser_Timer1_Wed,false);
        if (qGeyser_Timer1_Thu == "5") EEPROM.write(EEPROM_Geyser_Timer1_Thu,true);
        else EEPROM.write(EEPROM_Geyser_Timer1_Thu,false);       
        if (qGeyser_Timer1_Fri == "6") EEPROM.write(EEPROM_Geyser_Timer1_Fri,true);
        else EEPROM.write(EEPROM_Geyser_Timer1_Fri,false);   
        if (qGeyser_Timer1_Sat == "7") EEPROM.write(EEPROM_Geyser_Timer1_Sat,true);                   
        else EEPROM.write(EEPROM_Geyser_Timer1_Sat,false);    
    
        // TIMER 2   
        EEPROM.write(EEPROM_Geyser_Timer2_TimOn_Min,Get_Decimal_Mins_From_Web_Time_Value(qGeyser_TIM_2_ON) ); 
        EEPROM.write(EEPROM_Geyser_Timer2_TimOn_Hrs, Get_Decimal_Hours_From_Web_Time_Value(qGeyser_TIM_2_ON));             
        EEPROM.write(EEPROM_Geyser_Timer2_TimOff_Min,Get_Decimal_Mins_From_Web_Time_Value(qGeyser_TIM_2_OFF) ); 
        EEPROM.write(EEPROM_Geyser_Timer2_TimOff_Hrs, Get_Decimal_Hours_From_Web_Time_Value(qGeyser_TIM_2_OFF)); 

        // Update Long timer parameters with new values
        GeyserTimeControlParameters[Timer2][TimerOn][Mins] = EEPROM.read(EEPROM_Geyser_Timer2_TimOn_Min);
        GeyserTimeControlParameters[Timer2][TimerOn][Hrs] = EEPROM.read(EEPROM_Geyser_Timer2_TimOn_Hrs); 
        GeyserTimeControlParameters[Timer2][TimerOff][Mins] = EEPROM.read(EEPROM_Geyser_Timer2_TimOff_Min); 
        GeyserTimeControlParameters[Timer2][TimerOff][Hrs] = EEPROM.read(EEPROM_Geyser_Timer2_TimOff_Hrs); 
        LongTimerON[Timer2] = 
          GeyserTimeControlParameters[Timer2][TimerOn][Mins] + (GeyserTimeControlParameters[Timer2][TimerOn][Hrs] * 60);   // = Timer ON in minutes [Geyser][Timer2]  
        LongTimerOFF[Timer2] = 
          GeyserTimeControlParameters[Timer2][TimerOff][Mins] + (GeyserTimeControlParameters[Timer2][TimerOff][Hrs] * 60);



                 
        if (qGeyser_Timer2_Sun == "1") EEPROM.write(EEPROM_Geyser_Timer2_Sun,true);
        else EEPROM.write(EEPROM_Geyser_Timer2_Sun,false);    
        if (qGeyser_Timer2_Mon == "2") EEPROM.write(EEPROM_Geyser_Timer2_Mon,true);      
        else EEPROM.write(EEPROM_Geyser_Timer2_Mon,false);     
        if (qGeyser_Timer2_Tue == "3") EEPROM.write(EEPROM_Geyser_Timer2_Tue,true);      
        else EEPROM.write(EEPROM_Geyser_Timer2_Tue,false);     
        if (qGeyser_Timer2_Wed == "4") EEPROM.write(EEPROM_Geyser_Timer2_Wed,true);      
        else EEPROM.write(EEPROM_Geyser_Timer2_Wed,false);     
        if (qGeyser_Timer2_Thu == "5") EEPROM.write(EEPROM_Geyser_Timer2_Thu,true);      
        else EEPROM.write(EEPROM_Geyser_Timer2_Thu,false);     
        if (qGeyser_Timer2_Fri == "6") EEPROM.write(EEPROM_Geyser_Timer2_Fri,true);      
        else EEPROM.write(EEPROM_Geyser_Timer2_Fri,false); 
        if (qGeyser_Timer2_Sat == "7") EEPROM.write(EEPROM_Geyser_Timer2_Sat,true);      
        else EEPROM.write(EEPROM_Geyser_Timer2_Sat,false);
    
        if (qGeyser_Timer1_Mode == "D") 
        {
          EEPROM.write(EEPROM_TimerEnableGeyserTimer1,false); 
          TimerEnable[Timer1] = false;   // turn off timer
          ActivateGeyser(GeyserOff);  // turn off relay
        }
        
        if (qGeyser_Timer1_Mode == "E") 
        {
          EEPROM.write(EEPROM_TimerEnableGeyserTimer1,true);  
          TimerEnable[Timer1] = true;   // turn on timer          
        }
        // -------------------------------------------
        if (qGeyser_Timer2_Mode == "D") 
        {
          EEPROM.write(EEPROM_TimerEnableGeyserTimer2,false); 
          TimerEnable[Timer2] = false;   // turn off timer if running
          ActivateGeyser(GeyserOff);  // turn off relay if on
          //}         
        }
        if (qGeyser_Timer2_Mode == "E") 
        {
          EEPROM.write(EEPROM_TimerEnableGeyserTimer2,true); 
          TimerEnable[Timer2] = true;     // turn on timer 
        }
        // ----------------------------------------------------   
        EEPROM.commit();
        DownloadFrom_EEPROM_GeyserControlParameters(false);   
        content = "<!DOCTYPE HTML>\r\n<html><form>";
        content += "<br>Success: Data has been uploaded to the device";  
        if(!AccessPointRunning)   // if WIFI connection
        {
          content += "<br><br><a href='http://" + GetIPaddress() + "/Heater'> Go to Heating Settings Screen  </a>";  
          content += "<br><br><a href='http://" + GetIPaddress() + "/Timer'>Go to Timer Settings Screen </a>";
          content += "<br><br><a href='http://" + GetIPaddress() + "/DisplaySystemSettings'>Go to System Settings Screen </a>";
          content += "<br><br><a href='http://" + GetIPaddress() + "/DataLog'>Go to Data Logging Screen </a>";   
          content += "<br><br><a href='http://" + GetIPaddress() + "'>Go to the Main Menu </a>";      
          content += "</form></html>";
          GeyserPages.send(200, "text/html", content);
        }
        else      // if AP connection
        {
          content += "<br><br><a href='http://192.168.8.200/Heater'> Go to Heating Settings Screen  </a>";  
          content += "<br><br><a href='http://192.168.8.200/Timer'>Go to Timer Settings Screen </a>";
          content += "<br><br><a href='http://192.168.8.200/DisplaySystemSettings'>Go to System Settings Screen </a>";
          content += "<br><br><a href='http://192.168.8.200/DataLog'>Go to Data Logging Screen </a>";   
          content += "<br><br><a href='http://192.168.8.200'>Go to Main Menu </a>"; 
          content += "</form></html>";
          serverAP.send(200, "text/html", content);
        }       
}  // end of void TimerSettingUpload()    // Geyser Timer control configuration page 
// ------------------------------------------------------------------------------------
void WiFiSetup()    // Wifi Configuration page
{
        String qsid;
        String qpass;
        String qname;
        String qWiFiManualEnable;
        String qpassAP;
        
        qsid = serverAP.arg("ssid");   // check input from web page when submitted
        qpass = serverAP.arg("pass");
        qname = serverAP.arg("name"); 
        qWiFiManualEnable = serverAP.arg("WiFiManualEnable");
        qpassAP = serverAP.arg("passAP");        
    
       // .................. Process Network SSID  .................................       
          if (qsid.length() > 0 ) // Only if something has been entered into the SSID web field
          {
            unsigned int i=EEPROM_WiFi_SSID;
            while(i < EEPROM_WiFi_SSID + MaxStringLength)  // Clear out all of the previous stored entries for SSID
            {
              EEPROM.write(i, 255);
              i++;
            }
            i=0;
            while(i < qsid.length())   // store new SSID to EEPROM
            {
              EEPROM.write(EEPROM_WiFi_SSID + i, qsid[i]);
              i++;
            }
          } // end of if (qsid.length() > 0 ) // Only if something has been entered into the SSID web field
            
       // .................. Process Network password  .................................          
          if (qpass.length() > 0) // Only if something has been entered into the Password web field
          {    
            unsigned int i = EEPROM_WiFi_Password;
            while(i < (EEPROM_WiFi_Password + MaxStringLength))
            {
              EEPROM.write(i, 255); // Clear out all of the previous stored entries for password
              i++;
            }         
            i = 0; 
            while(i < qpass.length())  // store new password to EEPROM
            {
              EEPROM.write(EEPROM_WiFi_Password + i, qpass[i]);
              i++;
            }           
          } // end of if (qpass.length() > 0) // Only if something has been entered into the Password web field
                         
       // ................... Process Device Name ..................................          
          if (qname.length() > 0) // Only if something has been entered into the Device name web field
          {    
            unsigned int i = EEPROM_HostName;
            while(i < (EEPROM_HostName + MaxStringLength))
            {
              EEPROM.write(i, 255); // Clear out all of the previous stored entries for password
              i++;
            }         
            i = 0; 
            while(i < qname.length())  // store new device name to EEPROM
            {
              EEPROM.write(EEPROM_HostName + i, qname[i]);
              i++;
            }
          }  // end of if (qname.length() > 0) // Only if something has been entered into the Device name web field
       
       //  ................. Process Wifi Enable ................................... 
          if (qWiFiManualEnable.length() > 0 ) // Only if something has been entered into the field
          {
            if(qWiFiManualEnable == "E")   WiFiManualEnable = true; else WiFiManualEnable = false;
            EEPROM.write(EEPROM_WiFiManualEnable, WiFiManualEnable);
          } // end of     if (qWifiManualEnable.length() > 0 ) // Only if something has been 

       // .......................... Process AP password ...........................................
          if(qpassAP.length() > 0)
          {
            unsigned int i = EEPROM_AP_Password;
            while(i < (EEPROM_AP_Password + MaxStringLength))
            {
              EEPROM.write(i, 255); // Clear out all of the previous stored entries for password
              i++;
            }         
            i = 0; 
            while(i < qpassAP.length())  // store new password to EEPROM
            {
              EEPROM.write(EEPROM_AP_Password + i, qpassAP[i]);
              i++;
            }            
          }  // end of if(qpassAP > 0)   
      
          EEPROM.commit();   // Lock in EEPROM changes                             
          content = "<!DOCTYPE HTML>\r\n<html><form>";
          content += "<br>Success: Data has been uploaded to the device";
          content += "<br><b><i>Note that the device will need resetting before any new WiFi names or WiFi Enable become active</b></i>"; 
          content += "<br><br><a href='http://192.168.8.200/Heater'> Go to Heating Settings Screen  </a>";  
          content += "<br><a href='http://192.168.8.200/Timer'>Go to Timer Settings Screen </a>";
          content += "<br><a href='http://192.168.8.200/DisplaySystemSettings'>Go to System Settings Screen </a>";
          content += "<br><a href='http://192.168.8.200/DataLog'>Go to Data Logging Screen </a>";   
          content += "<br><a href='http://192.168.8.200'>Go to Main Menu </a>"; 
          content += "</form></html>";
          serverAP.send(200, "text/html", content);
          HeartbeatTick();   // Feed the HW watchdog!          
}   // end of void WiFiSetup()  // Wifi Configuration page
 // ----------------------------------------------------------------------
void DisplaySystemSettings()  
{
    content = "<!DOCTYPE HTML> \r\n<html>";
    //content += "<meta http-equiv='refresh' content='10'>";  
    content += "<h3>" + HostName;
    if(AccessPointRunning) content += " - (Offline)"; else content += " - (Online)";
    content += " System Settings Screen</h3>";  
    //WebClock();   
    //content += "<br>Long Clock (minutes) = " + String(LongClockMinute) + "  ";        
    
    content += "<fieldset>"; 
    content += "<legend><b>Network</b></legend>";
    if(!AccessPointRunning)  // if WiFi Screen
    {
      content += "Start the Offline Configuration Tool to change the Wifi Settings if necessary";
      content += "<br>Note that this will stop the current WiFi connection. You must reconnect your mobile device or computer to the Controller setup network SSID and connect your web browser to http://192.168.8.200";
      content += "<br><br><a href='http://" + GetIPaddress() + "/StartAP '>Start Offline Configuration Server </a>";             
      content += "<br>Note that the Controller will reset after quiting the Offline Configuration Server";        
    }
    else content += "<a href='http://192.168.8.200/WiFiSettings '>Go to WiFi Setting Screen </a>";  // AP screen
     
    content += "<br><br>Currently stored SSID = " + esid;
    content += "<br>Currently stored Device Name = " + HostName + "<br>";
    
    if(AccessPointRunning)  // Only if AP
    {
      if (WiFi.status() == WL_CONNECTED) 
      content += "<br>Current WiFi connection is good<br>";
      else
      content += "<br>Current WiFi connection is broken<br>"; 
    }  
    content += "</fieldset>"; 
        // -------------------------------------------------------------------------------     
    content += "<fieldset>"; 
    content += "<legend><b>Program control flags & variables - read only (refresh page to update)</b></legend>";     
    content += "The last successful NTP update was at ";
    content += String(Last_NTP_Hour) + ":";
    if (Last_NTP_Minute > 9) content += String(Last_NTP_Minute); else content += "0" + String(Last_NTP_Minute);    
    content += "  Day:" + String(Last_NTP_Day) + "<br>";
    content += "<br>Diagnostic messages on the serial port are ";
    if (EnableMessages) content += "enabled"; else content += "disabled";     
    content += "<br>Geyser Timer 1 is "; 
    if(TimerActive[0]) content += "running "; else  content += "not running ";
    content += "<and Timer 2 is "; 
    if(TimerActive[1]) content += "running"; else  content += "not running";    
    content += "<br>Failed Temperature Measurement Counter = " + String(TempFailCounter);
    content += "</fieldset>";  
        // ------------------------------------------------------------------------------- 
    content += "<fieldset>"; 
    content += "<legend><b>DS18B20 Geyser Temperature Sensor addresses </b></legend>"; 
    content += "<i>Reboot the controller after any changes</i>";
    content += "<br><form action = 'SaveTemperatureSensorAddresses'>";

    byte NoOfSensors = sensors.getDeviceCount();
    byte Counter = 0;
    while(Counter < NoOfSensors)
    {
       if(Counter == 0)
       {    
          content += "Main Temp Sensor index no = " + String(TempSensorIndex[0]) + " "; 
          content += "<label>- Change Index No (0-4)</label> ";
          content += "<input type = 'number' name='SensorIndex1' + length=3 size = '3' min='0' max = '4'><br>";
       }
       if(Counter == 1)
       {
          content += "Skin Temp Sensor 1 index no = " + String(TempSensorIndex[1]) + " ";       
          content += "<label>- Change Index No (0-4)</label> ";
          content += "<input type = 'number' name='SensorIndex2' + length=3 size = '3' min='0' max = '4'><br>";
       }
       if(Counter == 2)
       {    
          content += "Skin Temp Sensor 2 index no = " + String(TempSensorIndex[2]) + " ";        
          content += "<label>- Change Index No (0-4)</label> ";
          content += "<input type = 'number' name='SensorIndex3' + length=3 size = '3' min='0' max = '4'><br>";
       }
       if(Counter == 3)
       {       
          content += "Skin Temp Sensor 3 index no = " + String(TempSensorIndex[3]) + " ";        
          content += "<label>- Change Index No (0-4)</label> ";
          content += "<input type = 'number' name='SensorIndex4' + length=3 size = '3' min='0' max = '4'><br>";
       }
       if(Counter == 4)
       {        
          content += "Skin Temp Sensor 4 index no = " + String(TempSensorIndex[4]) + " ";                  
          content += "<label>- Change Index No (0-4)</label> ";
          content += "<input type = 'number' name='SensorIndex5' + length=3 size = '3' min='0' max = '4'><br>";    
       }     
       Counter ++;
    }      
    content += "<br><input type = 'submit' value = 'Upload New Addresses to EEPROM'>"; 
    content += "</form> </html>";  
    content += "</fieldset>";  

    content += "<br><fieldset>";
    content += "<legend><b>Auto Switching On and Off times Of Local Display (Hours only)</b></legend>";
    content += "Display is only enabled inbetween these times&nbsp<i/>(Display is OFF if both times are the same)</i>";
    content += "<br><form action = 'SaveDisplayTimerOnOffSettings'>";    
    content += "<br>ON time =&nbsp&nbsp" + String(DisplayOnTime);
    content += "&nbsp&nbsp<label>New ON time (0-23)  </label> ";    
    content += "<input type = 'number' name='DisplayOn' + length=4 size = '3' min='0' max = '23'>";
    content += "<br>OFF time =&nbsp" + String(DisplayOffTime);
    content += "&nbsp&nbsp<label>New OFF time (0-23) </label> ";    
    content += "<input type = 'number' name='DisplayOff' + length=4 size = '3' min='0' max = '23'>";  
    content += "<br><br><input type = 'submit' value = 'Upload new Display Timer settings to EEPROM'>"; 
    content += "</form> </html>";           
    content += "</fieldset>";      
        // -------------------------------------------------------------------------------               
    if (!AccessPointRunning) // If WiFi Connection
    {
      content += "<br><a href='http://" + GetIPaddress() + "/Heater'> Go to Heating Settings Screen  </a>";  
      content += "<br><a href='http://" + GetIPaddress() + "/Timer'>Go to Timer Settings Screen </a>";
      content += "<br><a href='http://" + GetIPaddress() + "/DataLog'>Go to Data Logging Screen </a>";   
      content += "<br><a href='http://" + GetIPaddress() + "/Diagnostics'>Go to Diagnostics Screen </a>"; 
      content += "<br><a href='http://" + GetIPaddress() + "/Quit'>Reset the Controller </a>";        
      content += "<br><br><a href='http://" + GetIPaddress() + "'>Go to the Main Menu </a>";  
      content += "</form> </html>";       
      GeyserPages.send(200, "text/html", content);  // send message to AP Webpage
    }
    else     // If AP connection
    {
      content += "<br><a href='http://192.168.8.200/Heater'> Go to Heating Settings Screen  </a>";  
      content += "<br><a href='http://192.168.8.200/Timer'>Go to Timer Settings Screen </a>";
      content += "<br><a href='http://192.168.8.200/DataLog'>Go to Data Logging Screen </a>";   
      content += "<br><br><a href='http://192.168.8.200'>Go to Main Menu </a>"; 
      content += "</form> </html>";       
      serverAP.send(200, "text/html", content);  // send message to AP Webpage
    }
 }   // end of DisplaySystemSettings() 
// -------------------------------------------------------------------------- 
 void ManualSwitchGeyser()  
 { 
   content = "<!DOCTYPE HTML> \r\n<html>";     
     
   if(digitalRead(GeyserControlRelay) == GeyserOff) 
   {
     content += "<br>Geyser is switched ON";   
     GeyserManualControlTimer = GeyserManualControlTimerInterval; 
     SelfCheck = Disabled;   // temporarily disable self check 
     ActivateGeyser(GeyserOn);       
   }
   else
   {
     content += "<br>Geyser is switched Off";   
     ActivateGeyser(GeyserOff); 
     GeyserManualControlTimer = 0;    
   }
   if(!AccessPointRunning)  // If WiFi connection
   {
      content += "<br><br><a href='http://" + GetIPaddress() + "/Heater'> Go to Heating Settings Screen  </a>";  
      content += "<br><a href='http://" + GetIPaddress() + "/Timer'>Go to Timer Settings Screen </a>";
      content += "<br><a href='http://" + GetIPaddress() + "/DisplaySystemSettings'>Go to System Settings Screen </a>";
      content += "<br><a href='http://" + GetIPaddress() + "/DataLog'>Go to Data Logging Screen </a>";   
      content += "<br><br><a href='http://" + GetIPaddress() + "'>Go to the Main Menu </a>";             
      GeyserPages.send(200, "text/html", content); 
   }
   else   // AP connection
   {
      content += "<br><br><a href='http://192.168.8.200/Heater'> Go to Heating Settings Screen  </a>";  
      content += "<br><a href='http://192.168.8.200/Timer'>Go to Timer Settings Screen </a>";
      content += "<br><a href='http://192.168.8.200/DisplaySystemSettings'>Go to System Settings Screen </a>";
      content += "<br><a href='http://192.168.8.200/DataLog'>Go to Data Logging Screen </a>";   
      content += "<br><br><a href='http://192.168.8.200'>Go to Main Menu </a>";            
      serverAP.send(200, "text/html", content); 
   }
}   // end of  void ManualSwitchGeyser()  
  // ---------------------------------------------------------------------
 void WiFiSettings()
 {
        content = "<!DOCTYPE HTML>\r\n<html>";
        content += "<h3>" + HostName + " - (Offine) WiFi Configuration Page</h3>";
        content += "<form action='WiFiSetup' autocomplete = 'off'></p>";         
               
        content += "<fieldset>"; 
        content += "<legend><b>Current stored Network details</b></legend>"; 
        content += "IP address " + GetIPaddress() + "<br>";
        content += "SSID = " + esid + "<br>";
        content += "Device name = " + HostName; 
        content += "<br><br>Local WiFi is manually ";
        if (WiFiManualEnable == true) content += "Enabled"; else content += "Disabled";              
        content += "</fieldset>";   
        
        content += "<fieldset>";             
        content += "<legend><b>Detected Networks ... </b></legend>";
        content += st ;
        content += "</fieldset>"; 
                            
        content += "<fieldset>";       
        content += "<legend><b>Input new settings (blank fields will not be updated)</b></legend>";
        content += "<legend><b>Input new network settings (20 char max per field, blank fields will not be updated)</b></legend>";
        content += "<label>SSID:</label>  <input name='ssid' size = '22' length=20>"; 
        content += "<label>&nbsp&nbspPassword: </label> <input name='pass' size = '22' length=20>";
        content += "<br><label>Name:</label> <input name='name' size = '22' length=20>"; 
        content += "<label>&nbsp&nbspAP Password:</label> <input name='passAP' size = '22' length=20><br><br>"; 
        
        if(WiFiManualEnable == true) content += "<input type = 'radio' name = 'WiFiManualEnable' value = 'D'>Click to DISABLE WiFi";  
        else content += "<input type = 'radio' name = 'WiFiManualEnable' value = 'E'>Click to ENABLE WiFi";                  
        content += "</fieldset>";         
                
        content += "<p><input type='submit' value = 'Upload new settings to the device'> </form>";
        content += "<p><a href='http://192.168.8.200'> Go to the Main Menu </a>";
        content += "</form></html>"; 
        serverAP.send(200, "text/html", content); 
        HeartbeatTick();   // Feed the HW watchdog!        
 }
 // ------------------------------------------------------------------------------
void Quit()         // Close AP and reset device
 {
    content = "<!DOCTYPE HTML> \r\n<html>";
    content += "Device will now reset and attempt login";         
    if(AccessPointRunning) serverAP.send(200, "text/html", content);  // send message to AP Webpage
    else GeyserPages.send(200, "text/html", content);
    delay(2000);
    if(AccessPointRunning) ResetDevice("From Access Point"); else ResetDevice("From Web Page");
 }   // end of Quit()
// ---------------------------------------------------------------------------------
void HeartbeatTick()   // short heatbeat tick to keep the HW watchdog alive
{
  digitalWrite(Heartbeat_LED, LED_On);
  delay(2);
  digitalWrite(Heartbeat_LED, LED_Off); 
  delay(2);   
}
// ------------------------------------------------------------
void DownloadFrom_EEPROM_GeyserControlParameters(boolean Display)  // display = false = suppress Special messages
 {        // download from EEPROM to working memory the timer control parameters
                    // -------------------- TIMER 1-------------------------
    GeyserTimeControlParameters[Timer1][TimerOn][Mins] = EEPROM.read(EEPROM_Geyser_Timer1_TimOn_Min);
    GeyserTimeControlParameters[Timer1][TimerOn][Hrs] = EEPROM.read(EEPROM_Geyser_Timer1_TimOn_Hrs); 
    GeyserTimeControlParameters[Timer1][TimerOff][Mins] = EEPROM.read(EEPROM_Geyser_Timer1_TimOff_Min); 
    GeyserTimeControlParameters[Timer1][TimerOff][Hrs] = EEPROM.read(EEPROM_Geyser_Timer1_TimOff_Hrs); 
    LongTimerON[Timer1] = 
      GeyserTimeControlParameters[Timer1][TimerOn][Mins] + (GeyserTimeControlParameters[Timer1][TimerOn][Hrs] * 60);   // = Timer ON in minutes [Geyser][Timer1]  
    LongTimerOFF[Timer1] = 
      GeyserTimeControlParameters[Timer1][TimerOff][Mins] + (GeyserTimeControlParameters[Timer1][TimerOff][Hrs] * 60);
    if(Display)
    {
      Serial.println("");
      Serial.print("Geyser Timer 1:");
      Serial.print("   Time On = "); Serial.print(GeyserTimeControlParameters[Timer1][TimerOn][Hrs]); 
      Serial.print(":"); 
      if(GeyserTimeControlParameters[Timer1][TimerOn][Mins] < 10) Serial.print("0"); 
      Serial.print(GeyserTimeControlParameters[Timer1][TimerOn][Mins]);     
      Serial.print("   Time Off = "); Serial.print(GeyserTimeControlParameters[Timer1][TimerOff][Hrs]); 
      Serial.print(":"); 
      if(GeyserTimeControlParameters[Timer1][TimerOff][Mins] < 10) Serial.print("0");       
      Serial.println(GeyserTimeControlParameters[Timer1][TimerOff][Mins]);       
    }       
                  // ----------------------- TIMER 2 ---------------------
    GeyserTimeControlParameters[Timer2][TimerOn][Mins] = EEPROM.read(EEPROM_Geyser_Timer2_TimOn_Min); 
    GeyserTimeControlParameters[Timer2][TimerOn][Hrs] = EEPROM.read(EEPROM_Geyser_Timer2_TimOn_Hrs); 
    GeyserTimeControlParameters[Timer2][TimerOff][Mins] = EEPROM.read(EEPROM_Geyser_Timer2_TimOff_Min); 
    GeyserTimeControlParameters[Timer2][TimerOff][Hrs] = EEPROM.read(EEPROM_Geyser_Timer2_TimOff_Hrs); 
    LongTimerON[Timer2] = 
      GeyserTimeControlParameters[Timer2][TimerOn][Mins] + (GeyserTimeControlParameters[Timer2][TimerOn][Hrs] * 60);   // = Timer ON in minutes [Geyser][Timer2]  
    LongTimerOFF[Timer2] = 
      GeyserTimeControlParameters[Timer2][TimerOff][Mins] + (GeyserTimeControlParameters[Timer2][TimerOff][Hrs] * 60);    
    
    if(Display)
    {
      Serial.print("Geyser Timer 2:");     
      Serial.print("   Time On = "); Serial.print(GeyserTimeControlParameters[Timer2][TimerOn][Hrs]); 
      Serial.print(":"); 
      if(GeyserTimeControlParameters[Timer2][TimerOn][Mins] < 10) Serial.print("0");      
      Serial.print(GeyserTimeControlParameters[Timer2][TimerOn][Mins]); 
      Serial.print("   Time Off = "); Serial.print(GeyserTimeControlParameters[Timer2][TimerOff][Hrs]); 
      Serial.print(":"); 
      if(GeyserTimeControlParameters[Timer2][TimerOff][Mins] < 10) Serial.print("0");   
      Serial.println(GeyserTimeControlParameters[Timer2][TimerOff][Mins]);      
    }  
           
   byte Day;   // pointer
   if(Display) Serial.println("");  

   byte Timer_No = 0;
   while (Timer_No < 2)  // for each of 2 timers per output
   {
      if(Display)
      {
         Serial.print("Geyser Timer "); 
         Serial.print (Timer_No+1);
         Serial.print (" Enabled days:  ");
      }
 
      Day = 0;
      while (Day < 7)   // read each timer day flag from EEPROM
      {
        GeyserTimeControlParameters_Day_Flags[Timer_No][Day] = EEPROM.read(EEPROM_Geyser_Timer1_Sun + (Timer_No*7) + Day);
        if(Display)
        {
            Serial.print(GeyserTimeControlParameters_Day_Flags[Timer_No][Day]); 
            Serial.print(":");
        }
        Day ++;   
      }     // end of while (Day < 7)   // read each timer day flag from EEPROM
      if(Display)
      {
        Serial.println("");
      }
      Timer_No ++;
   }       // end of while (Timer_No < 2)  // for each of 2 timers per output
  
  if(Display)
  {
    Serial.println(""); Serial.println("Timer status");
  }
                                                          
    Setting1_ON_Setpoint = EEPROM.read (EEPROM_Setting1_ON_Setpoint); 
    Setting2_ON_Setpoint = EEPROM.read (EEPROM_Setting2_ON_Setpoint); 
    Setting1_OFF_Setpoint = EEPROM.read (EEPROM_Setting1_OFF_Setpoint); 
    Setting2_OFF_Setpoint = EEPROM.read (EEPROM_Setting2_OFF_Setpoint); 
    GeyserTempControlEnabled = EEPROM.read (EEPROM_GeyserTempControlEnabled);
    TimerEnable[Timer1] = EEPROM.read (EEPROM_TimerEnableGeyserTimer1);
    TimerEnable[Timer2] = EEPROM.read (EEPROM_TimerEnableGeyserTimer2);    
      
    if (Display)
    {
       Serial.print(F("Timer 1 is ")); 
       if (TimerEnable[Timer1]) Serial.println(F("enabled")); else Serial.println(F("disabled"));
       Serial.print(F("Timer 2 is ")); 
       if (TimerEnable[Timer2]) Serial.println(F("enabled")); else Serial.println(F("disabled"));
        Serial.print(F("Average Temperature ('C) = ")); Serial.println(AverageGeyserTemperature,1);
        Serial.print(F("Maximum Temperature ('C) = ")); Serial.println(TempMax,1);
        Serial.print(F("Minimum Temperature ('C) = ")); Serial.println(TempMin,1);  
        Serial.print(F("Set 1 Setpoint ON ('C = ")); Serial.println(Setting1_ON_Setpoint);
        Serial.print(F("Set 1 Setpoint OFF ('C)= ")); Serial.println(Setting1_OFF_Setpoint);      
        Serial.print(F("Set 2 Setpoint ON ('C) = ")); Serial.println(Setting2_ON_Setpoint);
        Serial.print(F("Set 2 Setpoint OFF ('C)= ")); Serial.println(Setting2_OFF_Setpoint);      
        Serial.print(F("Geyser Setpoints are set to "));
        if(GeyserTempMode() == SetpointGroup2) 
        Serial.println(F("Group 2")); 
        else Serial.println(F("Group 1"));      
        Serial.print(F("Geyser temperature control is ")); 
        if(GeyserTempControlEnabled) Serial.println(F("ON")); else Serial.println(F("OFF"));                 
    }   
}   // end of  void DownloadFrom_EEPROM_GeyserControlParameters()
  // -------------------------------------------------------------------- 
void ActivateGeyser(boolean State)  
{     
  digitalWrite(GeyserControlRelay, State);   // activate Geyser relay
  if(State == GeyserOff)  // Geyser is being turned off
  {
    GeyserOnTimer += (millis() - GeyserOnRunningTimer);  // get time in milliseconds that it has been turned on
    Serial.print (F("Geyser has been on for (S) ")); Serial.println(GeyserOnTimer/1000);
  }
  else GeyserOnRunningTimer = millis();   // relay is being turned on - reset running timers to current time 

}    // end of void ActivateGeyser(byte Output_No, boolean State, boolean Set_Output_Timer) 
// ------------------------------------------------------------
void ResetDevice(String Reason)  // Reset the device - method depends on whether HW watchdog is fitted
{
 WiFi.softAPdisconnect(true);     // precautionary. Ensure that the soft Access point server is disconnected
 WiFi.mode(WIFI_STA);             // Set WIFI module to STA mode 
 delay(100);
 if (EnableMessages)
 {
    Serial.print("Forced reset -  ");
    Serial.println(Reason);
 }
 //if (WatchDogType == Hardware)
 // {
 //   Serial.println(F("Reset by external hardware watchdog"));
 //   while(true) // hang up and let the hardware watchdog kick in after some seconds
 //   {
 //     delay(1);
 //   }
 // }  // end of  if (WatchDogType == Hardware)
 //else        // HW watchdog not fitted so use internal reset command
 // {
 //   Serial.println(F("Reset by Software"));
    delay(10);
    ESP.restart();  
    while(true) delay(10);
  //}  // end of else  
} // end of void ResetDevice(String Reason)
  // -------------------------------------------------------------------
String Control_Timer_Time_To_String (byte Relay_No, byte TimerNo, byte TimeOn_Off )
{             // convert the Timer control parameter to a string value (for web page display)
  String T ;
  String T1;
  switch (Relay_No)
  {
    case 0:
    T =  String(GeyserTimeControlParameters[TimerNo][TimeOn_Off][1]);   // convert hours
    if (T.length() == 1) T = "0" + T;                           // if single digit add leading zero
    T1 = String(GeyserTimeControlParameters[TimerNo][TimeOn_Off][0]);   // convert minutes
    if (T1.length() == 1) T1 = "0" + T1;                        // if single digit add leading zero
    T = T + ":" + T1;           // combine digits with separator
    break;    
  } // end of Switch (Relay_No)
  return T;   
} // end of String Control_Timer_Time_To_String(byte Relay_No, byte TimerNo, byte TimeOn_Off, ) 
// -----------------------------------------------------------------------------
void Timer_Page_Geyser()  
{
        const byte TimeOn = 0;
        const byte TimeOff =1;
        const byte Timer1 = 0;
        const byte Timer2 = 1;  
        const byte Geyser = 0;      
        String Tim1 = Control_Timer_Time_To_String(Geyser,Timer1, TimeOn);    // convert time values to a string
        String Tim2 = Control_Timer_Time_To_String(Geyser,Timer1, TimeOff);
          
        content += "<h3>" + HostName;
        if(AccessPointRunning) content += " - (Offline)"; else content += " - (Online)";
        content += " Timer Control Settings</h3>";  
        content += "Timers 1 & 2 control when the heater is automatically controlled by the Setpoint values.";
        content += "<br><i>Note that the heater itself is not directly controlled by the timers, only the setpoint enables.</i><br><br>";   
       
        WebClock();
                
        if (content += "<p></p><form action = 'TimerSettingUpload'>");       
        content += "<fieldset>";  
        content += "<legend><b>TIMER 1 - Timer 1 & Timer 2 must NOT overlap</b></legend>";     
        content += "Timer 1 ON: <input type = 'time' name = 'Geyser_Timer_1_ON'  value = ";
        content += Tim1 + ">";      
        content += "&nbsp Timer 1 OFF: <input type = 'time' name = 'Geyser_Timer_1_OFF' value = ";
        content += Tim2 + ">"; 
                   
        content += "<br><br><input type = 'checkbox' name = 'Geyser_Timer1_Sunday' value = '1' ";
        if (GeyserTimeControlParameters_Day_Flags[Timer1][0]) content += " checked";
        content += ">Sunday&nbsp&nbsp"; 
              
        content += "<input type = 'checkbox' name = 'Geyser_Timer1_Monday'  value = '2' ";
        if (GeyserTimeControlParameters_Day_Flags[Timer1][1]) content += " checked";
        content += ">Monday&nbsp&nbsp"; 
        
        content += "<input type = 'checkbox' name = 'Geyser_Timer1_Tuesday' value = '3' ";
        if (GeyserTimeControlParameters_Day_Flags[Timer1][2]) content += " checked";
        content += ">Tuesday&nbsp&nbsp"; 
                
        content += "<input type = 'checkbox' name = 'Geyser_Timer1_Wednesday'  value = '4' ";
        if (GeyserTimeControlParameters_Day_Flags[Timer1][3]) content += " checked";
        content += ">Wednesday&nbsp&nbsp"; 
                
        content += "<input type = 'checkbox' name = 'Geyser_Timer1_Thursday' value = '5' ";
        if (GeyserTimeControlParameters_Day_Flags[Timer1][4]) content += " checked";
        content += ">Thursday&nbsp&nbsp"; 
                
        content += "<input type = 'checkbox' name = 'Geyser_Timer1_Friday'  value = '6' ";
        if (GeyserTimeControlParameters_Day_Flags[Timer1][5]) content += " checked";
        content += ">Friday&nbsp&nbsp"; 
                
        content += "<input type = 'checkbox' name = 'Geyser_Timer1_Saturday' value = '7' ";
        if (GeyserTimeControlParameters_Day_Flags[Timer1][6]) content += " checked";
        content += ">Saturday";         
        content += "</fieldset><br>";

        content += "<fieldset>";  
        content += "<legend><b>TIMER 2 - Timer 1 & Timer 2 must NOT overlap</b></legend>"; 
        Tim1 = Control_Timer_Time_To_String(Geyser,Timer2, TimeOn);    // convert time values to a string
        Tim2 = Control_Timer_Time_To_String(Geyser,Timer2, TimeOff);       
        content += "Timer 2 ON: <input type = 'time' name = 'Geyser_Timer_2_ON' value = ";
        content += Tim1 + ">"; 
        content += "&nbsp Timer 2 OFF: <input type = 'time' name = 'Geyser_Timer_2_OFF' value = "; 
        content += Tim2 + ">"; 
        
        content += "<br><br><input type = 'checkbox' name = 'Geyser_Timer2_Sunday' value = '1' ";
        if(GeyserTimeControlParameters_Day_Flags[Timer2][0]) content += " checked";                
        content += ">Sunday&nbsp&nbsp";
        
        content += "<input type = 'checkbox' name = 'Geyser_Timer2_Monday'  value = '2' ";
        if (GeyserTimeControlParameters_Day_Flags[Timer2][1]) content += " checked";        
        content += ">Monday&nbsp&nbsp";
        
        content += "<input type = 'checkbox' name = 'Geyser_Timer2_Tuesday' value = '3' ";
        if (GeyserTimeControlParameters_Day_Flags[Timer2][2]) content += " checked";
        content += ">Tuesday&nbsp&nbsp";
                
        content += "<input type = 'checkbox' name = 'Geyser_Timer2_Wednesday'  value = '4' ";
        if (GeyserTimeControlParameters_Day_Flags[Timer2][3]) content += " checked";
        content += ">Wednesday&nbsp&nbsp";        
        
        content += "<input type = 'checkbox' name = 'Geyser_Timer2_Thursday' value = '5' ";
        if (GeyserTimeControlParameters_Day_Flags[Timer2][4]) content += " checked";
        content += ">Thursday&nbsp&nbsp";        
        
        content += "<input type = 'checkbox' name = 'Geyser_Timer2_Friday'  value = '6' ";
        if (GeyserTimeControlParameters_Day_Flags[Timer2][5]) content += "checked";
        content += ">Friday&nbsp&nbsp";
                
        content += "<input type = 'checkbox' name = 'Geyser_Timer2_Saturday' value = '7' ";
        if (GeyserTimeControlParameters_Day_Flags[Timer2][6]) content += " checked";
        content += ">Saturday";        
        content += "</fieldset><br>";

        content += "<fieldset>";                      
        content += "<legend><b>Timer Operational Modes</b></legend>"; 
        content += "Timer 1 is ";
        if(TimerEnable[Timer1]) 
        {
          content += "ENABLED ";
          content += "<input type = 'radio' name = 'Geyser_Timer1_Mode' value = 'D'>Click to disable timer";  
        }
        else 
        {
          content += "DISABLED ";
          content += "<input type = 'radio' name = 'Geyser_Timer1_Mode' value = 'E'>Click to enable timer";
        }
        
        content += "<br><br>Timer 2 is ";
        if(TimerEnable[Timer2]) 
        {
          content += "ENABLED ";
          content += "<input type = 'radio' name = 'Geyser_Timer2_Mode' value = 'D'>Click to disable timer";  
        }
        else 
        {
          content += "DISABLED ";
          content += "<input type = 'radio' name = 'Geyser_Timer2_Mode' value = 'E'>Click to enable timer";          
        }
        content += "</fieldset>";        
                                 
        content += "<br> <input type = 'submit' value = 'Press to upload new settings'>";
        
        if(!AccessPointRunning) // If WiFi connection
        {
          content += "<br><br><a href='http://" + GetIPaddress() + "/Heater'> Go to Heating Settings screen  </a>";     
          content += "<br><a href='http://" + GetIPaddress() + "/DataLog'>Go to Data Logging Screen </a>"; 
          content += "<br><a href='http://" + GetIPaddress() + "/DisplaySystemSettings'>Go to System Settings Screen </a>";  
          content += "<br><a href='http://" + GetIPaddress() + "'>Go to the Main Menu </a>";          
        }
         
        else   // If AP connection
        {
          content += "<br><br><a href='http://192.168.8.200/Heater'> Go to Heating Settings Screen  </a>";     
          content += "<br><br><a href='http://192.168.8.200/DataLog'>Go to Data Logging Screen </a>"; 
          content += "<br><a href='http://192.168.8.200/DisplaySystemSettings'>Go to System Settings Screen </a>"; 
          content += "<br><a href='http://192.168.8.200/Quit'>Close Configuration Tool & Reset the Controller </a>";  
          content += "<br><a href='http://192.168.8.200'>Go to Main Menu </a>";                   
        }
        content += "</form> </html>";    
}   // end of void Timer_Page_Geyser()   // Common page layout for Output timer 1 set up
        // ----------------------------------------------------------------------- 
byte Get_Decimal_Hours_From_Web_Time_Value(String Time_Value)   // Input string = hrs:min(xx:yy) format from Access point web page input.
{                                                               // Returns decimal value. Input must be in the correct format
  byte Decimal_Value; 
  String Hours = String(Time_Value.charAt(0));
  Hours += String(Time_Value.charAt(1));
  Decimal_Value = Hours.toInt();
  return Decimal_Value;
}   // end of byte Get_Decimal_Hours_From_Web_Time_Value(String Time_Value) 
   // ----------------------------------------------------------------------------   
byte Get_Decimal_Mins_From_Web_Time_Value(String Time_Value)   // Input string = hrs:min(xx:yy) format from Access point web page input.
{                                                              // Returns decimal value.  Input must be in the correct format
  byte Decimal_Value;  
  String Minutes = String(Time_Value.charAt(3));
  Minutes += String(Time_Value.charAt(4));
  Decimal_Value = Minutes.toInt();
  return Decimal_Value;  
}   // end of  byte Get_Decimal_Mins_From_Web_Time_Value(String Time_Value) 
   // ----------------------------------------------------------------------------   
byte Convert_From_String_To_Decimal_Number(String String_Number)   // Input string number = xxx format from Access point web page input 
{                                                                  // (max value = 256). Returns decimal value
  unsigned int Decimal_Value;
  if (String_Number.length() > 3) 
  {  // String too many characters
    Decimal_Value = 0;
  }
  else 
  {
    Decimal_Value = String_Number.toInt();   // convert string to int (must be numeric string!)
    if (Decimal_Value > 255) Decimal_Value = 255;  
  }
  return byte(Decimal_Value);   
}   // end of byte Convert_From_String_To_Decimal_Number(String String_Number)   
   // ---------------------------------------------------------------------------- 
void Set_Up_Access_Point()    // set up & create an access point and web page to allow the login details
                              // to be changed - will reset the board after the changes have been entered
                              // Normal login to WiFi network using the stored details
{
  GeyserPages.stop();
  WiFi.mode(WIFI_AP_STA);
  if (EnableMessages) 
  { 
      Serial.println(""); 
      Serial.print(F("Starting up "));
      Serial.print(HostName);
      Serial.println(F(" Setup AP"));
  }
  HeartbeatTick();   // Feed the HW watchdog! 
  int n = WiFi.scanNetworks();
  st = "<ol>";
  for (int i = 0; i < n; ++i)
    {
      // add SSID and RSSI to string st for each network found
      st += "<li>";
      st += WiFi.SSID(i);
      HeartbeatTick();   // Feed the HW watchdog!
      st += " (";
      st += WiFi.RSSI(i);
      st += ")";
      st += (WiFi.encryptionType(i) == ENC_TYPE_NONE)?" ":"*";
      st += "</li>";
    }   // end of for (int i = 0; i < n; ++i)
  st += "</ol>";

    // ************* delete these 4 lines if you wish to use default IP address of 192.168.4.1
  IPAddress ip(192,168,8,200);        // Give Access point a preferred IP Address
  IPAddress gateway(192,168,8,200);   // Give Access point a preferred gateway address
  IPAddress subnet(255,255,255,0);    // Give Access point a preferred subnet address
  WiFi.softAPConfig(ip,gateway,subnet);
    // ***************************************************************************************
  ssid = HostName + " - Setup" ;  // AP server name is same as the device name
  WiFi.softAP(ssid.c_str(), passphrase.c_str());     // Create Access Point
  HeartbeatTick();   // Feed the HW watchdog!
  //Upload_Device_Configuration_Web_Page();  // Create and upload the configuration web page
  
  serverAP.begin();   // Start the access point webserver
  if (EnableMessages)  
  {
      Serial.print(F("Connect to Access Point ")); Serial.println(ssid); 
      Serial.print(F("Web page IP address ")); Serial.println(WiFi.softAPIP());
      Serial.println(F("for Device Setup")); 
  }
}   // end of void Set_Up_Access_Point(void) 
  // *****************************************************************************
void WebClock()
{
  content+= "Controller clock = ";
  content += String(ClockHour) + ":";      
  if (ClockMinute < 10) content += "0"; 
  content += String(ClockMinute) + " "; 
  if(NTP_Resolved || RTC_BoardPresent) content += dayStr(ClockWeekDay);
  if(RTC_BoardPresent) content += " - " + String(RTC_MonthDay) + "/" +  String(RTC_Month) + "/" + String(RTC_Year); 
}  // end of void WebClock
//-------------------------------------------------------------
void MainGeyserPage() // Main page for Geyser display page 
{
  content = "<!DOCTYPE HTML> <html>";  
  content += "<meta http-equiv='refresh' content='5'>";      
  content += "<form>";        
  content += "<h3>" + HostName;
  if(AccessPointRunning) content += " - (Offline)"; else content += " - (Online)"; 
  content +=" Main Display Screen</h3>";
  content += Firmware_Version + "<br>";
  
  WebClock();
       
  content += "<p></p><fieldset>";
  content += "<fieldset>";
  content += "<legend><b>Geyser Temperature</b></legend>";
  String Temp = String(AverageGeyserTemperature);
  content += "Geyser Temperature = " + Temp.substring(0,4) + "'C<br>";

  byte NoOfSensors = sensors.getDeviceCount(); 
  if(NoOfSensors > 1) 
  {
    byte i = 1;    
    while(i < NoOfSensors)
    {
      Temp = String (SkinTemp[i-1]);
      content += ("Skin ")+ String(i) + " - " + Temp.substring (0,4) + "'C";
      if(i < (NoOfSensors-1)) content += " :: ";
      i++;
    }
  }  // end of if(NoOfSensors > 1 
  content += "</fieldset>";
  
  content += "<fieldset>";
  content += "<legend><b>Geyser Status</b></legend>";
  content += "Automatic Temperature Control is ";
  if (GeyserTempControlEnabled == Enabled) content += "Enabled"; else content += "Disabled";
  content += "<br>Heater is currently ";
  if (digitalRead(GeyserControlRelay) == GeyserOff) content += "OFF"; else content += "ON";
  content += "</fieldset>";

  content += "<fieldset>";
  content += "<legend><b>Water Temperature Logs (Over last 24 hours)</b></legend>";
  content += "Maximum Temperature reached = " + String(TempMax) + "'C";
  content += "<br>Minimum Temperature reached =  " + String(TempMin) + "'C";
  content += "</fieldset>";
 
  content += "<fieldset>"; 
  content += "<legend><b>Temperature Setpoints</b></legend>";  
  
  content += "<fieldset>"; 
  content += "<legend><b> SET 1 (Default) ";  
  if(GeyserTempMode() == SetpointGroup1) content += " (ACTIVE)";
  content += "</b></legend>";     
  content += "Heater will switch ON if water temperature is below " + String(Setting1_ON_Setpoint) + "'C";
  content += "<br>Heater will switch OFF if water temperature is above  " + String(Setting1_OFF_Setpoint) + "'C";
  content += "</fieldset>";
    
  content += "<fieldset>"; 
  content += "<legend><b>SET 2 (Only when Timer 2 is running) "; 
  if(GeyserTempMode() == SetpointGroup2) content += " (ACTIVE)";
  content += "</b></legend>";     
  content += "Heater will switch ON if water temperature is below " + String(Setting2_ON_Setpoint) + "'C";
  content += "<br>Heater will switch OFF if water temperature is above  "+ String(Setting2_OFF_Setpoint) + "'C"; 
  content += "</fieldset>";
  
  content += "</fieldset>";
  
  content += "<fieldset>";
  content += "<legend><b>Timer Control</b></legend>";  
  content += "Timer 1 is ";
  if(TimerEnable[Timer1]) content += "Enabled"; else content += "Disabled";
  content += "&nbsp&nbsp&nbspTimer 2 is ";
  if(TimerEnable[Timer2]) content += "Enabled"; else content += "Disabled";         
  content += "</fieldset>";
         
  if(!AccessPointRunning) // If WiFi Connection
  {
    content += "<br><a href='http://" + GetIPaddress() + "/Heater'> Go to Heating Settings Screen  </a>";  
    content += "<br><a href='http://" + GetIPaddress() + "/Timer'>Go to Timer Settings Screen </a>";
    content += "<br><a href='http://" + GetIPaddress() + "/DisplaySystemSettings'>Go to System Settings Screen </a>";  
    content += "<br><a href='http://" + GetIPaddress() + "/DataLog'>Go to Data Logging Screen </a>";    
    content += "</form></html>";         
    GeyserPages.send(200, "text/html", content);    
  }
  else   // AP connection
  {
    content += "<br><a href='http://192.168.8.200/Heater'> Go to Heating Settings Screen  </a>";     
    content += "<br><a href='http://192.168.8.200/Timer'>Go to Timer Settings Screen </a>"; 
    content += "<br><a href='http://192.168.8.200/DisplaySystemSettings'>Go to System Settings Screen </a>"; 
    content += "<br><a href='http://192.168.8.200/Quit'>Close Configuration Tool & Reset the Controller </a>";    
    content += "</form></html>";         
    serverAP.send(200, "text/html", content);     
  }  
  content += "</fieldset>";       
  
       
} // end of void MainGeyserPage() // Main page for Geyser display page ....
// -----------------------------------------------------------------------------
void Heater() 
{
  content = "<!DOCTYPE HTML> <html>";   
  content += "<form action = 'HeaterSettingUpload'>";        
  content += "<h3>" + HostName;
  if(AccessPointRunning) content += " - (Offline)"; else content += " - (Online)";  
  content += " Heating Settings Screen</h3>";
  
  WebClock();
  
  content += "<p></p><fieldset>";
  content += "<legend><b>Normal Heater Setpoints (Except when Timer 2 is running)</b></legend>";
  content += "<i>Note that ON temperature setpoint MUST be LOWER value than OFF value</i>";
  content += "<br><br>Heater will switch ON when Temperature is below " + String(Setting1_ON_Setpoint) + "'C";
  content += "<br><label>Enter new ON value (25-70'C)</label> ";    
  content += "<input type = 'number' name='HeaterONDay' + length=3 size = '3' min='25' max = '70'>";
  
  content += "<br>Heater will switch OFF when Temperature is above " + String(Setting1_OFF_Setpoint) + "'C"; 
  content += "<br><label>Enter new OFF value (25-70'C)</label> ";    
  content += "<input type = 'number' name='HeaterOFFDay' + length=3 size = '3' min='25' max = '70'>";  
  content += "</fieldset>"; 

  content += "<fieldset>";
  content += "<legend><b>Timer 2 Heater Setpoints</b></legend>";
  content += "<i>Note that ON temperature setpoint MUST be LOWER value than OFF value</i>";
  content += "<br><br>Heater will switch ON when Temperature is below " + String(Setting2_ON_Setpoint) + "'C";
  content += "<br><label>Enter new ON value (25-70 'C)</label> ";    
  content += "<input type = 'number' name='HeaterONNight' + length=3 size = '3' min='25' max = '70'>";
  
  content += "<br>Heater will switch OFF when Temperature is above " + String(Setting2_OFF_Setpoint) + "'C"; 
  content += "<br><label>Enter new OFF value (25-70 'C)</label> ";    
  content += "<input type = 'number' name='HeaterOFFNight' + length=3 size = '3' min='25' max = '70'>"; 
  
  content += "<br><br>Thermostat control is currently ";
  if (GeyserTempControlEnabled == Enabled) 
  {
    content += "ENABLED "; 
    content += "<input type = 'radio' name = 'GeyserTempControlEnable' value = 'Disable'>Click to disable";
  }
  else 
  {
    content += "DISABLED ";
    content += "<input type = 'radio' name = 'GeyserTempControlEnable' value = 'Enable'>Click to enable";
  }
  content += "</fieldset>"; 

  content += "<fieldset>";   
  content += "<legend><b>Manual Control of Geyser Heater</b></legend>"; 
  content += "<i>NOTE - Thermostat control (see above) MUST be DISABLED before using Manual Control"; 
  content += "<br>WARNING - Heater will switch off after ";
  content += String(GeyserManualControlTimerInterval) + " minutes without Setpoint control</i>";      
  content += "<br><br><input type = 'submit' formaction = 'ManualSwitchGeyser' ";
  if(digitalRead(GeyserControlRelay) == GeyserOn) content += "value = ' Geyser is ON'>"; else content += "value = 'Geyser is OFF'>";
  content += " Click to Toggle Geyser ON & OFF<br>";      
  content += "</fieldset>"; 

  if(AccessPointRunning)   // only if AP
  {
    content += "<input type = 'submit' formaction = 'CloseAP' value = 'Close Access Point and reset Device'><br> "; 
  } 
  
  content += "<br> <input type = 'submit' value = 'Press to upload new values'>";  
  if (!AccessPointRunning)   // If Wifi connection
  {
    content += "<br><br><a href='http://" + GetIPaddress() + "/Timer'>Go to Timer Settings Screen </a>";
    content += "<br><a href='http://" + GetIPaddress() + "/DisplaySystemSettings'>Go to System Settings Screen </a>";
    content += "<br><a href='http://" + GetIPaddress() + "/DataLog'>Go to Data Logging Screen </a>";   
    content += "<br><br><a href='http://" + GetIPaddress() + "'>Go to the Main Menu </a>";  
    content += "</form></html>"; 
    GeyserPages.send(200, "text/html", content);
  }
  else   // AP connection
  {
    content += "<br><br><a href='http://192.168.8.200/Timer'>Go to Timer Settings Screen </a>";
    content += "<br><a href='http://192.168.8.200/DisplaySystemSettings'>Go to System Settings Screen </a>";
    content += "<br><a href='http://192.168.8.200/DataLog'>Go to Data Logging Screen </a>";   
    content += "<br><br><a href='http://192.168.8.200'>Go to the Main Menu </a>"; 
    content += "</form></html>";   
    serverAP.send(200, "text/html", content); 
  }     
}  // end of void Heater()  
// -----------------------------------------------------------------------------
void Timer() 
{   
    content = "<!DOCTYPE HTML> \r\n<html>";
    Timer_Page_Geyser();
    if(!AccessPointRunning) // if WiFi connection
    {
      GeyserPages.send(200, "text/html", content);

      content += "<br><br><a href='http://" + GetIPaddress() + "/Heater'> Go to Heating Settings Screen  </a>";       
      content += "<br><a href='http://" + GetIPaddress() + "/DisplaySystemSettings'>Go to System Settings Screen </a>";
      content += "<br><a href='http://" + GetIPaddress() + "/DataLog'>Go to Data Logging Screen </a>";   
      content += "<br><br><a href='http://" + GetIPaddress() + "'>Go to the Main Menu </a>";  
      
      content += "</form></html>";
      GeyserPages.send(200, "text/html", content);
    }
    else  // AP connection
    {
      serverAP.send(200, "text/html", content);
      content += "<br><br><a href='http://192.168.8.200/Heater'> Go to Heating Settings Screen  </a>";       
      content += "<br><a href='http://192.168.8.200/DisplaySystemSettings'>Go to System Settings Screen </a>";
      content += "<br><a href='http://192.168.8.200/DataLog'>Go to Data Logging Screen </a>";   
      content += "<br><br><a href='http://192.168.8.200'>Go to the Main Menu </a>"; 

      content += "</form></html>";
      serverAP.send(200, "text/html", content);
    }      
}  // end of void Timer () 
// -------------------------------------------------------------------------------------   
void UpdateLogs()    // update the 24 hour looging array for water temperature and geyser time on.
{      
        // --------------------- Update Temperature Logs --------
        //byte Pointer = 0;
        //byte NoOfPoints = 0;
        //float HourlyAverage = 0.0;
        //while(Pointer < 60)  // sum the last 60 average per minute temperature readings
        //{
          //if(MinuteTempLog[Pointer]> 0) 
          //{
            //HourlyAverage += MinuteTempLog[Pointer]; // if a valid reading in log
            //NoOfPoints ++;  // No of real data points = value used to calculate the average
          //}
          //Pointer ++;
        //}
        //if(NoOfPoints == 0) HourlyAverage = AverageGeyserTemperature;
        //else HourlyAverage = HourlyAverage / NoOfPoints;  // == average temperature (or real measurements) over last hour
        //TempLog[LogPointer] = HourlyAverage;  // store calculate hourly average temperature in hourly Temp Log array       
        TempLog[LogPointer] = AverageGeyserTemperature;  // store average temperature in hourly Temp Log array
                // ---------------------- Update Geyser ON log ---------     
        byte Temp = ClockHour;
        if (Temp == 0) Temp = 24;
        
        TimeLogHr[LogPointer] = Temp; 
        TimeLogMin[LogPointer] = ClockMinute;
        // ------------------------------------------------------------
        byte NoOfSensors = sensors.getDeviceCount();   
        byte i = 0;   // store skin temperatures
        while (i < NoOfSensors-1)
        {
          SkinLog[LogPointer][i] = SkinTemp[i];
          i++;
        }
        // ------------------------------------------------------------        
        if (EnableMessages)
        {
          Serial.print(LogPointer); Serial.print(F(": Store log values @ ")); 
          if(Temp < 10) Serial.print("0");
          Serial.print(Temp);  
          Serial.print(":"); 
          if(TimeLogMin[LogPointer] < 10) Serial.print("0"); 
          Serial.print(TimeLogMin[LogPointer]);
          Serial.print(F("   Temp stored as ")); Serial.println(TempLog[LogPointer],1); 
        }
        
        if (digitalRead(GeyserControlRelay) == GeyserOn)  // if Geyser is ON
        {         
          GeyserOnTimer += ((millis() - GeyserOnRunningTimer))/1000;  // get time in seconds that it has been turned on           
        }
        else  GeyserOnTimer = GeyserOnTimer/1000;   // geyser is OFF, convert ms to S
        
        GeyserPowerLog[LogPointer] = byte(GeyserOnTimer / 60);  // store On time minutes in log array  
        if( GeyserPowerLog[LogPointer] > 60) GeyserPowerLog[LogPointer] = 60;  // cap reading at 60 incase of overflow errors         
        GeyserOnRunningTimer = millis();   //reset On timers 
        GeyserOnTimer = 0;
        if (EnableMessages)
        {
          Serial.print(F("Geyser ON timer (mins) logged as ")); 
          Serial.println(GeyserPowerLog[LogPointer]); 
        }
        LogPointer ++;   // update log pointer
        if(LogPointer >= 24) LogPointer = 0;
}  // end of void UpdateLogs()    // update the 24 hour looging a
//---------------------------------------------------------------------------------------------------
void DataLog()   // Display 24 hour data log on web page
{
  byte LogIndex = LogPointer;
  byte NoOfSensors;
  content = "<!DOCTYPE HTML> <html>";  
  content += "<meta http-equiv='refresh' content='5'>";  
  //content += "<form action = 'ResetLog'>";        
  content += "<h3>" + HostName;
  if(AccessPointRunning) content += " - (Offline)"; else content += " - (Online)";
  content += " Data Logging Screen</h3>";
  
  WebClock();
  
  content += "<p></p><fieldset>";
  content += "<legend><b>Statistics</b></legend>";
  content += "Total time (hr:min) over the last 24 hours that Geyser has been ON = <br>";
  content += String(TotalGeyserMinutes()/60) + ":" + String(TotalGeyserMinutes() % 60) + " = " + String(float(TotalGeyserMinutes()*100.0/1440.0)) + "% of total time possible (1440 minutes)";
  content += "</fieldset>";
    
  content += "<p></p><fieldset>";
  content += "<legend><b>Last 24 hour geyser measurement record - Hourly Log (Most recent is top of the list)</b></legend>";
  content += "Time recorded - Temperature - Time ON (Mins) - ";
  NoOfSensors = sensors.getDeviceCount(); 
  if(NoOfSensors > 1)
  {
    byte SkinNo = 1;
    while (SkinNo < NoOfSensors)
    {
      content += "Skin " + String(SkinNo);
      if(SkinNo < NoOfSensors -1) content += " - ";
      SkinNo ++; 
    }
  }
  content += "<br>";
  
  if (LogIndex == 0) LogIndex = 23; else LogIndex = LogIndex -1;
  byte LoopCounter = 0;  
  while (LoopCounter < 24)
  {
    if(TimeLogHr[LogIndex] > 0)  // only display if valid data
    {
      content += "&nbsp &nbsp &nbsp";
      if(TimeLogHr[LogIndex] < 10) content += "0"; 
      content += String (TimeLogHr[LogIndex]) + ":";
      if(TimeLogMin[LogIndex] < 10) content += "0"; 
      content += String (TimeLogMin[LogIndex]) ;
      content += "&nbsp &nbsp &nbsp  &nbsp &nbsp &nbsp&nbsp&nbsp&nbsp&nbsp&nbsp&nbsp"; 
      content += String(TempLog[LogIndex]) + "&nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp"; 
      if(GeyserPowerLog[LogIndex] < 10) content += "0"; 
      content +=  String (GeyserPowerLog[LogIndex]) + "&nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp &nbsp";

      NoOfSensors = sensors.getDeviceCount(); 
      if(NoOfSensors > 1)
      {
        byte SkinNo = 1;
        while (SkinNo < NoOfSensors)
        {
          content += String (SkinLog[LogIndex][SkinNo-1]);
          if(SkinNo < NoOfSensors -1) content += "&nbsp &nbsp &nbsp";
          SkinNo ++; 
        }  // end of while (SkinNo < NoOfSensors)
        //content += "<br>";
      }  // end of if(NoOfSensors > 1) 
      content += "<br>";    
      LogIndex --;
      if (LogIndex == 255) LogIndex = 23; 
    }  // end of  if(TimeLogHr[LogIndex] > 0)  // only display if valid data            
    LoopCounter ++;       
  }  // end of while (LoopCounter < 24)
  content += "</fieldset>";
  content += "<form action = 'ResetLog'>";   
  content += "<br> <input type = 'submit' value = 'Reset the Data Log'>"; 
  content += "</form></html>";  
  content += "<form action = 'ManualUpdateLog'>";       
  content += "<br><input type = 'submit' value = 'Add New Point to Data Log'>"; 
  content += "</form></html>";  
  content += "<br><a href='http://" + GetIPaddress() + "'>Go to the Main Menu </a>";   
  GeyserPages.send(200, "text/html", content);  
}  // end of void DataLog()
// --------------------------------------------------------------------------------
void ManualUpdateLog()
{
  UpdateLogs();
  content = "<!DOCTYPE HTML>\r\n<html><form>";
  content += "<br>A New Data Point has been added to the Data Log<br>";

    if (!AccessPointRunning)  // If WiFi Connecton
    {
      content += "<br><a href='http://" + GetIPaddress() + "/Heater'> Go to Heating Settings Screen  </a>";  
      content += "<br><br><a href='http://" + GetIPaddress() + "/Timer'>Go to Timer Settings Screen </a>";
      content += "<br><br><a href='http://" + GetIPaddress() + "/DisplaySystemSettings'>Go to System Settings Screen </a>";
      content += "<br><br><a href='http://" + GetIPaddress() + "/DataLog'>Go to Data Logging Screen </a>";   
      content += "<br><br><a href='http://" + GetIPaddress() + "'>Go to the Main Menu </a>";  
      //content += "</form></html>";
      GeyserPages.send(200, "text/html", content); 
    }
    else    // AP connection
    {
      content += "<br><br><a href='http://192.168.8.200/Heater'> Go to Heating Settings Screen  </a>";  
      content += "<br><br><a href='http://192.168.8.200/Timer'>Go to Timer Settings Screen </a>";
      content += "<br><br><a href='http://192.168.8.200/DisplaySystemSettings'>Go to System Settings Screen </a>";
      content += "<br><br><a href='http://192.168.8.200/DataLog'>Go to Data Logging Screen </a>";   
      content += "<br><br><a href='http://192.168.8.200'>Go to Main Menu </a>"; 
      //content += "</form></html>";
      serverAP.send(200, "text/html", content); 
    }
}  // end of void ManualUpdateLog()
// --------------------------------------------------------------------------------
void ResetLog()
{
    byte i = 0;
    while (i < 24)  // set all members of log array to zero
    {
       GeyserPowerLog[i] = 0;  
       TempLog[i] = 0;   
       TimeLogHr[i] = 0;
       TimeLogMin[i] = 0;  
       LogPointer = 0;
       i++;    
    }
    //UpdateMinuteLog(true);  // reset the Hourly average variables
    content = "<!DOCTYPE HTML> \r\n<html>";     
    content += "All logged data has been reset ...";

    content += "<br><br><a href='http://" + GetIPaddress() + "/DisplaySystemSettings'>Go to System Settings Screen </a>";     
    content += "<br><a href='http://" + GetIPaddress() + "/Timer'>Go to Timer Settings Screen </a>";
    content += "<br><a href='http://" + GetIPaddress() + "/DisplaySystemSettings'>Go to System Settings Screen </a>";
    content += "<br><a href='http://" + GetIPaddress() + "/DataLog'>Go to Data Logging Screen </a>";       
    content += "<br><a href='http://" + GetIPaddress() + "'>Go to the Main Menu </a>";   
                
    GeyserPages.send(200, "text/html", content); 
}
// ----------------------------------------------------------------------------------------
String GetIPaddress()  // get IP address of Controller from Wifi setup.  Used in Web page hyperlink construction
{
  IPAddress ip = WiFi.localIP();
  String ipStr = String(ip[0]) + '.' + String(ip[1]) + '.' + String(ip[2]) + '.' + String(ip[3]);
  return ipStr;
}
// ----------------------------------------------------------------------------------------
void StartAP()
{
    content = "<!DOCTYPE HTML> \r\n<html>";     
    content += "The Offline Configuration Tool will now start.";  
    content += "<br><br>Note that this Wifi connection will now be broken";
    content += "<br>Connect your mobile device or computer to the"; 
    content += "<br>Controller Setup network SSID and";
    content += "<br>then connect a web browser to http://192.168.8.200";
    content += "<br><br>Note that the controller will reset on quitting the Configuration Tool";
    GeyserPages.send(200, "text/html", content);     
    Soft_AP_Switch  = true;   // Start AP
}
// ------------ RTC FUNCTIONS -----------------------------
// Convert normal decimal numbers to binary coded decimal */
byte decToBcd(byte val)
{
  return ( (val/10*16) + (val%10) );
}
// --------------------------------------------------------
// Convert binary coded decimal to normal decimal numbers
byte bcdToDec(byte val)
{
  return ( (val/16*10) + (val%16) );
}
// --------------------------------------------------------
void GetTimeDateFromRTC(boolean PrintValues)        // Gets the time from the DS3231
{
  Wire.beginTransmission(DS3231_I2C_ADDRESS);
  Wire.write(zero);             // Reset the register pointer
  Wire.endTransmission();
  Wire.requestFrom(DS3231_I2C_ADDRESS, 7);
  RTC_Second = bcdToDec(Wire.read() & 0x7f);
  RTC_Minute = bcdToDec(Wire.read());
  RTC_Hour   = bcdToDec(Wire.read() & 0x3f);  
  RTC_Day  = bcdToDec(Wire.read());
  RTC_MonthDay = bcdToDec(Wire.read());
  RTC_Month = bcdToDec(Wire.read());
  RTC_Year  = bcdToDec(Wire.read())+2000;
  
  ClockMinute = RTC_Minute;  // load RTC to soft clock
  ClockHour = RTC_Hour;
  ClockWeekDay = RTC_Day;
  
  if (EnableMessages == Enabled and PrintValues == true)
  {
    Serial.print(F("RTC = "));       
    Serial.print(RTC_Hour); // print the hour
    Serial.print(':');
      
    if ( RTC_Minute < 10 ) Serial.print('0');
    Serial.print(RTC_Minute); // print the minute 
    Serial.print(':');     
    if ( RTC_Second < 10 ) Serial.print('0');
    Serial.println(RTC_Second); // print the second
      
    Serial.print(F("RTC Date is "));
    Serial.print(dayStr(RTC_Day));
    Serial.print("-");
    Serial.print(RTC_MonthDay); // print the Month day
    Serial.print("-");
    Serial.print(RTC_Month); // print the Month
    Serial.print("-");  
    Serial.println(RTC_Year); // print the Year
  } 
} // end of void GetTimeDateFromRTC(boolean PrintValues)        // Gets the time from the DS3231
// -----------------------------------------------------
void UpdateRTC(boolean PrintValues)    // Update the real time clock using NTP data
{
    // print the hour, minute and second:
      if(PrintValues == true && EnableMessages == Enabled)
      {
        Serial.print(F("NTP Time is "));       // UTC is the time at Greenwich Meridian (GMT)      
        Serial.print(hour()); // print the hour
        Serial.print(':');
      
        if ( minute() < 10 ) Serial.print('0');
        Serial.print(minute()); // print the minute 
        Serial.print(':');     
        if ( (second()) < 10 ) Serial.print('0');
        Serial.println(second()); // print the second
      
        Serial.print(F("NTP Date is "));
        Serial.print(day()); // print the Month day
        Serial.print("-");
        Serial.print(month()); // print the Month
        Serial.print("-");  
        Serial.println(year()); // print the Year
      }
                                                // update the RTC
      Wire.beginTransmission(DS3231_I2C_ADDRESS);   
      Wire.write(zero);
      Wire.write(decToBcd(second()) & 0x7f);    // 0 to bit 7 starts the clock
      Wire.write(decToBcd(minute()));
      Wire.write(decToBcd(hour())); 
      Wire.write(decToBcd(weekday()));     // day of week
      Wire.write(decToBcd(day()));
      Wire.write(decToBcd(month()));
      Wire.write(decToBcd(year()-2000));     
      Wire.endTransmission(); 
      
      ClockMinute = minute();  // Update Soft Clock
      ClockHour = hour();
      ClockWeekDay = weekday();
      RTC_Year = year();
      RTC_Month = month();
      RTC_MonthDay = day();  
      LongClockMinute = ClockMinute + (ClockHour * 60);   
}   // end of void UpdateRTC(boolean PrintValues)    // Update the real time clock using NTP data
// ------------------------------------------------------
boolean CheckFor_RTC_Fitted()  // returns true if RTC fitted to circuit
{
    GetTimeDateFromRTC(false);
    boolean Fitted = true;
    if(RTC_Minute > 59) Fitted = false;
    else if(RTC_Hour > 23) Fitted = false;  
       else if(RTC_Day == 0 || RTC_Day > 7)   Fitted = false;
           else if(RTC_MonthDay > 31) Fitted = false;
              else if(RTC_Month > 12) Fitted = false;
    return Fitted;    
}
// ------------------------------------------------------------------------------------
void PrintTimerParameters()
{
  Serial.println("Timer 1"); 
  Serial.print(F("TimerEnable[Timer1]  = ")); Serial.println(TimerEnable[Timer1]);
  Serial.print(F("ClockWeekDay  = ")); Serial.println(ClockWeekDay);
  Serial.print(F("TimerActive[Timer1]  = ")); Serial.println(TimerActive[Timer1]);
  Serial.print(F("GeyserTimeControlParameters_Day_Flags[Timer1][ClockWeekDay-1]  = ")); Serial.println(GeyserTimeControlParameters_Day_Flags[Timer1][ClockWeekDay-1]);
  Serial.print(F("LongClockMinute  = ")); Serial.println(LongClockMinute);
  Serial.print(F("LongTimerON[Timer1]  = ")); Serial.println(LongTimerON[Timer1]);
  Serial.print(F("LongClockMinute  = ")); Serial.println(LongClockMinute);
  Serial.print(F("LongTimerOFF[Timer1]  = ")); Serial.println(LongTimerOFF[Timer1]);
  Serial.print(F("GeyserTempControlEnabled  = ")); Serial.println(GeyserTempControlEnabled);
  Serial.println("Timer 2  = "); 
  Serial.print(F("TimerEnable[Timer2]  = ")); Serial.println(TimerEnable[Timer2]);
  Serial.print(F("TimerActive[Timer2]  = ")); Serial.println(TimerActive[Timer2]);
  Serial.print(F("GeyserTimeControlParameters_Day_Flags[Timer2[ClockWeekDay-1]  = ")); Serial.println(GeyserTimeControlParameters_Day_Flags[Timer2][ClockWeekDay-1]);
  Serial.print(F("LongTimerON[Timer2]  = ")); Serial.println(LongTimerON[Timer2]);
  Serial.print(F("LongTimerOFF[Timer2]  = ")); Serial.println(LongTimerOFF[Timer2]);

  Serial.print(F("GeyserManualControlTimer  = ")); Serial.println(GeyserManualControlTimer);
  Serial.print(F("Self Check  = ")); Serial.println(SelfCheck);
  Serial.println(""); 
}
// ------------------------------------------------------------------------------------
int TotalGeyserMinutes()   // returns the number of minutes that the Geyser has been ON
{                        // over the 24 hour logging array
  byte pointer = 0;
  int TotalMinutes = 0;
  while (pointer < 24)
  {
    TotalMinutes += GeyserPowerLog[pointer];
    pointer ++;
  }
  return TotalMinutes;  
}
// ------------------------------------------------------------------------
void Diagnostics()      // Timer web server diagnostic screen
{
  content = "<!DOCTYPE HTML> \r\n<html>";  
  content += "<meta http-equiv='refresh' content='2'><br><br>";     
  
  WebClock();
  
  content += "<br><br>TIMER DIAGNOSTICS.";  
  content += "<br><br><b>TIMER 1</b>"; 
  content += "<br>TimerEnable[Timer1]  = " + String(TimerEnable[Timer1]);
  content += "<br>ClockWeekDay  = " + String(ClockWeekDay);
  content += "<br>TimerActive[Timer1]  = " + String(TimerActive[Timer1]);
  content += "<br>GeyserTimeControlParameters_Day_Flags[Timer1][ClockWeekDay-1]  = " + String(GeyserTimeControlParameters_Day_Flags[Timer1][ClockWeekDay-1]);
  content += "<br>LongClockMinute  = " + String(LongClockMinute);
  content += "<br>LongTimerON[Timer1]  = " + String(LongTimerON[Timer1]);
  content += "<br>LongTimerOFF[Timer1  = " + String(LongTimerOFF[Timer1]);

  content += "<br><br><b>TIMER 2</b>"; 
  content += "<br>TimerEnable[Timer2]  = " + String(TimerEnable[Timer2]);
  content += "<br>ClockWeekDay  = " + String(ClockWeekDay);  
  content += "<br>TimerActive[Timer2]  = " + String(TimerActive[Timer2]);
  content += "<br>GeyserTimeControlParameters_Day_Flags[Timer2[ClockWeekDay-1]  = " + String(GeyserTimeControlParameters_Day_Flags[Timer2][ClockWeekDay-1]);
  content += "<br>LongTimerON[Timer2]  = " + String(LongTimerON[Timer2]);
  content += "<br>LongTimerOFF[Timer2]  = " + String(LongTimerOFF[Timer2]);

  content += "<br><br>LongClockMinute  = " + String(LongClockMinute);
  content += "<br>GeyserTempControlEnabled  = " + String(GeyserTempControlEnabled);
  content += "<br>GeyserManualControlTimer(min)  = " + String(GeyserManualControlTimer);
  content += "<br>SelfCheck  = " + String(SelfCheck);

  content += "<br><br><a href='http://" + GetIPaddress() + "/DisplaySystemSettings'>Go to System Settings Screen </a>";     
  content += "<br><a href='http://" + GetIPaddress() + "/Timer'>Go to Timer Settings Screen </a>";
  content += "<br><a href='http://" + GetIPaddress() + "/DisplaySystemSettings'>Go to System Settings Screen </a>";
  content += "<br><a href='http://" + GetIPaddress() + "/DataLog'>Go to Data Logging Screen </a>";       
  content += "<br><br><a href='http://" + GetIPaddress() + "'>Go to the Main Menu </a>";  
  
  GeyserPages.send(200, "text/html", content);   
}
// -----------------------------------------------------------------
void printAddress(DeviceAddress deviceAddress)
{
  for (uint8_t i = 0; i < 8; i++)
  {
    Serial.print("0x");
    if (deviceAddress[i] < 0x10) Serial.print("0");
    Serial.print(deviceAddress[i], HEX);
    if (i < 7) Serial.print(", ");
  }
  Serial.println("");
}
// ------------------------------------------------------------------------------
void SaveTemperatureSensorAddresses()
{
    String qSensorIndex1;
    String qSensorIndex2;
    String qSensorIndex3;
    String qSensorIndex4;
    String qSensorIndex5;    
    if (!AccessPointRunning)   // If WiFi connection
    {   
          qSensorIndex1 = GeyserPages.arg("SensorIndex1");   
          qSensorIndex2 = GeyserPages.arg("SensorIndex2");  
          qSensorIndex3 = GeyserPages.arg("SensorIndex3");
          qSensorIndex4 = GeyserPages.arg("SensorIndex4");  
          qSensorIndex5 = GeyserPages.arg("SensorIndex5");             
    }
    else    // AP connection
    {
          qSensorIndex1 = serverAP.arg("SensorIndex1");   
          qSensorIndex2 = serverAP.arg("SensorIndex2");  
          qSensorIndex3 = serverAP.arg("SensorIndex3");
          qSensorIndex4 = serverAP.arg("SensorIndex4");  
          qSensorIndex5 = serverAP.arg("SensorIndex5");   
    }     
      boolean Flash = false;
      if (qSensorIndex1.length() > 0)  // if data has been inputted to field
      {
       TempSensorIndex[0] = qSensorIndex1.toInt();
       EEPROM.write(EEPROM_TempSensorIndex,TempSensorIndex[0]);
       Flash = Enabled;          
      }
      if (qSensorIndex2.length() > 0)  // if data has been inputted to field
      {
       TempSensorIndex[1] = qSensorIndex2.toInt();
       EEPROM.write(EEPROM_TempSensorIndex+1,TempSensorIndex[1]);
       Flash = Enabled;          
      }
      if (qSensorIndex3.length() > 0)  // if data has been inputted to field
      {
       TempSensorIndex[2] = qSensorIndex3.toInt();
       EEPROM.write(EEPROM_TempSensorIndex+2,TempSensorIndex[2]);
       Flash = Enabled;          
      }
      if (qSensorIndex4.length() > 0)  // if data has been inputted to field
      {
       TempSensorIndex[3] = qSensorIndex4.toInt();
       EEPROM.write(EEPROM_TempSensorIndex+3,TempSensorIndex[3]);
       Flash = Enabled;          
      }
      if (qSensorIndex5.length() > 0)  // if data has been inputted to field
      {
       TempSensorIndex[4] = qSensorIndex5.toInt();
       EEPROM.write(EEPROM_TempSensorIndex+4,TempSensorIndex[4]);
       Flash = Enabled;          
      }
      if(Flash) EEPROM.commit();  // lock in EEPROM changes if any made
  
    content = "<!DOCTYPE HTML> \r\n<html>";     
    content += "The Temperature Sensor index numbers have been saved to EEPROM and working memory<br>"; 
  
    if(!AccessPointRunning) // If WiFi Connection
    {
      content += "<br><br><a href='http://" + GetIPaddress() + "/Heater'> Go to Heating Settings Screen  </a>";  
      content += "<br><br><a href='http://" + GetIPaddress() + "/Timer'>Go to Timer Settings Screen </a>";
      content += "<br><br><a href='http://" + GetIPaddress() + "/DisplaySystemSettings'>Go to System Settings Screen </a>";  
      content += "<br><br><a href='http://" + GetIPaddress() + "/DataLog'>Go to Data Logging Screen </a>";   
      content += "<br><br><a href='http://" + GetIPaddress() + "'>Go to Main Screen </a>";         
      content += "</form></html>";         
      GeyserPages.send(200, "text/html", content);    
    }
    else   // AP connection
    {
      content += "<br><br><a href='http://192.168.8.200/Heater'> Go to Heating Settings Screen  </a>";     
      content += "<br><br><a href='http://192.168.8.200/Timer'>Go to Timer Settings Screen </a>"; 
      content += "<br><br><a href='http://192.168.8.200/DisplaySystemSettings'>Go to System Settings Screen </a>"; 
      content += "<br><br><a href='http://192.168.8.200/Quit'>Close Configuration Tool & Reset the Controller </a>";
      content += "<br><br><a href='http://192.168.8.200'>Go to Main Screen </a>";           
      content += "</form></html>";         
      serverAP.send(200, "text/html", content);     
    }     
}   // end of void SaveTemperatureSensorAddresses()
// --------------------------------------------------------------
String NibbleTo_BCD(byte Nibble)   // Nibble = 0 to 15
{
    String BCD = "";
    if(Nibble == 10) BCD = "A";
    else if (Nibble == 11) BCD = "B";
    else if (Nibble == 12) BCD = "C";
    else if (Nibble == 13) BCD = "D";
    else if (Nibble == 14) BCD = "E";
    else if (Nibble == 15) BCD = "F";
    else BCD = String(Nibble); 
    return BCD;
}
//---------------------------------------------------------------------
String BinToBCD(byte val)
{
    String Byte = "0x" + NibbleTo_BCD(val/16) + NibbleTo_BCD(val%16);
    return Byte;
}
// ------------------------------------------------------------------------
boolean ShutdownCheck()  // check if water temp too hot and shut down heater if necessary
{
    if (AverageGeyserTemperature >= ShutdownTemperature)
    {
      if(OLED)
      {
        oled.clear();
        oled.set2X();
        oled.println(" WARNING");
        oled.println("HIGH TEMP");
        oled.set1X();
        oled.println("");
        oled.println("   Heater disabled");
        ActivateGeyser(GeyserOff);
        oled.setCursor(7 * oled.fontWidth(), 7);
        oled.print(AverageGeyserTemperature); oled.print("'C");
      }
      else
      Serial.println("WARNING - HIGH GEYSER HEATER ALARM");
      return true;
    }  // end of  if (Reading >= ShutdownTemperature)
    else return false;
}
// --------------------------------------------------------------------------------------
void SaveDisplayTimerOnOffSettings()
{
    String qDisplayOn;
    String qDisplayOff;
    
    if(AccessPointRunning)
    {  
      qDisplayOn = serverAP.arg("DisplayOn");
      qDisplayOff = serverAP.arg("DisplayOff");  
    }
    else
    { 
      qDisplayOn = GeyserPages.arg("DisplayOn");
      qDisplayOff = GeyserPages.arg("DisplayOff");           
    }
    boolean EEPROM_Update = false;  

    if(qDisplayOn.length() > 0)  //a value has been entered into parameter field
    {
      byte temp = qDisplayOn.toInt();
      DisplayOnTime = temp;
      EEPROM.write(EEPROM_DisplayOnTime,DisplayOnTime);
      EEPROM_Update = true;
    }
    if(qDisplayOff.length() > 0)  //a value has been entered into parameter field
    {
      byte temp = qDisplayOff.toInt();
      DisplayOffTime = temp;
      EEPROM.write(EEPROM_DisplayOffTime,DisplayOffTime);
      EEPROM_Update = true; 
    }
    if(EEPROM_Update) EEPROM.commit();   // lock in EEPROM 

    content = "<!DOCTYPE HTML> \r\n<html>";     
    content += "The Display On/Off times have been saved to EEPROM<br>"; 
  
    if(!AccessPointRunning) // If WiFi Connection
    {
      content += "<br><br><a href='http://" + GetIPaddress() + "/Heater'> Go to Heating Settings Screen  </a>";  
      content += "<br><br><a href='http://" + GetIPaddress() + "/Timer'>Go to Timer Settings Screen </a>";
      content += "<br><br><a href='http://" + GetIPaddress() + "/DisplaySystemSettings'>Go to System Settings Screen </a>";  
      content += "<br><br><a href='http://" + GetIPaddress() + "/DataLog'>Go to Data Logging Screen </a>";   
      content += "<br><br><a href='http://" + GetIPaddress() + "'>Go to Main Screen </a>";         
      content += "</form></html>";         
      GeyserPages.send(200, "text/html", content);    
    }
    else   // AP connection
    {
      content += "<br><br><a href='http://192.168.8.200/Heater'> Go to Heating Settings Screen  </a>";     
      content += "<br><br><a href='http://192.168.8.200/Timer'>Go to Timer Settings Screen </a>"; 
      content += "<br><br><a href='http://192.168.8.200/DisplaySystemSettings'>Go to System Settings Screen </a>"; 
      content += "<br><br><a href='http://192.168.8.200/Quit'>Close Configuration Tool & Reset the Controller </a>";
      content += "<br><br><a href='http://192.168.8.200'>Go to Main Screen </a>";           
      content += "</form></html>";         
      serverAP.send(200, "text/html", content);     
    }        
}
// --------------------------------------------------------------------
