/**
   Webserver for monitoring ESP-Now based sensor stations passed to this server
   by a ESP-Now gateway over hardwired serial.
   
   Author: Marcell Stoer
   December 2018
   
*/

// Load WIFI Library
#include <WiFi.h>
// Load NTP Time server library
#include <ezTime.h>
// Load SD Card Libraries
#include "FS.h"
#include "SD.h"
#include "SPI.h"
//Load other libraries
#include <HardwareSerial.h>
#include <string.h>
// Load these two last, for function and variable definitions
#include "MainStationESP32.h"
#include "SensorStationMAC.h"

HardwareSerial Ser2(2); // use 2nd hardware serial port for communication with the ESPNow gateway

// sensor data structure, same format as used on sensor stations
struct SENSOR_DATA sensorData;

// Set web server port number to 80
WiFiServer server(80);

// set up timezone object for ezTime
Timezone VILand;  // sets up time zone object
String LastSensorTime;

long wifistartsecs; // save the start time of wifi connection

void setup() {

  char strbuf[30];

  Serial.begin(115200); Serial.println();
  Ser2.begin(sgBAUD_RATE);

  // Connect to Wi-Fi network with SSID and password
  Serial.print("Connecting to ");
  Serial.println(ssid);
  WiFi.mode (WIFI_STA);
  WiFi.begin(ssid, password);
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
  }
  wifistartsecs = millis() / 1000.0; //record wifi start time

  // set up SD card
  if (!SD.begin()) {
    Serial.println("Card Mount Failed");
    return;
  }
  uint8_t cardType = SD.cardType();

  if (cardType == CARD_NONE) {
    Serial.println("No SD card attached");
    return;
  }

  // create data files and write header info in each

  for ( int i = 0; i < MAXSTNS; i++ ) {
    String dfilepath =  "/" + dfilenames[i];
    if ( !SD.exists( dfilepath.c_str() ) ) { // don't overwrite existing files in case of power cycle/restart
      writeFile(SD, dfilepath.c_str(), sensorstn[i].name); // will create file, overwriting existing name
      appendFile(SD, dfilepath.c_str(), "\nDate-Time\t Temperature\t Humidity\t Pressure\t Light\n");
    }
  }

  // Print local IP address and start web server
  Serial.println("");
  Serial.println("WiFi connected.");
  Serial.print("IP address: ");  Serial.println(WiFi.localIP());
  server.begin();

  // ezTime begin with default parameters and sync
  waitForSync();
  Serial.println("UTC: " + UTC.dateTime());
  VILand.setLocation("America/Vancouver"); // set local time zone
  Serial.println("Local Time: " + VILand.dateTime("Y-m-d H:i:s"));
}

//
// Main loop, check webserver for requests and check for ESP Now messages
//
int heartBeat;

void loop() {
  WiFiClient client = server.available();   // Listen for incoming clients

  if (millis() - heartBeat > 30000) {
    Serial.println("Waiting for ESP-NOW messages from Serial Gateway ...");
    heartBeat = millis();
    checkwifi(); // checks wifi connection and reconnects if not working
    wifiuptimeprint();
  }

  // check for web client
  if (client) {
    handlewebclient( client );
    // Close the connection
    client.stop();
    Serial.println("Client disconnected.");
    Serial.println("");
  }

  // check for ESP-NOW message data
  while (Ser2.available()) {
    if ( Ser2.read() == '$' ) {
      while ( ! Ser2.available() ) {
        delay(1);
      }
      if ( Ser2.read() == '$' ) {
        readSerial();
      }
    }
  }

  // record server uptime
  stationuptimecheck();
}

//
// server uptime check
//
void stationuptimecheck() {
  long secsup = millis() / 1000.0;
  stationuptime.secs = secsup % 60;
  stationuptime.mins = (secsup / 60) % 60;
  stationuptime.hrs = (secsup / (60 * 60)) % 24;
  stationuptime.days = (secsup / (60 * 60 * 24));
}

//
// wifi uptime check
//
void wifiuptimeprint() {
  char strbuf[128]; 
  long secsup = millis() / 1000.0 - wifistartsecs;
  wifiuptime.secs = secsup % 60;
  wifiuptime.mins = (secsup / 60) % 60;
  wifiuptime.hrs = (secsup / (60 * 60)) % 24;
  wifiuptime.days = (secsup / (60 * 60 * 24));
  sprintf(strbuf, "WiFi Uptime: %3d Days %2d hours %2d minutes %2d seconds", stationuptime.days, stationuptime.hrs, stationuptime.mins, stationuptime.secs);
  Serial.println(strbuf);
}

//
// Checks WiFi connection and reconnects if necessary
//

void checkwifi() {
  if (WiFi.status() != WL_CONNECTED) {
    WiFi.mode (WIFI_STA);
    WiFi.begin(ssid, password);  // need to check ESP32 library if there is a better way
    while (WiFi.status() != WL_CONNECTED) {
      delay(500);
      Serial.print(".");
    }
    wifistartsecs = millis()/1000.0; //record the new wifi start time
  }
}

//
// read serial data from hard wired link to ESPNow gateway
//
void readSerial() {
  uint8_t mac[6];

  while (Ser2.available() < 6) {
    delay(1);
  }
  mac[0] = Ser2.read();
  mac[1] = Ser2.read();
  mac[2] = Ser2.read();
  mac[3] = Ser2.read();
  mac[4] = Ser2.read();
  mac[5] = Ser2.read();

  while (Ser2.available() < 1) {
    delay(1);
  }
  byte len =  Ser2.read();

  while (Ser2.available() < len) {
    delay(1);
  }
  Ser2.readBytes((char*)&sensorData, len);

  LastSensorTime = VILand.dateTime("Y-m-d H:i:s");
  Serial.println("Last Sensor Data Received: " + LastSensorTime);
  printMACaddr(mac);
  findLocation( mac );
}

//
// prints sensor date to serial (monitor)
//
void printSensorData() {
  Serial.print("Temperature is: ");  Serial.println(sensorData.temp);
  Serial.print("Humidity is: ");  Serial.println(sensorData.humidity);
  Serial.print("Pressure is: ");  Serial.println(sensorData.pressure);
  Serial.print("Light Lvl is: ");  Serial.println(sensorData.lightlvl);
}

//
// used to compare two MAC addresses
// returns 1 if equal
//
int isEqual(uint8_t* addr1, uint8_t* addr2)
{
  //   return(memcmp(addr1, addr2, sizeof(addr1)) == 0); // for some reason sizeof is returning 4
  return (memcmp(addr1, addr2, 6) == 0);
}

//
// Find Location of sensor station based on its MAC address
//
int findLocation( uint8_t* addr) {
  for ( int i = 0; i < MAXSTNS; i++ ) {
    if (isEqual(addr, sensorstn[i].mac)) {
      sensorstn[i].sdata.temp = sensorData.temp;
      sensorstn[i].sdata.humidity = sensorData.humidity + hsenscor[i]; // add correction factor
      sensorstn[i].sdata.pressure = sensorData.pressure;
      sensorstn[i].sdata.lightlvl = sensorData.lightlvl;
      strcpy(sensorstn[i].sensdtime, LastSensorTime.c_str() );
      Serial.print("Sensor Station is :"); Serial.println(sensorstn[i].name);
      // write data to file
      writesensordata(i);
      return 1;
    }
  }
}

//
// writes sensor data to file
//
void writesensordata(int i ) {
  char dataString[256] = {0};

  String dfilepath = "/" + dfilenames[i];
  File fh = SD.open(dfilepath.c_str());
  sprintf(dataString, "%20s\t%7.3f\t%7.3f\t%9.3f\t%7.3f\n", LastSensorTime.c_str(), sensorstn[i].sdata.temp, sensorstn[i].sdata.humidity, sensorstn[i].sdata.pressure, sensorstn[i].sdata.lightlvl);
  appendFile(SD, dfilepath.c_str(), dataString);
}

//
// prints MAC address out in a Pretty way with leading zeroes
//
void printMACaddr(uint8_t* mac) {
  char dataString[50] = {0};
  sprintf(dataString, "%02X:%02X:%02X:%02X:%02X:%02X", mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
  Serial.print("Sensor data received from MAC address: "); Serial.println(dataString);
}

//
// handles web client requests
//
void handlewebclient(WiFiClient client) {
  bool filesent = false;

  Serial.println("New Client");           // Print New Client found to serial monitor
  while (client.connected()) {            // loop while the client's connected
    if (client.available()) {             // if there's bytes to read from the client,
      String req = client.readStringUntil('\r'); // check request
      Serial.println(req);
      // check if a data file is requested and then send it
      for ( int i = 0; i < MAXSTNS; i++ ) {
        if (req.indexOf(dfilenames[i]) != -1) {
          Serial.println("Sending file as requested");
          sendwebfile(client, i); // need and index to which file is being sent if more than 1 file
          filesent = true;
          break; // break out of for loop once file is sent
        }
      }
      if (!filesent) { // not asking for a file, so send webpage
        sendwebpage(client);
      }
      break; // all done, exit loop
    }
  }
}

//
// prints HTML table of sensor data
//
void printwebtable(WiFiClient client) {
  char tablerow[256];

  client.println("<table style=\"width:100%\">");
  client.println("<tr> <th>Location</th><th>Temperature (C)</th><th>Humidity (%RH)</th><th>Pressure (mbar)</th><th>Light</th><th>Date & Time</th></tr>");
  for ( int i = 0; i < MAXSTNS; i++ ) {
    createtablerow(tablerow, i);
    client.println(tablerow);
    memset(tablerow, 0, sizeof(tablerow)); // clear string after use
  }
  client.println("</table>");
}

//
// creates an HTML table row
//
void createtablerow(char *tablerow, int rn) {
  char strbuf[16];

  strcpy(tablerow, "<tr> <td>"); // begin of row and first element
  strcat(tablerow, sensorstn[rn].name);
  strcat(tablerow, "</td><td>"); // start 2nd element
  if (sensorstn[rn].sdata.temp < -99 ) { // get rid of -ve values indicating no data received
    strcat(tablerow, "-");
  } else {
    dtostrf(sensorstn[rn].sdata.temp, 5, 1, strbuf);
    strcat(tablerow, strbuf);
  }
  strcat(tablerow, "</td><td>"); // start 3rd element
  if (sensorstn[rn].sdata.humidity < -99 ) { // get rid of -ve values indicating no data received
    strcat(tablerow, "-");
  } else {
    dtostrf(sensorstn[rn].sdata.humidity, 5, 1, strbuf);
    strcat(tablerow, strbuf);
  }
  strcat(tablerow, "</td><td>"); // start 4th element
  if (sensorstn[rn].sdata.pressure < -99 ) { // get rid of -ve values indicating no data received
    strcat(tablerow, "-");
  } else {
    dtostrf(sensorstn[rn].sdata.pressure, 7, 1, strbuf);
    strcat(tablerow, strbuf);
  }
  strcat(tablerow, "</td><td>"); // start 5th element
  if (sensorstn[rn].sdata.lightlvl < -99 ) { // get rid of -ve values indicating no data received
    strcat(tablerow, "-");
  } else {
    dtostrf(sensorstn[rn].sdata.lightlvl, 5, 1, strbuf);
    strcat(tablerow, strbuf);
  }
  strcat(tablerow, "</td><td>"); // start 6th element, the date & time
  strcat(tablerow, sensorstn[rn].sensdtime);
  strcat(tablerow, "</td></tr>"); // end of row
}

//
// Sends formatted web page to client
//
void sendwebpage(WiFiClient client) {
  char strbuf[256];

  // HTTP headers always start with a response code (e.g. HTTP/1.1 200 OK)
  // and a content-type so the client knows what's coming, then a blank line:
  client.println("HTTP/1.1 200 OK");
  client.println("Content-type:text/html");
  client.println("Connection: close");
  client.println();

  // Display the HTML web page
  client.println("<!DOCTYPE html><html>");
  client.println("<head><meta name=\"viewport\" content=\"width=device-width, initial-scale=1\">");
  client.println("<link rel=\"icon\" href=\"data:,\">");
  // Some style settings
  client.println("<style>html { font-family: Helvetica; display: inline-block; margin: 0px auto; text-align: center;}");
  client.println("</style></head>");

  // Web Page Heading
  client.println("<body><h1>ESP32 Local Sensor Monitor</h1>");

  printwebtable(client); // prints the sensor data in tabular format

  client.println("<br><br>");
  client.println("Current Time: " + VILand.dateTime("Y-m-d H:i:s") );
  sprintf(strbuf, "%3d Days %2d hours %2d minutes %2d seconds", stationuptime.days, stationuptime.hrs, stationuptime.mins, stationuptime.secs);
  client.println("<br>Station Uptime: ");
  client.println(strbuf);
  //client.println("<br>");
  // print out wifi uptime as well 
  sprintf(strbuf, "%3d Days %2d hours %2d minutes %2d seconds", wifiuptime.days, wifiuptime.hrs, wifiuptime.mins, wifiuptime.secs);
  client.println("<br>WiFi Uptime: ");
  client.println(strbuf);
  client.println("<br>");

  // The HTTP response ends with another blank line
  client.println();

  client.println("<br><br>Data from each sensor can be downloaded below.<br>");
  // links to file downloads
  for ( int i = 0; i < MAXSTNS; i++ ) {
    strcpy(strbuf, "<a href=\"/");
    strcat(strbuf, dfilenames[i].c_str());
    strcat(strbuf, "\" download> ");
    strcat(strbuf, sensorstn[i].name);
    strcat(strbuf, "</a><br><br>"); // add extra line for space between download file names
    client.println(strbuf);
  }
  client.println("---<br>");

  // end/close web page
  client.println("</body></html>");

  // The HTTP response ends with another blank line
  client.println();
}

//
// Sends requests data file to client
//
void sendwebfile(WiFiClient client, int i) {

  client.write("HTTP/1.0 200 OK\r\nContent-Type: text/html\r\n\r\n");
  String dfilepath =  "/" + dfilenames[i];
  File fh = SD.open(dfilepath.c_str());

  if (fh) {
    byte clientBuf[64];
    int clientCount = 0;

    while (fh.available()) {
      clientBuf[clientCount] = fh.read();
      clientCount++;

      if (clientCount > 63) {
        Serial.println("Packet");
        client.write(clientBuf, 64);
        clientCount = 0;
      }
    }

    if (clientCount > 0) {
      client.write(clientBuf, clientCount);
    }
    fh.close();
  }
  else {
    Serial.println("file open failed");
  }
}

//
// SD Card file related functions
//
void readFile(fs::FS &fs, const char * path) {
  Serial.printf("Reading file: %s\n", path);

  File file = fs.open(path);
  if (!file) {
    Serial.println("Failed to open file for reading");
    return;
  }

  Serial.print("Read from file: ");
  while (file.available()) {
    Serial.write(file.read());
  }
  file.close();
}

void writeFile(fs::FS &fs, const char * path, const char * message) {
  Serial.printf("Writing file: %s\n", path);

  File file = fs.open(path, FILE_WRITE); // will overwrite existing file
  if (!file) {
    Serial.println("Failed to open file for writing");
    return;
  }
  if (file.print(message)) {
    Serial.println("File written");
  } else {
    Serial.println("Write failed");
  }
  file.close();
}

void appendFile(fs::FS &fs, const char * path, const char * message) {
  Serial.printf("Appending to file: %s\n", path);

  File file = fs.open(path, FILE_APPEND);
  if (!file) {
    Serial.println("Failed to open file for appending");
    return;
  }
  if (file.print(message)) {
    Serial.println("Message appended");
  } else {
    Serial.println("Append failed");
  }
  file.close();
}
