// Copyright 2016 Steve Quinn
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
//
// Requires ESP8266WiFi Library found here: https://github.com/ekstrand/ESP8266wifi/
// Requires Arduino IDE support for ESP8266 found here: http://esp8266.github.io/Arduino/versions/2.2.0-rc1/doc/installing.html
// Requires FS file system support found in ESP8266 Core or here : https://github.com/esp8266/Arduino/blob/master/cores/esp8266/FS.h
// Requires ESP8266WebServer Library found here:  http://esp8266.github.io/Arduino/versions/2.2.0-rc1/doc/installing.html
// Requires ESP8266mDNS Library found here:  http://esp8266.github.io/Arduino/versions/2.2.0-rc1/doc/installing.html
// Also reference the following : https://github.com/esp8266/arduino-esp8266fs-plugin
// 
// Purpose
// ESP8266 Simple WiFi Access Point with led output. Led illuminates when a WiFi Client has connected.
// To create more dummy networks. Upload the SPIFFs and give it a new 'NETWORK_SSID_DEFAULT' string.
// There is a simple web page which can be served up via the following URL in your browser 'DUMMYSVR.local'
//
// Compiled on Arduino IDE 1.6.9
//
// From Tools Menu
// Board: 'Generic ESP8266 Module'
// Flash Mode: 'DIO'
// Flash Frequency '40MHz'
// CPU Frequency '80 MHz'
// Flash Size: '1M (512KSPIFFS)'
// Debug Port: 'Disabled'
// Debug Level: 'None'
// Reset Method: 'ck'
// Upload Speed: '115200'
//
//
// Maintenance History
// 26/11/16 : DummyNetwork1 Used to create a dummy WiFi network to test the Temperature/Humidity IoT Sensor
//
// 
// Note
// After install of arduino-esp8266fs-plugin, create a subdirectory named 'data' within the directory holding this sketch.
// Any files residing here will be uploaded to the SPIFFS area on the ESP8266 when Tools->ESP8266 Sketch Data Upload
// Is selected. Though you still need to follow the Flash/Reset button process on the programmer. The file system is uploaded
// Using a different offset to the code which starts from 'flash at 0x00000000'
//

#define DEBUG_GENERAL  // Undefine this for general debug information via the serial port. 
#define DEBUG_WEB // Undefine this for comprehensive debug information on web interface via the serial port. Requires DEBUG_GENERAL.
#define DEBUG_MDNS // Undefine this for comprehensive debug information on mDNS support via the serial port. Requires DEBUG_GENERAL.


#include <ESP8266WiFi.h>
#include <string.h>
#include "FS.h"
#include <stdlib.h>
#include <ESP8266WebServer.h>
#include <ESP8266mDNS.h>



//EDIT THESE LINES TO MATCH YOUR SETUP
#define NETWORK_SSID_DEFAULT                "DUMMYNET1"
#define NETWORK_PASSWORD_DEFAULT            "PASSWORD"
char network_ssid[30];
char network_password[30];

//LED on ESP8266 GPIO0
const int lightPin = 0;

const char* host = "DUMMYSVR";
ESP8266WebServer server(80);
static bool hasSPIFFS = false;
unsigned char softap_stations_cnt = 0;
unsigned char softap_stations_cnt_old = 1;

void setup() {
  // Start the serial line for debugging
  // This is enabled or the TX/RX port will require 10K pull ups to stop oscillations of the I/Ps which makes the ESP8266-01 pull more current
  //#ifdef DEBUG_GENERAL
  Serial.begin(115200);
  //Serial.setDebugOutput(true);
  delay(100);
  //#endif
  #ifdef DEBUG_GENERAL
  Serial.println("Starting...");
  #endif
  

  // Set up default security parameters. If all else fails.
  strcpy(network_ssid,NETWORK_SSID_DEFAULT);
  strcpy(network_password,NETWORK_PASSWORD_DEFAULT);
  
  // Initialize the light as an output and set to LOW (off)
  pinMode(lightPin, OUTPUT);
  digitalWrite(lightPin, HIGH);

  // Start filing subsystem
  SPIFFS.begin();
  delay(2000);

  #ifdef DEBUG_GENERAL
  Serial.println("In AP Mode");
  #endif
  hasSPIFFS = true;
 
  // Start wifi subsystem
  WiFi.mode(WIFI_AP);
  WiFi.encryptionType(ENC_TYPE_WEP);
  WiFi.softAP((const char *)network_ssid, (const char *)network_password);

  IPAddress myIP = WiFi.softAPIP();
  #ifdef DEBUG_GENERAL
  Serial.print("AP IP address: "); Serial.println(myIP);
  #endif    
  // Set up HTTP server
  //server.on("/0", HTTP_GET, handleNetworkConfig);
  server.onNotFound(handleNotFound);
  server.begin();
  #ifdef DEBUG_WEB
  Serial.println("HTTP server started");
  #endif
  
  if (MDNS.begin(host)) {
    MDNS.addService("http", "tcp", 80);
    #ifdef DEBUG_MDNS
    Serial.println("MDNS responder started");
    Serial.print("You can now connect to http://");
    Serial.print(host);
    Serial.println(".local");
    #endif
  }
    
  
  //wait a bit before starting the main loop
  delay(2000);
}





void loop(){
  server.handleClient();
  softap_stations_cnt = WiFi.softAPgetStationNum();
  //softap_stations_cnt = wifi_softap_get_station_num(); // Count of stations which are connected to ESP8266 soft-AP
  if (softap_stations_cnt == 0) {
    digitalWrite(lightPin, HIGH);
  } else {
    digitalWrite(lightPin, LOW);
  }
  #ifdef DEBUG_GENERAL
  if (softap_stations_cnt != softap_stations_cnt_old) {
    softap_stations_cnt_old = softap_stations_cnt;
    Serial.print("Client Count : ");
    Serial.println(softap_stations_cnt);
  }
  #endif
  delay(10); 
}



void returnOK(String mess) {
  #ifdef DEBUG_WEB
  Serial.println("returnOK");  
  #endif
  if (mess.length() > 0)
    server.send(200, "text/html", mess);
  else  
    server.send(200, "text/plain", "");
}

void returnFail(String msg) {
  #ifdef DEBUG_WEB
  Serial.println("returnFail");  
  #endif
  server.send(500, "text/plain", msg + "\r\n");
}

bool loadFromSPIFFS(String path){
  String dataType = "text/plain";
  #ifdef DEBUG_WEB
  Serial.println("loadFromSPIFFS");  
  #endif
  if(path.endsWith("/")) path += "index.htm";

  if(path.endsWith(".src")) path = path.substring(0, path.lastIndexOf("."));
  else if(path.endsWith(".htm")) dataType = "text/html";
  else if(path.endsWith(".css")) dataType = "text/css";
  else if(path.endsWith(".js")) dataType = "application/javascript";
  else if(path.endsWith(".json")) dataType = "application/json";
  else if(path.endsWith(".png")) dataType = "image/png";
  else if(path.endsWith(".gif")) dataType = "image/gif";
  else if(path.endsWith(".jpg")) dataType = "image/jpeg";
  else if(path.endsWith(".ico")) dataType = "image/x-icon";
  else if(path.endsWith(".xml")) dataType = "text/xml";
  else if(path.endsWith(".pdf")) dataType = "application/pdf";
  else if(path.endsWith(".zip")) dataType = "application/zip";
  else if(path.endsWith(".png")) dataType = "image/png";

  File dataFile = SPIFFS.open(path.c_str(),"r");

  if (!dataFile)
    return false;

  if (server.hasArg("download")) dataType = "application/octet-stream";

  if (server.streamFile(dataFile, dataType) != dataFile.size()) {
    #ifdef DEBUG_WEB
    Serial.println("Sent less data than expected!");
    #endif
  }

  dataFile.close();
  return true;
}


void handleNotFound(){
  #ifdef DEBUG_WEB
  Serial.println("handleNotFound");
  #endif
  if(hasSPIFFS && loadFromSPIFFS(server.uri())) return;
  String message = "SPIFFS Not Detected\n\n";
  message += "URI: ";
  message += server.uri();
  message += "\nMethod: ";
  message += (server.method() == HTTP_GET)?"GET":"POST";
  message += "\nArguments: ";
  message += server.args();
  message += "\n";
  for (uint8_t i=0; i<server.args(); i++){
    message += " NAME:"+server.argName(i) + "\n VALUE:" + server.arg(i) + "\n";
  }
  server.send(404, "text/plain", message);
  #ifdef DEBUG_WEB
  Serial.print(message);
  #endif
}









