/////////////////////////////////////////////////////////////////////////////////
//Copyright (c) 2009  Andrew L. Sandoval
//
//Permission is hereby granted, free of charge, to any person obtaining a copy
//of this software and associated documentation files (the "Software"), to deal
//in the Software without restriction, including without limitation the rights
//to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//copies of the Software, and to permit persons to whom the Software is
//furnished to do so, subject to the following conditions:
//
//The above copyright notice and this permission notice shall be included in
//all copies or substantial portions of the Software.
//
//THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
//THE SOFTWARE.
/////////////////////////////////////////////////////////////////////////////////
//
//
//
// Digital Window Sticker - Microcontroller Code
// Andrew L. Sandoval - October 2009
// FAT16 & SD CARD support code Written by William Greiman, published under GPL
// FAT16 Project: http://code.google.com/p/fat16lib/
//
#include <avr/pgmspace.h>
#include <Fat16.h>

class ALS48x16Display
{
private:
  int m_iDataPin;
  int m_iWriteClockPin;
  int m_iCS2Pin;
  int m_iCS1Pin;
  
  ALS48x16Display& WriteBit(bool bOn)
  {
    digitalWrite(m_iWriteClockPin, LOW);
    digitalWrite(m_iDataPin, bOn ? HIGH : LOW);
    digitalWrite(m_iWriteClockPin, HIGH);
    return *this;
  }
    
  ALS48x16Display& WriteDataNibble(int iCS, unsigned char ucAddress,
    unsigned char ucData)
  {
    digitalWrite(iCS, LOW);
    WriteBit(true);     // 1
    WriteBit(false);    // 0
    WriteBit(true);     // 1 WRITE
    //
    // Send 6-bit address start
    //
    for(int i = 0; i < 7; i++)
    {
      int iBit = ucAddress << i+1;
      WriteBit(iBit & 0x80);  // High bit
    }
    
    //
    // Send 4-bits of data
    //
    for(int i = 0; i < 4; i++)
    {
      int iBit = ucData >> i;
      WriteBit(iBit & 0x01);  // Low bit  (Opposite Addr)
    }
    digitalWrite(iCS, HIGH);
    return *this;
  }
  
  ALS48x16Display& WriteDataByte(int iCS, unsigned char ucAddress,
    unsigned char ucData)
  {
    digitalWrite(iCS, LOW);
    WriteBit(true);     // 1
    WriteBit(false);    // 0
    WriteBit(true);     // 1 WRITE
    //
    // Send 7-bit address start
    //
    for(int i = 0; i < 7; i++)
    {
      int iBit = ucAddress << i+1;
      WriteBit(iBit & 0x80);  // High bit
    }
    
    //
    // Send 8-bits of data
    //
    for(int i = 0; i < 8; i++)
    {
      int iBit = ucData >> i;
      WriteBit(iBit & 0x01);  // Low bit  (Opposite Addr)
    }
    digitalWrite(iCS, HIGH);
    return *this;
  }
public:
  ALS48x16Display(int iDataPin = 7, int iWriteClockPin = 6,
    int iCS2Pin = 5, int iCS1Pin = 4) :
    m_iDataPin(iDataPin), m_iWriteClockPin(iWriteClockPin),
    m_iCS2Pin(iCS2Pin), m_iCS1Pin(iCS1Pin)
  {
  }
  
  void SendCommand(unsigned char ucCommand, int iCS)
  {
    digitalWrite(m_iWriteClockPin, HIGH);  // Reset CLK
    digitalWrite(iCS, LOW);    // Enable target
    WriteBit(true);
    WriteBit(false);
    WriteBit(false);    // 100
    for(int i = 0; i < 8; i++)
    {
      int iBit = ucCommand << i;
      WriteBit(iBit & 0x80);  // High bit
    }
    WriteBit(false);          // Lame, but 9 bits needed for command
    digitalWrite(iCS, HIGH);  // Reset CS to HIGH
  }
  
  void Initialize()
  {
    //
    // Setup Display Pin-modes and start values
    //
    pinMode(m_iDataPin, OUTPUT);
    pinMode(m_iWriteClockPin, OUTPUT);
    pinMode(m_iCS2Pin, OUTPUT);
    pinMode(m_iCS1Pin, OUTPUT);
    digitalWrite(m_iCS2Pin, HIGH);
    digitalWrite(m_iCS1Pin, HIGH);
    digitalWrite(m_iWriteClockPin, HIGH);
    
    //
    // Set Modes for each display
    //
    SendCommand(0x00, m_iCS1Pin);   // Disable System
    SendCommand(0x00, m_iCS2Pin);   // Disable System on 2
    SendCommand(0x2c, m_iCS1Pin);   // ?
    SendCommand(0x2c, m_iCS2Pin);
    SendCommand(0x14, m_iCS1Pin);  // Master Mode
    SendCommand(0x10, m_iCS2Pin);
    SendCommand(0x01, m_iCS1Pin);  // Sys-On
    SendCommand(0x01, m_iCS2Pin);
    SendCommand(0x03, m_iCS1Pin);
    SendCommand(0x03, m_iCS2Pin);
    delay(100);
  }
  
  ALS48x16Display& Blank(bool bReverse)
  {
    unsigned char ucFill[96];
    memset(ucFill, 0, sizeof(ucFill));
    
    DisplayBuffer(ucFill, bReverse);
    return *this;
  }
  
  void DisplayBufferOnSide(const unsigned char *pucData, int iCS,
    bool bReverse = false)
  {
    // pucData must point to a buffer that is 0x60 bytes
    // arranged by vertical rows of bits
    digitalWrite(iCS, LOW);
    WriteBit(true);     // 1
    WriteBit(false);    // 0
    WriteBit(true);     // 1 WRITE
    
    //
    // Send 7-bit address start
    //
    for(int i = 0; i < 7; i++)
    {
      WriteBit(0);
    }
    //
    // Send Data a byte at a time
    //
    for(int y = 0; y < 24; ++y) // row
    {
      unsigned char ucByte = *pucData++;
      unsigned char ucByte2 = *pucData++;
      for(int i = 0; i < 8; ++i)
      {
        bool b = (ucByte & (1 << i));
        WriteBit(bReverse ? !b : b);
      }
      for(int i2 = 0; i2 < 8; ++i2)
      {
        bool b = (ucByte2 & (1 << i2));
        WriteBit(bReverse ? !b : b);
      }
    }
    digitalWrite(iCS, HIGH);
  }
  
  void DisplayBuffer(const unsigned char *pucData192, bool bReverse=false)
  {
    // pucData192 must by 192 bytes!
    DisplayBufferOnSide(pucData192, m_iCS1Pin, bReverse);
    DisplayBufferOnSide(pucData192+48, m_iCS2Pin, bReverse);    
  }
};

ALS48x16Display g_ledDisplay;

const unsigned char g_pucNoFilesImage[] =
{
  0x00, 0x00, 0xff, 0x7c, 0x03, 0x44, 0x0e, 0x44,
  0x18, 0x7c, 0x60, 0x00, 0xff, 0x40, 0x00, 0x00,
  0x00, 0x7c, 0xfc, 0x44, 0x84, 0x28, 0x84, 0x10,
  0x84, 0x00, 0x78, 0x3c, 0x00, 0x40, 0x00, 0x38,
  0x00, 0x40, 0x00, 0x3c, 0x00, 0x00, 0x00, 0x5c,
  0xff, 0x54, 0x09, 0x54, 0x09, 0x74, 0x09, 0x00,
  0x09, 0x00, 0x00, 0x5c, 0xfd, 0x00, 0x00, 0x00,
  0x00, 0x7c, 0xff, 0x14, 0x00, 0x04, 0x00, 0x70,
  0x78, 0x50, 0x94, 0x70, 0x94, 0x00, 0x94, 0x70,
  0x98, 0x40, 0x00, 0x70, 0x00, 0x00, 0x9c, 0x70,
  0x94, 0x10, 0xa4, 0x60, 0xe4, 0x00, 0x00, 0x70,
  0x00, 0x50, 0xbf, 0x7c, 0x00, 0x00, 0x00, 0x00
};

const unsigned char g_pucNoCard[] =
{
  0x41, 0xfe, 0x41, 0x82, 0x7f, 0x82, 0x41, 0x82,
  0x41, 0x82, 0x00, 0x00, 0x7f, 0xfe, 0x03, 0x12,
  0x06, 0x12, 0x1c, 0x12, 0x7f, 0xfe, 0x00, 0x00,
  0x4f, 0xfe, 0x49, 0x12, 0x49, 0x32, 0x49, 0x52,
  0x79, 0x9e, 0x00, 0x00, 0x7f, 0xfe, 0x49, 0x82,
  0x49, 0x82, 0x41, 0x44, 0x00, 0x38, 0x7f, 0x00,
  0x09, 0x00, 0x19, 0xde, 0x29, 0xde, 0x4f, 0x00,
  0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x7f, 0x00,
  0x01, 0x00, 0x01, 0x00, 0xf4, 0x5f, 0x14, 0x50,
  0xd4, 0x55, 0x54, 0x55, 0x54, 0x57, 0x14, 0x50,
  0xd4, 0x57, 0x54, 0x54, 0x54, 0x54, 0x94, 0x53,
  0x14, 0x48, 0xf4, 0x47, 0x04, 0x40, 0xfc, 0x7f
};

SdCard g_SDCard;
bool g_bReadFromSDCard = false;
bool g_bNoCard = true;

void Fat16Setup()
{
  if(g_bNoCard)
  {
    Serial.println("No card available to initialize!");
    return;
  }
  if(!g_SDCard.init())
  {
    Serial.println("Failed to initialize SD Card!!!");
    g_bReadFromSDCard = false;
    return;
  }
  if(!Fat16::init(g_SDCard))
  {
    Serial.println("Failed to initialize FAT16 filesystem!");
    g_bReadFromSDCard = false;
    return;
  }
  g_bReadFromSDCard = true;
}

#define SDCARD_DETECT_PIN  2  // When low a card is present
                              // Could be interrupt driver on 2, not necessary though

void setup()
{
  //
  // Debugging:
  //
  Serial.begin(9600);
  g_ledDisplay.Initialize();
 
  //
  // Card Detection:
  //
  pinMode(SDCARD_DETECT_PIN, INPUT);
  digitalWrite(SDCARD_DETECT_PIN, HIGH);
  g_bNoCard = digitalRead(2);
  if(g_bNoCard) 
  {
     Serial.println("No Card found duning setup...");
  }
  else
  {
    Serial.println("SD Card found.");
    //
    // Setup SD Card
    //
    Fat16Setup();
  }
}

void loop()
{
  g_bNoCard = digitalRead(SDCARD_DETECT_PIN);
  if(g_bNoCard)
  {
     Serial.println("No SD Card!");
     g_ledDisplay.DisplayBuffer(g_pucNoCard);
     delay(800);  // Wait for a card (could spin waiting for interrupt)
     while(g_bNoCard)
     {
       delay(200);
       g_bNoCard = digitalRead(SDCARD_DETECT_PIN);
     }
     Serial.println("Card Inserted!  Trying to initialize!");
     Fat16Setup();  // Can't get out of this just to fail again...
  }
  
  if(!g_bReadFromSDCard)
  {
    //
    // Flash the whole matrix on this error!
    // Card Present but can't be read...
    //
    g_ledDisplay.Blank(true);
    delay(1000);
    g_ledDisplay.Blank(false);
    delay(1000);
    Fat16Setup();  // Try again...
    return;
  }
  
  //
  // Loop through the files on the SD card, display and delay
  // Max 512 files in root directory, FAT16 limitation
  //
  for(unsigned int i = 0; i < 512; ++i)
  {
    Fat16 file;
    char cFilename[15];
    sprintf(cFilename, "%u.DWS", i);
    if(!file.open(cFilename, O_READ))
    {
      Serial.print(cFilename);
      Serial.println(" can not be opened!");
      // Reached the end of the numbered DWS files, loop or display no files
      if(!i)
      {
        g_ledDisplay.DisplayBuffer(g_pucNoFilesImage);
        delay(1000);
      }
      return;  // Start at 0 again
    }
    Serial.print("Opened file ");
    Serial.println(cFilename);
    
    unsigned char pucImage[96];
    unsigned long ulDelay = 1000;
    int16_t iRead = file.read(pucImage, sizeof(pucImage));
    Serial.print(iRead, DEC);
    Serial.println(" bytes read from card.");
    iRead = file.read(&ulDelay, sizeof(ulDelay));
    file.close();
    
    Serial.print(iRead, DEC);
    Serial.println(" bytes read from card.");
    g_ledDisplay.DisplayBuffer(pucImage);
    delay(ulDelay);
  }
}

