
/*
 * Dice.cpp - Library for Reading dice face.
 * Created by Calvin Ochs, January 15, 2019
 * 
 * 0 0 0
 * 1 2 0
 * 5 4 3
 */
#include "Arduino.h"
#include "Dice.h"

// parameters: (pin1, pin2, pin3, pin4, pin5, calibration numbers 1-5, amount of times to loop, data pin, clock pin, latch pin)

Dice::Dice(const int pin1,const int pin2,const int pin3,const int pin4,const int pin5, int cal1, int cal2, int cal3, int cal4, int cal5, int lop, 
                                                                                         int serialDataPin, int clockPin, int latchPin)
{

  // set attributes
    _numberOfShiftRegisters = 5;
    
    _clockPin = clockPin;
    _serialDataPin = serialDataPin;
    _latchPin = latchPin;
    
    // define pins as outputs
    pinMode(clockPin, OUTPUT);
    pinMode(serialDataPin, OUTPUT);
    pinMode(latchPin, OUTPUT);
    
    // set pins low
    digitalWrite(clockPin, LOW);
    digitalWrite(serialDataPin, LOW);
    digitalWrite(latchPin, LOW);
    
    // allocates the specified number of bytes and initializes them to zero
    _digitalValues = (uint8_t *)malloc(numberOfShiftRegisters * sizeof(uint8_t));
    memset(_digitalValues, 0, numberOfShiftRegisters * sizeof(uint8_t));
    
    setAll(_digitalValues); // reset shift register
    
  _pin1 = pin1;
  _pin2 = pin2;
  _pin3 = pin3;
  _pin4 = pin4;
  _pin5 = pin5;
  _cal1 = cal1;
  _cal2 = cal2;
  _cal3 = cal3;
  _cal4 = cal4;
  _cal5 = cal5;

  dot1 = 0;
  dot2 = 0;
  dot3 = 0;
  dot4 = 0;
  dot5 = 0;

  p1 = 0;
  p2 = 0;
  p3 = 0;
  p4 = 0;
  p5 = 0;
  _loop = lop;
  dice = 0;
  
  currentFace = "00000";
  face1 = "01000";
  face2 = "00100";
  face2r = "00001";
  face3 = "01001";
  face3r = "01100";
  face4 = "00101";
  face5 = "01101";
  face6 = "00111";
  face6r = "10101";
  
}


//reads the face of the dice
int Dice::ReadFace()
{
  p1 = 0;
  p2 = 0;
  p3 = 0;
  p4 = 0;
  p5 = 0;
  dot1 = 0;
  dot2 = 0;
  dot3 = 0;
  dot4 = 0;
  dot5 = 0;
  currentFace = "00000";
  
  for (int i = 0; i < _loop; i++)
  {
    set(1,HIGH);
    p1 += analogRead(_pin1);
    set(1,LOW);
    set(2,HIGH);
    p2 += analogRead(_pin2);
    set(2,LOW);
    set(3,HIGH);
    p3 += analogRead(_pin3);
    set(3,LOW);
    set(4,HIGH);
    p4 += analogRead(_pin4);
    set(4,LOW);
    set(5,HIGH);
    p5 += analogRead(_pin5);
    set(5,LOW);
    delay(10);
  }
    p1 = p1 / _loop;
    p2 = p2 / _loop;
    p3 = p3 / _loop;
    p4 = p4 / _loop;
    p5 = p5 / _loop;
    
    if (p1>_cal1) dot1=1; else dot1=0;
    if (p2>_cal2) dot2=1; else dot2=0;
    if (p3>_cal3) dot3=1; else dot3=0;
    if (p4>_cal4) dot4=1; else dot4=0;
    if (p5>_cal5) dot5=1; else dot5=0;

    currentFace = String(dot1)+String(dot2)+String(dot3)+String(dot4)+String(dot5);
    
    if (currentFace == face1) dice = 1;
    else if (currentFace == face2) dice = 2;
    else if (currentFace == face2r) dice = 2;
    else if (currentFace == face3) dice = 3;
    else if (currentFace == face3r) dice = 3;
    else if (currentFace == face4) dice = 4;
    else if (currentFace == face5) dice = 5;
    else if (currentFace == face6) dice = 6;
    else if (currentFace == face6r) dice = 6;
    else if (currentFace == face6r) dice = 6;
    else dice = 0;

    return dice;
    
     
}

//returns value from pip location
int Dice::One()
{
  set(1,HIGH);
  p1 = analogRead(_pin1);
  set(1,LOW);
  return p1;  
}

int Dice::Two()
{
  set(2,HIGH);
  p2 = analogRead(_pin2);
  set(2,LOW);
  return p2;  
}

int Dice::Three()
{
  set(3,HIGH);
  p3 = analogRead(_pin3);
  set(3,LOW);
  return p3;  
}

int Dice::Four()
{
  set(4,HIGH);
  p4 = analogRead(_pin4);
  set(4,LOW);
  return p4;  
}

int Dice::Five()
{
  set(5,HIGH);
  p5 = analogRead(_pin5);
  set(5,LOW);
  return p5;  
}

//returns the current dice number
int Dice::getDice()
{
  return dice;
}

// Set all pins of the shift registers at once.
// digitalVAlues is a uint8_t array where the length is equal to the number of shift registers.
void Dice::setAll(uint8_t * digitalValues)
{
    int byte;
    
    for (byte = _numberOfShiftRegisters - 1; byte >= 0; byte--) {
        shiftOut(_serialDataPin, _clockPin, MSBFIRST, digitalValues[byte]);
    }
    
    _digitalValues = digitalValues; 
    
    digitalWrite(_latchPin, HIGH); 
    digitalWrite(_latchPin, LOW); 
}


// Retrieve all states of the shift registers' output pins.
// The returned array's length is equal to the numbe of shift registers.
uint8_t * Dice::getAll()
{
    return _digitalValues; 
}


// Set a specific pin to either HIGH (1) or LOW (0).
// The pin parameter is a positive, zero-based integer, indicating which pin to set.
void Dice::set(int pin, uint8_t value)
{
    setNoUpdate(pin, value);
    updateRegisters();
}

// Updates the shift register pins to the stored output values.
void Dice::updateRegisters()
{
   setAll(_digitalValues);
}


// Equivalent to set(int pin, uint8_t value), except the physical shift register is not updated.
// Should be used in combination with updateRegisters().
void Dice::setNoUpdate(int pin, uint8_t value)
{
    if (value == 1) {
        _digitalValues[pin / 8] |= 1 << (pin % 8);
    }
    else {
        _digitalValues[pin / 8] &= ~(1 << (pin % 8));
    }
}


// Returns the state of the given pin.
// Either HIGH (1) or LOW (0)
uint8_t Dice::get(int pin)
{
    return (_digitalValues[pin / 8] >> (pin % 8)) & 1;
}


// Sets all pins of all shift registers to HIGH (1).
void Dice::setAllHigh()
{
    int i; 
    for (i = 0; i < _numberOfShiftRegisters; i++) {
        _digitalValues[i] = 255;
    }
    setAll(_digitalValues); 
}


// Sets all pins of all shift registers to LOW (0).
void Dice::setAllLow()
{
    int i; 
    for (i = 0; i < _numberOfShiftRegisters; i++) {
        _digitalValues[i] = 0;
    }
    setAll(_digitalValues); 
}
