/*
    FreeRTOS V9.0.0 - Copyright (C) 2016 Real Time Engineers Ltd.
    All rights reserved

    VISIT http://www.FreeRTOS.org TO ENSURE YOU ARE USING THE LATEST VERSION.

    This file is part of the FreeRTOS distribution.

    FreeRTOS is free software; you can redistribute it and/or modify it under
    the terms of the GNU General Public License (version 2) as published by the
    Free Software Foundation >>>> AND MODIFIED BY <<<< the FreeRTOS exception.

    ***************************************************************************
    >>!   NOTE: The modification to the GPL is included to allow you to     !<<
    >>!   distribute a combined work that includes FreeRTOS without being   !<<
    >>!   obliged to provide the source code for proprietary components     !<<
    >>!   outside of the FreeRTOS kernel.                                   !<<
    ***************************************************************************

    FreeRTOS is distributed in the hope that it will be useful, but WITHOUT ANY
    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
    FOR A PARTICULAR PURPOSE.  Full license text is available on the following
    link: http://www.freertos.org/a00114.html

    ***************************************************************************
     *                                                                       *
     *    FreeRTOS provides completely free yet professionally developed,    *
     *    robust, strictly quality controlled, supported, and cross          *
     *    platform software that is more than just the market leader, it     *
     *    is the industry's de facto standard.                               *
     *                                                                       *
     *    Help yourself get started quickly while simultaneously helping     *
     *    to support the FreeRTOS project by purchasing a FreeRTOS           *
     *    tutorial book, reference manual, or both:                          *
     *    http://www.FreeRTOS.org/Documentation                              *
     *                                                                       *
    ***************************************************************************

    http://www.FreeRTOS.org/FAQHelp.html - Having a problem?  Start by reading
    the FAQ page "My application does not run, what could be wrong?".  Have you
    defined configASSERT()?

    http://www.FreeRTOS.org/support - In return for receiving this top quality
    embedded software for free we request you assist our global community by
    participating in the support forum.

    http://www.FreeRTOS.org/training - Investing in training allows your team to
    be as productive as possible as early as possible.  Now you can receive
    FreeRTOS training directly from Richard Barry, CEO of Real Time Engineers
    Ltd, and the world's leading authority on the world's leading RTOS.

    http://www.FreeRTOS.org/plus - A selection of FreeRTOS ecosystem products,
    including FreeRTOS+Trace - an indispensable productivity tool, a DOS
    compatible FAT file system, and our tiny thread aware UDP/IP stack.

    http://www.FreeRTOS.org/labs - Where new FreeRTOS products go to incubate.
    Come and try FreeRTOS+TCP, our new open source TCP/IP stack for FreeRTOS.

    http://www.OpenRTOS.com - Real Time Engineers ltd. license FreeRTOS to High
    Integrity Systems ltd. to sell under the OpenRTOS brand.  Low cost OpenRTOS
    licenses offer ticketed support, indemnification and commercial middleware.

    http://www.SafeRTOS.com - High Integrity Systems also provide a safety
    engineered and independently SIL3 certified version for use in safety and
    mission critical applications that require provable dependability.

    1 tab == 4 spaces!
*/

/******************************************************************************
 * NOTE 1:  This project provides three demo applications.  A simple blinky
 * style project, a more comprehensive test and demo application, and an
 * lwIP example.  The mainSELECTED_APPLICATION setting in main.c is used to
 * select between the three.  See the notes on using mainSELECTED_APPLICATION
 * in main.c.  This file implements the simply blinky style version.
 *
 * NOTE 2:  This file only contains the source code that is specific to the
 * basic demo.  Generic functions, such FreeRTOS hook functions, and functions
 * required to configure the hardware are defined in main.c.
 ******************************************************************************
 *
 * main_blinky() creates one queue, and two tasks.  It then starts the
 * scheduler.
 *
 * The Queue Send Task:
 * The queue send task is implemented by the prvQueueSendTask() function in
 * this file.  prvQueueSendTask() sits in a loop that causes it to repeatedly
 * block for 200 milliseconds, before sending the value 100 to the queue that
 * was created within main_blinky().  Once the value is sent, the task loops
 * back around to block for another 200 milliseconds...and so on.
 *
 * The Queue Receive Task:
 * The queue receive task is implemented by the prvQueueReceiveTask() function
 * in this file.  prvQueueReceiveTask() sits in a loop where it repeatedly
 * blocks on attempts to read data from the queue that was created within
 * main_blinky().  When data is received, the task checks the value of the
 * data, and if the value equals the expected 100, toggles an LED.  The 'block
 * time' parameter passed to the queue receive function specifies that the
 * task should be held in the Blocked state indefinitely to wait for data to
 * be available on the queue.  The queue receive task will only leave the
 * Blocked state when the queue send task writes to the queue.  As the queue
 * send task writes to the queue every 200 milliseconds, the queue receive
 * task leaves the Blocked state every 200 milliseconds, and therefore toggles
 * the LED every 200 milliseconds.
 */

/* Kernel includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"

/* Standard demo includes. */
#include "tasksing.h"
#include "servoPWM.h"

/* Priorities at which the tasks are created. */
#define mainQUEUE_RECEIVE_TASK_PRIORITY		( tskIDLE_PRIORITY + 2 )
#define	mainQUEUE_SEND_TASK_PRIORITY		( tskIDLE_PRIORITY + 1 )

/* The rate at which data is sent to the queue.  The 200ms value is converted
to ticks using the portTICK_PERIOD_MS constant. */
#define mainQUEUE_SEND_FREQUENCY_MS			( 200 / portTICK_PERIOD_MS )

/* The rate at which the blink rate changes */
#define mainBLINK_RATE_CHANGE_FREQUENCY_MS	( 2000 / portTICK_PERIOD_MS )

/* The code to be run.
 * 0: original
 * 1: single task
 * 2: variable blink rate
 */
#define mainSELECTED_BLINK_MODE	3

/* The number of items the queue can hold.  This is 1 as the receive task
will remove items as they are added, meaning the send task should always find
the queue empty. */
#define mainQUEUE_LENGTH					( 1 )

/* The LED toggled by the Rx task. */
#define mainTASK_LED						( 0 )

/*-----------------------------------------------------------*/

/*
 * The tasks as described in the comments at the top of this file.
 */
static void prvQueueReceiveTask( void *pvParameters );
static void prvQueueSendTask( void *pvParameters );
static void prvSingleTask( void *pvParameters );
static void prvQueueSendRateTask( void *pvParameters );
static void prvSingleTaskVariable( void *pvParameters );
static void prvHdlFsmTask( void *pvParameters );

static void prvPanSweepTask( void *pvParameters );
static void prvTiltSweepTask( void *pvParameters );


/*-----------------------------------------------------------*/

/* The queue used by both tasks. */
static QueueHandle_t xQueue = NULL;

/*-----------------------------------------------------------*/

void main_tasksing( void )
{
	/* Create the queue. */
	xQueue = xQueueCreate( mainQUEUE_LENGTH, sizeof( uint32_t ) );

	if( xQueue != NULL )
	{
		uint8_t panDelay = 128;
		xTaskCreate( prvPanSweepTask, "Pan", configMINIMAL_STACK_SIZE, &panDelay, mainQUEUE_SEND_TASK_PRIORITY, NULL );
		uint8_t tiltDelay = 64;
		xTaskCreate( prvTiltSweepTask, "Tilt", configMINIMAL_STACK_SIZE, &tiltDelay, mainQUEUE_SEND_TASK_PRIORITY, NULL );
#if 0
#if ( mainSELECTED_BLINK_MODE == 0 )
		/* Start the two tasks as described in the comments at the top of this
		file. */
		xTaskCreate( prvQueueReceiveTask,				/* The function that implements the task. */
					"Rx", 								/* The text name assigned to the task - for debug only as it is not used by the kernel. */
					configMINIMAL_STACK_SIZE, 			/* The size of the stack to allocate to the task. */
					NULL, 								/* The parameter passed to the task - not used in this case. */
					mainQUEUE_RECEIVE_TASK_PRIORITY, 	/* The priority assigned to the task. */
					NULL );								/* The task handle is not required, so NULL is passed. */

		uint8_t ledToggleDelay = 31;
		xTaskCreate( prvQueueSendTask, "TX", configMINIMAL_STACK_SIZE, &ledToggleDelay, mainQUEUE_SEND_TASK_PRIORITY, NULL );
#endif
#if ( mainSELECTED_BLINK_MODE == 1 )
		uint8_t ledToggleDelay = 31;
		xTaskCreate( prvSingleTask, "Blink", configMINIMAL_STACK_SIZE, &ledToggleDelay, mainQUEUE_SEND_TASK_PRIORITY, NULL );
#endif
#if ( mainSELECTED_BLINK_MODE == 2 )
		xTaskCreate( prvQueueSendRateTask,				/* The function that implements the task. */
					"RateTX", 								/* The text name assigned to the task - for debug only as it is not used by the kernel. */
					configMINIMAL_STACK_SIZE, 			/* The size of the stack to allocate to the task. */
					NULL, 								/* The parameter passed to the task - not used in this case. */
					mainQUEUE_SEND_TASK_PRIORITY, 	/* The priority assigned to the task. */
					NULL );								/* The task handle is not required, so NULL is passed. */
		xTaskCreate( prvSingleTaskVariable, "RxBlink", configMINIMAL_STACK_SIZE, NULL, mainQUEUE_RECEIVE_TASK_PRIORITY, NULL );
#endif
#if ( mainSELECTED_BLINK_MODE == 3 )
		uint16_t ledToggleDelay = 2000;
		xTaskCreate( prvHdlFsmTask,				/* The function that implements the task. */
					"HdlFsm", 								/* The text name assigned to the task - for debug only as it is not used by the kernel. */
					configMINIMAL_STACK_SIZE, 			/* The size of the stack to allocate to the task. */
					&ledToggleDelay, 								/* The parameter passed to the task - not used in this case. */
					mainQUEUE_SEND_TASK_PRIORITY, 	/* The priority assigned to the task. */
					NULL );								/* The task handle is not required, so NULL is passed. */
		//xTaskCreate( prvSingleTaskVariable, "RxBlink", configMINIMAL_STACK_SIZE, NULL, mainQUEUE_RECEIVE_TASK_PRIORITY, NULL );
#endif
#endif
		/* Start the tasks and timer running. */
		vTaskStartScheduler();
	}

	/* If all is well, the scheduler will now be running, and the following
	line will never be reached.  If the following line does execute, then
	there was either insufficient FreeRTOS heap memory available for the idle
	and/or timer tasks to be created, or vTaskStartScheduler() was called from
	User mode.  See the memory management section on the FreeRTOS web site for
	more details on the FreeRTOS heap http://www.freertos.org/a00111.html.  The
	mode from which main() is called is set in the C start up code and must be
	a privileged mode (not user mode). */
	for( ;; );
}
/*-----------------------------------------------------------*/
static void prvPanSweepTask( void *pvParameters )
{
static unsigned char position=90;
static signed char step=1;
TickType_t xNextWakeTime;

	uint8_t taskDelayMS = *((uint8_t*) pvParameters);

	portTASK_USES_FLOATING_POINT(); //just in case this isn't true for us.

	/* Initialise xNextWakeTime - this only needs to be done once. */
	xNextWakeTime = xTaskGetTickCount();

	PWM_Freq_Duty(PWM_X_BASE,CLK_PERIOD,SERVO_FREQ,Duty_val(position));
	PWM_START(PWM_X_BASE);

	for( ;; )
	{
		/* Place this task in the blocked state until it is time to run again. */
		//vTaskDelayUntil( &xNextWakeTime, mainQUEUE_SEND_FREQUENCY_MS );
		vTaskDelayUntil( &xNextWakeTime, taskDelayMS / portTICK_PERIOD_MS );
		position+=step;
		if (position >= 90)
			step = -1;
		else if (position == 45)
			step = 1;
		PWM_Freq_Duty(PWM_X_BASE,CLK_PERIOD,SERVO_FREQ,Duty_val(position));
	}
}
/*-----------------------------------------------------------*/
static void prvTiltSweepTask( void *pvParameters )
{
static unsigned char position=90;
static signed char step=1;
TickType_t xNextWakeTime;

	uint8_t taskDelayMS = *((uint8_t*) pvParameters);

	portTASK_USES_FLOATING_POINT(); //just in case this isn't true for us.

	/* Initialise xNextWakeTime - this only needs to be done once. */
	xNextWakeTime = xTaskGetTickCount();

	PWM_Freq_Duty(PWM_Z_BASE,CLK_PERIOD,SERVO_FREQ,Duty_val(position));
	PWM_START(PWM_Z_BASE);

	for( ;; )
	{
		/* Place this task in the blocked state until it is time to run again. */
		//vTaskDelayUntil( &xNextWakeTime, mainQUEUE_SEND_FREQUENCY_MS );
		vTaskDelayUntil( &xNextWakeTime, taskDelayMS / portTICK_PERIOD_MS );
		position+=step;
		if (position >= 125)
			step = -1;
		else if (position == 55)
			step = 1;
		PWM_Freq_Duty(PWM_Z_BASE,CLK_PERIOD,SERVO_FREQ,Duty_val(position));
	}
}
/*-----------------------------------------------------------*/
/*-----------------------------------------------------------*/
/*-----------------------------------------------------------*/
/*-----------------------------------------------------------*/
/*-----------------------------------------------------------*/
/*-----------------------------------------------------------*/
#if 0
static void prvSingleTask( void *pvParameters )
{
TickType_t xNextWakeTime;

	uint8_t taskDelayMS = *((uint8_t*) pvParameters);

	/* Initialise xNextWakeTime - this only needs to be done once. */
	xNextWakeTime = xTaskGetTickCount();

	for( ;; )
	{
		/* Place this task in the blocked state until it is time to run again. */
		//vTaskDelayUntil( &xNextWakeTime, mainQUEUE_SEND_FREQUENCY_MS );
		vTaskDelayUntil( &xNextWakeTime, taskDelayMS / portTICK_PERIOD_MS );
		vParTestToggleLED( mainTASK_LED );
	}
}
/*-----------------------------------------------------------*/

static void prvQueueSendRateTask( void *pvParameters )
{
TickType_t xNextWakeTime;
const unsigned long ulValueToSend = 100UL;

	/* Remove compiler warning about unused parameter. */
	( void ) pvParameters;

	for( ;; )
	{
		/* Place this task in the blocked state until it is time to run again. */
		vTaskDelayUntil( &xNextWakeTime, mainBLINK_RATE_CHANGE_FREQUENCY_MS );
		xQueueSend( xQueue, &ulValueToSend, 0U );
	}
}
/*-----------------------------------------------------------*/

static void prvSingleTaskVariable( void *pvParameters )
{
TickType_t xNextWakeTime;
unsigned long ulReceivedValue;
const unsigned long ulExpectedValue = 100UL;
static uint8_t ulBlinkRate = 250;

	/* Remove compiler warning about unused parameter. */
	( void ) pvParameters;

	for( ;; )
	{
		vTaskDelayUntil( &xNextWakeTime, ulBlinkRate / portTICK_PERIOD_MS );
		vParTestToggleLED( mainTASK_LED );
		/* Wait until something arrives in the queue - this task will block
		indefinitely provided INCLUDE_vTaskSuspend is set to 1 in
		FreeRTOSConfig.h. */
		xQueueReceive( xQueue, &ulReceivedValue, 0 );

		/*  To get here something must have been received from the queue, but
		is it the expected value?  If it is, toggle the LED. */
		if( ulReceivedValue == ulExpectedValue )
		{
			ulBlinkRate >>=1;
			if (ulBlinkRate<31) ulBlinkRate=250;
			ulReceivedValue = 0U;
		}
	}
}

/*-----------------------------------------------------------*/

static void prvQueueSendTask( void *pvParameters )
{
TickType_t xNextWakeTime;
const unsigned long ulValueToSend = 100UL;

	/* Remove compiler warning about unused parameter. */
	//( void ) pvParameters;

	uint8_t taskDelayMS = *((uint8_t*) pvParameters);

	/* Initialise xNextWakeTime - this only needs to be done once. */
	xNextWakeTime = xTaskGetTickCount();

	for( ;; )
	{
		/* Place this task in the blocked state until it is time to run again. */
		//vTaskDelayUntil( &xNextWakeTime, mainQUEUE_SEND_FREQUENCY_MS );
		vTaskDelayUntil( &xNextWakeTime, taskDelayMS / portTICK_PERIOD_MS );

		/* Send to the queue - causing the queue receive task to unblock and
		toggle the LED.  0 is used as the block time so the sending operation
		will not block - it shouldn't need to block as the queue should always
		be empty at this point in the code. */
		xQueueSend( xQueue, &ulValueToSend, 0U );
	}
}
/*-----------------------------------------------------------*/

static void prvQueueReceiveTask( void *pvParameters )
{
unsigned long ulReceivedValue;
const unsigned long ulExpectedValue = 100UL;

	/* Remove compiler warning about unused parameter. */
	( void ) pvParameters;

	for( ;; )
	{
		/* Wait until something arrives in the queue - this task will block
		indefinitely provided INCLUDE_vTaskSuspend is set to 1 in
		FreeRTOSConfig.h. */
		xQueueReceive( xQueue, &ulReceivedValue, portMAX_DELAY );

		/*  To get here something must have been received from the queue, but
		is it the expected value?  If it is, toggle the LED. */
		if( ulReceivedValue == ulExpectedValue )
		{
			vParTestToggleLED( mainTASK_LED );
			ulReceivedValue = 0U;
		}
	}
}
/*-----------------------------------------------------------*/

static void prvHdlFsmTask( void *pvParameters )
{
TickType_t xNextWakeTime;
static uint8_t prev_state = 0;

	uint16_t taskDelayMS = *((uint16_t*) pvParameters);

	/* Initialise xNextWakeTime - this only needs to be done once. */
	xNextWakeTime = xTaskGetTickCount();

	vParTestHDLSetFsmPATTERN(0x1234);

	for( ;; )
	{
		/* Place this task in the blocked state until it is time to run again. */
		//vTaskDelayUntil( &xNextWakeTime, mainQUEUE_SEND_FREQUENCY_MS );
		vTaskDelayUntil( &xNextWakeTime, taskDelayMS / portTICK_PERIOD_MS );
		//vParTestToggleLED( mainTASK_LED );
		prev_state ++;
		if ( prev_state > 10 )
			prev_state = 0xff ;
		else if ( prev_state == 0xff )
			prev_state = 0x0 ;
		vParTestHDLSetFsmSETTINGS(prev_state);
	}
}

/*-----------------------------------------------------------*/
#endif
