/**********************************************
 PIR Activated Keychain Camera
 Copyright: Doug Paradis - 2010
 All rights reserved
 This code may be used for private use, as long
 as, copyright notice retained.
 Acknowledgements: Some code based on TI App note
 SLAA335 authored by Mike Mitchell
P1.3 (input - pin 5) --> PIR signal (low = motion)
P1.4 (output - pin 6) --> Shutter switch
P1.5 (output - pin 7) -->  Mode switch
 compiled using IAR Embedded Workbench
**********************************************/
#include  "msp430.h"
unsigned char shot_cnt = 60;            // num of shots available
unsigned long int cntr_val = 120000;          // 120000 ~= 1 sec
unsigned int vlo_counts = 0;            // # of VLO clocks in 1 Mhz
unsigned char band_flg = 0;             // flg indicating timing PIR
unsigned char trip_flg = 0;   // flag to monitor if PIR signal is'nt long enough

// Function protos                   
unsigned int cal_VLO(void);
void take_photo(void);
void interval_delay(int delay_time, char wait_flg);
void chg_mode_to_compressed(void);
void turn_off(void);
void turn_on(void);
void main(void)
{
  WDTCTL = WDTPW + WDTHOLD;              // Stop watchdog timer
  
// Ports
  P1SEL = 0;                               // all p1 pins i/o                                                     
  P1SEL |= BIT0;                           // pin P1.0 selected as ACLK
  P1DIR |= 0xf7;                           // all p1 pins output except p1.3
  P2SEL = 0;                               // necessary for p2
  P2DIR |= BIT6 + BIT7;                    // all p2 pins output
  P1OUT &= ~(BIT4) + ~(BIT5);              // p1.4 and p1.5 output low
  P2OUT |= BIT6 + BIT7;                    // all p2 pins high
   
// Clocks  --  Setup DCO and VLO
  BCSCTL1 = CALBC1_1MHZ;                // Use 1Mhz cal data for DCO
  DCOCTL = CALDCO_1MHZ;                 // Use 1Mhz cal data for DCO
  BCSCTL3 = LFXT1S_2;                   // Use VLO for ACLK
  
// VLO calc
  vlo_counts = cal_VLO();               // # of VLO clocks in 1 MHz
      
// chg mode to compressed HR 
  chg_mode_to_compressed();   
   
// 30 sec startup delay
  interval_delay(30,1);           // 30 sec delay before allowing 1st photo
  TACCTL0 &= ~CCIFG;              // Clear CCIFG  
    
// turn on p1 interrupts    
  P1IE |= BIT3;                // Enable p1 interrupt for p1.3  
  P1IES |= BIT3;               // Interrupt edge select, high-low (falling edge)
  P1IFG = 0;                   // clear p1 interrrupts
  
  _BIS_SR(LPM3_bits + GIE);    // Enter LPM3 with interrupts enabled
    
}
/*************** subroutines ***************/
// delay_time = len of delay, wait_flg = wait for interrupt or not
void interval_delay(int delay_time, char wait_flg)
{
  unsigned int sec_int;                       
  signed long temp = delay_time * cntr_val;     
  // TimerA setup 
  TACTL = TASSEL_1 + ID_3 + TACLR;              // TA = ACLK, divide by 8
  TACCTL0 = CCIE;                              // Enable CCR0 interrupt
  
  // Divide 1 Mhz by counts to get # of VLO counts in period
  sec_int = 0;
  do {
    temp -= vlo_counts;
    sec_int++;
  } while (temp > 0);
  TACCR0 = sec_int;                     // TACCR0 period for delay
 
  // Start timer
  TACTL |= MC_1;                        // Up mode
  
  // wait for timer or not
  if (wait_flg == 1)
  {
    while ((TACCTL0 & CCIFG) == 0);       // Wait for next capture
  }
 
}

unsigned int cal_VLO (void)
{
  unsigned int first_cap, vlo_counts;
  BCSCTL1 |= DIVA_3;                    // Divide ACLK by 8
  TACCTL0 = CM_1 + CCIS_1 + CAP;        // Capture on ACLK
  TACTL = TASSEL_2 + MC_2 + TACLR;      // Start TA, MCLK(DCO), Continuous
  while ((TACCTL0 & CCIFG) == 0);       // Wait until capture
  TACCR0 = 0;                           // Ignore first capture
  TACCTL0 &= ~CCIFG;                    // Clear CCIFG
  while ((TACCTL0 & CCIFG) == 0);       // Wait for next capture
  first_cap = TACCR0;                   // Save first capture
  TACCTL0 &= ~CCIFG;                    // Clear CCIFG
  while ((TACCTL0 & CCIFG) == 0);       // Wait for next capture
  vlo_counts = (TACCR0 - first_cap);    // # of VLO clocks in 1 Mhz
  
  return vlo_counts;
}

void take_photo (void)
{
  if (shot_cnt > 0)
  { 
    // turn off p1 interrupt
    P1IE &= ~BIT3;                           // Disable p1 interrupt for p1.3
    P1IFG = 0;                               // reset all p1 interrupts
    // turn camera on
    turn_on();
    // take photo
    P1OUT |= BIT4;                           // set p1.4 to one - shutter
    interval_delay(1,1);
    P1OUT &= ~BIT4;
    shot_cnt--;
    // turn camera off
    turn_off();
    // if camera full - go to LPM4
    if (shot_cnt == 0 )
    {
      LPM4;
    } 
    // post photos PIR lockout period
    interval_delay (10,1);                // wait 10 sec
    P1IE |= BIT3;                         // Enable p1 interrupt for p1.3
    P1IFG = 0;                            // reset all p1 interrupts
    
  }
}
 
void chg_mode_to_compressed(void)
{
// set mode to compressed high resolution
    // turn off p1 interrupt
    P1IE &= ~BIT3;                       // Disable p1 interrupt for p1.3
    P1IFG = 0;                           // reset all p1 interrupts
    interval_delay(1,1);
    // push mode switch 8 times
    for (int j = 1; j <= 8; j++)       
    {  
      P1OUT |= BIT5;                   // set p1.5 to one - mode switch
      interval_delay(1,1);
      P1OUT &= ~BIT5;                  // set p1.5 to zero
      interval_delay(1,1);
    }
    // push shutter once
    P1OUT |= BIT4;                     // set p1.4 to one - shutter
    interval_delay(1,1);
    P1OUT &= ~BIT4;                    // set p1.4 to zero 
    interval_delay(1,1);
    // turn camera off
    turn_off();
    // PIR lockout period
    interval_delay(15,1);       // 15 sec lockout before next pair of photos
    // turn on p1 interrupt
    P1IE |= BIT3;                      // Enable p1 interrupt for p1.3
    P1IFG = 0;                         // reset all p1 interrupts
    
}   

void turn_on(void)
{
  cntr_val = 60000;                // 60000 ~= 0.5 sec  
  interval_delay(1,1);
  // hit mode switch once
  P1OUT |= BIT5;                   // set p1.5 to one - mode switch
  interval_delay(1,1);
  P1OUT &= ~BIT5;                  // set p1.5 to zero
  interval_delay(1,1);
  cntr_val = 120000;               // 120000 ~= 1 sec
}

void turn_off(void)
{
  interval_delay(1,1);
  // hit mode switch once
  P1OUT |= BIT5;                   // set p1.5 to one - mode switch
  interval_delay(1,1);
  P1OUT &= ~BIT5;                  // set p1.5 to zero
  // hit shutter switch once
  interval_delay(1,1);
  P1OUT |= BIT4;                   // set p1.4 to one - shutter
  interval_delay(1,1);
  P1OUT &= ~BIT4;                  // set p1.4 to zero 
  interval_delay(1,1);
}

/******************* ISR ********************/
// Timer A0 interrupt service routine
#pragma vector=TIMERA0_VECTOR
__interrupt void Timer_A0 (void)
{ 
  P1OUT ^= BIT1;                        // toggle p1.1 - troubleshooting
  if (trip_flg == 1) 
  {
    trip_flg = 0;
  }
  // if band time is reached and p1.3 still 0
  else if ((band_flg == 1) & ((P1IN & BIT3) == 0)) 
  {
    band_flg =0;
    take_photo();
  } 
 
}  
  
// P1.0 interrupt service routine
#pragma vector=PORT1_VECTOR
__interrupt void port_1 (void)
{
    if (trip_flg == 0)
    {
      band_flg = 1;
      // set band time
      cntr_val = 15000;        // 15000 ~= 0.125 sec  
      interval_delay(1,0);
      cntr_val = 120000;       // 120000 ~= 1 sec
    }
    else 
    {
      trip_flg = 1;
    } 
  P1IFG = 0;              // clear p1 interrupts   
   
}
  


