/*
* Includes
*/
#include <stdlib.h>
#include <stdio.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h> 
#include "adelay.h"
#include "lcd.h"
/* 
*  Define the buttons, no need to switch to input as they're all inputs by default
*  DDRX = I/O settings,
*  PORTx = Turn on/off internal pull up resistors
*  PINx = Input register
*/
#define BUTTON_PORT PORTB      			
#define BUTTON_PIN  PINB

#define BTN_SHOOT_PIN 		PINC       
#define BTN_SHOOT_BIT  		PC3       

#define BTN_LEFT_PIN 		PINB    
#define BTN_LEFT_BIT 		PB1    
#define BTN_RIGHT_PIN		PINB  
#define BTN_RIGHT_BIT		PB2        
#define BTN_UP_PIN 			PINB      
#define BTN_UP_BIT 			PB3    
#define BTN_DOWN_PIN 		PINB         
#define BTN_DOWN_BIT 		PB4         
#define BTN_MODE_PIN 		PINB     
#define BTN_MODE_BIT 		PB5          
//Define output ports/bits
#define OUTPUT_PORT PORTD
#define IROUT_BIT PD7

//PortD Output
/****************************************/
/****************************************/
/****** Variable Definitions     ********/
/****************************************/
/****************************************/
uint8_t count; 					//count loop timer for shoot function
volatile uint8_t menu = 0;				//menu mode
uint8_t submenu = 0;			//submenu mode
uint8_t doonce = 0;				//do-once flag
uint8_t xpos = 0;				//x-position on LCD
uint8_t triggerstart = 0;		//trigger start flag
volatile uint8_t trigger_continue = 0;  //interval continue flag
uint8_t intervalstart = 0;		//interval start flag
volatile uint8_t interval_continue = 0;  //interval continue flag
volatile uint8_t showintervalstatus = 0;
volatile uint8_t total_shots = 0;  //User preset number of shots
volatile uint16_t shotcount=0;
volatile uint16_t timeremain=0;
volatile uint16_t millisecond_counter=0;
volatile uint16_t triggercondition=0;
volatile uint16_t interval=0;
volatile uint8_t sleep_counter=0;
volatile uint8_t conditionmet = 0;
uint8_t interval_secunit = 0;	//number of single seconds in interval
uint8_t interval_secten = 0;	//number of tens of seconds in interval
uint8_t interval_minunit = 0;	//number of single minutes in interval
uint8_t interval_minten = 0;	//number of tens of minutes in interval
uint8_t interval_hour = 0;		//number of hours in interval

uint8_t trigger_thousands = 0;	//number of thousands in trigger condition
uint8_t trigger_hundreds = 0;	//number of hundreds in trigger condition
uint8_t trigger_tens = 0;		//number of tens in trigger condition
uint8_t trigger_units = 0;		//number of units in trigger condition

uint8_t delay_hundreds = 0;		//hundreds of milliseconds in delay
uint8_t delay_tens = 0;			//tens of milliseconds in delay
uint8_t delay_units = 0;		//single milliseconds in delay

uint8_t triggered = 0;
volatile uint8_t trigger_operator_select = 0;
char buffer[15];				//buffer for text/numbers for interval conversion, etc
volatile uint16_t ADC_Value;	//current Analog-Digital conversion value

//io init
void init_io(void)
{
	//All Ouputs on
	DDRD = 0xff;
	DDRB |= ( 1 << PB0);
	DDRC |= ( 1 << PC1);
	DDRC |= ( 1 << PC2);
	
	//Turn on pullups
	PORTB |= (1 << PB0);
	PORTB |= (1 << BTN_LEFT_BIT);
	PORTB |= (1 << BTN_UP_BIT);
	PORTB |= (1 << BTN_DOWN_BIT);
	PORTB |= (1 << BTN_RIGHT_BIT);
	PORTB |= (1 << BTN_MODE_BIT);
	PORTC |= (1 << BTN_SHOOT_BIT);
	//Interrupt on INT0
	MCUCR |= (1 << ISC01);
	
}
void timer2_init(void)
{	
	//Set timer 2 to CTC mode
	TCCR2 |= (1 << WGM21);
	//Set the timer 2 compare value (A) to interrupt every 10ms
	OCR2 = 124;
	//Set clock frequency to external crystal, prescaled to 256
	TCCR2 |= (1 << CS22) | (1 << CS21);
	//Set the interrupts to enable on match
	TIMSK |= (1 << OCIE2);
}

void timer2_deinit(void)
{
	//Stop the clock!
	TCCR2 |= (0 << CS22) | (0 << CS21);
}

void ADC_INIT(void)
{
					ADCSRA |= (1 << ADPS2) | (1 << ADPS0); // Set ADC prescaler to 32 - 125kHz sample rate @ 4MHz

					ADMUX |= (1 << REFS0); // Set ADC reference to AVCC

					// No MUX values needed to be changed to use ADC0

					ADCSRA |= (1 << ADEN);  // Enable ADC
	
					ADCSRA |= (1 << ADIE);  // Enable ADC Interrupt
					sei();   // Enable Global Interrupts
}

void ADC_DEINIT(void)
{
					ADCSRA |= ~(1 << ADEN);  // Disable ADC
	
					ADCSRA |= ~(1 << ADIE);  // Disable ADC Interrupt
					cli();   // Disable Global Interrupts
}

void shoot(void)
{
		for(count=0; count <=16; count++)
		{
		OUTPUT_PORT ^= (1 << IROUT_BIT); //turn on
		Delay_us(15.625);
		OUTPUT_PORT ^= (1 << IROUT_BIT); //turn off
		Delay_us(15.625);
		}
		
		Delay_us(7300);
		
		for(count=0; count <=16; count++)
		{
		OUTPUT_PORT ^= (1 << IROUT_BIT); //turn on
		Delay_us(15.625);
		OUTPUT_PORT ^= (1 << IROUT_BIT); //turn off
		Delay_us(15.625);
		}
}

int is_button_pressed(void)
{
	//Check for button press.  Order of precedence is shoot, mode, up, down, left ,right.
	
	//Shoot button pressed
	if((PINC & (1 << BTN_SHOOT_BIT)) == 0)
	{
		Delay_ms(30);
		if((PINC & (1 << BTN_SHOOT_BIT)) == 0)
			return 1;
	}
	//Mode button pressed
	else if((PINB & (1 << BTN_MODE_BIT)) == 0)
	{
		Delay_ms(30);
		if((PINB & (1 << BTN_MODE_BIT)) == 0)
			return 2;
	}
	//Up button pressed
	else if((PINB & (1 << BTN_UP_BIT)) == 0)
	{
		Delay_ms(30);
		if((PINB & (1 << BTN_UP_BIT)) == 0)
			return 3;
	}
	//Down button pressed
	else if((PINB & (1 << BTN_DOWN_BIT)) == 0)
	{
		Delay_ms(30);
		if((PINB & (1 << BTN_DOWN_BIT)) == 0)
			return 4;
	}
	//Left button pressed
	else if((PINB & (1 << BTN_LEFT_BIT)) == 0)
	{
		Delay_ms(30);
		if((PINB & (1 << BTN_LEFT_BIT)) == 0)
			return 5;
	}
	//Right button pressed
	else if((PINB & (1 << BTN_RIGHT_BIT)) == 0)
	{
		Delay_ms(30);
		if((PINB & (1 << BTN_RIGHT_BIT)) == 0)
			return 6;
	}

	return 0;
}
/****************************************/
/****************************************/
/******* MANUAL SHOOT MENU **************/
/****************************************/
/****************************************/
void load_manual_menu(void)
{
	lcd_clrscr();
	//Mode 1 is manual mode
	lcd_puts("Mode: Manual \n");
	lcd_puts("Press shoot");
	//If no buttons are pressed, do nothing
	//Shoot if this option is selected
	while(is_button_pressed() == 0)
	{
	}
		if(is_button_pressed() == 1)
		{
			lcd_clrscr();
			shoot();
			lcd_puts("Shot Taken!");
			Delay_ms(500);
		}
		//Change Mode
		if(is_button_pressed() == 2)
		{
			menu = 1;
			Delay_ms(500);
		}
}
/****************************************/
/****** INTERVAL TIMER FUNCTIONS ********/
/****************************************/
/****************************************/
uint16_t calculateinterval(uint8_t ihour, uint8_t iminten, uint8_t iminunit, uint8_t isecten, uint8_t isecunit)
{
//We want the interval in seconds, note that the maximum possible output for this fuction is (4271*9)=38439
uint16_t iseconds;
iseconds = (ihour * 3600) + (iminten * 10 * 60) + (iminunit * 60) + (isecten * 10) + (isecunit);
return iseconds;
}

void checkintervaltimes(void)
{
	if(interval_hour == 10)
		interval_hour = 0;
	if(interval_minten == 10)
		interval_minten = 0;
	if(interval_minunit == 10)
		interval_minunit = 0;
	if(interval_secunit==10)
		interval_secunit = 0;
	if(interval_secten == 10)
		interval_secten = 0;
											
	if(interval_hour == 255)
		interval_hour = 9;
	if(interval_minten == 255)
		interval_minten = 9;
	if(interval_minunit == 255)
		interval_minunit = 9;
	if(interval_secunit==255)
		interval_secunit = 9;
	if(interval_secten == 255)
		interval_secten = 9;
}

void start_interval_shoot(void)
{
	//Initialise and start timer2, enable sleep modes and set sleep mode to awake on timer compare or pin change, enable global interrupts
	sei();
	shoot();
	shotcount = 1;
	timer2_init();
	//Shooting functionality is done in interrupts
	while(1)
	{
		lcd_clrscr();
		//display the shooting screen
		lcd_puts("Shot No.:0\n");
		lcd_puts("Timer:   0");
		//display remaining seconds
		lcd_gotoxy(9,1);
		sprintf(buffer,"%05d",timeremain);
		lcd_puts(buffer); 
		//display shot count
		lcd_gotoxy(9,0);
		sprintf(buffer,"%05d",shotcount);
		lcd_puts(buffer);
		Delay_ms(500);

		sleep_counter++;
		
		if(is_button_pressed() == 1)
		{
			break;
		}
		if(sleep_counter >= 15)
		{
			lcd_command(LCD_DISP_OFF);
			while(is_button_pressed() == 0)
			{ }
		}
		
		lcd_command(LCD_DISP_ON);
	}
		timer2_deinit();
	cli();	
	GICR &= ~(1 << INT1);
	shotcount = 0;
	interval = 0;
	intervalstart = 0;
	timeremain = 0;
	millisecond_counter = 0;
}

void load_interval_menu(void)
{
	//Cursor on
	lcd_command(LCD_DISP_ON_CURSOR);
	//Mode 2 is interval mode					
	lcd_clrscr();
	lcd_puts("Mode: Interval \n");
	lcd_puts("H:MM:SS  :  :");
	lcd_gotoxy(8,1);
	//put each interval number into a string and output to the LCD
	itoa( interval_hour , buffer, 10);
	lcd_puts(buffer);
	lcd_gotoxy(10,1);
	itoa( interval_minten , buffer, 10);
	lcd_puts(buffer);
	lcd_gotoxy(11,1);
	itoa( interval_minunit , buffer, 10);
	lcd_puts(buffer);
	lcd_gotoxy(13,1);
	itoa( interval_secten , buffer, 10);
	lcd_puts(buffer);
	lcd_gotoxy(14,1);
	itoa( interval_secunit , buffer, 10);
	lcd_puts(buffer);
	//grab the interval, conveniently put into seconds
	interval = calculateinterval(interval_hour, interval_minten, interval_minunit, interval_secten, interval_secunit);
	//do once to set the initial xpos
	if (doonce == 0)
	{
		xpos = 14;
		doonce = 1;
		lcd_gotoxy(xpos,1);
	}
	//If no buttons are pressed, do nothing
	while(is_button_pressed() == 0)
	{
		lcd_gotoxy(xpos,1);
	}
	//If SHOOT pushed, start shooting
	if(is_button_pressed() == 1)
	{
		//initialise variables
		timeremain = interval;
		intervalstart = 1;
		if(interval == 0)
			intervalstart = 0;
		if(intervalstart == 1)
		{
			Delay_ms(500);
			start_interval_shoot();
		}
	}
	//If MODE pushed, change mode to 3 and return
	if(is_button_pressed() == 2)
	{	
		//reset the doonce flag
		doonce = 0;
		menu = 2;
		Delay_ms(500);
	}
	
	if(is_button_pressed() == 3)
	{
		//depending which xposition we are on the screen, increment the relevant value
		switch(xpos)
		{
			case 8: 
				interval_hour++;
				break;
			case 10:
				interval_minten++;
				break;
			case 11:
				interval_minunit++;
				break;
			case 13:
				interval_secten++;
				break;
			case 14:
				interval_secunit++;
				break;
		}
		//let's just double check that nothing is too large or small (0-9 is the valid range)
		checkintervaltimes();
	}
	//DOWN BUTTON
	if(is_button_pressed() == 4)
	{	
		//depending which xposition we are on the screen, decrement the relevant value
		switch(xpos)
		{
			case 8: 
				interval_hour--;
				break;
			case 10:
				interval_minten--;
				break;
			case 11:
				interval_minunit--;
				break;
			case 13:
				interval_secten--;
				break;
			case 14:
				interval_secunit--;
				break;
		}
		//let's just double check that nothing is too large or small (0-9 is the valid range)
		checkintervaltimes();
	}
	//LEFT BUTTON
	if(is_button_pressed() == 5)
	{			
		//go one left, if we get to a semi-colon (i.e. H:MM:SS) go one more, loop if we get to the end
		xpos--;
		if(xpos==12)
			xpos--;
		if(xpos==9)
			xpos--;
		if(xpos==7)
			xpos = 14;
		//update position
		lcd_gotoxy(xpos,1);
	}
	//RIGHT BUTTON
	if(is_button_pressed() == 6)
	{	
		//go one right, if we get to a semi-colon (i.e. H:MM:SS) go one more, loop if we get to the end
		xpos++;
		if(xpos==9)
			xpos++;
		if(xpos==12)
			xpos++;
		if(xpos==15)
			xpos = 8;
		//update position
		lcd_gotoxy(xpos,1);
	}
	return;
	//the menu loop should bring us back here though!
}
/****************************************/
/****** EXTERNAL TRIGGER FUNCTIONS ******/
/****************************************/
/****************************************/
void checktriggertimes(void)
{
	if(trigger_thousands == 2)
		trigger_thousands = 0;
	if(trigger_hundreds == 10)
		trigger_hundreds = 0;
	if(trigger_tens == 10)
		trigger_tens = 0;
	if(trigger_units == 10)
		trigger_units = 0;
	
	if(trigger_thousands == 255)
		trigger_thousands = 0;
	if(trigger_hundreds == 255)
		trigger_hundreds = 9;
	if(trigger_tens == 255)
		trigger_tens = 9;
	if(trigger_units == 255)
		trigger_units = 9;
	
	if(trigger_thousands > 0)
	{
		if(trigger_hundreds > 0)
			trigger_hundreds = 0;
		if(trigger_tens >= 3)
			trigger_tens = 2;
		if(trigger_units >= 5)
			trigger_units = 4;
	}
	
	if(trigger_operator_select == 3)
		trigger_operator_select = 0;
	if(trigger_operator_select == 255)
		trigger_operator_select = 2;
}

void checkdelaytimes(void)
{
	if(delay_hundreds == 10)
		delay_hundreds = 0;
	if(delay_tens == 10)
		delay_tens = 0;
	if(delay_units == 10)
		delay_units = 0;
			
	if(delay_hundreds == 255)
		delay_hundreds = 9;
	if(delay_tens == 255)
		delay_tens = 9;
	if(delay_units == 255)
		delay_units = 9;

}

void waitfortrigger(uint16_t cond)
{	
	ADCSRA |= (1 << ADSC);
	while(triggered == 0)
	{
		lcd_gotoxy(0,0);
		lcd_puts("Waiting...\n");
		lcd_command(LCD_DISP_ON);
		lcd_gotoxy(12,0);
		//format the string to show leading zeroes, so it looks like: Current Val: 0000 on the screen (because we don't clear the screen)
		sprintf(buffer,"%04d",ADC_Value);
		lcd_puts(buffer);
		lcd_puts("Current Val:");
		switch(trigger_operator_select)
		{
		case 0:
			if(ADC_Value > cond)
			triggered = 1;
			break;
		case 1:
			if(ADC_Value == cond)
			triggered = 1;
			break;
		case 2:
			if(ADC_Value < cond)
			triggered = 1;
			break;
		}
	}
	return;
}

void start_triggered_shoot(void)
{
		ADC_INIT();
		//set the trigger and delay variables to what they need to be
		uint16_t totaldelay = delay_hundreds*100+delay_tens*10+delay_units;
		lcd_clrscr();
		//wait for the trigger to work
		waitfortrigger(triggercondition);
		//display an update on the screen
		lcd_puts("Triggered...\n");
		lcd_puts("Delaying...\n");
		
		Delay_ms(totaldelay);
		ADC_DEINIT();
		
		shoot();
		
		lcd_puts("Shot Taken!");
		Delay_ms(1000);
		submenu = 0;
		triggered = 0;
		triggerstart = 0;
		doonce = 0;
		xpos = 12;
		return;
}

void set_delay_menu(void)
{
	//we haven't left the trigger function yet, so do-once should be 1 for the first time this function is called
	if(doonce == 2)
	{
		xpos = 0;
		doonce = 3;
		Delay_ms(400);
	}
	//Set Delay
	lcd_clrscr();
	//Show the cursor
	lcd_command(LCD_DISP_ON_CURSOR);
	lcd_puts("Set Delay (ms): \n");
	lcd_gotoxy(0,1);
	itoa(delay_hundreds , buffer, 10);
	lcd_puts(buffer);
	lcd_gotoxy(1,1);
	itoa( delay_tens , buffer, 10);
	lcd_puts(buffer);
	lcd_gotoxy(2,1);
	itoa( delay_units , buffer, 10);
	lcd_puts(buffer);
	while(is_button_pressed() == 0)
	{
		lcd_gotoxy(xpos,1);
	}
	if(is_button_pressed() == 1)
	{
		triggerstart = 1;
		return;
	}
	//if mode button pressed, go back one screen
	if(is_button_pressed() == 2)
	{	
		//turn off the ADC
		ADC_DEINIT();
		//remember to set the x position back to where we can see it!
		xpos = 12;
		doonce = 1;
		submenu = 1;
		Delay_ms(500);
		return;
	}
	//UP BUTTON
	if(is_button_pressed() == 3)
	{
		switch(xpos)
		{
		case 0: 
			delay_hundreds++;
			break;
		case 1:
			delay_tens++;
			break;
		case 2:
			delay_units++;
			break;
		}	
		checkdelaytimes();
	}
	//DOWN BUTTON
	if(is_button_pressed() == 4)
	{	
		switch(xpos)
		{
		case 0: 
			delay_hundreds--;
			break;
		case 1:
			delay_tens--;
			break;
		case 2:
			delay_units--;
			break;
		}
			
		checkdelaytimes();
	}
	
	//LEFT BUTTON
	if(is_button_pressed() == 5)
	{			
		xpos--;
		if(xpos==255)
			xpos=2;
				
		lcd_gotoxy(xpos,1);
	}
//RIGHT BUTTON
	if(is_button_pressed() == 6)
	{					
		xpos++;
		if(xpos==3)
			xpos=0;
		
		lcd_gotoxy(xpos,1);
	}
	return;
}	

void set_trigger(void)
{
	lcd_command(LCD_DISP_ON_CURSOR);
	lcd_clrscr();
	lcd_puts("Trigger On: ");
	lcd_gotoxy(11,0);
	switch(trigger_operator_select)
	{
		case 0:
			lcd_puts(">");
		case 1:
			lcd_puts("=");
		case 2:
			lcd_puts("<");
	}
	lcd_gotoxy(12,0);
	itoa( trigger_thousands , buffer, 10);
	lcd_puts(buffer);
	lcd_gotoxy(13,0);
	itoa( trigger_hundreds , buffer, 10);
	lcd_puts(buffer);
	lcd_gotoxy(14,0);
	itoa( trigger_tens , buffer, 10);
	lcd_puts(buffer);
	lcd_gotoxy(15,0);
	itoa( trigger_units , buffer, 10);
	lcd_puts(buffer);
	triggercondition = (trigger_thousands*1000)+(trigger_hundreds*100)+(trigger_tens*10)+(trigger_units);
	
	if(doonce == 1)
	{
		xpos = 11;
		doonce = 2;
		lcd_gotoxy(xpos,0);
	}
	while(is_button_pressed() == 0)
	{
		lcd_gotoxy(xpos,0);
	}
	if(is_button_pressed() == 1)
	{
		//go to the sub menu - delay setting
		submenu = 2;
		Delay_ms(500);
		return;
	}
	//MODE Changed
	if(is_button_pressed() == 2)
	{
		doonce = 0;
		submenu = 0;
		Delay_ms(500);
		return;
	}
	//UP BUTTON
	if(is_button_pressed() == 3)
	{
		switch(xpos)
		{
		case 11:
			trigger_operator_select++;
			break;
		case 12: 
			trigger_thousands++;
			break;
		case 13: 
			trigger_hundreds++;
			break;
		case 14:
			trigger_tens++;
			break;
		case 15:
			trigger_units++;
			break;
		}
		
		checktriggertimes();
	}
	
	//DOWN BUTTON
	if(is_button_pressed() == 4)
	{	
		switch(xpos)
		{
		case 11: 
			trigger_operator_select--;
			break;
		case 12: 
			trigger_thousands--;
			break;
		case 13: 
			trigger_hundreds--;
			break;
		case 14:
			trigger_tens--;
			break;
		case 15:
			trigger_units--;
			break;
		}
		
		checktriggertimes();
	}
	//LEFT BUTTON
	if(is_button_pressed() == 5)
	{			
		xpos--;
		if(xpos==10)
			xpos=15;
		lcd_gotoxy(xpos,1);
	}
	
	//RIGHT BUTTON
	if(is_button_pressed() == 6)
	{					
		xpos++;
		if(xpos==16)
			xpos=11;
		lcd_gotoxy(xpos,1);
	}
}

void load_trigger_menu(void)
{
	//Set up the ADC once
	if(doonce == 0)
	{
		ADC_INIT();
		lcd_clrscr();
		lcd_puts("Mode: Trigger \n");
		Delay_ms(500);
		doonce = 1;
	}
	
	ADCSRA |= (1 << ADSC);  // Start A2D Conversions	
	
	//If no buttons are pressed, do nothing
	while(is_button_pressed() == 0)
	{
		lcd_clrscr();
		lcd_puts("Current Val: \n");
		lcd_command(LCD_DISP_ON);
		lcd_gotoxy(12,0);
		//format the string to show leading zeroes, so it looks like: Current Val: 0000 on the screen (because we don't clear the screen)
		sprintf(buffer,"%04d",ADC_Value);
		lcd_puts(buffer);
		lcd_gotoxy(xpos,1);
		Delay_ms(100);
	}
	if(is_button_pressed() == 1)
	{
		//go to the sub menu - delay setting
		submenu = 1;
		Delay_ms(500);
		return;
	}
	//MODE Changed
	if(is_button_pressed() == 2)
	{
		doonce = 0;
		menu = 0;
		submenu = 0;
		ADC_DEINIT();
		//turn off the LEDs
		PORTC &= ~(1 << PC2);
		PORTC &= ~(1 << PC1);
		Delay_ms(500);
		return;
	}
}

int main(void)
{
	init_io();
	OUTPUT_PORT &= ~(1 << IROUT_BIT);
	lcd_init(LCD_DISP_ON);
	
	lcd_clrscr();
	lcd_puts(" Software v1.00\n");
	lcd_puts(" By Whiternoise \n");
	Delay_ms(2000);
	lcd_clrscr();
	
while(1)
{
while(is_button_pressed() == 0)
{
	//Functions should set the value for menu and submenu before they finish - i.e. the menu option will remain the same
	switch(menu)
	{
		/* This is the first menu, manual triggering */
		case 0:
			load_manual_menu();
		break;
		
		/* This is the second menu, interval timing */
		case 1:
			load_interval_menu();
		break;
		
		case 2:
			if(submenu == 2)
			{
				if(triggerstart == 1)
				{
				start_triggered_shoot();
				submenu = 0;
				break;
				}
				
				set_delay_menu();
			}
			if(submenu == 1)
			{
				set_trigger();
			}
			if(submenu == 0)
			{
				load_trigger_menu();
			}
		break;
		
		default:
			load_manual_menu();
		break;
	}
}
}
return 1;
}
// Run when timer 2 compares at a rate of 15625Hz = 125^2, CTC of 125 gives an interrupt frequency of 125Hz, thus setting a second "wraparound" timer of 125 gives us 1Hz operator we want
ISR(TIMER2_COMP_vect)
{
	//1/125 milliseconds
	millisecond_counter++;
	//Run every second
	if(millisecond_counter == 125)
	{
		millisecond_counter = 0;
			//decrease the time remaining
			timeremain--;
			//if the time remaining is zero...
			if(timeremain <= 0)
			{
				//take a picture, increase shot count and reset the countdown to interval
				shoot();
				shotcount++;
				timeremain = interval;
			}
	}
}
//Run when ADC finishes a conversion
ISR(ADC_vect)
{
   ADC_Value = ADC;
   ADCSRA |= (1 << ADSC);  // Start A2D Conversions
   
   switch(trigger_operator_select)
    {
		case 0:
		if(ADC_Value > triggercondition)
			conditionmet = 1;
		else
			conditionmet = 0;
		case 1:
		if(ADC_Value == triggercondition)
			conditionmet = 1;
		else
			conditionmet = 0;
		case 2:
		if(ADC_Value < triggercondition)
			conditionmet = 1;
		else
			conditionmet = 0;
    }
    if(conditionmet == 1)
    {
	PORTC |= (1 << PC1); // Turn on LED1
	PORTC &= ~(1 << PC2); // Turn off LED2
    }
    else if(conditionmet == 0)
	{
	PORTC	  &= ~(1 << PC1); // Turn off LED1
	PORTC	  |= (1 << PC2); // Turn on LED2
	}
}
