/*******************************************************************************
* Title: LCD_Setup.c                                                           *
* Version: 1.0                                                                 *
* Author: Jeff Nybo                                                            *
* Date: January 10, 2011                                                       *
*                                                                              *
* Description:                                                                 *
* This file contains all the functions that are used to initialize an HD44780  *
* (or compatible model) Display and operate it correctly.                      *
*******************************************************************************/

/*******************************************************************************
* Function: LCD_Init(void)                                                     * 
*                                                                              *
* Variables:                                                                   *
* N/A                                                                          *
*                                                                              *
* Description:                                                                 *
* This function will initialize the LCD Display and set it up for operation as *
* specified by the user.                                                       *
*******************************************************************************/
void LCD_Init(void)
{
  _Delay_ms(20);  //Allow for LCD to stabilize
  
  LCD_CMD(0x28);  //4-Bit Data Interface, 2 Line Display, 5x8 Font
  LCD_CMD(0x0C);  //Automatically shift cursor right
  LCD_CMD(0x06);  //Display ON, No Cursor, No Blinking of Cursor  
  LCD_CLEAR;      //Clear LCD Display   
}  

/*******************************************************************************
* Function: LCD_CMD(byte command)                                              * 
*                                                                              *
* Variables:                                                                   *
* command -> Stores the value of the command that is to be sent to the LCD     *
*                                                                              *
* Description:                                                                 *
* This function will send a command to the LCD Display, it is used for setting *
* up and controlling the operations of the LCD Display.                        *
*******************************************************************************/
void LCD_CMD(byte command)
{
 PORTB = 0xF0 & command; //Set PORTB equal to the high nibble of command
 LCD_DC = 0;  //Instruct LCD that a command will be written
 LCD_RW = 0;  //Prepare the LCD to receive data
 
 LCD_PULSE;   //Send the high nibble of command to the LCD
 
 PORTB = (0x0F & command) << 4; //Set PORTB equal to the low nibble of command
 
 LCD_PULSE;   //Send the low nibble of command to the LCD
 
 _Delay_us(40); //Allow the LCD enough time to process the command
}  

/*******************************************************************************
* Function: LCD_Send(byte data)                                                * 
*                                                                              *
* Variables:                                                                   *
* data -> Stores the value of the data that is to be sent to the LCD           *
*                                                                              *
* Description:                                                                 *
* This function is used for sending data to the LCD, whatever data is sent to  *
* the LCD will be displayed in ASCII format at whatever location the cursor    *
* has been positioned at.                                                      *
*******************************************************************************/
void LCD_Send(byte data)
{ 
 PORTB = 0xF0 & data; //Set PORTB equal to the high nibble of data
 LCD_DC = 1;  //Instruct LCD that data will be written
 LCD_RW = 0;  //Prepare the LCD to receive data
 
 LCD_PULSE;   //Send the high nibble of data to the LCD
 
 PORTB = (0x0F & data) << 4; //Set PORTB equal to the low nibble of data
 
 LCD_PULSE;   //Send the low nibble of data to the LCD
 
 _Delay_us(40); //Allow the LCD enough time to process the command
} 

/*******************************************************************************
* Function: LCD_Pos(byte x, byte y)                                            * 
*                                                                              *
* Variables:                                                                   *
* x -> Stores the x-coordinate of the location of the cursor                   *
* y -> Stores the y-coordinate of the position of the cursor                   *
*                                                                              *
* Description:                                                                 *
* This finction will move the cursor to the xy coordinate as specified by the  *
* user. It will convert the xy coordinates into one command and send it to the *
* LCD Display.                                                                 *
*******************************************************************************/
void LCD_Pos(byte x, byte y)
{
  byte pos;
  
  pos = (y == 1) ? 0x80 : 0xC0;   //Get the y-coordinate and store it in pos
  pos |= (x - 1);   //Get the x-coordinate and store it in pos
  
  LCD_CMD(pos);   //Set the cursor to the desired location
}

/*******************************************************************************
* Function: LCD_vString(char str[])                                            * 
*                                                                              *
* Variables:                                                                   *
* str[] -> Stores the value of the variable that is to be displayed            *
*                                                                              *
* Description:                                                                 *
* This function will display the char string variable that is passed into it   *
* on the LCD Display.                                                          *
*******************************************************************************/
void LCD_vString(char str[])
{
 byte max = 0;
 byte i;
 
 while (str[max] != '\0') //Loop until the end of the string has been reached
 {
  max++; //Get the length of the string
 }  
 
 for (i = 0;i < max;i++) //Loop until the end of the string is reached
 {
  LCD_Send(str[i]); //Display each character in the string on the LCD
 }  
}

/*******************************************************************************
* Function: LCD_String(auto const far rom char *str)                           * 
*                                                                              *
* Variables:                                                                   *
* *str -> Points to the constant string that is passed into the function       *
*                                                                              *
* Description:                                                                 *
* This function will display a constant string for the user on the LCD. I have *
* not yet figured out how to combine this and the prior function to be used as *
* one. Sort of like a printf statement.                                        *
*******************************************************************************/
void LCD_String(auto const far rom char *str)
{
 byte max = 0;
 byte i;
 
 while (str[max] != '\0') //Loop until the end of the string has been reached
 {
  max++; //Get the length of the string
 }  
 
 for (i = 0;i < max;i++)  //Loop until the end of the string is reached
 {
  LCD_Send(str[i]); //Display each character in the string on the LCD
 }  
}  
