/*=========================================================================
   This file is part of the Cardboard Robot SDK.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import "CBDofVector.h"
#import "CBRobot.h"
#import "CBMotorParameters.h"

@implementation CBDofVector

@synthesize m1;
@synthesize m2;
@synthesize m3;

/** Returns a CBDofVector object initialized with the specified motor
 *  positions (M1, M2, M3).
 */
- (id)initWithM1:(double)m1Pos andM2:(double)m2Pos andM3:(double)m3Pos {
    self = [super init];
    if (self) {
        m1 = m1Pos;
        m2 = m2Pos;
        m3 = m3Pos;
    }
    return self;
}

/** Returns a vector initialized to the specified motor positions (M1, M2, M3). */
+ (CBDofVector *)vectorWithM1:(double)m1Pos andM2:(double)m2Pos andM3:(double)m3Pos {
    return [[[CBDofVector alloc] initWithM1:m1Pos andM2:m2Pos andM3:m3Pos] autorelease];
}

/** Returns a vector initialized to (0, 0, 0) */
+ (CBDofVector *)zero {
    return [CBDofVector vectorWithM1:0 andM2:0 andM3:0];
}

/** Returns the value of the component with the given index (0, 1, or 2) */
- (double)componentWithIndex:(int)component {
    switch (component) {
        case 0:
            return m1;
        case 1:
            return m2;
        case 2:
            return m3;
        default:
            [NSException raise:@"CBBadArgument" format:@"Component index must be between 0 and 2"];
            return NAN;
    }
}

/** Returns a vector with the specified component (0, 1, or 2) set to the given
 *  value.
 */
- (CBDofVector *)setComponentWithIndex:(int)component toValue:(double)value {
    switch (component) {
        case 0:
            return [CBDofVector vectorWithM1:value andM2:m2 andM3:m3];
        case 1:
            return [CBDofVector vectorWithM1:m1 andM2:value andM3:m3];
        case 2:
            return [CBDofVector vectorWithM1:m1 andM2:m2 andM3:value];
        default:
            [NSException raise:@"CBBadArgument" format:@"Component index must be between 0 and 2"];
            return nil;
    }
}

// Inherited from CBVector
- (CBDofVector *)pointAsDofVectorForRobot:(CBRobot *)robot {
    return self;    // (no conversion needed)
}

/** Returns a string representation of this point */
- (NSString *)description {
    return [NSString stringWithFormat:@"(%.06f, %.06f, %.06f)", m1, m2, m3];
}

@end
