/*=========================================================================
   This file is part of the Cardboard Robot SDK.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import "CBArmSpeed.h"

@implementation CBArmSpeed

@synthesize m1Speed;
@synthesize m2Speed;
@synthesize m3Speed;
@synthesize m4Speed;

/** Returns a CBArmSpeed object initialized with the specified speed for each
 *  motor (in radians per second).
 */
- (id)initWithM1Speed:(double)m1
           andM2Speed:(double)m2
           andM3Speed:(double)m3
           andM4Speed:(double)m4 {
    
    if (m1 < 0 || m2 < 0 || m3 < 0 || m4 < 0) {
        [NSException raise:@"CBBadArgument" format:@"One or more speed components was negative"];
    }
    
    self = [super init];
    if (self) {
        m1Speed = m1;
        m2Speed = m2;
        m3Speed = m3;
        m4Speed = m4;
    }
    return self;
}

/** Returns a CBArmSpeed object with the specified speed for each motor
 *  (in radians per second).
 */
+ (CBArmSpeed *)armSpeedWithM1Speed:(double)m1
                         andM2Speed:(double)m2
                         andM3Speed:(double)m3
                         andM4Speed:(double)m4 {
    
    return [[[CBArmSpeed alloc] initWithM1Speed:m1 andM2Speed:m2 andM3Speed:m3 andM4Speed:m4] autorelease];
}

/** Returns a CBArmSpeed object that represents a speed of zero */
+ (CBArmSpeed *)zero {
    return [CBArmSpeed armSpeedWithM1Speed:0 andM2Speed:0 andM3Speed:0 andM4Speed:0];
}

/** Returns a string description of this CBArmSpeed object */
- (NSString *)description {
    return [NSString stringWithFormat:@"(%.06f, %.06f, %.06f, %.06f)", m1Speed, m2Speed, m3Speed, m4Speed];
}

/** Returns the value of the component with the given index (0, 1, 2,
 *  or 3) */
- (double)componentWithIndex:(int)component {
    switch (component) {
        case 0:
            return m1Speed;
        case 1:
            return m2Speed;
        case 2:
            return m3Speed;
        case 3:
            return m4Speed;
        default:
            [NSException raise:@"CBBadArgument" format:@"Component must be between 0 and 2"];
            return NAN;
    }
}

/** Returns a CBArmSpeed with the specified component (0, 1, 2, 3) set
 *  to the given value.
 */
- (CBArmSpeed *)setComponentWithIndex:(int)component toValue:(double)value {
    if (component < 0 || component > 3) {
        [NSException raise:@"CBBadArgument" format:@"Component must be between 0 and 2"];
    }
    if (value < 0) {
        [NSException raise:@"CBBadArgument" format:@"Attempted to set a speed component to a negative value"];
    }
    double m1 = component == 0 ? value : m1Speed;
    double m2 = component == 1 ? value : m2Speed;
    double m3 = component == 2 ? value : m3Speed;
    double m4 = component == 3 ? value : m4Speed;
    return [CBArmSpeed armSpeedWithM1Speed:m1 andM2Speed:m2 andM3Speed:m3 andM4Speed:m4];
}

@end
