/*=========================================================================
   This file is part of the Cardboard Robot SDK.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import "CBArmPosition.h"
#import "CBDofVector.h"
#import "CBMotorParameters.h"
#import "CBRobot.h"

@implementation CBArmPosition

@synthesize tipPosition;
@synthesize m4;

- (void)dealloc {
    [tipPosition release]; tipPosition = nil;
    [super dealloc];
}

/** Returns a CBArmPosition object initialized with the specified tip position
 *  and motor #4 position (in radians).
 */
- (id)initWithTipPosition:(id <CBVector>)tipPos andM4:(double)m4Pos {
    self = [super init];
    if (self) {
        tipPosition = [tipPos retain];
        m4 = m4Pos;
    }
    return self;
}

/** Returns a CBArmPosition object initialized with the given motor positions
 *  (specified in radians).
 */
- (id)initWithM1:(double)m1Pos andM2:(double)m2Pos andM3:(double)m3Pos andM4:(double)m4Pos {
    return [self initWithTipPosition:[CBDofVector vectorWithM1:m1Pos andM2:m2Pos andM3:m3Pos] andM4:m4Pos];
}

/** Returns a CBArmPosition object with the specified tip position and motor #4
 *  position (in radians).
 */
+ (CBArmPosition *)armPositionWithTipPosition:(id <CBVector>)tipPos andM4:(double)m4Pos {
    return [[[CBArmPosition alloc] initWithTipPosition:tipPos andM4:m4Pos] autorelease];
}

/** Returns a CBArmPosition object initialized with the given motor positions
 *  (specified in radians).
 */
+ (CBArmPosition *)armPositionWithM1:(double)m1Pos andM2:(double)m2Pos andM3:(double)m3Pos andM4:(double)m4Pos {
    return [self armPositionWithTipPosition:[CBDofVector vectorWithM1:m1Pos andM2:m2Pos andM3:m3Pos] andM4:m4Pos];
}

/** Returns a CBArmPosition object which represents the home position / zero */
+ (CBArmPosition *)zero {
    return [CBArmPosition armPositionWithTipPosition:[CBDofVector zero] andM4:0];
}

/** Returns a string representation of this position */
- (NSString *)description {
    return [NSString stringWithFormat:@"(tip:%@, m4:%.06f)", tipPosition, m4];
}

@end
