/*=========================================================================
   This file is part of the Cardboard Robot Console application.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import "CBJoystickSetupWindowController.h"
#import "CBJoystickAxisViewController.h"
#import "CBJoystickButtonViewController.h"
#import "CBJoystickManager.h"

/* Private members */
@interface CBJoystickSetupWindowController() {
    NSMutableArray *viewControllers;    // Array of view controllers for each view in the list
    DDHidJoystick *selectedJoystick;    // The currently selected joystick
    
    IBOutlet NSView *scrolledView;      // The view within the NSScrollView
    IBOutlet NSArrayController *joysticksController;    // The array of joysticks displayed in the combo box
    IBOutlet NSComboBox *joysticksBox;  // The joysticks combo box
}

- (void)setSelectedJoystick:(DDHidJoystick *)joystick;
- (void)refreshJoystickProperties;

- (IBAction)selectedJoystickChanged:(id)sender;
- (IBAction)refreshJoysticks:(id)sender;

@end


@implementation CBJoystickSetupWindowController

- (void)dealloc {
    [viewControllers release]; viewControllers = nil;
    [super dealloc];
}


/** Called when the window is loaded */
- (void)windowDidLoad {
    [super windowDidLoad];
    
    viewControllers = [[NSMutableArray alloc] init];
    [self refreshJoysticks:nil];
}

/** Called when the window is about to close */
- (void)windowWillClose:(NSNotification *)notification {
    [[NSNotificationCenter defaultCenter] postNotificationName:CB_NOTIFICATION_JOYSTICK_SETUP_WINDOW_WILL_CLOSE object:self];
}

/** Sets the joystick currently being configured */
- (void)setSelectedJoystick:(DDHidJoystick *)joystick {
    if (joystick != selectedJoystick) {
        [selectedJoystick release];
        selectedJoystick = [joystick retain];
        [self refreshJoystickProperties];
    }
}

/** Generates, or re-generates the properties for the currently selected joystick */
- (void)refreshJoystickProperties {
    
    // Remove all old subviews
    for (NSViewController *viewController in viewControllers) {
        [[viewController view] removeFromSuperview];
    }
    [viewControllers removeAllObjects];
    
    // Create subviews for each joystick axis
    for (int stick = 0; stick < [selectedJoystick countOfSticks]; stick ++) {
        DDHidJoystickStick *stickObj = [selectedJoystick objectInSticksAtIndex:stick];
        int axisCount = [stickObj countOfStickElements] + ([stickObj xAxisElement] ? 1 : 0) + ([stickObj yAxisElement] ? 1 : 0);
        for (int axis = 0; axis < axisCount; axis ++) {
            CBJoystickAxisViewController *c = [[[CBJoystickAxisViewController alloc] initWithNibName:@"JoystickAxisView" bundle:nil andStick:stick andAxis:axis forJoystick:selectedJoystick] autorelease];
            [viewControllers addObject:c];
        }
    }
    
    // Create subviews for each joystick button
    for (int button = 0; button < [selectedJoystick numberOfButtons]; button ++) {
        CBJoystickButtonViewController *c = [[[CBJoystickButtonViewController alloc] initWithNibName:@"JoystickButtonView" bundle:nil andButton:button forJoystick:selectedJoystick] autorelease];
        [viewControllers addObject:c];
    }
    
    // Load each view so we can determine its size, etc.
    NSMutableArray *views = [NSMutableArray array];
    for (NSViewController *c in viewControllers) {
        [c loadView];
        [views addObject:[c view]];
    }
    
    // Compute the total size of the scrolling frame's contents
    // (this needs to be done before adding the subviews, or they will clip)
    NSSize totalSize = NSMakeSize([[views objectAtIndex:0] frame].size.width, 0);
    for (NSView *view in views) {
        totalSize.height += [view frame].size.height;
    }
    [scrolledView setFrame:NSMakeRect(0, 0, totalSize.width, totalSize.height)];
    
    // Position each subview in the scrolling frame
    NSPoint nextOrigin = NSMakePoint(0, 0);
    for (NSView *view in views) {
        [view setFrameOrigin:nextOrigin];
        [scrolledView addSubview:view];
        nextOrigin.y += [view frame].size.height;
     
        // Place a separator between rows
        NSBox *b = [[[NSBox alloc] initWithFrame:NSMakeRect(0, nextOrigin.y, totalSize.width, 1)] autorelease];
        [b setBoxType:NSBoxCustom];
        [b setBorderColor:[NSColor controlShadowColor]];
        [scrolledView addSubview:b];
    }
}

/** Called when a new joystick is selected in the combo box */
- (IBAction)selectedJoystickChanged:(id)sender {
    NSInteger selectedIndex = [joysticksBox indexOfSelectedItem];
    if (selectedIndex >= 0) {
        DDHidJoystick *joystick = [[joysticksController arrangedObjects] objectAtIndex:selectedIndex];
        [self setSelectedJoystick:joystick];
    }
}

/** Called when the "refresh joysticks" button is clicked */
- (IBAction)refreshJoysticks:(id)sender {
    [[CBJoystickManager sharedInstance] refreshJoysticks];
    
    // Remove the old joystick entries
    [joysticksController removeObjectsAtArrangedObjectIndexes:[NSIndexSet indexSetWithIndexesInRange:NSMakeRange(0, [[joysticksController arrangedObjects] count])]];
    
    // Add the new joystick entries
    [joysticksController addObjects:[[CBJoystickManager sharedInstance] joysticks]];
    
    if ([joysticksBox indexOfSelectedItem] < 0 && [joysticksBox numberOfItems] > 0) {
        [joysticksBox selectItemAtIndex:0];
        [self selectedJoystickChanged:self];
    }
}

@end
