/*=========================================================================
   This file is part of the Cardboard Robot Console application.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import <CoreFoundation/CoreFoundation.h>
#import "CBJoystickManager.h"
#import "CBJoystickListener.h"


/* Private members */
@interface CBJoystickManager() {
    NSMutableArray *listeners;  // NON-RETAINING array of listeners
}

@end


@implementation CBJoystickManager

@synthesize joysticks;

- (void)dealloc {
    // (NOTE: don't call stopListening - it crashes for disconnected devices)
    [joysticks makeObjectsPerformSelector:@selector(setDelegate:) withObject:nil];

    [listeners release]; listeners = nil;
    [joysticks release]; joysticks = nil;
    [super dealloc];
}

- (id)init {
    self = [super init];
    if (self) {
        // Create a non-retaining array of listeners
        CFArrayCallBacks callbacks = { 0, NULL, NULL, CFCopyDescription, CFEqual };
        listeners = (NSMutableArray *)CFArrayCreateMutable(NULL, 0, &callbacks);
        if (listeners == nil) { NSAssert(NO, @"Failed to allocate listener object"); return nil; }
    }
    return self;
}

/** Returns the shared Joystick Manager instance */
+ (CBJoystickManager *)sharedInstance {
    static CBJoystickManager *sharedInstance = nil;
    @synchronized(self) {
        if (sharedInstance == nil) {
            sharedInstance = [[CBJoystickManager alloc] init];
        }
        return sharedInstance;
    }
}

/** Registers a new listener for joystick events.  This method does NOT
 *  retain the listener, so be sure to call removeListener before the
 *  listener object is deallocated.
 */
- (void)addJoystickListener:(id <CBJoystickListener>)listener {
    [listeners addObject:listener];
}

/** Un-registers the specified listener */
- (void)removeJoystickListener:(id <CBJoystickListener>)listener {
    [listeners removeObject:listener];
}

/** Refreshes the list of joysticks */
- (void)refreshJoysticks {
    
    // Release the old joysticks
    // (NOTE: don't call stopListening - it crashes for disconnected devices)
    [joysticks makeObjectsPerformSelector:@selector(setDelegate:) withObject:nil];
    [joysticks release];
    
    // Start listening to the new ones
    joysticks = [[DDHidJoystick allJoysticks] retain];
    [joysticks makeObjectsPerformSelector:@selector(setDelegate:) withObject:self];
    [joysticks makeObjectsPerformSelector:@selector(startListening)];
}

/** Called when the x-axis of a stick is changed */
- (void)ddhidJoystick:(DDHidJoystick *)joystick
                stick:(unsigned int)stick
             xChanged:(int)value {
    
    for (id <CBJoystickListener> object in listeners) {
        if ([object respondsToSelector:@selector(joystick:stick:axis:valueChanged:)]) {
            [object joystick:joystick stick:stick axis:0 valueChanged:value / (double)DDHID_JOYSTICK_VALUE_MAX];
        }
    }
}

/** Called when the y-axis of a stick is changed */
- (void)ddhidJoystick:(DDHidJoystick *)joystick
                stick:(unsigned int)stick
             yChanged:(int)value {
    
    // (Offset the axis number by any x axis present)
    DDHidJoystickStick *stickObj = [joystick objectInSticksAtIndex:stick];
    int axis = [stickObj xAxisElement] ? 1 : 0;
    
    for (id <CBJoystickListener> object in listeners) {
        
        if ([object respondsToSelector:@selector(joystick:stick:axis:valueChanged:)]) {
            [object joystick:joystick stick:stick axis:axis valueChanged:value / (double)DDHID_JOYSTICK_VALUE_MAX];
        }
    }
}

/** Called when an axis other than x or y on a stick is changed */
- (void)ddhidJoystick:(DDHidJoystick *)joystick
                stick:(unsigned int)stick
            otherAxis:(unsigned int)axis
         valueChanged:(int)value {
    
    // (Offset the axis number by any x or y axis present)
    DDHidJoystickStick *stickObj = [joystick objectInSticksAtIndex:stick];
    if ([stickObj xAxisElement]) { axis += 1; }
    if ([stickObj yAxisElement]) { axis += 1; }
    
    for (id <CBJoystickListener> object in listeners) {
        
        if ([object respondsToSelector:@selector(joystick:stick:axis:valueChanged:)]) {
            [object joystick:joystick stick:stick axis:axis valueChanged:value / (double)DDHID_JOYSTICK_VALUE_MAX];
        }
    }
}

/** Called when a joystick button is pressed */
- (void)ddhidJoystick:(DDHidJoystick *)joystick
           buttonDown:(unsigned int)buttonNumber {

    for (id <CBJoystickListener> object in listeners) {
        
        if ([object respondsToSelector:@selector(joystick:buttonDown:)]) {
            [object joystick:joystick buttonDown:buttonNumber];
        }
    }
}

/** Called when a joystick button is released */
- (void)ddhidJoystick:(DDHidJoystick *)joystick
             buttonUp:(unsigned int)buttonNumber {
    
    for (id <CBJoystickListener> object in listeners) {
        
        if ([object respondsToSelector:@selector(joystick:buttonUp:)]) {
            [object joystick:joystick buttonUp:buttonNumber];
        }
    }
}

@end
