/*=========================================================================
   This file is part of the Cardboard Robot Console application.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import "CBJoystickButtonViewController.h"
#import "CBJoystickManager.h"
#import "CBJoystickSettings.h"

/* Private members */
@interface CBJoystickButtonViewController() {
    DDHidJoystick *representedJoystick;     // The joystick this controller represents
    int representedButton;                  // The button this controller represents
    NSMutableArray *commands;               // Array of commands, corresponding to menu items
    
    IBOutlet NSButton *buttonButton;
    IBOutlet NSTextField *buttonLabel;
    IBOutlet NSPopUpButton *popupButton;
}

- (void)addCommand:(NSString *)command withTitle:(NSString *)title ;

- (IBAction)commandChanged:(id)sender;
- (IBAction)buttonClicked:(id)sender;

@end


@implementation CBJoystickButtonViewController

- (void)dealloc {
    [[CBJoystickManager sharedInstance] removeJoystickListener:self];
    
    [representedJoystick release]; representedJoystick = nil;
    [commands release]; commands = nil;
    
    [super dealloc];
}

- (id)init {
    [NSException raise:@"CBBadCall" format:@"Wrong init method used"];
    return nil;
}

- (id)initWithCoder:(NSCoder *)aDecoder {
    [NSException raise:@"CBBadCall" format:@"Wrong init method used"];
    return nil;
}

- (id)initWithNibName:(NSString *)nibNameOrNil bundle:(NSBundle *)nibBundleOrNil {
    [NSException raise:@"CBBadCall" format:@"Wrong init method used"];
    return nil;
}

- (id)initWithNibName:(NSString *)nibNameOrNil bundle:(NSBundle *)nibBundleOrNil andButton:(int)button forJoystick:(DDHidJoystick *)joystick {
    self = [super initWithNibName:nibNameOrNil bundle:nibBundleOrNil];
    if (self) {
        representedJoystick = [joystick retain];
        representedButton = button;
        [[CBJoystickManager sharedInstance] addJoystickListener:self];
        
        commands = [[NSMutableArray alloc] init];
    }
    return self;
}

- (void)awakeFromNib {
    [super awakeFromNib];
    [buttonLabel setStringValue:[NSString stringWithFormat:@"Button %i", representedButton]];
    
    [self addCommand:nil withTitle:@"Does Nothing"];
    [self addCommand:CB_JOYSTICK_COMMAND_INCREASE_M1 withTitle:@"Increases M1"];
    [self addCommand:CB_JOYSTICK_COMMAND_DECREASE_M1 withTitle:@"Decreases M1"];
    [self addCommand:CB_JOYSTICK_COMMAND_INCREASE_M2 withTitle:@"Increases M2"];
    [self addCommand:CB_JOYSTICK_COMMAND_DECREASE_M2 withTitle:@"Decreases M2"];
    [self addCommand:CB_JOYSTICK_COMMAND_INCREASE_M3 withTitle:@"Increases M3"];
    [self addCommand:CB_JOYSTICK_COMMAND_DECREASE_M3 withTitle:@"Decreases M3"];
    [self addCommand:CB_JOYSTICK_COMMAND_INCREASE_M4 withTitle:@"Increases M4"];
    [self addCommand:CB_JOYSTICK_COMMAND_DECREASE_M4 withTitle:@"Decreases M4"];
    
    // Obtain the current command from the joystick settings
    NSString *currentCommand = [[CBJoystickSettings sharedInstance] commandForButton:representedButton ofJoystick:representedJoystick];
    if (currentCommand != nil) {
        NSUInteger index = [commands indexOfObject:currentCommand];
        if (index != NSNotFound) {
            [popupButton selectItemAtIndex:index];
        }
        else {
            [popupButton selectItemAtIndex:0];
        }
    }
}

/** Adds a command as an option on the pop-up button */
- (void)addCommand:(NSString *)command withTitle:(NSString *)title {
    NSMenuItem *menuItem = [[[NSMenuItem alloc] initWithTitle:title action:nil keyEquivalent:@""] autorelease];
    [menuItem setTag:[commands count]]; // (tag = index in command array)
    if (command == nil) {
        [commands addObject:[NSNull null]];
    }
    else {
        [commands addObject:command];
    }
    [[popupButton menu] addItem:menuItem];
}

/** Called when a new command is selected */
- (IBAction)commandChanged:(id)sender {
    // Save the command that was selected
    if ([popupButton selectedTag] >= 0) {
        id command = [commands objectAtIndex:[popupButton selectedTag]];
        if (command == [NSNull null]) { command = nil; }
        [[CBJoystickSettings sharedInstance] setCommand:command forButton:representedButton ofJoystick:representedJoystick];
    }
}

/** Called when the user clicks the button */
- (IBAction)buttonClicked:(id)sender {
    
    // If the user clicks the button, return immediately to the off state
    [buttonButton setState:NSOffState];
}

/** Called when a joystick button is pressed */
- (void)joystick:(DDHidJoystick *)joystick
      buttonDown:(unsigned)buttonNumber {
    
    if (joystick == representedJoystick && buttonNumber == representedButton) {
        [buttonButton setState:NSOnState];
    }
}

/** Called when a joystick button is released */
- (void)joystick:(DDHidJoystick *)joystick
        buttonUp:(unsigned)buttonNumber {
    
    if (joystick == representedJoystick && buttonNumber == representedButton) {
        [buttonButton setState:NSOffState];
    }
}

@end
