/*=========================================================================
   This file is part of the Cardboard Robot Console application.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import "CBJoystickAxisViewController.h"
#import "CBJoystickManager.h"
#import "CBJoystickSettings.h"

/* Private members */
@interface CBJoystickAxisViewController() {
    DDHidJoystick *representedJoystick; // The joystick this controller represents
    int representedStick;               // The stick this controller represents
    int representedAxis;                // The axis this controller represents
    NSMutableArray *commands;           // Array of commands, corresponding to menu items
    
    IBOutlet NSLevelIndicator *levelIndicator;
    IBOutlet NSTextField *axisLabel;
    IBOutlet NSPopUpButton *popupButton;
}

- (void)addCommand:(NSString *)command withTitle:(NSString *)title;
- (IBAction)commandChanged:(id)sender;

@end


@implementation CBJoystickAxisViewController

- (void)dealloc {
    [[CBJoystickManager sharedInstance] removeJoystickListener:self];
    
    [representedJoystick release]; representedJoystick = nil;
    [commands release]; commands = nil;
    
    [super dealloc];
}

- (id)init {
    [NSException raise:@"CBBadCall" format:@"Wrong init method used"];
    return nil;
}

- (id)initWithCoder:(NSCoder *)aDecoder {
    [NSException raise:@"CBBadCall" format:@"Wrong init method used"];
    return nil;
}

- (id)initWithNibName:(NSString *)nibNameOrNil bundle:(NSBundle *)nibBundleOrNil {
    [NSException raise:@"CBBadCall" format:@"Wrong init method used"];
    return nil;
}

- (id)initWithNibName:(NSString *)nibNameOrNil bundle:(NSBundle *)nibBundleOrNil andStick:(int)stickNumber andAxis:(int)axisNumber forJoystick:(DDHidJoystick *)joystick {
    
    self = [super initWithNibName:nibNameOrNil bundle:nibBundleOrNil];
    if (self) {
        representedStick = stickNumber;
        representedAxis = axisNumber;
        representedJoystick = [joystick retain];
        [[CBJoystickManager sharedInstance] addJoystickListener:self];
        
        commands = [[NSMutableArray alloc] init];
    }
    return self;
}

- (void)awakeFromNib {
    [super awakeFromNib];
    [axisLabel setStringValue:[NSString stringWithFormat:@"Stick %i: Axis %i", representedStick, representedAxis]];
    
    [self addCommand:nil withTitle:@"Does Nothing"];
    [self addCommand:CB_JOYSTICK_COMMAND_CONTROL_M1 withTitle:@"Controls M1"];
    [self addCommand:CB_JOYSTICK_COMMAND_CONTROL_M2 withTitle:@"Controls M2"];
    [self addCommand:CB_JOYSTICK_COMMAND_CONTROL_M3 withTitle:@"Controls M3"];
    [self addCommand:CB_JOYSTICK_COMMAND_CONTROL_M4 withTitle:@"Controls M4"];
    
    // Obtain the current command from the joystick settings
    NSString *currentCommand = [[CBJoystickSettings sharedInstance] commandForStick:representedStick andAxis:representedAxis ofJoystick:representedJoystick];
    if (currentCommand != nil) {
        NSUInteger index = [commands indexOfObject:currentCommand];
        if (index != NSNotFound) {
            [popupButton selectItemAtIndex:index];
        }
        else {
            [popupButton selectItemAtIndex:0];
        }
    }
}

/** Adds a command as an option on the pop-up button */
- (void)addCommand:(NSString *)command withTitle:(NSString *)title {
    NSMenuItem *menuItem = [[[NSMenuItem alloc] initWithTitle:title action:nil keyEquivalent:@""] autorelease];
    [menuItem setTag:[commands count]]; // (tag = index in command array)
    if (command == nil) {
        [commands addObject:[NSNull null]];
    }
    else {
        [commands addObject:command];
    }
    [[popupButton menu] addItem:menuItem];
}

/** Called when a new command is selected */
- (IBAction)commandChanged:(id)sender {
    // Save the command that was selected
    if ([popupButton selectedTag] >= 0) {
        id command = [commands objectAtIndex:[popupButton selectedTag]];
        if (command == [NSNull null]) { command = nil; }
        [[CBJoystickSettings sharedInstance] setCommand:command forStick:representedStick andAxis:representedAxis ofJoystick:representedJoystick];
    }
}

/** Called when the joystick is moved */
- (void)joystick:(DDHidJoystick *)joystick
           stick:(unsigned)stickNumber
            axis:(unsigned)axisNumber
    valueChanged:(double)value {
    
    if (joystick == representedJoystick &&
        stickNumber == representedStick &&
        axisNumber == representedAxis) {
        
        [levelIndicator setDoubleValue:value];
    }
}

@end
