/*=========================================================================
   This file is part of the Cardboard Robot Console application.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import "CBSpeedTransformer.h"
#import "CBMainWindowController.h"


/* Private members */
@interface CBSpeedTransformer() {
    CBRobot *robot;
}

@property (assign, nonatomic) CBUnit unit1;
@property (assign, nonatomic) CBUnit unit2;
@property (assign, nonatomic) CBUnit unit3;
@property (assign, nonatomic) CBUnit unit4;

- (void)updateUnits;
- (void)updateComponents;

- (void)setSpeedComponent:(int)componentIndex toValue:(double)value;

- (double)dofConversionFactorForMotor:(int)motor;

@end


@implementation CBSpeedTransformer

@synthesize robot;
@synthesize speed;
@synthesize dofUnit;

@synthesize component1;
@synthesize component2;
@synthesize component3;
@synthesize component4;

@synthesize unit1;
@synthesize unit2;
@synthesize unit3;
@synthesize unit4;

- (void)dealloc {
    [robot release]; robot = nil;
    [speed release]; speed = nil;
    [super dealloc];
}

- (id)init {
    self = [super init];
    if (self) {
        robot = [[[CBMainWindowController sharedInstance] robot] retain];
        dofUnit = CBUnitRadians;
        [self setSpeed:[CBArmSpeed zero]];
        [self updateUnits];
        [self updateComponents];
    }
    return self;
}

+ (NSSet *)keyPathsForValuesAffectingComponent1 {
    return [NSSet setWithObjects:@"speed", @"uniformComponent", @"dofUnit", nil];
}

+ (NSSet *)keyPathsForValuesAffectingComponent2 {
    return [NSSet setWithObjects:@"speed", @"uniformComponent", @"dofUnit", nil];
}

+ (NSSet *)keyPathsForValuesAffectingComponent3 {
    return [NSSet setWithObjects:@"speed", @"uniformComponent", @"dofUnit", nil];
}

+ (NSSet *)keyPathsForValuesAffectingComponent4 {
    return [NSSet setWithObjects:@"speed", @"uniformComponent", @"dofUnit", nil];
}

+ (NSSet *)keyPathsForValuesAffectingSpeed {
    return [NSSet setWithObjects:@"component1", @"component2", @"component3", @"component4", @"uniformComponent", nil];
}

+ (NSSet *)keyPathsForValuesAffectingUniformComponent {
    return [NSSet setWithObjects:@"speed", @"component1", nil];     // (setting components 2, 3 or 4 doesn't affect the uniform component)
}

// (property documented in header)
- (void)setDofUnit:(CBUnit)aDofUnit {
    if (aDofUnit != dofUnit) {
        dofUnit = aDofUnit;
        [self updateUnits];
        [self updateComponents];
    }
}

/** Updates the unit1, unit2, etc. properties based on the current
 *  parameters.
 */
- (void)updateUnits {
    [self setUnit1:dofUnit];
    [self setUnit2:dofUnit];
    [self setUnit3:dofUnit];
    [self setUnit4:dofUnit];
}

/** Updates the components from the current position & unit setting */
- (void)updateComponents {
    if (speed != nil) {
        component1 = [speed m1Speed] * [self dofConversionFactorForMotor:1];    // (do not call setComponent1 or stack will overflow)
        component2 = [speed m2Speed] * [self dofConversionFactorForMotor:2];
        component3 = [speed m3Speed] * [self dofConversionFactorForMotor:3];
        component4 = [speed m4Speed] * [self dofConversionFactorForMotor:4];
    }
}

// (property documented in header)
- (void)setSpeed:(CBArmSpeed *)aSpeed {
    if (speed != aSpeed) {
        [speed release]; speed = nil;
        speed = [aSpeed retain];
        [self updateComponents];
    }
}

// (property documented in header)
- (void)setComponent1:(double)value {
    [self setSpeedComponent:0 toValue:value];
    component1 = value;
}

// (property documented in header)
- (void)setComponent2:(double)value {
    [self setSpeedComponent:1 toValue:value];
    component2 = value;
}

// (property documented in header)
- (void)setComponent3:(double)value {
    [self setSpeedComponent:2 toValue:value];
    component3 = value;
}

// (property documented in header)
- (void)setComponent4:(double)value {
    [self setSpeedComponent:3 toValue:value];
    component4 = value;
}

// (property documented in header)
- (void)setUniformComponent:(double)value {
    NSAssert(component2 == component1 && component3 == component1 && component4 == component1, @"Uniform component being written, but components are not uniform");
    [self setComponent1:value];
    [self setComponent2:value];
    [self setComponent3:value];
    [self setComponent4:value];
}

// (property documented in header)
- (double)uniformComponent {
    NSAssert(component2 == component1 && component3 == component1 && component4 == component1, @"Uniform component being read, but components are not uniform");
    return component1;
}

/** Helper method to setComponent1, etc. */
- (void)setSpeedComponent:(int)componentIndex toValue:(double)value {
    NSAssert(speed != nil, @"No position is set");
    if (speed) {
        value /= [self dofConversionFactorForMotor:(componentIndex + 1)];
        CBArmSpeed *newSpeed = [speed setComponentWithIndex:componentIndex toValue:value];
        [speed release]; speed = nil;
        speed = [newSpeed retain];
    }
}

/** Returns the conversion factor (radians to target unit) for the specified motor */
- (double)dofConversionFactorForMotor:(int)motor {
    switch (dofUnit) {
        case CBUnitRadians:
            return 1.0;
        case CBUnitDegrees:
            return 180.0 / pi;
        case CBUnitSteps:
            return [robot stepsFromAngle:1.0 forMotor:motor];
        default:
            [NSException raise:@"CBInvalidOperation" format:@"Unknown unit: %i", dofUnit];
            return NAN;
    }
}

@end
