/*=========================================================================
   This file is part of the Cardboard Robot Console application.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import "CBCompiledProgram.h"
#import "CBMainWindowController.h"
#import "CBProgram.h"
#import "CBProgramEntry.h"
#import "CBScalarSpeedTransformer.h"


/* Private members */
@interface CBProgram()

- (void)startObservingEntry:(CBProgramEntry *)entry;
- (void)stopObservingEntry:(CBProgramEntry *)entry;

@end


@implementation CBProgram

@synthesize dofUnit;
@synthesize entries;

- (void)dealloc {
    
    // Stop listening for changes (in case an entry outlives this object)
    for (CBProgramEntry *entry in entries) {
        [self stopObservingEntry:entry];
    }
    
    [self removeWindowController:[CBMainWindowController sharedInstance]];
    [entries release]; entries = nil;
}

- (id)init {
    self = [super init];
    if (self) {
        dofUnit = CBUnitDegrees;
        entries = [[NSMutableArray alloc] init];
    }
    return self;
}

- (void)makeWindowControllers {
    [self addWindowController:[CBMainWindowController sharedInstance]];
}

/** Sets the current dof unit */
- (void)setDofUnit:(CBUnit)aDofUnit {
    // Propagate the change to the position transformer of each entry
    for (CBProgramEntry *entry in entries) {
        [entry setDofUnit:aDofUnit];
    }
    dofUnit = aDofUnit;
}

// (Returns the program as an NSData to be saved)
- (NSData *)dataOfType:(NSString *)typeName error:(NSError **)outError {
    @try {
        CBRobot *robot = [[CBMainWindowController sharedInstance] robot];
        NSAssert(robot != nil, @"Robot instance is nil");
        double degsPerRad = 180.0 / pi;
        
        CBPositionTransformer *pt = [[[CBPositionTransformer alloc] init] autorelease];
        [pt setDofUnit:CBUnitDegrees];

        // Build an array of lines to save to the file
        NSMutableArray *lines = [NSMutableArray array];
        for (CBProgramEntry *entry in entries) {
            
            // Use a position transformer to translate the position to the unit / components to be saved
            [pt setPosition:[[entry positionTransformer] position]];
            NSString *line = [NSString stringWithFormat:@"%f, %f, %f, %f, %f, %f",
                              [pt component1], [pt component2], [pt component3], [pt component4],
                              [[entry speedTransformer] speed] * degsPerRad, [entry pause]];
            [lines addObject:line];
        }
        return [[lines componentsJoinedByString:@"\n"] dataUsingEncoding:NSASCIIStringEncoding];
    }
    @catch (NSException *e) {
        NSLog(@"An exception occurred: %@", e);
        return nil;     // UI will display an error to user - don't crash the application
    }
}

// (Reads the program from an NSData object)
- (BOOL)readFromData:(NSData *)data ofType:(NSString *)typeName error:(NSError **)outError {
    @try {
        CBRobot *robot = [[CBMainWindowController sharedInstance] robot];
        NSAssert(robot != nil, @"Robot instance is nil");
        double radsPerDeg = pi / 180.0;
        
        // Remove all existing entries (using KVC, so display is updated and entries are unregistered properly)
        for (int index = (int)[entries count] - 1; index >= 0; index --) {
            [self removeObjectFromEntriesAtIndex:index];
        }

        NSString *entireFile = [[[NSString alloc] initWithData:data encoding:NSUTF8StringEncoding] autorelease];
        NSArray *lines = [entireFile componentsSeparatedByCharactersInSet:[NSCharacterSet newlineCharacterSet]];
        NSCharacterSet *commaCharSet = [NSCharacterSet characterSetWithCharactersInString:@","];
        for (NSString *line in lines) {
            if ([line length] > 0) {    // (skip the empty line between a /r and /n, and any other empty lines)
                @autoreleasepool {
                    NSArray *parts = [line componentsSeparatedByCharactersInSet:commaCharSet];
                    
                    double m1 = ([parts count] < 1) ? 0.0 : [[parts objectAtIndex:0] doubleValue];
                    double m2 = ([parts count] < 2) ? 0.0 : [[parts objectAtIndex:1] doubleValue];
                    double m3 = ([parts count] < 3) ? 0.0 : [[parts objectAtIndex:2] doubleValue];
                    double m4 = ([parts count] < 4) ? 0.0 : [[parts objectAtIndex:3] doubleValue];
                    double speed = ([parts count] < 5) ? 10.0 : [[parts objectAtIndex:4] doubleValue];
                    double pause = ([parts count] < 6) ? 0.0 : [[parts objectAtIndex:5] doubleValue];
                    
                    CBProgramEntry *entry = [[[CBProgramEntry alloc] init] autorelease];
                    [[entry speedTransformer] setSpeed:speed * radsPerDeg];
                    [entry setPause:pause];
                    
                    [[entry positionTransformer] setDofUnit:dofUnit];
                    [[entry positionTransformer] setPosition:[CBArmPosition armPositionWithM1:m1 * radsPerDeg
                                                                                        andM2:m2 * radsPerDeg
                                                                                        andM3:m3 * radsPerDeg
                                                                                        andM4:m4 * radsPerDeg]];
                    
                    [self insertObject:entry inEntriesAtIndex:[entries count]];
                }
            }
        }
        
        // Don't undo past a load / revert operation
        [[self undoManager] removeAllActions];
        
        return YES;
    }
    @catch (NSException *e) {
        NSLog(@"An exception occurred: %@", e);
        return NO;      // UI will display an error to user - don't crash the application
    }
}

/** Inserts a row (KVC-compliant) */
- (void)insertObject:(CBProgramEntry *)object inEntriesAtIndex:(NSUInteger)index {
    [object setDofUnit:dofUnit];
    [entries insertObject:object atIndex:index];
    [self startObservingEntry:object];
    
    // Add the inverse operation to the undo stack
    NSUndoManager *undoManager = [self undoManager];
    [[undoManager prepareWithInvocationTarget:self] removeObjectFromEntriesAtIndex:index];
    [undoManager setActionName:@"Add Row"];
}

/** Removes a row (KVC-compliant) */
- (void)removeObjectFromEntriesAtIndex:(NSUInteger)index {
    CBProgramEntry *entry = [[[entries objectAtIndex:index] retain] autorelease];
    [self stopObservingEntry:entry];
    [entries removeObjectAtIndex:index];
    
    // Add the inverse operation to the undo stack
    NSUndoManager *undoManager = [self undoManager];
    [[undoManager prepareWithInvocationTarget:self] insertObject:entry inEntriesAtIndex:index];
    [undoManager setActionName:@"Remove Row"];
}

/** Starts listening for changes on the specified program entry */
- (void)startObservingEntry:(CBProgramEntry *)entry {
    [[entry positionTransformer] addObserver:self forKeyPath:@"position" options:(NSKeyValueObservingOptionOld | NSKeyValueObservingOptionNew) context:NULL];
    [[entry speedTransformer] addObserver:self forKeyPath:@"speed" options:(NSKeyValueObservingOptionOld | NSKeyValueObservingOptionNew) context:NULL];
    [entry addObserver:self forKeyPath:@"pause" options:(NSKeyValueObservingOptionOld | NSKeyValueObservingOptionNew) context:NULL];
}

/** Stops listening for changes on the specified program entry */
- (void)stopObservingEntry:(CBProgramEntry *)entry {
    [[entry positionTransformer] removeObserver:self forKeyPath:@"position"];
    [[entry speedTransformer] removeObserver:self forKeyPath:@"speed"];
    [entry removeObserver:self forKeyPath:@"pause"];
}

/** Called when an observed value changes */
- (void)observeValueForKeyPath:(NSString *)keyPath ofObject:(id)object change:(NSDictionary *)change context:(void *)context {
    id oldValue = [change objectForKey:NSKeyValueChangeOldKey];
    id newValue = [change objectForKey:NSKeyValueChangeNewKey];
    
    // (NSNull is used to represent nil in the dictionary)
    if (oldValue == [NSNull null]) {
        oldValue = nil;
    }
    if (newValue == [NSNull null]) {
        newValue = nil;
    }
    
    // (On unit change, we get notifications about changes when values haven't
    // actually changed.  Change to DOF unit => change to transformed value =>
    // change to original value)
    if (newValue != oldValue) {
        
        // Special case for NSNumber - different number instances can have the same value
        if (![newValue respondsToSelector:@selector(doubleValue)] ||
                ![oldValue respondsToSelector:@selector(doubleValue)] ||
                [newValue doubleValue] != [oldValue doubleValue]) {
            
            // Add the inverse operation to the undo stack
            NSUndoManager *undoManager = [self undoManager];
            [[undoManager prepareWithInvocationTarget:object] setValue:oldValue forKeyPath:keyPath];
            [undoManager setActionName:@"Edit"];
        }
    }
}

@end
