/*=========================================================================
   This file is part of the Cardboard Robot Console application.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import "CBPositionTransformer.h"
#import "CBDofVector.h"
#import "CBMainWindowController.h"


/* Private members */
@interface CBPositionTransformer() {
    CBRobot *robot;
}

@property (assign, nonatomic) CBUnit unit1;
@property (assign, nonatomic) CBUnit unit2;
@property (assign, nonatomic) CBUnit unit3;
@property (assign, nonatomic) CBUnit unit4;

- (void)updateUnits;
- (void)updateComponents;

- (void)setPositionComponent:(int)componentIndex toValue:(double)value;

- (double)dofConversionFactorForMotor:(int)motor;

@end


@implementation CBPositionTransformer

@synthesize coordinateSystem;
@synthesize dofUnit;
@synthesize position;

@synthesize component1;
@synthesize component2;
@synthesize component3;
@synthesize component4;

@synthesize unit1;
@synthesize unit2;
@synthesize unit3;
@synthesize unit4;

- (void)dealloc {
    [robot release]; robot = nil;
    [position release]; position = nil;
    [super dealloc];
}

- (id)init {
    self = [super init];
    if (self) {
        robot = [[[CBMainWindowController sharedInstance] robot] retain];
        dofUnit = CBUnitRadians;
        coordinateSystem = CBCoordinateSystemDof;
        [self setPosition:[CBArmPosition zero]];
        [self updateUnits];
        [self updateComponents];
    }
    return self;
}

+ (NSSet *)keyPathsForValuesAffectingComponent1 {
    return [NSSet setWithObjects:@"position", @"dofUnit", nil];
}

+ (NSSet *)keyPathsForValuesAffectingComponent2 {
    return [NSSet setWithObjects:@"position", @"dofUnit", nil];
}

+ (NSSet *)keyPathsForValuesAffectingComponent3 {
    return [NSSet setWithObjects:@"position", @"dofUnit", nil];
}

+ (NSSet *)keyPathsForValuesAffectingComponent4 {
    return [NSSet setWithObjects:@"position", @"dofUnit", nil];
}

+ (NSSet *)keyPathsForValuesAffectingPosition {
    return [NSSet setWithObjects:@"component1", @"component2", @"component3", @"component4", nil];
}

// (property documented in header)
- (void)setDofUnit:(CBUnit)aDofUnit {
    if (aDofUnit != dofUnit) {
        dofUnit = aDofUnit;
        [self updateUnits];
        [self updateComponents];
    }
}

/** Updates the unit1, unit2, etc. properties based on the current
 *  parameters.
 */
- (void)updateUnits {
    [self setUnit1:dofUnit];
    [self setUnit2:dofUnit];
    [self setUnit3:dofUnit];
    [self setUnit4:dofUnit];
}

/** Updates the components from the current position & unit setting */
- (void)updateComponents {
    if (position != nil) {
        switch (coordinateSystem) {
            case CBCoordinateSystemDof:
            {
                CBDofVector *v = [[position tipPosition] pointAsDofVectorForRobot:robot];
                component1 = [v m1] * [self dofConversionFactorForMotor:1];    // (do not call setComponent1 or stack will overflow)
                component2 = [v m2] * [self dofConversionFactorForMotor:2];
                component3 = [v m3] * [self dofConversionFactorForMotor:3];
                component4 = [position m4] * [self dofConversionFactorForMotor:4];
                break;
            }
            default:
                NSAssert(nil, @"Unknown coordinate system");
                return;
        }
    }
}

// (property documented in header)
- (void)setPosition:(CBArmPosition *)aPosition {
    if (position != aPosition) {
        [position release]; position = nil;
        position = [aPosition retain];
        [self updateComponents];
    }
}

// (property documented in header)
- (void)setComponent1:(double)value {
    [self setPositionComponent:0 toValue:value];
    component1 = value;
}

// (property documented in header)
- (void)setComponent2:(double)value {
    [self setPositionComponent:1 toValue:value];
    component2 = value;
}

// (property documented in header)
- (void)setComponent3:(double)value {
    [self setPositionComponent:2 toValue:value];
    component3 = value;
}

// (property documented in header)
- (void)setComponent4:(double)value {
    [self setPositionComponent:3 toValue:value];
    component4 = value;
}

/** Helper method to setComponent1, etc. */
- (void)setPositionComponent:(int)componentIndex toValue:(double)value {
    NSAssert(position != nil, @"No position is set");
    if (position) {
        value /= [self dofConversionFactorForMotor:(componentIndex + 1)];
        
        // Obtain the current tip / M4
        CBDofVector *tip = [[position tipPosition] pointAsDofVectorForRobot:robot];
        double m4 = [position m4];
        
        // Compute a new tip / M4
        if (componentIndex >= 0 && componentIndex < 3) {
            tip = [tip setComponentWithIndex:componentIndex toValue:value];
        }
        else if (componentIndex == 3) {
            m4 = value;
        }
        else {
            NSAssert(NO, @"Invalid component index: %i", componentIndex);
        }
        
        // Set the new position
        CBArmPosition *newPosition = [CBArmPosition armPositionWithTipPosition:tip andM4:m4];
        if (newPosition != nil) {
            [position release]; position = nil;
            position = [newPosition retain];
        }
    }
}

/** Returns the conversion factor (radians to target unit) for the specified motor */
- (double)dofConversionFactorForMotor:(int)motor {
    switch (dofUnit) {
        case CBUnitRadians:
            return 1.0;
        case CBUnitDegrees:
            return 180.0 / pi;
        case CBUnitSteps:
            return [robot stepsFromAngle:1.0 forMotor:motor];
        default:
            [NSException raise:@"CBInvalidOperation" format:@"Unknown unit: %i", dofUnit];
            return NAN;
    }
}

@end
