
/*
 * This program plays phone sounds.
 * 
 * It is designed to offload sound functionality from the Sketch blunoPhone
 * A serial connection is established on pins 
 * 

 * 
 * Sound files are converted to wav here:
 * https://audio.online-convert.com/convert-to-wav
 * 8 bit
 * 16000 Hz
 * mono
 * Advanced options
 * PCM unsigned 8-bit
 * 
 * The sound files are then stored on a micro SD Card in 8.3 format.  Meaning:
 * the file name has a maximum of 8 characters and the extension in ".wav" which
 * is 3 charachters.
 * 
 * The micro SD Card controller is connected to the the Arduino using a SPI
 * interface.  The SD Card is automatically mounted upon startup.
 * 
 * The main loop simply listens for instructions coming from the other Arduino
 * on D3.  The library SoftwareSerial.h is used for communication.  Most commands
 * simply specify which sound file to play. But the command COMMAND_PLAY_RANDOM_MESSAGE
 * results in a random answering machine message to be played.
 * 
 * Note: do *not* connect anything to pin A0.  This pin is used to seed the
 * random number generator so it needs to float unconnected.
 * 
 * 
 * Optionally, you can connect some LEDs to pind D5, D6, D7, and D8 if
 * you want to blink based on commands from the other Arduino.
 */



#include "Arduino.h"
#include "SD.h"             // SD Card library
#include "SPI.h"            // Connect SD Card controller via SPI bus
#include "TMRpcm.h"         // Play the audio files
#include <SoftwareSerial.h> // communicate with another Arduino

/************** general configuration *********/

// at startup play dialtone without waiting for bluno
//#define AUTO_PLAY_DIALTONE

// How loud should sound be? Range: 0-7  (3 is good for one listener.  5 for
// holding up the phone for a small group to hear)
#define SOUND_VOLUME 5 // normally 3-5 but sometimes 2 so I don't bother others

/**************************
 * Phone dialing settings
 **************************/

#define SOFT_SERIAL_RX         3  // talk to other Arduino over D2, D3
#define SOFT_SERIAL_TX         2

// Speaker wires: white/brown(gnd) and white/orange(R1 -> D9)
#define SOUND_OUTPUT_PIN       9    // pin 9 is PWM using the correct timer
#define SDCARD_CS_PIN          4    // SPI CS pin for SDCard with sound files

#define YELLOW_LED             8    // yellow LED
#define GREEN_LED              7    // green  LED
#define RED_LED                6    // red    LED
#define BLUE_LED               5    // white  LED


/************* Sound settings *********/

// commands from the other Arduino telling us what to do

#define COMMAND_PLAY_DIALTONE         1
#define COMMAND_PLAY_HELLO            2
#define COMMAND_PLAY_OFFHOOK          3
#define COMMAND_PLAY_RINGING          4
#define COMMAND_PLAY_WRONGNUMBER      5
#define COMMAND_PLAY_FACILITY_TROUBLE 6
#define COMMAND_PLAY_RANDOM_MESSAGE   7
#define COMMAND_STOP_AUDIO            8
#define COMMAND_BLINK_LIKE_CRAZY      9
#define COMMAND_STOP_BLINK_LIKE_CRAZY 10


// index IDs for retrieving files from SDCard
#define DIALTONE_FILE_INDEX         0
#define HELLO_FILE_INDEX            1
#define OFFHOOK_FILE_INDEX          2
#define RINGING_FILE_INDEX          3
#define WRONGNUMBER2_FILE_INDEX     4
#define FACILITY_TROUBLE_FILE_INDEX 5
#define RANDOM_MESSAGE              6
#define STOP_AUDIO                  7

// Blink all the LEDs like crazy if asked to
bool blinkLikeCrazy = false;

// blink rates when connected spaced to look like random splatte
#define BLUE_BLINK_DELAY   300   // blink rate 0.3 seconds
#define YELLOW_BLINK_DELAY 500   // blink rate 0.4 seconds
#define RED_BLINK_DELAY    800   // blink rate 0.5 seconds
#define GREEN_BLINK_DELAY  1300   // blink rate 0.5 seconds
long timeOfLastBlueBlink = 0;
long timeOfLastGreenBlink = 0;
long timeOfLastYellowBlink = 0;
long timeOfLastRedBlink = 0;

// Store Sound file name string in program space FLASH to avoid with low memory issues
const char dialToneFile[]        PROGMEM = "dltone18.wav";
//const char helloFile[]         PROGMEM = "hello.wav";
const char helloFile[]           PROGMEM = "tmhello.wav";
const char offHookFile[]         PROGMEM = "offhook.wav";
const char ringingFile[]         PROGMEM = "ringing.wav";
const char wrongNumberFile[]     PROGMEM = "wrngnum2.wav";
const char facilityTroubleFile[] PROGMEM = "fctytrbl.wav";


const  char* const soundFiles [] PROGMEM = {
    dialToneFile,
    helloFile,
    offHookFile,
    ringingFile,
    wrongNumberFile,
    facilityTroubleFile
};

#define NUM_SOUND_FILES (sizeof(soundFiles) / sizeof(char *))


// Some randomly funny answering machine messages to play back when dialing
// the phone number shown on the dial (aka dialing myself)
const char answeringMachine1File[]    PROGMEM = "cantfind.wav";
const char answeringMachine2File[]    PROGMEM = "columbo.wav";
const char answeringMachine3File[]    PROGMEM = "communct.wav";
const char answeringMachine4File[]    PROGMEM = "dadada.wav";
const char answeringMachine5File[]    PROGMEM = "duwap.wav";
const char answeringMachine6File[]    PROGMEM = "japnes.wav";
const char answeringMachine7File[]    PROGMEM = "mi.wav";
const char answeringMachine8File[]    PROGMEM = "million.wav";
const char answeringMachine9File[]    PROGMEM = "montybus.wav";
const char answeringMachine10File[]   PROGMEM = "queen.wav";
const char answeringMachine11File[]   PROGMEM = "silyvoic.wav";


const  char* const answeringMachineFiles [] PROGMEM = {
    answeringMachine1File,
    answeringMachine2File,
    answeringMachine3File,
    answeringMachine4File,
    answeringMachine5File,
    answeringMachine6File,
    answeringMachine7File,
    answeringMachine8File,
    answeringMachine9File,
    answeringMachine10File,
    answeringMachine11File
};

#define NUM_ANSWERINGMACHINE_FILES (sizeof(answeringMachineFiles) / sizeof(char *))


char currentlyPlayingSound [20]; // file name or the currently playing sound

TMRpcm audio; //structure for playing sounds

File rootDir; // top directory on the SDCard

SoftwareSerial mySerial(SOFT_SERIAL_RX, SOFT_SERIAL_TX); // RX, TX

#define STATE_INITIALIZING          0
#define STATE_IDLE                  1
#define STATE_RECEIVED_DATA         2
#define STATE_RECEIVING_DATA        3

int state = STATE_INITIALIZING;
int previousState = -1;

int dataReceived =0;
bool isDataReadEnabled = false;
int  dataInputPinState = 0;
int  dataInputLastState = 0;
int  dataInputInputTrueState = 0;
bool isDataReadEnabledLastState = false;
long timeOfDataReadEnabledStateChange = 0;
long timeOfDataPinStateChange = 0;

bool isDataReadEnabledTrueState = false;
bool wasReceivingData = false;
bool dataWasHigh = false;
int  receivedMessage = 0;

int pulseCount = 0;

void setup() {

  Serial.begin(115200);

  // set the data rate for the SoftwareSerial port to other Arduino
  mySerial.begin(4800);

  pinMode(SDCARD_CS_PIN,          OUTPUT);
  pinMode(SOUND_OUTPUT_PIN,       OUTPUT);
  
  pinMode(YELLOW_LED,             OUTPUT);
  pinMode(GREEN_LED,              OUTPUT);
  pinMode(RED_LED,                OUTPUT);
  pinMode(BLUE_LED,               OUTPUT);

  digitalWrite(SDCARD_CS_PIN,      LOW);
  digitalWrite(SOUND_OUTPUT_PIN,   LOW);
  
  digitalWrite(YELLOW_LED,         LOW);
  digitalWrite(GREEN_LED,          LOW);
  digitalWrite(RED_LED,            LOW);
  digitalWrite(BLUE_LED,           LOW);

  // for randomizing the list of answering machine messages
  randomSeed(analogRead(0)); // there should be noise on A0 sufficient for random seed

  Serial.print("Mounting SD card...");

  if (!SD.begin(SDCARD_CS_PIN)) {
      //If the SS pin is in a LOW state it will send a Fail message Serial.println("SD fail");
    Serial.println("Mount failed!");
    while (1);
  }
  Serial.println("Mount complete.");

  audio.speakerPin = SOUND_OUTPUT_PIN; //The pin where you will put the speaker, usually the 9
  audio.setVolume(SOUND_VOLUME);       // How loud? 0 to 7
  audio.quality(1); //only accepts 1 or 0, 1 is for better quality

#ifdef AUTO_PLAY_DIALTONE
  playSound (DIALTONE_FILE_INDEX, true);  // play dialtone
#endif

  state = STATE_IDLE;
}


void loop() {

  
  switch (state) {
    case STATE_IDLE:

      if(previousState != state)
        Serial.println("STATE_IDLE");

      previousState = state;

      if (mySerial.available()) {
        receivedMessage = mySerial.read();
        Serial.print ("msg: ");
        Serial.println (receivedMessage);
        state = STATE_RECEIVED_DATA;
      }

      break;


    case STATE_RECEIVED_DATA:

    if(previousState != state) {
          Serial.println("STATE_RECEIVED_DATA");

      previousState = state;

      switch (receivedMessage) {
        case COMMAND_PLAY_DIALTONE:
           playSound (DIALTONE_FILE_INDEX, true);
           break;
           
        case COMMAND_PLAY_HELLO:
           playSound (HELLO_FILE_INDEX, false);
           break;
           
        case COMMAND_PLAY_OFFHOOK:
           playSound (OFFHOOK_FILE_INDEX, true);
           break;
           
        case COMMAND_PLAY_RINGING:
           playSound (RINGING_FILE_INDEX, true);
           break;
           
        case COMMAND_PLAY_WRONGNUMBER:
           playSound (WRONGNUMBER2_FILE_INDEX, false);
           break;
           
        case COMMAND_PLAY_FACILITY_TROUBLE:
           playSound (FACILITY_TROUBLE_FILE_INDEX, false);
           break;
           
        case COMMAND_PLAY_RANDOM_MESSAGE:
           playSound (RANDOM_MESSAGE, false);
           break;
           
        case COMMAND_STOP_AUDIO:
           playSound (STOP_AUDIO, false);
           break;
           
        case COMMAND_BLINK_LIKE_CRAZY:
           blinkLikeCrazy = true;
           break;
           
        case COMMAND_STOP_BLINK_LIKE_CRAZY:
           blinkLikeCrazy = false;
           break;
           
        default:
           break;
      }
    }

      state = STATE_IDLE;
      break;

    default:
        break;
  }

  if (blinkLikeCrazy) {
      if (timeToUpdateBlueBlink()) {
        blink (1, true, BLUE_LED);
      }
      if (timeToUpdateGreenBlink()) {
        blink (1, true, GREEN_LED);
      }
      if (timeToUpdateYellowBlink()) {
        blink (1, true, YELLOW_LED);
      }
      if (timeToUpdateRedBlink()) {
        blink (1, true, RED_LED);
      }
  }
} // loop


void playSound (int soundIndex, bool repeat) {
  
  long randomIndex;
  
  if (soundIndex == RANDOM_MESSAGE) {

    // randomly pick a funny machine message

    randomIndex = random(0, NUM_ANSWERINGMACHINE_FILES-1);
    strcpy_P(currentlyPlayingSound,
      (char*)pgm_read_word(&(answeringMachineFiles[randomIndex])));
      
  } else if (soundIndex == STOP_AUDIO ||
              soundIndex < 0 || soundIndex >= NUM_SOUND_FILES) {

    Serial.println ("stop sound");

    audio.stopPlayback();
    audio.disable();

    return;
    
  } else {
      // grab the file name from FLASH
    strcpy_P(currentlyPlayingSound,
      (char*)pgm_read_word(&(soundFiles[soundIndex])));
  }

  Serial.print   ("playing : ");
  Serial.println (currentlyPlayingSound);
  
  audio.play(currentlyPlayingSound);
  audio.loop(repeat);

}


void blink (int numBlinks, bool fast) {
  blink (numBlinks, fast, GREEN_LED);
}

void blink (int numBlinks, bool fast, int pin) {
  int i;

  if (numBlinks <0)
    return;

  for (i=0; i<numBlinks*2; i++) {
    if (i%2 == 0) {
           digitalWrite(pin, LOW);
    }
    else {
           digitalWrite(pin, HIGH);
    }

    if (fast)
      delay(100);
    else
      delay(300);
  }

  digitalWrite(pin, LOW);

  //delay(1000);

}

// after connected we wildly blink the blue LED
bool timeToUpdateBlueBlink () {
    long now = millis();
    if ((now - timeOfLastBlueBlink) > BLUE_BLINK_DELAY) {
      timeOfLastBlueBlink = now;
      return true;
    } else {
      return false;
    }
}

// after connected we wildly blink the green LED
bool timeToUpdateGreenBlink () {
    long now = millis();
    if ((now - timeOfLastGreenBlink) > GREEN_BLINK_DELAY) {
      timeOfLastGreenBlink = now;
      return true;
    } else {
      return false;
    }
}

// after connected we wildly blink the yellow LED
bool timeToUpdateYellowBlink () {
    long now = millis();
    if ((now - timeOfLastYellowBlink) > YELLOW_BLINK_DELAY) {
      timeOfLastYellowBlink = now;
      return true;
    } else {
      return false;
    }
}

// after connected we wildly blink the red LED
bool timeToUpdateRedBlink () {
    long now = millis();
    if ((now - timeOfLastRedBlink) > RED_BLINK_DELAY) {
      timeOfLastRedBlink = now;
      return true;
    } else {
      return false;
    }
}

