
/*
 * This program uses an antique dial phone as input.  If the correct phone number is found
 * it will reach out over BLE bluetooth and send commands to a known peripheral
 * in the room.  The peripheral would typically be a wearable with an LED -- possibly
 * a nametag that starts flashing when contacted.
 * 
 * This Sketch is to be run on a Bluno Nano that is housed inside the phone (BLUno Central).
 * This is designed to communicate using BLE Blootooth with another Sketch "blunoNameTag"
 * that runs on battery powered "wearable" Bluno Beetles housed in "name tags"
 * held by people scattered around the room.
 *
 * I built two: one for a 1970s phone that I sent to my family on the west coast, another
 * for a 1930s vintage Bell 202 antuque phone for my family on the east coast.
 * 
 * Optionally phone sounds can be played from a seperate Arduino that would also be housed
 * inside the phone. There are various reasons to do so that are described later in comments.
 * Define OFFLOAD_SOUND if you want another Arduino to play the sounds using the Sketch
 * nanoPhoneSounds.
 * 
 * Another Sketch "blunoBasic" is used for configuring BLE parameters.
 * 
 * Note: do *not* connect anything to pin A0.  This pin is used to seed the
 * random number generator so it needs to float unconnected.
 * 
 * This program is large for an Arduino.  RAM is maxed out so, if you add more features
 * or increase the number of variables, you might encounter some strange behaviour such as
 * program reset and unexpected reboot.  The program uses some advanced techinques to
 * conserve RAM (and some hacks to conserve RAM).  (Seriously consider offloading sound 
 * functionality to another Arduino if you have RAM issues.)
 * 
 * The BLUno BLE appears to be designed for communication between only two Bluetooth
 * endpoints: one central and one peripheral.  The design of this program has one central
 * BLUno and many peripherals. Only one peripheral is connected at any given time but there
 * are many possible for selection.  It works but it's not perfect.  Usually it connects
 * fairly quickly (within 15 seconds) but sometimes it takes much longer (a couple of
 * minutes).  The peripherals (name tags) have a watchdog that forces a reboot occasionally.
 * This seemed to help but even that's not perfect.  You may need to switch off/on all
 * peripherals as well as the central phone to get a solid connection.
 * 
 * This sketch dynamically uses the "AT+BIND" command followed by "AT+RESTART" to connect
 * with the desired peripheral. That sequence does not always work.  It seems to be a lot
 * more stable if called twice in a loop instead of just once. Unfortunately, every time
 * through the loop adds an extra second of delay.  Consider calling AT+BIND three times
 * for better stability if you are not conserned about the extra delay.
 * 
 * When programming a Sketch into the BLUno central or any peripheral via the USB
 * cable, I found it be most stable if I power off all other BLUnos in the room.  Because
 * of the serial connection over Bluetooth from one BLUno to another, the
 * programmer seems to get confused on which BLUno is target to be programmed.
 * 
 * When messing with AT commands by hand on the Serial Monitor, I found it best to run
 * the Sketch blunoBasic.
 * 
 */

/* 
 * 
 * You will need to know the MAC address of all peripherals as well as the central BLUno.
 * Do the following to grab the MAC address:
 * - connect to serial port
 * - set "No line ending"
 * - type "+++" and send
 *     [should say "Enter AT Mode"]
 * - set "Both NL & CR"
 * - AT+MAC=?
 * 
 * Copy/paste the peripheral MAC addresses into blunoPhone Sketch.  Copy/paste the
 * MAC address of the central BLUno into the BlunoNameTag Sketch.
 * 
 * Here are some note on my devices, yours will be different. 
 * TMW central BlUno MAC is: 
 *   East Coast beetle: 0xF045DA10B9B3
 *   West Coast beetle: 0xF045DA10B2FE
 *   East Coast nano: 0xB4994C50236A (proto) or 0xB4994C502C9C
 *   
 * For BLE, the central BLUno is configured differently from the BLUno peripherals.
 * 
 * On each peripheral force it to bind only to the Central:
 * 
 *  - AT+CMODE=UNIQUE 
 *  - AT+BIND=0xB4994C50236A or 0xF045DA10B2FE
 *  - AT+NAME=8  (set to the ID number of the Bluno)
 *  - AT+ROLE=ROLE_PERIPHERAL
 *  - AT+USBDEBUG=OFF
 *  - AT+BLUNODEBUG=OFF
 *  - AT+IBEACONS=OFF
 *
 * Set the ID to an integer to match the id in phoneBook on blunoPhone
 *     (note the default NAME is "Bluno")
 *     for example, for ID 8 (Eric's nametag) do this:
 *  AT+NAME=8
 *  
 * To exit AT mode:
 * - AT+EXIT
 *
 * 
 * ******************
 * 
 * To configure BLE on central Bluno (inside the phone) run the sketch blunoBasic
 * 
 * Do the following:
 * - Open serial monitor and set mode to "No line ending"
 * - +++
 * - Set mode to "Both NL & CR"
 * 
 * - AT+MAC=?
 *     [save the response hex number for programming AT+BIND=0x___ 
 *      on all the Bluno Beetle peripherals]
 * - AT+ROLE=ROLE_CENTRAL
 * - AT+CMODE=UNIQUE
 * - AT+USBDEBUG=OFF
 * - AT+BLUNODEBUG=OFF
 * - AT+USBDEBUG=OFF
 * - AT+EXIT
 *   
 * 
 * Note: when programming the central BLUno, make sure to power off all Bluno Beetle
 *       periperals.  Sometimes the Bluetooth BLE connection can actually
 *       cause you to program the wrong device (the Beetle peripheral) because serial
 *       communications to the central nano are forwarded over Bluetooth BLE
 *       to it's bound (AT+BIND) BLUno Beetle.
 *      
 * 
 */

#include "Arduino.h"
#include "PlainProtocol.h"

/************** general configuration *********/
//#define WEST_COAST          // uncomment to set parameters for west-coast phone
//#define QUICK_DIAL        // Take the first number dialed as the ID to dial
//#define DEBUG_DISABLE_BLE // print the states without actually trying to connect to peripheral
//#define AUTO_DIAL 5  // if defined, skip right to dialing this phone number

// You can off-load sounds to a second arduino.  There are a several reasons you might
// want to do this.
//
// 1) RAM.  This Sketch is a fairly complex program for an Arduino. We are running into
// the upper limits of RAM usage ("Low memory available" warning). Sound functionality
// imports SPI and SDCard drivers and uses a bunch of extra variables. With just a few
// more variables or array entries and the program begins acting very very strange.
// Possibly rebooting!
// I implemented a bunch of tricks and hacks to minimize RAM usage.  Not sure how many more
// shortcuts are available if functionality grows.
//
// 2) Pins. Maybe you are using a Bluno Beetle for BLE.  That part doesn't 
// have pins for SPI and sound output.  Go ahead and play sounds on a standard
// Arduino.  (I've used a Nano, which works great!)
//
// 3) Two Arduinos boot faster than one.  You will hear the dialtone sooner
// if you use one arduino for bluetooth and a different one for sounds.
// Initializing the SPI driver and mounting the SDCard to access sounds
// takes at least a second or two.  Initializing BLE takes a couple of
// seconds.  These two tasks can happen in parrallel if you have two
// Arduinos.  Avoid the confusing silence before the dialtone.
//
// So you can define OFFLOAD_SOUND and run the sketch "nanoPhoneSounds" on
// an arduino connected for serial communication on D2 and D3 (or whatever
// pins you choose)
//
//#define OFFLOAD_SOUND

/**************************
 * Phone dialing settings
 **************************/

// on-hook (power switch) orange & brown
// Speaker wires: white/brown(gnd) and white/orange(R1 -> D9 nano)

#ifdef OFFLOAD_SOUND
// Wheel moving wires: white/green(D5) & white/blue(+5) [or ground if normally connected]
#define WHEEL_MOVING_INPUT_PIN 8 // Bluno: 5    // If high then the dial is active
// Pulse wires: blue(gnd) green(D4)
#define PULSE_INPUT_PIN        7 // Bluno: 4    // dial pulses come in on this pin

#define SOFT_SERIAL_RX 2  // talk to another Arduino over D2, D3
#define SOFT_SERIAL_TX 3

#else // OFFLOAD_SOUND

// Wheel moving wires: white/green(D8) & white/blue(ground) [or +5 if normally disconnected]
#define WHEEL_MOVING_INPUT_PIN 8    // If high then the dial is active
// Pulse wires: blue(gnd) green(D4)
#define PULSE_INPUT_PIN        7    // dial pulses come in on this pin

// Speaker wires: white/brown(gnd) and white/orange(R1 -> D9)
#define SOUND_OUTPUT_PIN       9    // pin 9 is PWM using the correct timer
#define SDCARD_CS_PIN          4    // SPI CS pin for SDCard with sound files
#endif // OFFLOAD_SOUND

#define ON_BOARD_LED          13  // LED soldered onto the bluno beetle


//************

#define BLINK_SPEED         300   // 300ms delay blinking LEDs
#define FAST_BLINK_SPEED    100   // 100ms delay quickly blinking LEDs
#define COMMUNICATION_SPEED 20    // 15ms sending data to other Arduino

#define DEBOUNCE_DELAY 10  // wait 10ms for the circuit to settle during puls transition

#ifdef QUICK_DIAL
#define NUM_DIGITS_IN_PHONE_NUMBER 1  // for debugging: dial on the first digit 
#else
#define NUM_DIGITS_IN_PHONE_NUMBER 6  // use a fake 6-digit phone number to avoid confusion with a real phone 
#endif
long numberDialed = 0;       // save all digits of the number dialed
long currentCallNumber = 0;  // The number we just dialed
int pulseCount   = 0;        // count the pulses resulting in the current digit

// variables for counting pulses
int pulseInputPinState   = 0;  // current state of WHEEL_MOVING_INPUT_PIN
int pulseInputLastState  = 0;  // what was the pin value last time through the loop
int pulseInputTrueState  = 0;  // debounced state of WHEEL_MOVING_INPUT_PIN
long timeOfPulsePinStateChange = 0;   // save the time for debouncing
bool pulseWasHigh        = false; // state of pulse last time through the loop
int needToSaveNewDigit   = 0;
int numDigits = 0;

//#define PING_DELAY 5000 // ping peripheral every 20 seconds
#define PING_DELAY 3000 // ping peripheral every 20 seconds
long timeOfLastPing = 0;

#define ACTIVITY_TIMEOUT 10000 // "hang up and try again" after 10 sec inactivity
#define CONNECTED_ACTIVITY_TIMEOUT 240000 // bebebebebebe after 4:00 connection
long timeOfLastActivity = 0;

#define RING_NO_REPLY_TIMEOUT 90000 // "facility trouble, hang up and try again" after 90 sec inactivity
long timeOfBeginRinging = 0;

#define BLE_RESTART_DELAY 10000  // restart BLE device every 10 seconds
long timeOfLastBLERestart = 0;

// variables for decisions about if a person is moving the dial
int isWheelMovingState      = 0; // current state of RED_LED
int isWheelMovingLastState  = 0; // what was the pin value last time through the loop
int isWheelMovingTrueState  = 0; // debounced state of RED_LED
bool wasDialing             = false;
long timeOfWheelMoveStateChange = 0; // save the time for debouncing

#define STATE_STARTUP           0
#define STATE_WAITING_FOR_DIGIT 1
#define STATE_COUNTING_PULSES   2
#define STATE_DIAL_COMPLETE     3
#define STATE_BLE_CONNECTED     4
#define STATE_BLE_CONFIRMED     5
#define STATE_UNKNOWN_NUMBER    6
#define STATE_DIALED_MY_NUMBER  7
#define STATE_MY_NUMBER         8
#define STATE_OFF_HOOK_TOO_LONG 9
#define STATE_NO_REPLY          10

int state = STATE_STARTUP;
int previousState = -1;

// Signals to send to other Arduino that plays our sounds
#define COMMAND_PLAY_DIALTONE         1
#define COMMAND_PLAY_HELLO            2
#define COMMAND_PLAY_OFFHOOK          3
#define COMMAND_PLAY_RINGING          4
#define COMMAND_PLAY_WRONGNUMBER      5
#define COMMAND_PLAY_FACILITY_TROUBLE 6
#define COMMAND_PLAY_RANDOM_MESSAGE   7
#define COMMAND_STOP_AUDIO            8
#define COMMAND_BLINK_LIKE_CRAZY      9
#define COMMAND_STOP_BLINK_LIKE_CRAZY 10

/**************************************
 * Bluetooth communication  settings
 **************************************/

typedef struct {
  int  id;            // ID sticker on the peripheral and AT+NAME=?
  long phoneNumber;   // number to dial
  long peripheralMAC; // peripheral result of AT+MAC=?
} phoneLookup_t;

#define MY_PHONE_NUMBER 869869 // TMW-TMW

// All MAC addresses start with 0xF045.  Save some bytes by prefixing later.
#define MAC_PREFIX "0xF045"

#ifdef WEST_COAST
West Coast nametags
phoneLookup_t phoneBook [] = {
  // 48-bit MAC fits into 32-bit long integer using MAC_PREFIX
  {5, 448025, 0xDA10B983}, // 5 Don     0xF045DA10B983
  {6, 448876, 0xDA10B9A1}, // 6 Linnea  0xF045DA10B9A1
  {8, 448997, 0xDA10B39D}, // 8 Eric    0xF045DA10B39D

  {-1, -1, 0}, // done
};
#else
// East Coast nametags

phoneLookup_t phoneBook [] = {
  // 48-bit MAC fits into 32-bit long integer using MAC_PREFIX
  {0, 869340, 0xDA10B9B3}, // 0 Dan     0xF045DA10B9B3
  {1, 869541, 0xDA10B39C}, // 1 Kristin 0xF045DA10B39C
  {2, 869342, 0xDA10B55A}, // 2 David   0xF045DA10B55A
  {3, 869263, 0xDA10B06F}, // 3 Chris   0xF045DA10B06F
  {4, 869244, 0xDA10B7E2}, // 4 Briana  0xF045DA10B7E2
  {5, 869345, 0xDA10B3F2}, // 5 Diane   0xF045DA10B3F2
  {6, 869776, 0xDA10B1CE}, // 6 Steven  0xF045DA10B1CE
  {7, 869647, 0xDA10B78F}, // 7 Mike    0xF045DA10B78F
  {8, 869748, 0xDA10B51C}, // 8 Sandy   0xF045DA10B51C
  {9, MY_PHONE_NUMBER, 0xDA10BAA4}, // 9 TMW  0xF045DA10BAA4    (does not work on battery)

  {-1, -1, 0}, // done
};
#endif

int receivedID = -1;
int expectedID   = -2;
String receivedIDString = "";

//PlainProtocol constructor, define the Serial port and the baudrate.
PlainProtocol myBLUNO(Serial,115200);

#ifdef OFFLOAD_SOUND

#include <SoftwareSerial.h>

SoftwareSerial mySerial(SOFT_SERIAL_RX, SOFT_SERIAL_TX); // RX, TX

#else // OFFLOAD_SOUND

#include "SD.h"             // SDCard library
#include "SPI.h"
#include "TMRpcm.h"         // Play the audio files#include <SoftwareSerial.h>

// How loud should sound be? Range: 0-7  (3 is good for one listener.  5 for
// holding up the phone for a group to hear)
#define SOUND_VOLUME 5 // normally 3-5 but sometimes 2 so I don't bother others

// Store Sound file name string in program space FLASH to avoid with low memory issues
const char dialToneFile[]        PROGMEM = "dltone18.wav";
//const char helloFile[]         PROGMEM = "hello.wav";
const char helloFile[]           PROGMEM = "tmhello.wav";
const char offHookFile[]         PROGMEM = "offhook.wav";
const char ringingFile[]         PROGMEM = "ringing.wav";
const char wrongNumberFile[]     PROGMEM = "wrngnum2.wav";
const char facilityTroubleFile[] PROGMEM = "fctytrbl.wav";

// index IDs for retrieving files from SDCard
#define DIALTONE_FILE_INDEX         0
#define HELLO_FILE_INDEX            1
#define OFFHOOK_FILE_INDEX          2
#define RINGING_FILE_INDEX          3
#define WRONGNUMBER2_FILE_INDEX     4
#define FACILITY_TROUBLE_FILE_INDEX 5
#define RANDOM_MESSAGE              6
#define STOP_AUDIO                  7

const  char* const soundFiles [] PROGMEM = {
    dialToneFile,
    helloFile,
    offHookFile,
    ringingFile,
    wrongNumberFile,
    facilityTroubleFile
};

#define NUM_SOUND_FILES (sizeof(soundFiles) / sizeof(char *))


// Some randomly funny answering machine messages to play back when dialing
// the phone number shown on the dial (aka dialing myself)
const char answeringMachine1File[]    PROGMEM = "cantfind.wav";
const char answeringMachine2File[]    PROGMEM = "columbo.wav";
const char answeringMachine3File[]    PROGMEM = "communct.wav";
const char answeringMachine4File[]    PROGMEM = "dadada.wav";
const char answeringMachine5File[]    PROGMEM = "duwap.wav";
const char answeringMachine6File[]    PROGMEM = "japnes.wav";
const char answeringMachine7File[]    PROGMEM = "mi.wav";
const char answeringMachine8File[]    PROGMEM = "million.wav";
const char answeringMachine9File[]    PROGMEM = "montybus.wav";
const char answeringMachine10File[]   PROGMEM = "queen.wav";
const char answeringMachine11File[]   PROGMEM = "silyvoic.wav";


const  char* const answeringMachineFiles [] PROGMEM = {
    answeringMachine1File,
    answeringMachine2File,
    answeringMachine3File,
    answeringMachine4File,
    answeringMachine5File,
    answeringMachine6File,
    answeringMachine7File,
    answeringMachine8File,
    answeringMachine9File,
    answeringMachine10File,
    answeringMachine11File
};

#define NUM_ANSWERINGMACHINE_FILES (sizeof(answeringMachineFiles) / sizeof(char *))

char currentlyPlayingSound [20]; // file name or the currently playing sound

TMRpcm audio; //structure for playing sounds

File rootDir; // top directory on the SDCard

#endif // OFFLOAD_SOUND


void setup() {


#ifdef DEBUG_DISABLE_BLE
  Serial.begin(115200);

#else
  myBLUNO.init();

//Note: it would be good to play a pre-dialtone sound here.
// atBind takes a couple of seconds:

  // Fake MAC. Unbind all periperals before a number is dialed
  atBind("0x000000000001");

#endif

#ifdef OFFLOAD_SOUND
  // set the data rate for the SoftwareSerial port to other Arduino
  mySerial.begin(4800);
#else // OFFLOAD_SOUND
  pinMode(SDCARD_CS_PIN,          OUTPUT);
  pinMode(SOUND_OUTPUT_PIN,       OUTPUT);

  digitalWrite(SDCARD_CS_PIN,      LOW);
  digitalWrite(SOUND_OUTPUT_PIN,   LOW);
#endif // OFFLOAD_SOUND

  pinMode(WHEEL_MOVING_INPUT_PIN, INPUT);
  pinMode(PULSE_INPUT_PIN,        INPUT);
  pinMode(ON_BOARD_LED,           OUTPUT);
  
  digitalWrite(ON_BOARD_LED,          LOW);

#ifndef OFFLOAD_SOUND
  // for randomizing the list of answering machine messages
  randomSeed(analogRead(0)); // there should be noise on A0 sufficient for random seed

  Serial.print("Mounting SD card...");

  if (!SD.begin(SDCARD_CS_PIN)) {
      //If the SS pin is in a LOW state it will send a Fail message Serial.println("SD fail");
    Serial.println("Mount failed!");
    while (1);
  }
  Serial.println("Mount complete.");

  audio.speakerPin = SOUND_OUTPUT_PIN; //The pin where you will put the speaker, usually the 9
  audio.setVolume(SOUND_VOLUME);       // How loud? 0 to 7
  audio.quality(1); //only accepts 1 or 0, 1 is for better quality

  playSound (DIALTONE_FILE_INDEX, true);  // play dialtone

#else // OFFLOAD_SOUND

  delay (1000);  // wait for sound Arduino to finish booting
#endif // OFFLOAD_SOUND


}


void loop() {

  int digit;

  isWheelMovingState = digitalRead (WHEEL_MOVING_INPUT_PIN);
  pulseInputPinState = digitalRead(PULSE_INPUT_PIN);


  if (isWheelMovingLastState != isWheelMovingState) {
    timeOfWheelMoveStateChange = millis();
  }


  if ((millis() - timeOfWheelMoveStateChange) > DEBOUNCE_DELAY) {
    // debounce - this happens once it's stablized
    if (isWheelMovingState != isWheelMovingTrueState) {
      // this means that the switch has either just gone from closed->open or vice versa.
      isWheelMovingTrueState = isWheelMovingState;
#ifdef DEBUG_DISABLE_BLE
      Serial.print ("wheel moving: ");
      Serial.println (isWheelMovingTrueState);
#endif
    }
  }

  if (activityWatchdogTimeout()) {
    state = STATE_OFF_HOOK_TOO_LONG;
  }

  switch (state) {
    case STATE_STARTUP:

#ifdef DEBUG_DISABLE_BLE
        if(previousState != state)
          Serial.println("STATE_STARTUP");
#endif
        previousState = state;

#ifdef AUTO_DIAL
        // for debugging, skip actual dialing the phone
        // and just bind with the peripheral  (call the name tag)
        currentCallNumber = AUTO_DIAL;
        state = STATE_DIAL_COMPLETE;
#else // AUTO_DIAL
      //if (isWheelMovingTrueState) {
      //  state = STATE_COUNTING_PULSES;
      //}
      state = STATE_WAITING_FOR_DIGIT;
#endif // AUTO_DIAL

      // tell nano to play dialtone
      playSound (COMMAND_PLAY_DIALTONE);
  
      break;

    case STATE_WAITING_FOR_DIGIT:

        if(previousState != state) {
          timeOfLastActivity = millis();
#ifdef DEBUG_DISABLE_BLE
          Serial.println("STATE_WAITING_FOR_DIGIT");
#endif
        }

        previousState = state;


         if (numDigits >= NUM_DIGITS_IN_PHONE_NUMBER) {
    #ifdef DEBUG_DISABLE_BLE        
           Serial.print("Dialed phone number: ");
           Serial.println(numberDialed);
    #endif
           currentCallNumber = numberDialed;
           numberDialed = 0;
           numDigits    = 0;

           state = STATE_DIAL_COMPLETE;
    
         }
         
         if (isWheelMovingTrueState) {
        
           if (!wasDialing) {
    #ifdef DEBUG_DISABLE_BLE
            Serial.println("Begin new number...");
    #endif
            receivedID = -1;
            expectedID   = -2;
            playSound (COMMAND_STOP_AUDIO); // stop playing
          }

          pulseCount = 0;


          state = STATE_COUNTING_PULSES;
          //wasDialing = true;
      }
      break;

    case STATE_COUNTING_PULSES:

      if(previousState != state) {
        timeOfLastActivity = millis();
#ifdef DEBUG_DISABLE_BLE
        Serial.println("STATE_COUNTING_PULSES");
#endif
      }
      
      previousState = state;

      if (isWheelMovingTrueState) {

        // wheel is turning now count the pulses
    
         if (pulseInputLastState != pulseInputPinState) {
            timeOfPulsePinStateChange = millis();
         }
    
         if ((millis() - timeOfPulsePinStateChange) > DEBOUNCE_DELAY) {
            // debounce - this happens once it's stablized
            if (pulseInputTrueState != pulseInputPinState) {
            // this means that the switch has either just gone from closed->open or vice versa.
              pulseInputTrueState = pulseInputPinState;
              Serial.print ("pulse: ");
              Serial.println (pulseInputTrueState);
            }
                
           if (pulseInputTrueState) {
    
            //digitalWrite(PULSE_COUNT_OUTPUT_PIN, HIGH);
    
            if (!pulseWasHigh) {
             pulseWasHigh = true;
             pulseCount++;
    #ifdef DEBUG_DISABLE_BLE
             Serial.println(pulseCount);
    #endif
           }
         } else {
        
            if (pulseWasHigh) {
               pulseWasHigh = false;
            }
         }
        }
      } else {

            // isWheelMovingTrueState is false

          if (pulseCount != 0) {  // 0 pulses would be a glitch
            digit = pulseCount % 10; // 10 pulses means the digit was 0
            numDigits++;
            numberDialed = numberDialed * 10 + digit;
    
    #ifdef DEBUG_DISABLE_BLE        
            Serial.println("...found digit.");
            Serial.print("Phone Number so far: ");
            Serial.print(numberDialed);
            Serial.println();
    #endif

            pulseCount = 0;
            
            // Note: if all digits were dialed STATE_WAITING_FOR_DIGIT will
            // go to STATE_DIAL_COMPLETE
            state = STATE_WAITING_FOR_DIGIT;
                        
            //wasDialing = false;

          }
    
          //wasDialing = false;
       }
      break;

    case STATE_DIAL_COMPLETE:

      if(previousState != state) {
        timeOfLastActivity = millis();
#ifdef DEBUG_DISABLE_BLE
        Serial.println("STATE_DIAL_COMPLETE");
#endif
      }
        
      previousState = state;



      if (currentCallNumber == MY_PHONE_NUMBER) {
        // Special case.  This is TMW's phone number.
        // tell nano to play a random answering machine message
        playSound (COMMAND_PLAY_RANDOM_MESSAGE);
        //state = STATE_MY_NUMBER;
      }  else {
        // tell nano to play ringing sounds
        playSound (COMMAND_PLAY_RINGING);
      }
      
      if (0 == callPhoneNumber (currentCallNumber)) {
         state = STATE_BLE_CONNECTED;
#ifdef DEBUG_DISABLE_BLE
          Serial.println("atRestart");
#else
         atRestart(); // get ready to receive reply
         delay (500);
#endif
      } else {
         state = STATE_UNKNOWN_NUMBER;
      }
      break;

    case STATE_BLE_CONNECTED:

      if(previousState != state) {
        timeOfBeginRinging = millis();
      }
     
      previousState = state;
      
      if (timeToPing()) {
        //atBind();  // important! restart BLE device for stable connection for listening
        //OUCH removed callPhoneNumber (currentCallNumber);
        delay(500);
//        blink(expectedID, BLINK_SPEED);
//      myBLUNO.write("CALLING", expectedID);
        myBLUNO.write("HELLO", expectedID);     // ask peripheral for ID

      Serial.println();
      //delay(100);
        //blink(2, BLINK_SPEED);
      }

      receiveBLE ();

      if (receivedID == expectedID) {
          state = STATE_BLE_CONFIRMED;
      }

      if (ringingWatchdogTimeout()) {
        // abort if we don't hear back in a long time
        state = STATE_NO_REPLY;
      }
      break;

    case STATE_BLE_CONFIRMED:
#ifdef DEBUG_DISABLE_BLE
        if(previousState != state)
          Serial.println("STATE_BLE_CONFIRMED");
#endif

      if (previousState != state) {
        playSound (COMMAND_PLAY_HELLO);
      }

      if (timeToPing()) {
        myBLUNO.write("CONNECTED!");     // tell other periperal to turn on the LED
        Serial.println();   //print line feed character to complete msg
      }
      previousState = state;

      break;

    case STATE_NO_REPLY:
        if(previousState != state) {
          timeOfLastActivity = millis();
#ifdef DEBUG_DISABLE_BLE
          Serial.println("STATE_NO_REPLY");
#endif
        
        playSound (COMMAND_PLAY_FACILITY_TROUBLE);
      }
      previousState = state;

      break;

    case STATE_UNKNOWN_NUMBER:
        if(previousState != state) {
          timeOfLastActivity = millis();
#ifdef DEBUG_DISABLE_BLE
          Serial.println("STATE_UNKNOWN_NUMBER");
#endif
        
        playSound (COMMAND_PLAY_WRONGNUMBER);
      }
      previousState = state;

      break;
#if 0
    case STATE_MY_NUMBER:
      if(previousState != state) {
          timeOfLastActivity = millis();
          playSound (COMMAND_PLAY_RANDOM_MESSAGE);
          delay (100);
#ifdef OFFLOAD_SOUND
          mySerial.write(COMMAND_BLINK_LIKE_CRAZY);
#endif
      }
      previousState = state;

      break;
#endif // 0

    case STATE_OFF_HOOK_TOO_LONG:
#ifdef DEBUG_DISABLE_BLE
        if(previousState != state)
          Serial.println("STATE_OFF_HOOK_TOO_LONG");
#endif

      if (previousState != state) {
        playSound (COMMAND_PLAY_OFFHOOK);
      }
      previousState = state;
      break;

    default:
        break;
  }
  
   isWheelMovingLastState = isWheelMovingState;
   pulseInputLastState = pulseInputPinState;

#if 1
  if (state == STATE_DIAL_COMPLETE && timeToRestartBLE()) {
#ifdef DEBUG_DISABLE_BLE
     Serial.println("atRestart");
#else
     atRestart();  // restart the BLE device for stable connection for listening
     delay (100);
#endif
  }
#endif
} // loop

int callPhoneNumber (long phoneNumber) {

  phoneLookup_t *phoneBookEntry;
  int index = 0;
  
  do {
    phoneBookEntry = &phoneBook[index++];
    if (phoneBookEntry->id < 0) {
      // past the last number, nothing found
#ifdef DEBUG_DISABLE_BLE        
      Serial.println ("not a valid number");
#endif
      return -1;
    }
    
    if (phoneBookEntry->phoneNumber == phoneNumber
#ifdef QUICK_DIAL
        || phoneBookEntry->id == phoneNumber // debugging: we dialed the ID not the phone number
#endif
    ) {

      String mac;
      mac = String(phoneBookEntry->peripheralMAC, HEX);
      mac.toUpperCase();
      mac = MAC_PREFIX + mac;
      
#ifdef DEBUG_DISABLE_BLE        
      Serial.print ("calling id: ");
      Serial.println (phoneBookEntry->id);
      Serial.print (" ");
      Serial.println (mac);
#else
      //myBLUNO.write("LED", 0);     // tell previous periperal to turn off the LED

      atBind (mac);
      //myBLUNO.write("HELLO", phoneBookEntry->id);
      //Serial.println();
#endif

      expectedID = phoneBookEntry->id;

      break;
    } else {

    }
  } while (1);

  return 0;
}

// send a message to the other Arduino to play a sound
void playSound (int sound) {

#ifdef DEBUG_DISABLE_BLE
    Serial.print ("playSound ");
    Serial.println (sound);
#endif

#ifdef OFFLOAD_SOUND
    mySerial.write(sound);  // send integer indicating sound to play
#else // OFFLOAD_SOUND
     switch (sound) {
        case COMMAND_PLAY_DIALTONE:
           playSound (DIALTONE_FILE_INDEX, true);
           break;
           
        case COMMAND_PLAY_HELLO:
           playSound (HELLO_FILE_INDEX, false);
           break;
           
        case COMMAND_PLAY_OFFHOOK:
           playSound (OFFHOOK_FILE_INDEX, true);
           break;
           
        case COMMAND_PLAY_RINGING:
           playSound (RINGING_FILE_INDEX, true);
           break;
           
        case COMMAND_PLAY_WRONGNUMBER:
           playSound (WRONGNUMBER2_FILE_INDEX, false);
           break;
           
        case COMMAND_PLAY_FACILITY_TROUBLE:
           playSound (FACILITY_TROUBLE_FILE_INDEX, false);
           break;
           
        case COMMAND_PLAY_RANDOM_MESSAGE:
           playSound (RANDOM_MESSAGE, false);
           break;
           
        case COMMAND_STOP_AUDIO:
           playSound (STOP_AUDIO, false);
           break;
           
        default:
           break;
      }

#endif // OFFLOAD_SOUND

}

#ifndef OFFLOAD_SOUND
void playSound (int soundIndex, bool repeat) {
  
  long randomIndex;
  
  if (soundIndex == RANDOM_MESSAGE) {

    // randomly pick a funny machine message

    randomIndex = random(0, NUM_ANSWERINGMACHINE_FILES-1);
    strcpy_P(currentlyPlayingSound,
      (char*)pgm_read_word(&(answeringMachineFiles[randomIndex])));
      
  } else if (soundIndex == STOP_AUDIO ||
              soundIndex < 0 || soundIndex >= NUM_SOUND_FILES) {

    //Serial.println ("stop sound");

    audio.stopPlayback();
    audio.disable();

    return;
    
  } else {
      // grab the file name from FLASH
    strcpy_P(currentlyPlayingSound,
      (char*)pgm_read_word(&(soundFiles[soundIndex])));
  }

  //Serial.print   ("playing : ");
  //Serial.println (currentlyPlayingSound);
  
  audio.play(currentlyPlayingSound);
  audio.loop(repeat);

}

#endif // OFFLOAD_SOUND

void atBind(String peripheralAddr) {

  int i;

// seems like we need to do this twice to work 100%
  for (i=0; i<2; i++) {
    Serial.println("AT+EXIT");
    delay(100); //can't be omitted, same below
    Serial.print("+++");// Enter AT mode of itself
    delay(700); // 700 milliseconds MUST be added or AT will not work
    Serial.print("AT+BIND=");
    Serial.println(peripheralAddr);  delay(100);
    Serial.println("AT+RESTART");  delay(100);
    //Serial.read(); // throw away possible garbage
    //delay(5000);
    //Serial.println("AT+EXIT");
  }
}

void atRestart() {
  Serial.println("AT+EXIT");
  delay(100); //can't be omitted, same below
  Serial.print("+++");// Enter AT mode of itself
  delay(700); // 700 milliseconds MUST be added or AT mode not work
  Serial.println("AT+RESTART");       delay(100);
  delay(100);
  //Serial.read();  // throw out garbage data
}

bool timeToPing () {

    long now = millis();
    if ((now - timeOfLastPing) > PING_DELAY) {
      timeOfLastPing = now;
      return true;
    } else {
      return false;
    }
}

bool timeToRestartBLE () {
    long now = millis();
    if ((now - timeOfLastBLERestart) > BLE_RESTART_DELAY) {
      timeOfLastBLERestart = now;
      return true;
    } else {
      return false;
    }
}


bool activityWatchdogTimeout () {

    long now = millis();
    if ((now - timeOfLastActivity) > CONNECTED_ACTIVITY_TIMEOUT) {
      return true;
    } else {
      return false;
    }
}

// If no one answers after a long time, abort and tell user to try again
bool ringingWatchdogTimeout () {

    long now = millis();
    if ((now - timeOfBeginRinging) > RING_NO_REPLY_TIMEOUT) {
      return true;
    } else {
      return false;
    }
}

void receiveBLE () {
  //delay(500);
  //blink(2, FAST_BLINK_SPEED);
    if (myBLUNO.available()) {    //receive valid command
      //delay(500);
      //blink(3, FAST_BLINK_SPEED);
        if (myBLUNO.equals("ID")){       //if the command name is "DISP"
          receivedID = myBLUNO.read();   //read the string to local value
          delay(1000);
          blink(receivedID, BLINK_SPEED);
          delay(1000);
          blink(receivedID, BLINK_SPEED);
          delay(1000);
        }
  }
  
}

// blink LED or send pulses pausing millisec between pulses
void blink (int numBlinks, int millisec) {
  blink (numBlinks, millisec, ON_BOARD_LED);
}

void blink (int numBlinks, int millisec, int pin) {
  int i;

  if (numBlinks <0)
    return;

  for (i=0; i<numBlinks*2; i++) {
    if (i%2 == 0) {
           digitalWrite(pin, LOW);
    }
    else {
           digitalWrite(pin, HIGH);
    }

    delay(millisec);

  }

  digitalWrite(pin, LOW);

  //delay(1000);

}

