
#include "Arduino.h"
#include "PlainProtocol.h"

/*
 * This program implements a wearable "name tag" that runs on a DFRobot Bluno Beetle.
 * It uses BLE Bluetooth connectivity to communicate with a central Bluno that
 * is hidden inside an antique telephone.
 * 
 * In the idle state -- when there is no connection to the phone -- a blue LED
 * gently pulses glowing brighter then dimmer.
 * 
 * As communication handshake begins, a couple of LEDs of different colors light up
 * and the blue LED glows full brightnes no longer pulsing.
 * 
 * Once communincation is fully established, all LEDs begin blinking in a crazy fast
 * quasi-random pattern.
 * 
 * It is all related to a The Mad Wrapper Christmas game to match a gift to a person.
 * 
 * PlainProtocol library can be found on GitHub:
 * 
 * git clone https://github.com/DFRobot/BlunoAccessoryShieldDemo.git
 * 
 * ***********************
 * Configure BLE settings using the simple sketch blunoBasic
 * 
 * On all peripheral BlUno's do the following to grab the ID:
 * - connect to serial port
 * - set "No line ending"
 * - type "+++" and send
 *     [should say "Enter AT Mode"]
 * - set "Both NL & CR"
 * - AT+MAC=?
 * 
 * Note: TMW central BlUno MAC is: 
 *   East Coast beetle: 0xF045DA10B9B3
 *   West Coast beetle: 0xF045DA10B2FE
 *   East Coast nano 1: 0xB4994C50236A
 *   East Coast nano 2: 0xB4994C502C9C
 *   
 * So on each peripheral force it to bind only to the Central:
 *  - AT+CMODE=UNIQUE 
 *  - AT+BIND=0xB4994C50236A or 0xF045DA10B2FE
 *  - AT+NAME=8  (set to the ID number of the Bluno)
 *  - AT+ROLE=ROLE_PERIPHERAL
 *  - AT+USBDEBUG=OFF
 *  - AT+BLUNODEBUG=OFF
 *  - AT+IBEACONS=OFF
 *
 * Set the ID to an integer to match the id in phoneBook on blunoPhone
 *     (note the default NAME is "Bluno")
 *     for example, for ID 8 (Eric's nametag) do this:
 *  AT+NAME=8
 *  
 * To exit AT mode:
 * - AT+EXIT
 *
 */

// uncomment this to send AT+BIND=0x_____ to bind to the phone
//    it only needs to be done once since the AT parameters are saved
//    when you restart
#define INITIALIZE_CONNECTION
//#define START_WITH_BLINK   // blink all LEDs during startup  (Only for debugging!)

// A couple of reasons to not glow the blue LED when this is on:
// 1) Save battery power
// 2) I wanted to hide one nametag that will light up when TWM phone number is dialed
//#define DISABLE_SLOW_GLOW  // Uncomment if you don't want to continuously glow blue.


// Define for phone sent to the west coast.  Otherwise east coast.
//#define WEST_COAST
//#define USE_LEDS_FOR_DEBUG  // use the green, yellow, and red LEDs for debugging

#define ONBOARD_LED 13
#define GREEN_LED    2  // green LED
#define BLUE_LED     3  // blue  LED
#define YELLOW_LED   4  // yellow LED
#define RED_LED      5  // red    LED

// these values result in relaxing glow breathing rate
#define GLOW_MIN 15 //15           // min brightness of glow
#define GLOW_MAX 100 //120          // max brightness of glow
#define GLOW_UPDATE_DELAY 40 // when to increment
#define GLOW_STEP_SIZE 2

#define BLINK_MIN 0           // min brightness of glow
#define BLINK_MAX 200          // max brightness of glow

// blink rates when connected spaced to look like random orchestrated splatter
#define BLUE_BLINK_DELAY   300   // blink rate 0.3 seconds
#define YELLOW_BLINK_DELAY 500   // blink rate 0.5 seconds
#define RED_BLINK_DELAY    800   // blink rate 0.8 seconds
#define GREEN_BLINK_DELAY  1300  // blink rate 1.3 seconds
long timeOfLastBlueBlink = 0;
long timeOfLastGreenBlink = 0;
long timeOfLastYellowBlink = 0;
long timeOfLastRedBlink = 0;

long timeOfLastPulseUpdate = 0;
int glowValue = 0;              // led brightness 0-255
int glowIncrement = GLOW_STEP_SIZE; // Either + or - depending on up/down glow

#define BLE_UPDATE_DELAY 100  // check for news from centeral every 100ms
long timeOfLastBLECheck = 0;

#define BLE_RESTART_DELAY 30000  // restart BLE device every 10 seconds
//#define BLE_RESTART_DELAY 60000  // restart BLE device every 10 seconds
long timeOfLastBLERestart = 0;

//PlainProtocol constructor, define the Serial port and the baudrate.
PlainProtocol myBLUNO(Serial,115200);

#ifdef WEST_COAST
char *centralMAC = "0xF045DA10B2FE";  // Bluno Beetle
#else
//char *centralMAC = "0xF045DA10B9B3"; // Bluno Beetle
char *centralMAC = "0xB4994C50236A";  // Bluno Nano 1
//char *centralMAC = "0xB4994C502C9C";  // Bluno Nano 2

#endif

int receivedID;

int ledValue=0;
String theString = "";
String ledString = "";
String myName = "";  //The Bluno AT+NAME seeded with 01, 02, 03, matching sticker 
int    myID = -1;
int    personCalled = -1;
String rssi = "";

// Variables will change :
int ledState = LOW;             // ledState used to set the LED

#define STATE_INITIALIZING  0
#define STATE_DISCONNECTING 1
#define STATE_DISCONNECTED  2
#define STATE_CONNECTING    3
#define STATE_CONNECTED     4
#define STATE_RESET         5

int state = STATE_INITIALIZING;
int previousState = -1;


unsigned long previousMillis = 0;   

// if we haven't heard from central in 15 seconds go to disconnected state
#define WATCHDOG_TIMEOUT 60000
long timeOfLastPing = 0;

#define SEND_DELAY 5000 // ping central every 5 seconds
long timeOfLastSend = 0;

#define RSSI_CHECK_DELAY 20000  // connected from central for 10sec? state=Connected
long timeOfLastRSSICheck = 0;


// function to call to reset the Arduino
void(* resetArduino) (void) = 0;

void setup() {

    myBLUNO.init();
  //Serial.begin(115200);

  getBLEName();
  //Serial.print("myID = ");
  //Serial.println(myID);
  //delay(500);

#ifdef INITIALIZE_CONNECTION
   atBind(centralMAC);
#endif
  
  pinMode(ONBOARD_LED,  OUTPUT);
  pinMode(BLUE_LED, OUTPUT);
  pinMode(GREEN_LED,  OUTPUT);
  pinMode(YELLOW_LED, OUTPUT);
  pinMode(RED_LED,    OUTPUT);
  
  digitalWrite(ONBOARD_LED, LOW);
  digitalWrite(BLUE_LED, LOW);
  digitalWrite(GREEN_LED, LOW);
  digitalWrite(YELLOW_LED, LOW);
  digitalWrite(RED_LED, LOW);
  //analogWrite(BLUE_LED, 10);  // max = 255

  delay(100);

#ifdef START_WITH_BLINK
  blink(1, false, GREEN_LED);
  blink(1, false, YELLOW_LED);
  blink(1, false, RED_LED);
  blink(1, false, BLUE_LED);
  //delay (1000);
#endif

#ifdef USE_LEDS_FOR_DEBUG
// blink out the value of AT+NAME that was previously programmed
  blink(myID, true, GREEN_LED);
#endif

  state = STATE_DISCONNECTED;
}

void loop() {
  // put your main code here, to run repeatedly:

    unsigned long currentMillis = millis();

  if (watchDogDidTimeout() && !isCentralConnected()) {
     digitalWrite(ONBOARD_LED, LOW);
     //digitalWrite(BLUE_LED, LOW);
     //analogWrite(BLUE_LED, 0);
#if 0
     state = STATE_DISCONNECTING;
#else
     state = STATE_RESET;
#endif
  }

  if (timeToCheckRSSI()) {
#ifdef USE_LEDS_FOR_DEBUG
      blink (2, true, YELLOW_LED);
#endif

      if (state == STATE_CONNECTING || state == STATE_CONNECTED) {
          blink (2, false, BLUE_LED);
      }
    if (isCentralConnected()) {

      if (state == STATE_DISCONNECTED || state == STATE_DISCONNECTING) {
        state = STATE_CONNECTING;
      }
    } else {
#ifdef USE_LEDS_FOR_DEBUG
        digitalWrite(YELLOW_LED, LOW);
#endif
        state = STATE_DISCONNECTING;
      }
    }


  switch (state) {
    case STATE_RESET:
        // reset the arduino and start fresh
        resetArduino();
        // NOT REACHED
        
    break;
    
    case STATE_DISCONNECTING:
#ifdef USE_LEDS_FOR_DEBUG
        blink(2, false, RED_LED);
#endif
        //Serial.println("DISCONNECTING");
        //atBind(centralMAC);
        atRestart();

        state = STATE_DISCONNECTED;
#ifdef USE_LEDS_FOR_DEBUG
        digitalWrite(RED_LED, HIGH);
#endif

        previousState = state;

        break;
        
    case STATE_DISCONNECTED:
      if (timeToUpdateGlow()) {
#ifdef DISABLE_SLOW_GLOW
        glowValue = 0;
#else // DISABLE_SLOW_GLOW
        glowValue += glowIncrement;
        if (glowValue <= GLOW_MIN) {
          glowValue = GLOW_MIN;
          glowIncrement = GLOW_STEP_SIZE;  // +1 so start glowing brighter
        } else if (glowValue >= GLOW_MAX) {
          glowValue = GLOW_MAX;
          glowIncrement = -GLOW_STEP_SIZE;  // start dimming now
        }
#endif // DISABLE_SLOW_GLOW
        analogWrite(BLUE_LED, glowValue);
      }

#ifndef USE_LEDS_FOR_DEBUG
      digitalWrite(RED_LED,    LOW);
      digitalWrite(YELLOW_LED, LOW);
#endif
      previousState = state;
      
    break;
    
    case STATE_CONNECTING:

      digitalWrite(YELLOW_LED, HIGH);

      if (timeToSend()) {
         myBLUNO.write("ID", myID);      // reply with my ID
         Serial.println();   //print line feed character to complete msg
      }
      
       if (timeForBlueBlink()) {
          blink (1, false, BLUE_LED);
          delay (500);
      }

      
        if (state != previousState) {
          resetAllWatchdogs();
          previousState = state;
        }

     break;
    
    case STATE_CONNECTED:

#ifdef USE_LEDS_FOR_DEBUG
      digitalWrite(YELLOW_LED, HIGH);
#endif

      if (timeForBlueBlink()) {
#ifdef USE_LEDS_FOR_DEBUG
        blink (6, true, BLUE_LED);
#else
        blink (BLUE_LED);
#endif // USE_LEDS_FOR_DEBUG
      }

#ifndef USE_LEDS_FOR_DEBUG
      if (timeForGreenBlink()) {
        blink (GREEN_LED);
      }
      if (timeForYellowBlink()) {
        blink (YELLOW_LED);
      }
      if (timeForRedBlink()) {
        blink (RED_LED);
      }
#endif // USE_LEDS_FOR_DEBUG

      
        if (state != previousState) {
          resetAllWatchdogs();
          previousState = state;
        }

      break;
 
  }

  if (timeToCheckBLE()) {
   if (myBLUNO.available()) {    //receive valid command from phone
        if (myBLUNO.equals("HELLO")){
          receivedID = myBLUNO.read();   //read the string to local value
          if (receivedID != myID) {
            blink (2, true, RED_LED);
            // central is trying to contact someone else, reset ourself
            atRestart();
            delay(5000);
            state=STATE_DISCONNECTING;
          } else {
            myBLUNO.write("ID", myID);        // reply with my ID
            Serial.println();   //print line feed character to complete msg
            watchDogReset();
            state = STATE_CONNECTING;
            delay (1000);  // pause to let the message go out            
          }
        }
        else if (myBLUNO.equals("CONNECTED!")) {
          digitalWrite(ONBOARD_LED, HIGH);
          watchDogReset();

          state = STATE_CONNECTED;

        }
        else if (myBLUNO.equals("CALLING")) {
          personCalled = myBLUNO.read();   // who was being called?
          if (personCalled == myID) {
            myBLUNO.write("ID", myID);      // reply with my ID
            Serial.println();   //print line feed character to complete msg
            watchDogReset();
            state = STATE_CONNECTING;
            delay (1000);  // pause to let the message go out
          }
        } else {
          previousMillis = currentMillis;
        }
        
    }

  } // check BLE?

} // loop

void atBind(char* centralAddr) {
  Serial.println("AT+EXIT");
  delay(100); //can't be omitted, same below
  Serial.print("+++");// Enter AT mode of itself
  delay(700); // 700 milliseconds MUST be added or AT ode not work
  Serial.println("AT+CMODE=UNIQUE");  delay(100);
  Serial.print("AT+BIND=");
  Serial.println(centralAddr);         delay(100);
  Serial.println("AT+RESTART");       delay(100);
}

void atRestart() {
  Serial.println("AT+EXIT");
  delay(100); //can't be omitted, same below
  Serial.print("+++");// Enter AT mode of itself
  delay(700); // 700 milliseconds MUST be added or AT mode not work
  Serial.println("AT+RESTART");       delay(100);
  delay(100);
  //Serial.read();  // throw out garbage data
}

void getBLEName() {
  Serial.println("AT+EXIT");
  delay(100); //can't be omitted, same below
  Serial.readString();  // throw away existing stuff in serial
  Serial.print("+++");// Enter AT mode of itself
  delay(700); // 700 milliseconds MUST be added or AT mode not work
  
  Serial.readString();  // throw away existing stuff in serial
  
  Serial.println("AT+NAME=?");  delay(700);
  myName = Serial.readString();
  //Serial.println("AT+EXIT");  delay(100);
  Serial.println("AT+RESTART");

  int n = myName.indexOf('\n');
  if (n >0 ) {
    myName.setCharAt(n, '\0'); 
  }
  
  n = myName.indexOf('\r');
  if (n >0 ) {
    myName.setCharAt(n, '\0'); 
  }

  myID = myName.toInt();

}


// If cental is on and connected to me
// then RSSI signal strength value will be non-zero
// this will return true when the on-board "link" LED is on
bool isCentralConnected() {

  int rssiValue;
  
  Serial.println("AT+EXIT");
  delay(100); //can't be omitted, same below
  Serial.readString();  // throw away existing stuff in serial
  Serial.print("+++");// Enter AT mode of itself
  delay(700); // 700 milliseconds MUST be added or AT mode not work
  
  Serial.readString();  // throw away existing stuff in serial
  Serial.readString();  // throw away existing stuff in serial
  
  Serial.println("AT+RSSI=?");  delay(100);
  rssi = Serial.readString();
  Serial.println("AT+EXIT");  delay(100);
  //Serial.println("AT+RESTART");  delay(500);

   rssiValue = rssi.toInt();
   
   if (rssiValue == 0) {
    digitalWrite(GREEN_LED, LOW);
    return (false);  // "-000"
   } else {
    digitalWrite(GREEN_LED, HIGH);
    return (true);
   }

}

void blink (int pin) {
  digitalWrite(pin, HIGH);
  delay(100);
  digitalWrite(pin, LOW);
}

void blink (int numBlinks, bool fast) {
  blink (numBlinks, fast, ONBOARD_LED);
}

void blink (int numBlinks, bool fast, int pin) {
  int i;

  if (numBlinks < 0 || numBlinks > 20)
    return;
    
  for (i=0; i<numBlinks*2; i++) {
    if (i%2 == 0) {
      digitalWrite(pin, LOW);
    }
    else {
      digitalWrite(pin, HIGH);
    }

    if (fast)
      delay(100);
    else
      delay(300);
  }

  digitalWrite(pin, LOW);
}

bool watchDogReset () {

    timeOfLastPing = millis();    
}

bool watchDogDidTimeout () {

    long now = millis();
    if ((now - timeOfLastPing) > WATCHDOG_TIMEOUT) {
      timeOfLastPing = now;
      return true;
    } else {
      return false;
    }
    
}

// gently glow bright/dim when not connected
bool timeToUpdateGlow () {
    long now = millis();
    if ((now - timeOfLastPulseUpdate) > GLOW_UPDATE_DELAY) {
      timeOfLastPulseUpdate = now;
      return true;
    } else {
      return false;
    }
}

// after connected we wildly blink the blue LED
bool timeForBlueBlink () {
    long now = millis();
    if ((now - timeOfLastBlueBlink) > BLUE_BLINK_DELAY) {
      timeOfLastBlueBlink = now;
      return true;
    } else {
      return false;
    }
}

// after connected we wildly blink the green LED
bool timeForGreenBlink () {
    long now = millis();
    if ((now - timeOfLastGreenBlink) > GREEN_BLINK_DELAY) {
      timeOfLastGreenBlink = now;
      return true;
    } else {
      return false;
    }
}

// after connected we wildly blink the yellow LED
bool timeForYellowBlink () {
    long now = millis();
    if ((now - timeOfLastYellowBlink) > YELLOW_BLINK_DELAY) {
      timeOfLastYellowBlink = now;
      return true;
    } else {
      return false;
    }
}

// after connected we wildly blink the red LED
bool timeForRedBlink () {
    long now = millis();
    if ((now - timeOfLastRedBlink) > RED_BLINK_DELAY) {
      timeOfLastRedBlink = now;
      return true;
    } else {
      return false;
    }
}

bool timeToCheckRSSI () {
    long now = millis();
    if ((now - timeOfLastRSSICheck) > RSSI_CHECK_DELAY) {
      timeOfLastRSSICheck = now;
      return true;
    } else {
      return false;
    }
}

void resetRssiTime () {
    timeOfLastRSSICheck = millis();
}

bool timeToCheckBLE () {
    long now = millis();
    if ((now - timeOfLastBLECheck) > BLE_UPDATE_DELAY) {
      timeOfLastBLECheck = now;
      return true;
    } else {
      return false;
    }
}

void resetBLECheckTime () {
    timeOfLastBLECheck = millis();
}

bool timeToRestartBLE () {
    long now = millis();
    if ((now - timeOfLastBLERestart) > BLE_RESTART_DELAY) {
      timeOfLastBLERestart = now;
      return true;
    } else {
      return false;
    }
}

void resetRestartTime () {
    timeOfLastBLERestart = millis();  
}

void resetAllWatchdogs () {
  resetRssiTime ();
  resetBLECheckTime ();
  resetRestartTime ();
}

// when state == connecting, keep sending out our ID
bool timeToSend () {

    long now = millis();
    if ((now - timeOfLastSend) > SEND_DELAY) {
      timeOfLastSend = now;
      return true;
    } else {
      return false;
    }
}

