#include <Wire.h>

// A Battery reconditioner for the Arduino
// Written by Michael Suzuki

long LogInterval;
long SampInterval=120; 	//Number of seconds in the LogInterval

const float aRef=3.09;	//Aref
const float Vcc=4.9;	//Vcc
const float LoadRes=0.8;
const byte nsamp=10;

// Timing counters
unsigned long StartTime;		//Log Interval counter

// Analog pins
unsigned int PinV1=0;		//Battery voltage 1
unsigned int PinV2=1;		//Battery voltage 2

//Digital output
int Relay1=2;		//Discharge transistor 1
int Relay2=3;		//Discharge transistor 2
int RedLED=4;
int GreenLED=5;

unsigned int VoltValue = 0;  	// value coming from Battery voltage 
float Volt1Avg;					// Analog Value Read Avg 1
float Volt2Avg;					// Analog Value Read Avg 2

float StartBattV1;		// Start Battery voltage 1 in V	
float StartBattV2;		// Start Battery voltage 2 in V
float LoadBattV1;		// Load Batt 1
float LoadBattV2;		// Load Batt 2
float IntRes1;			// Internal Resistance Batt 1
float IntRes2;			// Internal Resistance Batt 2

boolean doDischarge;
byte j=0;

void error(char *str)
{
  Serial.print("error: ");
  Serial.println(str);
  while(1);
}

//Prototypes for utility functions
void printData();
float getBatteryVoltage(unsigned int batteryNum);
float getIntRes(float StartV, float LoadV);

void setup(void)
{
  analogReference(EXTERNAL); //3.3V connected to AREF thru 3.3K AREF=3VDC
  //discard first analogReads
  for (int i=0; i <= 5; i++){
	int dummy=analogRead(i);
  }
  pinMode(Relay1, OUTPUT);
  pinMode(Relay2, OUTPUT);
  pinMode(RedLED, OUTPUT); 
  pinMode(GreenLED, OUTPUT); 
  
  Serial.begin(9600);
  Serial.println();
  Serial.println("StartV1, LoadV1, IntRes1,StartV2, LoadV2, IntRes2");

  LogInterval=SampInterval*1000.0;	 //  LogInterval needs to be in milliseconds
  StartTime=millis(); //Initialize StartTime
}

void loop(void)
{
	// Get starting voltages
	StartBattV1 = getBatteryVoltage(PinV1);
	StartBattV2 = getBatteryVoltage(PinV2);
	
	// Activate Relay1 to get Internal Resistance
	digitalWrite(Relay2, HIGH);
	delay(2000);  
	LoadBattV1 = getBatteryVoltage(PinV1);
	LoadBattV2 = getBatteryVoltage(PinV2);
	digitalWrite(Relay2, LOW);
	IntRes1=getIntRes(StartBattV1, LoadBattV1);
	IntRes2=getIntRes(StartBattV2, LoadBattV2);
	printData();

	//Check for one or two batteries above 0.4 Volts
	doDischarge=false;
	if (StartBattV1>0.4)
	{
		if ((StartBattV2>0.4) || (StartBattV2<0.2))
		{
			doDischarge=true;
		}
	}
	if (StartBattV1<0.2)
	{
		if (StartBattV2>0.4)
		{
			doDischarge=true;
		}
	}
	// run if at least one battery is > 0.4 Volt
	if (doDischarge)
	{
		if ((StartBattV1>0.99)||(StartBattV1>0.99))
		{
			do
			{
				//Stage1, activate both relays
				digitalWrite(RedLED, HIGH);
				digitalWrite(Relay1, HIGH);
				delay(200);
				digitalWrite(Relay2, HIGH);
				LoadBattV1 = getBatteryVoltage(PinV1);
				LoadBattV2 = getBatteryVoltage(PinV2);
			} while ((millis()-StartTime)<long(LogInterval));
			digitalWrite(Relay1, LOW);
			digitalWrite(RedLED, LOW);
			delay(200);
			digitalWrite(Relay2, LOW);
		}
		else
		{
				LogInterval=SampInterval*2000.0;
				do
				{
					digitalWrite(RedLED, HIGH);
					digitalWrite(Relay1, HIGH);
					delay(200);
					LoadBattV1 = getBatteryVoltage(PinV1);
					LoadBattV2 = getBatteryVoltage(PinV2);
				} while ((millis()-StartTime)<long(LogInterval));
				digitalWrite(RedLED, LOW);
				digitalWrite(Relay1, LOW);
		}
		printData();
	}
	else
	{
		if ((StartBattV1>0.6)||(StartBattV1>0.6))
		{
			delay(12000);	// Give batteries time to settle
		}
		else
		{
			while(1)
			{
				digitalWrite(GreenLED, HIGH);
			}
		}
	}
	StartTime=millis(); //Initialize StartTime	
}

//////////////////////////////////////////////////////////////////////////////////
// Read analog input for specified battery and maps into a voltage
//////////////////////////////////////////////////////////////////////////////////
float getBatteryVoltage(unsigned int batteryNum)
{
	VoltValue=0.0;
	for (j=0;j<nsamp;j++)
		{    
				VoltValue += analogRead(PinV1); 
		}    
		return (VoltValue/nsamp)*aRef/1023;
}

//////////////////////////////////////////////////////////////////////////////////
// Calulate Internal Resistance
//////////////////////////////////////////////////////////////////////////////////
float getIntRes(float StartV, float LoadV)
{
	// Internal Resistance is in series with Battery Voltage.
	// Start Voltage is Battery Voltage
	// Load Voltage is voltage across Load Resistor. 
	// Internal Resistor and Load make Voltage divider.
	// I = E/R = LoadV/LoadRes = (StartV - LoadV)/Internal Resistance
	// Internal Res * LoadV = (StartV - LoadV) * LoadRes
	// Internal Res = (StartV*LoadRes)/LoadV - LoadRes
	return (StartV*LoadRes/LoadV)-LoadRes;
}

//////////////////////////////////////////////////////////////////////////////////
// Print Data
//////////////////////////////////////////////////////////////////////////////////
void printData()
{
	Serial.print("$, ");
	Serial.print(StartBattV1,2);
	Serial.print(", ");
	Serial.print(LoadBattV1,2);
	Serial.print(", ");    
	Serial.print(IntRes1,2);
	Serial.print(", ");    
	Serial.print(StartBattV2,2);
	Serial.print(", ");
	Serial.print(LoadBattV2,2);
	Serial.print(", ");    
	Serial.print(IntRes2,2);
	Serial.println();
	
	StartTime = millis();	//Get StartTime for next LogInterval
}